#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include <sys/types.h>

#include <idsa.h>

#include "sparse.h"

#define DEFAULT_PRI   13	/* 8(user)+5(notice) */

int parse_event(IDSA_EVENT * evt, char *buf)
{
  pid_t pid = 0;
  unsigned int pri = DEFAULT_PRI;
  int result = 0;

  int attime = 0;
  int athost = 0;
  int atservice = 0;
  int atmessage = 0;
  int atclose = 0;
  int atopen = 0;
  int atcolon = 0;

  int i = 0;

  if ((evt == NULL) || (buf == NULL)) {	/* unreasonable input */
#ifdef TRACE
    fprintf(stderr, "parse_event(): NULL data\n");
#endif
    return 1;
  }
#ifdef TRACE
  fprintf(stderr, "parse_event(): buffer=%s\n", buf);
#endif

  /* look for priority */
  if (buf[i] == '<') {		/* try to get hold of priority */
    i++;
    pri = atoi(buf + i);

    while (isdigit(buf[i]) && i < 5) {
      i++;
    }

    if (buf[i] == '>') {
      i++;
    } else {
      i = 0;
    }
  }
#ifdef TRACE
  fprintf(stderr, "parse_event(): priority=%d, start of text=%d\n", pri, atstart);
#endif

  /* look for date */
  /*                                           0123456789012345  */
  /* date prefix has the following structure: "XXX XX XX:XX:XX " */
  attime = i;
  if (strlen(buf + i) > 16) {
    if ((buf[i + 3] == ' ') && (buf[i + 6] == ' ') && (buf[i + 9] == ':') && (buf[i + 12] == ':')) {
      i += 16;
    }
  }
#ifdef TRACE
  fprintf(stderr, "parse_event(): skipped date, offset=%d\n", i);
#endif

  /* skip hostname */
  athost = i;
  while ((buf[i] != '\0') && (!isspace(buf[i]))) {
    i++;
  }
  if (buf[i] != '\0') {
    i++;
  }
#ifdef TRACE
  fprintf(stderr, "parse_event(): marker at %d, offset=%d=", mark, i);
#endif

  atservice = i;
  /* try and find the : after service[pid]: */
  while ((buf[i] != '\0') && (buf[i] != ':') && (!isspace(buf[i]))) {
    switch (buf[i]) {
    case '[':
      atopen = i;
      pid = atoi(buf + i + 1);
      break;
    case ']':
      atclose = i;
      break;
    }
    i++;
  }
#ifdef TRACE
  fprintf(stderr, "parse_event(): looked for colon, offset=%d", i);
#endif

  if (buf[i] == ':') {		/* sensible modern syslog:  <pri>...service[pid]: */
    atcolon = i;
    i++;
    switch (buf[i]) {
    case ' ':
    case '\t':			/* squid tries to be a bit too clever */
      atmessage = i + 1;	/* skip : and trailing whitespace */
      break;
    default:
      atmessage = i;		/* just skip colon, also traps \0 */
      break;
    }
  }

  /* funky green logic */

  if ((atservice > 0) && (atopen > atservice) && (atclose > atopen) && (atclose + 1 == atcolon)) {
    /* have "* *[*]:" prefix looks good - don't be fussy */
    buf[atopen] = '\0';
  } else if ((attime > 0) && (athost > attime) && (atservice > athost) && (atcolon > atservice)) {
    /* have "<*>??? ?? ??:??:?? * *:?" prefix - looks good too */
    buf[atcolon] = '\0';
  } else if ((attime > 0) && (athost > attime)) {
    /* have "<*>??? ?? ??:??:??" prefix - mumble, at least skip date, no service */
    atservice = 0;
    atmessage = athost;
  } else if ((attime > 0)) {
    /* have "<*>" prefix grumble, everything else is message */
    atservice = 0;
    atmessage = attime;
  } else {
    /* very bad, can't recognize anything at all */
    atservice = 0;
    atmessage = 0;
  }

#ifdef TRACE
  fprintf(stderr, "parse_event(): message=%s, full=%s\n", buf + atmessage, buf);
#endif

  result += idsa_event_syslog(evt, pri, buf + atmessage);
  if (atservice) {
    result += idsa_service(evt, buf + atservice);
    if (pid) {
      result += idsa_pid(evt, pid);
    }
  }
#ifdef TRACE
  idsa_event_dump(evt, stderr);
#endif

  return result;
}
