/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
Header file for classes that support GUI data structures. They are 
created to minimize interaction berween GUI and VTK. 
*/

#ifndef IANIMATOR_H
#define IANIMATOR_H

class iAnimatorScript;
class iAnimatorEventObserver;


#include "istring.h"
#include "ivtksource.h"


struct iAnimatorState 
{
	int mode;
	int nframes;
	int nBlendedFrames;
	int nTransitionFrames;
	float dphi;
	float dtheta;
	float dscale;
	float droll;
	double slideSpeed;
	float flybySpeed;
	float xsecPos;
	bool ifBoundingBox;
	bool ifColorBars;
	bool ifTimeLabel;
	int currec;
	int cameraProjection;
	double cameraPosition[3];
	double cameraFocalPoint[3];
	double cameraViewUp[3];
	double cameraParallelScale;
	double cameraClippingRange[2];
	iString titlePageFile;
	int titlePageNumBlendedFrames;
	int titlePageNumFrames;
	iString logoFile;
	float logoOpacity;
	int logoPosition;
};


class vtkPolyData;
class iCameraPath;
class vtkImageData;


#include "iobject.h"


class iAnimator : public iObject
{
	
public:
	
//virtual const char *GetClassName() {return "";}
//static int IsTypeOf(const char *type) { return 0; }
//virtual int IsA(const char *type) { return 0; }

	static iAnimator *New(iVTK *m);
	virtual ~iAnimator();

	virtual int animate();
	virtual int step(bool dumpImage = true);
	virtual void reset();
	virtual void resetCurrentFile();

	virtual iAnimatorScript* getScript(){ return animatorScript; }

	inline int getReturnCode(){ return returnCode; }

	inline int getMode(){ return state.mode; }
	inline int getNframes(){ return state.nframes; }
	inline float getDPhi(){ return state.dphi; }
	inline float getDTheta(){ return state.dtheta; }
	inline float getDScale(){ return state.dscale; }
	inline float getDRoll(){ return state.droll; }
	inline double getSlideSpeed(){ return fabs(state.slideSpeed); }
	inline float getFlybySpeed(){ return state.flybySpeed; }
	inline int getNumBlendedFrames(){ return state.nBlendedFrames; }
	inline int getNumTransitionFrames(){ return state.nTransitionFrames; }

	inline vtkImageData *getTitlePageImage(){ return titlePageImage; }
	inline iString getTitlePageFile(){ return state.titlePageFile; }
	inline int getNumTitlePageFrames(){ return state.titlePageNumFrames; }
	inline int getNumTitlePageBlendedFrames(){ return state.titlePageNumBlendedFrames; }
	inline vtkImageData *getLogoImage(){ return logoImage; }
	inline iString getLogoFile(){ return state.logoFile; }
	inline int getLogoPosition(){ return state.logoPosition; }
	inline float getLogoOpacity(){ return state.logoOpacity; }

	virtual void getInfo(bool &nr, int &cr, int &cf){ nr = newrec; cr = currec; cf = curframe; }
 
	void setMode(int ma);
	inline void setNframes(int na){ state.nframes = na; }
	inline void setDPhi(float va){ state.dphi = va; }
	inline void setDTheta(float va){ state.dtheta = va; }
	inline void setDScale(float va){ state.dscale = va; }
	inline void setDRoll(float va){ state.droll = va; }
	inline void setSlideSpeed(double va){ state.slideSpeed = va; }
	inline void setFlybySpeed(float va){ state.flybySpeed = va; }
	inline void setNumBlendedFrames(int na){ state.nBlendedFrames = na; }
	inline void setNumTransitionFrames(int na){ state.nTransitionFrames = na; }

	bool setTitlePageFile(iString s);
	inline void setNumTitlePageFrames(int n){ state.titlePageNumFrames = n; }
	inline void setNumTitlePageBlendedFrames(int n){ state.titlePageNumBlendedFrames = n; }
	bool setLogoFile(iString s);
	inline void setLogoPosition(int n){ state.logoPosition = n; }
	inline void setLogoOpacity(float v){ state.logoOpacity = v; }

	inline bool getUseScript(){ return useScript; }
	virtual void setUseScript(bool s){ useScript = s; }

	inline bool getRestoreCamera(){ return restoreCamera; }
	virtual void setRestoreCamera(bool s){ restoreCamera = s; }

	inline bool getInheritSettings(){ return inheritSettings; }
	virtual void setInheritSettings(bool s){ inheritSettings = s; }

	inline iString getScriptFileName(){ return scriptFileName; }
	virtual void setScriptFileName(iString s){ scriptFileName = s; }

	virtual void resetState();
	virtual void saveState();
	virtual void restoreState();
	virtual void copyState(iAnimator *);

	virtual void packState(iString &s);
	virtual void unpackState(iString s);

	virtual void setDebugMode(bool s);
	//
	//  Camera path funstions
	//
	virtual int getNumberOfPathHandles();
	virtual int getNumberOfPathSteps();
	virtual bool isPathLoop();
	virtual bool isFocalPointPathUsed();
	virtual bool isFocalPointPathToPoint();

	virtual void setNumberOfPathHandles(int n);
	virtual void setNumberOfPathSteps(int n);
	virtual void setPathLoop(bool s);
	virtual void useFocalPointPath(bool s);
	virtual void setFocalPointPathToPoint(bool s);

	virtual void setCameraPathColor(float r, float g, float b);

	virtual bool isCameraPathValid();

	virtual void playCameraPathDemo();

protected:

	iAnimator(iVTK *m);
	virtual void positionCameraOnThePath(int step);

private:

	bool started, debugMode, useScript, restoreCamera, inheritSettings;
	unsigned int seed;
	int returnCode;

	iString scriptFileName;

	iAnimatorState state;
	iAnimatorState state2;

	float randstep;
	bool newrec;
	int prevrec, currec, totframe;
	int curframe;
	//
	//  Internal variables
	//
	vtkFloat pos[3];
	float r, dphl0, dthl0, ramp;
	vtkFloat xc1[3], xc2[3], x[3];
	float v[3], t, dt0, dt;

	int pathStep;
	bool useFocalPath;
	vtkPolyData *pathData, *pathData2;
	iCameraPath *cameraPath, *focalPath;

	vtkImageData **blenderBase;
	int numBlenderBase;

	vtkImageData *titlePageImage, *logoImage;
	bool doingTitlePage;

	iAnimatorScript *animatorScript;
	iAnimatorEventObserver *animatorEventObserver;
	
};

#endif // IANIMATOR_H

