/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "iparticlesdataconverter.h"

#include "ilimits.h"
#include "ivisualobject.h"
#include "iparticles.h"
#include "ipiecewisefunction.h"
#include "imath.h"
#include "ipolygonaldata.h"

#include <vtkMath.h>
#include <vtkPoints.h>
#include <vtkObjectFactory.h>
#include <vtkFloatArray.h>
#include <vtkAppendPolyData.h>
#include <vtkSphereSource.h>
#include <vtkPointSource.h>
#include <vtkCellArray.h>
#include <vtkPointData.h>


void reportNullPointer(int);


//--------------------------------------------------------------------------
iParticlesDataConverter* iParticlesDataConverter::New(iVisualObject *m)
{
	return new iParticlesDataConverter(m);
}


iParticlesDataConverter::iParticlesDataConverter(iVisualObject *m) : iVisualObjectSource(m)
{
	vtkIdType l, n;
	int i;
	vtkFloat x[3], r;
	
	mode = 0;
	
	size = 0.0003;
	
	int nsym = 1;
	vtkFloatArray *newNormals;
	
	symPoints[0] = 0;
	symVerts[0] = 0;
	symPolys[0] = 0;
	symNorms[0] = 0;
	
	//
	//  Triangle
	//
	nsym = IPDC_MODE_TRIANGLE;
	auxTriangle = vtkPolyData::New();
	if(auxTriangle == 0) reportNullPointer(8901);
	static float x2[4][3]={{0.0,0.0,1.0}, {0.0,0.866,-0.5}, {0.866*0.866,-0.5*0.866,-0.5},
	{-0.866*0.866,-0.5*0.866,-0.5}};
	static vtkIdType pts[4][3]={{0,2,1}, {0,1,3}, {0,3,2}, {1,2,3}};
	
	vtkPoints *points = vtkPoints::New();
	if(points == 0) reportNullPointer(8902);
	vtkCellArray *polys = vtkCellArray::New();
	if(polys == 0) reportNullPointer(8903);
	// Load the point, cell, and data attributes.
	for(i=0; i<4; i++) points->InsertPoint(i,x2[i]);
	for(i=0; i<4; i++) polys->InsertNextCell(3,pts[i]);
	auxTriangle->SetPolys(polys);
	polys->Delete();
	auxTriangle->SetPoints(points);
	points->Delete();
	
	newNormals = vtkFloatArray::New();
	if(newNormals == 0) reportNullPointer(8904);
	newNormals->SetNumberOfComponents(3);
    newNormals->SetNumberOfTuples(4);
	for(l=0; l<4; l++)
	{
		float x01[3], x02[3], xn[3];
		for(i=0; i<3; i++) x01[i] = x2[pts[l][1]][i] - x2[pts[l][0]][i];
		for(i=0; i<3; i++) x02[i] = x2[pts[l][2]][i] - x2[pts[l][0]][i];
		vtkMath::Cross(x01,x02,xn);
		vtkMath::Normalize(xn);
		newNormals->SetTuple(l,xn);
	}
    auxTriangle->GetPointData()->SetNormals(newNormals);
	newNormals->Delete();
	
	symPoints[nsym] = auxTriangle->GetPoints();
	symVerts[nsym] = auxTriangle->GetVerts();
	symPolys[nsym] = auxTriangle->GetPolys();
	symNorms[nsym] = (vtkFloatArray *)auxTriangle->GetPointData()->GetNormals();
	//
	//  Sphere
	//
	nsym = IPDC_MODE_SPHERE;
	auxSphere = vtkSphereSource::New();
	if(auxSphere == 0) reportNullPointer(8905);
	auxSphere->SetThetaResolution(8);
	auxSphere->SetPhiResolution(16);
	auxSphere->SetCenter(0.0,0.0,0.0);
	auxSphere->SetRadius(1.0);
	auxSphere->Update();
	
	symPoints[nsym] = auxSphere->GetOutput()->GetPoints();
	symVerts[nsym] = auxSphere->GetOutput()->GetVerts();
	symPolys[nsym] = auxSphere->GetOutput()->GetPolys();
	symNorms[nsym] = (vtkFloatArray *)auxSphere->GetOutput()->GetPointData()->GetNormals();
	//
	//  Cluster
	//
	nsym = IPDC_MODE_CLUSTER;
	auxClustr = vtkPointSource::New();
	if(auxClustr == 0) reportNullPointer(8906);
	auxClustr->SetNumberOfPoints(100);
	auxClustr->SetCenter(0.0,0.0,0.0);
	auxClustr->SetRadius(1.0);
	auxClustr->Update();
	//
	//  Rearrange points radially to have a centrally concentrated cluster
	//
	vtkPoints *p = auxClustr->GetOutput()->GetPoints();
	n = p->GetNumberOfPoints();
	for(l=0; l<n; l++)
	{
		p->GetPoint(l,x);
		r = x[0]*x[0]+x[1]*x[1]+x[2]*x[2];
		r = pow(2*r/(1.0+r),2.0);
		x[0] = x[0]*r;
		x[1] = x[1]*r;
		x[2] = x[2]*r;
		p->SetPoint(l,x);
	}
	
	newNormals = vtkFloatArray::New();
	newNormals->SetNumberOfComponents(3);
    newNormals->SetNumberOfTuples(n);
	for(l=0; l<n; l++) newNormals->SetTuple3(l,0.0,0.0,1.0);
    auxClustr->GetOutput()->GetPointData()->SetNormals(newNormals);
	newNormals->Delete();
	
	symPoints[nsym] = auxClustr->GetOutput()->GetPoints();
	symVerts[nsym] = auxClustr->GetOutput()->GetVerts();
	symPolys[nsym] = auxClustr->GetOutput()->GetPolys();
	symNorms[nsym] = (vtkFloatArray *)auxClustr->GetOutput()->GetPointData()->GetNormals();
	//
	//  Galaxy
	//
	nsym = IPDC_MODE_GALAXY;
	auxGalaxy = vtkPointSource::New();
	if(auxGalaxy == 0) reportNullPointer(8908);
	auxGalaxy->SetNumberOfPoints(100);
	auxGalaxy->SetCenter(0.0,0.0,0.0);
	auxGalaxy->SetRadius(1.0);
	auxGalaxy->Update();
	//
	//  Rearrange points to have a spiral galaxy in z=0 plane.
	//
	p = auxGalaxy->GetOutput()->GetPoints();
	n = p->GetNumberOfPoints();
	for(l=0; l<n; l++)
	{
		p->GetPoint(l,x);
		r = x[0]*x[0]+x[1]*x[1]+x[2]*x[2];
		//		r1 = x[0]*x[0]+x[1]*x[1];
		//		phi = atan2(x[0],x[1]);
		r = (0.2*(1-r)+2*r)/(1.0+r);
		x[0] = x[0]*r;
		x[1] = x[1]*r;
		x[2] = 0.0;
		p->SetPoint(l,x);
	}
	
	newNormals = vtkFloatArray::New();
	newNormals->SetNumberOfComponents(3);
    newNormals->SetNumberOfTuples(n);
	for(l=0; l<n; l++) newNormals->SetTuple3(l,0.0,0.0,1.0);
    auxGalaxy->GetOutput()->GetPointData()->SetNormals(newNormals);
	newNormals->Delete();
	
	symPoints[nsym] = auxGalaxy->GetOutput()->GetPoints();
	symVerts[nsym] = auxGalaxy->GetOutput()->GetVerts();
	symPolys[nsym] = auxGalaxy->GetOutput()->GetPolys();
	symNorms[nsym] = (vtkFloatArray *)auxGalaxy->GetOutput()->GetPointData()->GetNormals();
	//
	//  The rest
	//
	for(i=++nsym; i<=MAXMODE; i++)
	{
		symPoints[i] = 0;
		symVerts[i] = 0;
		symPolys[i] = 0;
		symNorms[i] = 0;
	}
	
}


iParticlesDataConverter::~iParticlesDataConverter()
{
	
	auxTriangle->Delete();
	auxSphere->Delete();
	auxClustr->Delete();
	auxGalaxy->Delete();
	
}


void iParticlesDataConverter::setMode(int m)
{
	if(m>=0 && m<=MAXMODE) mode = m;
	this->Modified();
}


void iParticlesDataConverter::setSize(float s)
{
	if(s>=0.01 && s<=100.0) size = 0.001*s*s;
	this->Modified();
}


void iParticlesDataConverter::ExecuteData(vtkDataObject *)
{
	iPolygonalData *input = (iPolygonalData *)this->GetInput();
	iPolygonalData *output = (iPolygonalData *)this->GetOutput();
	
	if(input == 0) return;

	output->ShallowCopy(input);

	if(mode == IPDC_MODE_POINTS) return;
	
	vtkPoints *ipoi = input->GetPoints();
	if(ipoi == 0) return;
	int sop;
	switch(ipoi->GetDataType())
	{
	case VTK_FLOAT:
		{
			sop = sizeof(float);
			break;
		}
	case VTK_DOUBLE:
		{
			sop = sizeof(double);
			break;
		}
	default: return;
	}

	vtkDataArray *isca = input->GetPointData()->GetScalars();
	float *iscaArr = NULL;
	if(isca != NULL) iscaArr = (float *)isca->GetVoidPointer(0);
	
	vtkPoints *opoi = vtkPoints::New(ipoi->GetDataType());
	if(opoi == 0) reportNullPointer(8910);
	vtkCellArray *ocell = vtkCellArray::New();
	if(ocell == 0) reportNullPointer(8911);
	
	vtkCellArray *scell = 0;
	int ncell;
	switch (mode) 
	{
	case IPDC_MODE_TRIANGLE:
	case IPDC_MODE_SPHERE:	{ scell = symPolys[mode]; break; }
	case IPDC_MODE_CLUSTER:
	case IPDC_MODE_GALAXY:	{ scell = symVerts[mode]; break; }
	}
	
	vtkIdType ntot = ipoi->GetNumberOfPoints();
	
	int catt;
	float cattLo = 0.0, cattHi = 0.0;
	bool cattLog = false;
	catt = ((iParticles *)myObject)->getAttToSize();
	if(catt > 0)
	{
		cattLo = myObject->getLimits()->getAttLow(catt);
		cattHi = myObject->getLimits()->getAttHigh(catt);
		cattLog = ((iParticles *)myObject)->getLogScaleToSize();
		if(cattLog)
		{
			cattLo = log10(1.0e-30+fabs(cattLo));
			cattHi = log10(1.0e-30+fabs(cattHi));
		}
	}
	
	vtkIdType l; int i, j, np, nc;
	float *s0 = 0, latt;
	double x0[3], x1[3], x2[3], s;
	vtkIdType ncel1, *pcel1, pcel2[999];
	
	np = symPoints[mode]->GetNumberOfPoints();
	nc = scell->GetNumberOfCells(); 
	ncell = scell->GetMaxCellSize();
	//
	//  Check if lines specified too
	//
	vtkIdType npoff = 0;
	if(input->GetLines()!=0 && input->GetLines()->GetNumberOfCells()>0)
	{
		output->SetLines(input->GetLines());
		npoff = ntot;
	}
	else
	{
		//
		//  remove verts
		//
		output->SetVerts(NULL);
	}

	vtkIdType ntot2 = ntot*np + npoff;
	// Allocates and sets MaxId
	opoi->SetNumberOfPoints(ntot2);

	int natt;
	if(isca == 0) natt = 0; else natt = isca->GetNumberOfComponents();
	vtkFloatArray *osca = 0;
	float *oscaArr = 0;
	if(natt > 0) 
	{
		osca = vtkFloatArray::New();
		if(osca == 0) reportNullPointer(8912);
		osca->SetNumberOfComponents(natt);
		// Allocates and sets MaxId
		osca->SetNumberOfTuples(ntot2);
		oscaArr = (float *)osca->GetVoidPointer(0);
	}
	
	vtkFloatArray *onor = 0;
	float *onorArr = 0;
	if(symNorms[mode] != 0) 
	{
		onor = vtkFloatArray::New();
		if(onor == 0) reportNullPointer(8913);
		onor->SetNumberOfComponents(3);
		// Allocates and sets MaxId
		onor->SetNumberOfTuples(ntot2);
		onorArr = (float *)onor->GetVoidPointer(0);
	} 
	//
	//  If not enough memory - revert to points
	//
	if(ocell->Allocate(ocell->EstimateSize(ntot*nc,ncell)) == 0) 
	{
		opoi->Delete();
		if(natt > 0) osca->Delete();
		if(onor != 0) onor->Delete();
		ocell->Delete();
		output->Initialize();
		output->ShallowCopy(input);
		this->Modified();
		return;
	}
	//
	//  Direct access to glyph data
	//
	float *spArr = (float *)symPoints[mode]->GetVoidPointer(0);
	float *snArr = (float *)symNorms[mode]->GetVoidPointer(0);

	//
	//  Set fixed seed to make sure particles do not jump in an animation
	//
	vtkMath::RandomSeed(123456789);
	//
	//  If lines specified, add original points first
	//
	if(npoff > 0) 
	{
		memcpy(opoi->GetVoidPointer(0),ipoi->GetVoidPointer(0),ntot*3*sop);
		if(natt > 0) memcpy(oscaArr,iscaArr,ntot*natt*sizeof(float));
		if(onor) 
		{
			for(l=0; l<ntot; l++)
			{
				for(j=0; j<3; j++) onorArr[3*l+j] = snArr[j];  // this is faster than memcpy!
			}
		}
	}
	//
	//  Prepare for direct scaling
	//
	iParticles *myParticles = (iParticles *)myObject;
	iPiecewiseFunction *pFun = myParticles->getSizeFunction();

	bool doDirect = myParticles->getAttSizeDirect();
	double scale = 2.0/myParticles->getAttSizeBasicScale()*myParticles->getAttSizeExtraFactor();
	//
	//  Main loop
	//
	ocell->InitTraversal();
	double mat0[3][3], mat1[3][3], mat2[3][3], mata[3][3], mat[3][3];
	float phi1, phi2, phi3;
	vtkIdType ll = 0;
	ll = npoff;
	for(l=0; l<ntot; l++)
	{
		
		if(l%1000 == 0)
		{
			this->UpdateProgress((float)l/ntot);
			if(this->GetAbortExecute()) break;
		}

		ipoi->GetPoint(l,x0);
		s = size;
		if(natt > 0)
		{
			s0 = iscaArr + natt*l;
			if(catt > 0)
			{
				if(doDirect)
				{
					//
					//  use the attribute as the precise value of the size scaled by scale
					//
					s = scale*s0[catt];
				}
				else
				{
					latt = s0[catt];
					if(cattLog) latt = log10(1.0e-30+fabs(latt));
					latt = (latt-cattLo)/(cattHi-cattLo);
					s *= (10*pFun->getValue(latt));
				}
			}
		}
		
		if(mode==IPDC_MODE_GALAXY || mode==IPDC_MODE_TRIANGLE)
		{
			for(i=0; i<3; i++) for(j=0; j<3; j++) mat0[i][j] = mat1[i][j] = mat2[i][j] = 0.0;
			phi1 = vtkMath::Pi()*vtkMath::Random();
			phi2 = vtkMath::Pi()*vtkMath::Random();
			phi3 = vtkMath::Pi()*vtkMath::Random();
			mat0[0][0] = 1.0;
			mat0[1][1] = mat0[2][2] = cos(phi1);
			mat0[1][2] = sin(phi1);
			mat0[2][1] = -mat0[1][2];
			mat1[1][1] = 1.0;
			mat1[0][0] = mat1[2][2] = cos(phi2);
			mat1[0][2] = sin(phi2);
			mat1[2][0] = -mat1[0][2];
			mat2[2][2] = 1.0;
			mat2[0][0] = mat2[1][1] = cos(phi3);
			mat2[0][1] = sin(phi3);
			mat2[1][0] = -mat2[0][1];
			vtkMath::Multiply3x3(mat0,mat1,mata);
			vtkMath::Multiply3x3(mata,mat2,mat);
		}
		
		for(i=0; i<np; i++)
		{
			for(j=0; j<3; j++) x1[j] = spArr[3*i+j];
			if(mode==IPDC_MODE_GALAXY || mode==IPDC_MODE_TRIANGLE)
			{
				vtkMath::Multiply3x3(mat,x1,x2);
				for(j=0; j<3; j++) x1[j] = x2[j]; // this is the fastest assignment
			}
			for(j=0; j<3; j++) x1[j] = s*x1[j] + x0[j];
			opoi->SetPoint(ll,x1);
			for(j=0; j<natt; j++) oscaArr[natt*ll+j] = s0[j];
			if(onor) 
			{
				for(j=0; j<3; j++) onorArr[3*ll+j] = snArr[3*i+j];
			}
			ll++;
		}
		
		scell->InitTraversal();
		for(i=0; i<nc; i++)
		{
			scell->GetNextCell(ncel1,pcel1);
			for(j=0; j<ncel1; j++) pcel2[j] = pcel1[j] + npoff + np*l;
			ocell->InsertNextCell(ncel1,pcel2);
		}
		
	}
	
	output->SetPoints(opoi);
	opoi->Delete();
	
	switch (mode) 
	{
	case IPDC_MODE_TRIANGLE:
	case IPDC_MODE_SPHERE: { output->SetPolys(ocell); break; }
	case IPDC_MODE_CLUSTER:
	case IPDC_MODE_GALAXY: { output->SetVerts(ocell); break; }
	}
	ocell->Delete();
	
    if(natt > 0)
	{
		output->GetPointData()->SetScalars(osca);
		osca->Delete();
	}
	
	if(onor)
	{
		output->GetPointData()->SetNormals(onor);
		onor->Delete();
	}
	
	this->Modified();
	
}


float iParticlesDataConverter::getMemorySize()
{
	if(mode == IPDC_MODE_POINTS) return 0.0;
	return this->GetOutput()->GetActualMemorySize();
}

