/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "iorthoslicer.h"

#include "ivisualobject.h"
#include "ilimits.h"
#include "iuniformmeshdata.h"

#include <vtkPointData.h>
#include <vtkFloatArray.h>


void reportNullPointer(int);


iOrthoSlicer::iOrthoSlicer(iVisualObject *m) : iVisualObjectSource(m)
{

	Origin[0] = Origin[1] = Origin[2] = 0;
	Axis = 1; 
	resample = 1;

}


iOrthoSlicer::~iOrthoSlicer() 
{
}


void iOrthoSlicer::ExecuteData(vtkDataObject *)
{
    vtkPointData *pd, *outpd;
    vtkDataArray *inScalars;
    vtkFloatArray *newScalars;
    iUniformMeshData *input = (iUniformMeshData *)this->GetInput();
    iUniformMeshData *output = (iUniformMeshData *)this->GetOutput();
    int dim[3], dim0[3], numPts;
    vtkFloat ptorig[3], ptspace[3];
    vtkFloat outptorig[3], outptspace[3];
    int Uidx=(Axis+1)%3,  Vidx=(Axis+2)%3;
    
	if(input == 0) return;
	
	output->ShallowCopy(input);

	pd = input->GetPointData();
    outpd = output->GetPointData();
    if(pd->GetScalars() == 0) return;
    
    inScalars = pd->GetScalars();
    input->GetOrigin(ptorig);
    input->GetSpacing(ptspace);
    input->GetDimensions(dim0);

	int step = resample;
	step = ( dim0[Uidx]/8 > step ) ? step : dim0[Uidx]/8;
	step = ( dim0[Vidx]/8 > step ) ? step : dim0[Uidx]/8;
	if(step < 1) step = 1;

	dim[Uidx] = dim0[Uidx]/step;
	dim[Vidx] = dim0[Vidx]/step;
	dim[Axis] = 1;

	vtkIdType ind;
	float *data = (float *)input->GetScalarPointer();

    numPts = dim[Uidx]*dim[Vidx];
    newScalars=vtkFloatArray::New();
	if(newScalars == 0) reportNullPointer(6602);
	newScalars->SetNumberOfComponents(1);
    newScalars->SetNumberOfTuples(numPts);
    
	float Ufact = (dim0[Uidx]-1.0)/(dim[Uidx]-1.0);
	float Vfact = (dim0[Vidx]-1.0)/(dim[Vidx]-1.0);

	outptspace[Uidx] = ptspace[Uidx]*Ufact;
	outptspace[Vidx] = ptspace[Vidx]*Vfact;
	outptspace[Axis] = ptspace[Axis];

	outptorig[Uidx] = ptorig[Uidx];
	outptorig[Vidx] = ptorig[Vidx];
	outptorig[Axis] = Origin[Axis];

	output->SetDimensions(dim);
	output->SetOrigin(outptorig);
	output->SetSpacing(outptspace);

    int ijk[3], ijk1[3], ijk2[3];
    float ht = (Origin[Axis]-ptorig[Axis])/ptspace[Axis];//Normalized height
    ijk[Axis] = (int)floor(ht);					 	    //Grid Point below
    float ss = (ht-ijk[Axis]); //interpolation factor
	if(ijk[Axis] < 0) ijk[Axis] = 0;
	if(ijk[Axis] >= dim0[Axis]) ijk[Axis] = dim0[Axis] - 1;

	float lowscale, hiscale;

	ijk1[Axis] = ijk[Axis];
	ijk2[Axis] = ijk[Axis] + 1;
	if(ijk2[Axis] == dim0[Axis]) {
		if(myObject->getLimits()->periodic[Axis])
		{
			ijk2[Axis] -= dim0[Axis]; 
		}
		else
		{
			ijk2[Axis]--;
		}
	}

	for(ijk[Uidx]=0; ijk[Uidx]<dim[Uidx]; ijk[Uidx]++)
	{

		this->UpdateProgress((float)ijk[Uidx]/dim[Uidx]);
		if(this->GetAbortExecute()) break;

		for(ijk[Vidx]=0; ijk[Vidx]<dim[Vidx]; ijk[Vidx]++)
		{
			ijk1[Uidx] = ijk[Uidx]*step + step/2;
			ijk1[Vidx] = ijk[Vidx]*step + step/2;
			ind = ijk1[0] + (vtkIdType)dim0[0]*(ijk1[1]+(vtkIdType)dim0[1]*ijk1[2]);
			lowscale = *(data+ind);
			ijk2[Uidx] = ijk1[Uidx];
			ijk2[Vidx] = ijk1[Vidx];
			ind = ijk2[0] + (vtkIdType)dim0[0]*(ijk2[1]+(vtkIdType)dim0[1]*ijk2[2]);
			hiscale = *(data+ind);
			newScalars->SetTuple1(ijk[Uidx]+ijk[Vidx]*dim[Uidx],(1-ss)*lowscale+ss*hiscale);
		}
    }
    
    outpd->SetScalars(newScalars);
    newScalars->Delete();
    
}

