/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "iqtdefs.h"
#include "iqt.h"
#include "iqtextension.h"
#include "iqtwindow.h"
#include "ivtk.h"
#include "ivtkwindow.h"
#include "ienvironment.h"
#include "iqt_view.h"
#include "iqt_surf.h"
#include "iqt_xsec.h"
#include "iqt_volv.h"
#include "iqt_part.h"
#include "iqt_vect.h"
#include "iqt_tens.h"
#include "iqt_wins.h"

#include "ianimator.h"
#include "imath.h"
#include "idatareader.h"
#include "iparticles.h"
#include "isurface.h"
#include "itensor.h"
#include "itrackball.h"
#include "ivector.h"
#include "ivolume.h"
#include "ixsection.h"
#include "ilightkit.h"
#include "imarker.h"
#include "idialogprogress.h"
#include "icolorbars.h"
#include "ierror.h"

#include <vtkLightKit.h>

#include <qaction.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qcolordialog.h>
#include <qcombobox.h>
#include <qfiledialog.h>
#include <qimage.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlcdnumber.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qslider.h>
#include <qspinbox.h>
#include <qtabwidget.h>
#include <qtoolbutton.h>
#include <qvalidator.h>


#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkPlane.h>
#include <vtkRenderer.h>
#include <vtkActor.h>
#include <vtkMapper.h>
#include <vtkDataSet.h>
#include <vtkMatrix4x4.h>
#include <vtkPointData.h>

#ifndef I_VTK_VERSION_40
#include <vtkSplineWidget.h>
#endif


#define CURRENT_MARKER    iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMember()


//
// *********************************************************************************
//
//  
//   General view panel widgets
//
//
// *********************************************************************************
//
void iQT_View::init()
{

	myQT = (iQT *)parentWidget()->parentWidget()->parentWidget()->parentWidget()->parentWidget()->parentWidget();
	if(strcmp(myQT->name(),"iQT") != 0)
	{
		iQTWindow::getCurrentWindow()->popupError("Incorrect parenting in Tab Pages",IFRITERROR_INCORRECT_PARENTING_IN_TABS);
		exit(1);
	}

	clipPlaneSliderRenderTracking = true;
	//
	//   add Windows panel
	//
	TabLayoutWins = new QVBoxLayout(TabWidget->page(3), 0, 6, ""); 
	TabWidgetWins = new iQT_Wins(TabWidget->page(3),"TabWidgetWins");
	TabLayoutWins->addWidget(TabWidgetWins);

	//
	//  Make sliders look better
	//
	LayoutViewFrameRate->setStretchFactor(LayoutViewFrameRateLeft,3);
	LayoutViewFrameRate->addStretch(1);
	
	LayoutViewAnimFlyby->setStretchFactor(LayoutViewAnimFlybyLeft,3);
	LayoutViewAnimFlyby->addStretch(1);
	
	LayoutViewAnimXsec->setStretchFactor(LayoutViewAnimXsecLeft,3);
	LayoutViewAnimXsec->addStretch(1);
	
	LayoutViewAnimRotations->setStretchFactor(LayoutViewAnimRotationsLeft,3);
	LayoutViewAnimRotations->addStretch(1);
	
	LayoutViewAnimScale->setStretchFactor(LayoutViewAnimScaleLeft,3);
	LayoutViewAnimScale->addStretch(1);

	LayoutMarkMaterial->setStretchFactor(View_Mark_MaterialGroupBox,3);
	LayoutMarkMaterial->addStretch(1);

	//
	//  Configure MarkerPositionLineEdits
	//
	View_Mark_MarkerPositionXLineEdit->setValidator(new QDoubleValidator(-1.0,1.0,10,View_Mark_MarkerPositionXLineEdit));
	View_Mark_MarkerPositionYLineEdit->setValidator(new QDoubleValidator(-1.0,1.0,10,View_Mark_MarkerPositionYLineEdit));
	View_Mark_MarkerPositionZLineEdit->setValidator(new QDoubleValidator(-1.0,1.0,10,View_Mark_MarkerPositionZLineEdit));

	//
	//  Disable marker widgets
	//
	enableView_MarkWidgets(false,0);
#ifdef I_VTK_VERSION_40
	TabWidgetScene->removePage(TabWidgetScene->page(3));
#endif

}


void iQT_View::doRender()
{
	myQT->doRender();
}


void iQT_View::updateView_Widgets()
{
	//
	//  Correct main panel settings
	//
	
	//  iVisualObject checkboxes enabled/disabled 
	this->updateSwitchesSurf();
	this->updateSwitchesXsec();
	this->updateSwitchesVolv();
	this->updateSwitchesPart();
	this->updateSwitchesVect();
	this->updateSwitchesTens();
    
	CALL_FUNCTION1(View_Flyby,setOn,iVTKWindow::getCurrentWindow()->isInteractorStyleFlight());
	
	CALL_FUNCTION1(View_DesiredFrameRateSlider,setValue,round(iVTKWindow::getCurrentWindow()->getUpdateRate()));
	View_DesiredFrameRateLCD->display(iVTKWindow::getCurrentWindow()->getUpdateRate());
	
	if(iVTKWindow::getCurrentWindow()->isParallelProjection())
	{
		CALL_FUNCTION1(View_Projection,setButton,0);
		View_ViewAngleSlider->setEnabled(false);
		View_ViewAngleLCD->setEnabled(false);
	}
	else
	{
		CALL_FUNCTION1(View_Projection,setButton,1);
		View_ViewAngleSlider->setEnabled(true);
		View_ViewAngleLCD->setEnabled(true);
	}
	
	CALL_FUNCTION1(View_ViewAngleSlider,setValue,round(iVTKWindow::getCurrentWindow()->getRenderer()->GetActiveCamera()->GetViewAngle()));
	CALL_FUNCTION1(View_ViewAngleLCD,display,round(iVTKWindow::getCurrentWindow()->getRenderer()->GetActiveCamera()->GetViewAngle()));

	CALL_FUNCTION1(View_BoundingBoxButton,setChecked,iVTKWindow::getCurrentWindow()->isBoundingBoxVisible());
	CALL_FUNCTION1(View_TimeLabelButton,setChecked,iVTKWindow::getCurrentWindow()->isLabelVisible());
	CALL_FUNCTION1(View_RulerButton,setChecked,iVTKWindow::getCurrentWindow()->isRulerVisible());
	CALL_FUNCTION1(View_MeasuringBoxButton,setChecked,iVTKWindow::getCurrentWindow()->isMeasuringBoxVisible());
	CALL_FUNCTION1(View_ColorBarsButton,setChecked,iVTKWindow::getCurrentWindow()->isColorBarsVisible());

	CALL_FUNCTION1(View_BackgroundColorLabel,setPaletteBackgroundColor,iVTKWindow::getCurrentWindow()->getBackgroundColor());
	//
	//  Correct light widgets settings
	//
    View_Light_KeyDirection->blockSignals(true);
	View_Light_KeyDirection->setAngles(iVTKWindow::getCurrentWindow()->getLightKit()->GetKeyLightAzimuth(),iVTKWindow::getCurrentWindow()->getLightKit()->GetKeyLightElevation());
    View_Light_KeyDirection->blockSignals(false);

    float lk = iVTKWindow::getCurrentWindow()->getLightKit()->GetKeyLightIntensity();
	CALL_FUNCTION1(View_Light_KeyBrightness,setValue,round(100.0*lk));
    CALL_FUNCTION1(View_Light_HeadBrightness,setValue,round(100.0*lk/iVTKWindow::getCurrentWindow()->getLightKit()->GetKeyToHeadRatio()));
    CALL_FUNCTION1(View_Light_FillBrightness,setValue,round(100.0*lk/iVTKWindow::getCurrentWindow()->getLightKit()->GetKeyToFillRatio()));
	//
	//  Set animation options
	//    
	//  1. Style panel
	//
    CALL_FUNCTION1(View_Anim_Style,setButton,iVTKWindow::getCurrentWindow()->getAnimator()->getMode());
    CALL_FUNCTION1(View_Anim_RestoreCameraCheckBox,setChecked,iVTKWindow::getCurrentWindow()->getAnimator()->getRestoreCamera());
    
    int vt1 = iVTKWindow::getCurrentWindow()->getAnimator()->getNframes();
    View_Anim_FramesLCD->display(vt1);
    int vt = 0;
    int vs = 1;
	int i;
    for(i=0; vt<vt1 && i<100; i++)
	{
		if(i < 40) vs = 1; else if(i < 70) vs = 2; else if(i < 100) vs = 10; else if(i<130) vs = 20; else vs = 100; 
		vt = vt + vs;
    }
    CALL_FUNCTION1(View_Anim_FramesSlider,setValue,i-1);
	//
	//  2. Settings panel
	//
    CALL_FUNCTION1(View_Anim_FlybySpeedSlider,setValue,round(10.0*(3.0+log10(iVTKWindow::getCurrentWindow()->getAnimator()->getFlybySpeed()))));

    CALL_FUNCTION1(View_Anim_XsectionMotionSlider,setValue,round(1000.0*iVTKWindow::getCurrentWindow()->getAnimator()->getSlideSpeed()));
    
    CALL_FUNCTION1(View_Anim_RotatePhiSlider,setValue,round(iVTKWindow::getCurrentWindow()->getAnimator()->getDPhi()*10.0));
    View_Anim_RotatePhiLCD->display(iVTKWindow::getCurrentWindow()->getAnimator()->getDPhi());
    
    CALL_FUNCTION1(View_Anim_RotateThetaSlider,setValue,round(iVTKWindow::getCurrentWindow()->getAnimator()->getDTheta()*10.0));
    View_Anim_RotateThetaLCD->display(iVTKWindow::getCurrentWindow()->getAnimator()->getDTheta());
    
    CALL_FUNCTION1(View_Anim_RotateRollSlider,setValue,round(iVTKWindow::getCurrentWindow()->getAnimator()->getDRoll()*10.0));
    View_Anim_RotateRollLCD->display(iVTKWindow::getCurrentWindow()->getAnimator()->getDRoll());
    
    CALL_FUNCTION1(View_Anim_ScaleSlider,setValue,round(log10(iVTKWindow::getCurrentWindow()->getAnimator()->getDScale())*100.0));
    View_Anim_ScaleLCD->display(iVTKWindow::getCurrentWindow()->getAnimator()->getDScale());
	//
	//  3. Script panel
	//
	CALL_FUNCTION1(View_Anim_Scripting,setChecked,iVTKWindow::getCurrentWindow()->getAnimator()->getUseScript());
    CALL_FUNCTION1(View_Anim_InheritSettingsCheckBox,setChecked,iVTKWindow::getCurrentWindow()->getAnimator()->getInheritSettings());
	//
	//  4. Advanced panel
	//
    CALL_FUNCTION1(View_Anim_BlendedEffectsSlider,setValue,iVTKWindow::getCurrentWindow()->getAnimator()->getNumBlendedFrames());
    View_Anim_BlendedEffectsLCD->display(iVTKWindow::getCurrentWindow()->getAnimator()->getNumBlendedFrames());
    CALL_FUNCTION1(View_Anim_TransitionEffectsSlider,setValue,iVTKWindow::getCurrentWindow()->getAnimator()->getNumTransitionFrames());
    View_Anim_TransitionEffectsLCD->display(iVTKWindow::getCurrentWindow()->getAnimator()->getNumTransitionFrames());
        
	CALL_FUNCTION1(View_Anim_CameraPathStepsSlider,setValue,round(10.0*log10((double)iVTKWindow::getCurrentWindow()->getAnimator()->getNumberOfPathSteps())));
	View_Anim_CameraPathStepsLCD->display(iVTKWindow::getCurrentWindow()->getAnimator()->getNumberOfPathSteps());

	CALL_FUNCTION1(View_Anim_CameraPathHandlesSpinBox,setValue,iVTKWindow::getCurrentWindow()->getAnimator()->getNumberOfPathHandles());

	CALL_FUNCTION1(View_Anim_CameraPathLoopCheckBox,setChecked,iVTKWindow::getCurrentWindow()->getAnimator()->isPathLoop());

	CALL_FUNCTION1(View_Anim_CameraPathFocalCheckBox,setChecked,iVTKWindow::getCurrentWindow()->getAnimator()->isFocalPointPathUsed());

	CALL_FUNCTION1(View_Anim_CameraPathFocalToPointCheckBox,setChecked,iVTKWindow::getCurrentWindow()->getAnimator()->isFocalPointPathToPoint());
	//
	//  5. Title/logo panel
	//
	QImage q; QPixmap p;

	CALL_FUNCTION1(View_Anim_TitlePageFileNameLineEdit,setText,iVTKWindow::getCurrentWindow()->getAnimator()->getTitlePageFile());
	CALL_FUNCTION1(View_Anim_TitlePageShowTimeSpinBox,setValue,round(iVTKWindow::getCurrentWindow()->getAnimator()->getNumTitlePageFrames()/30.0));
	CALL_FUNCTION1(View_Anim_TitlePageDissolveTimeSpinBox,setValue,round(iVTKWindow::getCurrentWindow()->getAnimator()->getNumTitlePageBlendedFrames()/30.0));
	if(!iVTKWindow::getCurrentWindow()->getAnimator()->getTitlePageFile().isEmpty())
	{
		iVTK::convertImage(iVTKWindow::getCurrentWindow()->getAnimator()->getTitlePageImage(),q);
		p.convertFromImage(q.smoothScale(64,999999,QImage::ScaleMin));
		View_Anim_TitlePageImagePixmapLabel->setPixmap(p);
	}
	else View_Anim_TitlePageImagePixmapLabel->setPixmap(0);

	CALL_FUNCTION1(View_Anim_LogoFileNameLineEdit,setText,iVTKWindow::getCurrentWindow()->getAnimator()->getLogoFile());
	CALL_FUNCTION1(View_Anim_LogoOpacityButtonGroup,setButton,round(4.0*(iVTKWindow::getCurrentWindow()->getAnimator()->getLogoOpacity()-0.25)));
	CALL_FUNCTION1(View_Anim_LogoPositionButtonGroup,setButton,iVTKWindow::getCurrentWindow()->getAnimator()->getLogoPosition());
	if(!iVTKWindow::getCurrentWindow()->getAnimator()->getLogoFile().isEmpty())
	{
		iVTK::convertImage(iVTKWindow::getCurrentWindow()->getAnimator()->getLogoImage(),q);
		p.convertFromImage(q.smoothScale(64,999999,QImage::ScaleMin));
		View_Anim_LogoImagePixmapLabel->setPixmap(p);
	}
	else View_Anim_LogoImagePixmapLabel->setPixmap(0);
	//
	//  One script for all windows
	//
	//	CALL_FUNCTION1(View_Anim_ScriptName,setText,iVTKWindow::getCurrentWindow()->getAnimator()->getScriptFileName());
	//
	//  Correct clip plane setting
	//
	CALL_FUNCTION1(View_Clip_ShowBox,setChecked,iVTKWindow::getCurrentWindow()->isClipPlaneVisible());
	if(View_Clip_ShowBox->isChecked()) updateView_Clip_Show(true);

	vtkFloat x[3];
	vtkFloat n[3], l;
	vtkPlane *pl = iVTKWindow::getCurrentWindow()->getClipPlane();
	pl->GetOrigin(x);
	pl->GetNormal(n);
	l = -(x[0]*n[0]+x[1]*n[1]+x[2]*n[2]);
	CALL_FUNCTION1(View_Clip_PositionSlider,setValue,round(100.0*l));
	CALL_FUNCTION1(View_Clip_PositionSliderCheckBox,setChecked,clipPlaneSliderRenderTracking);
	View_Clip_Direction->blockSignals(true);
	View_Clip_Direction->setDirection(n[0],n[1],n[2]);
	View_Clip_Direction->blockSignals(false);
	//
	//  Set Stereo options
	//
	CALL_FUNCTION1(View_Ster_Method,setButton,iVTKWindow::getCurrentWindow()->getWindow()->GetStereoType());
	if(!iVTKWindow::getCurrentWindow()->getWindow()->GetStereoCapableWindow())
	{
		CALL_FUNCTION1(View_Ster_Method,setButton,0);
		View_Ster_Method->setEnabled(false);
		View_Ster_EyeAngleSlider->setEnabled(false);
		View_Ster_EyeAngleLCD->setEnabled(false);
	}
	CALL_FUNCTION1(View_Ster_EyeAngleSlider,setValue,round(4.0*iVTKWindow::getCurrentWindow()->getRenderer()->GetActiveCamera()->GetEyeAngle()));
	View_Ster_EyeAngleLCD->display(iVTKWindow::getCurrentWindow()->getRenderer()->GetActiveCamera()->GetEyeAngle());
	//
	//  Markers options
	//
	this->updateView_MarkWidgets();
	//
	//  Set ColorBars panel
	//
	this->updateView_BarsWidgets();
	//
	//  Set Windows options
	//
	// ---  None to set  ----------------

	if(myQT->ext != NULL) myQT->ext->updateSwitches(-1);

	myQT->updateDataChannelComboBox(DISPLAYPAGE_VIEW);

}


void iQT_View::updateSwitchesSurf()
{

	bool dataPresent = iVTKWindow::getCurrentWindow()->getReader()->isThereMeshData(CURRENT_SURFACE->getDataChannel(),CURRENT_SURFACE->getDataStream());
	CALL_FUNCTION1(this->View_Surf,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->TabWidgetSurf->View_SurfLocal,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->actionShowSurf,setEnabled,dataPresent);

	bool isVisible = iVTKWindow::getCurrentWindow()->getSurfaceFamily()->isVisible();
	CALL_FUNCTION1(this->View_Surf,setChecked,isVisible);
	CALL_FUNCTION1(myQT->TabWidgetSurf->View_SurfLocal,setChecked,isVisible);
	CALL_FUNCTION1(myQT->actionShowSurf,setOn,isVisible);

}


void iQT_View::updateSwitchesXsec()
{

	bool dataPresent = iVTKWindow::getCurrentWindow()->getReader()->isThereMeshData(CURRENT_XSECTION->getDataChannel(),CURRENT_SURFACE->getDataStream());
	CALL_FUNCTION1(this->View_Xsec,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->TabWidgetXsec->View_XsecLocal,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->actionShowXsec,setEnabled,dataPresent);

	bool isVisible = iVTKWindow::getCurrentWindow()->getXsectionFamily()->isVisible();
	CALL_FUNCTION1(this->View_Xsec,setChecked,isVisible);
	CALL_FUNCTION1(myQT->TabWidgetXsec->View_XsecLocal,setChecked,isVisible);
	CALL_FUNCTION1(myQT->actionShowXsec,setOn,isVisible);

}


void iQT_View::updateSwitchesVolv()
{

	bool dataPresent = iVTKWindow::getCurrentWindow()->getReader()->isThereMeshData(CURRENT_VOLUME->getDataChannel(),CURRENT_SURFACE->getDataStream());
	CALL_FUNCTION1(this->View_Volv,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->TabWidgetVolv->View_VolvLocal,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->actionShowVolv,setEnabled,dataPresent);

	bool isVisible = iVTKWindow::getCurrentWindow()->getVolumeFamily()->isVisible();
	CALL_FUNCTION1(this->View_Volv,setChecked,isVisible);
	CALL_FUNCTION1(myQT->TabWidgetVolv->View_VolvLocal,setChecked,isVisible);
	CALL_FUNCTION1(myQT->actionShowVolv,setOn,isVisible);

}


void iQT_View::updateSwitchesPart()
{

	bool dataPresent = iVTKWindow::getCurrentWindow()->getReader()->isTherePartData(CURRENT_PARTICLES->getDataChannel(),CURRENT_SURFACE->getDataStream());
	CALL_FUNCTION1(this->View_Part,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->TabWidgetPart->View_PartLocal,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->actionShowPart,setEnabled,dataPresent);

	bool isVisible = iVTKWindow::getCurrentWindow()->getParticlesFamily()->isVisible();
	CALL_FUNCTION1(this->View_Part,setChecked,isVisible);
	CALL_FUNCTION1(myQT->TabWidgetPart->View_PartLocal,setChecked,isVisible);
	CALL_FUNCTION1(myQT->actionShowPart,setOn,isVisible);

}


void iQT_View::updateSwitchesVect()
{

	bool dataPresent = iVTKWindow::getCurrentWindow()->getReader()->isThereVectData(CURRENT_VECTOR->getDataChannel(),CURRENT_SURFACE->getDataStream());
	CALL_FUNCTION1(this->View_Vect,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->TabWidgetVect->View_VectLocal,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->actionShowVect,setEnabled,dataPresent);

	bool isVisible = iVTKWindow::getCurrentWindow()->getVectorFamily()->isVisible();
	CALL_FUNCTION1(this->View_Vect,setChecked,isVisible);
	CALL_FUNCTION1(myQT->TabWidgetVect->View_VectLocal,setChecked,isVisible);
	CALL_FUNCTION1(myQT->actionShowVect,setOn,isVisible);

}


void iQT_View::updateSwitchesTens()
{

	bool dataPresent = iVTKWindow::getCurrentWindow()->getReader()->isThereTensData(CURRENT_TENSOR->getDataChannel(),CURRENT_SURFACE->getDataStream());
	CALL_FUNCTION1(this->View_Tens,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->TabWidgetTens->View_TensLocal,setEnabled,dataPresent);
	CALL_FUNCTION1(myQT->actionShowTens,setEnabled,dataPresent);

	bool isVisible = iVTKWindow::getCurrentWindow()->getTensorFamily()->isVisible();
	CALL_FUNCTION1(this->View_Tens,setChecked,isVisible);
	CALL_FUNCTION1(myQT->TabWidgetTens->View_TensLocal,setChecked,isVisible);
	CALL_FUNCTION1(myQT->actionShowTens,setOn,isVisible);

}


void iQT_View::updateView_Surf(bool s)
{
	
	myQT->startTask();

	iVTKWindow::getCurrentWindow()->getSurfaceFamily()->show(s);

    CALL_FUNCTION1(View_Surf,setChecked,s);
    CALL_FUNCTION1(myQT->TabWidgetSurf->View_SurfLocal,setChecked,s);
	CALL_FUNCTION1(myQT->actionShowSurf,setOn,s);
    
    myQT->TabWidgetSurf->updateSurf_Widgets();
	if(myQT->Toolbar->hasMouse() && s) myQT->TabWidget->setCurrentPage(DISPLAYPAGE_SURF);

    myQT->render(true);
	myQT->endTask();
	
}


void iQT_View::updateView_Xsec(bool s)
{
	
	myQT->startTask();

	iVTKWindow::getCurrentWindow()->getXsectionFamily()->show(s);

    CALL_FUNCTION1(View_Xsec,setChecked,s);
    CALL_FUNCTION1(myQT->TabWidgetXsec->View_XsecLocal,setChecked,s);
	CALL_FUNCTION1(myQT->actionShowXsec,setOn,s);
    
    myQT->TabWidgetXsec->updateXsec_Widgets();
	if(myQT->Toolbar->hasMouse() && s) myQT->TabWidget->setCurrentPage(DISPLAYPAGE_XSEC);

    myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_Volv(bool s)
{
    
	myQT->startTask();

	iVTKWindow::getCurrentWindow()->getVolumeFamily()->show(s);

    CALL_FUNCTION1(View_Volv,setChecked,s);
    CALL_FUNCTION1(myQT->TabWidgetVolv->View_VolvLocal,setChecked,s);
	CALL_FUNCTION1(myQT->actionShowVolv,setOn,s);
    
    myQT->TabWidgetVolv->updateVolv_Widgets();
	if(myQT->Toolbar->hasMouse() && s) myQT->TabWidget->setCurrentPage(DISPLAYPAGE_VOLV);
	//
	// Reset the projection to orthogonal is VolumePro is used
	//
	if(CURRENT_VOLUME->getMode()==IVOLUME_MODE_VOLUMEPRO  && myQT->TabWidgetView->View_Volv->isChecked()) {
		CALL_FUNCTION1(myQT->TabWidgetView->View_Projection,setEnabled,false);
		if(myQT->TabWidgetView->View_Projection->id(myQT->TabWidgetView->View_Projection->selected()) == 1) {
			CALL_FUNCTION1(myQT->TabWidgetView->View_Projection,setButton,0);
			iVTKWindow::getCurrentWindow()->getRenderer()->GetActiveCamera()->ParallelProjectionOn(); 
			myQT->popupInfo("VolumePro board only suports the Orthogonal projection.\n Setting the projection to Orthogonal.\n");
		}
    } else CALL_FUNCTION1(myQT->TabWidgetView->View_Projection,setEnabled,true);
	
    myQT->render(true);
	myQT->endTask();
	
}


void iQT_View::updateView_Part(bool s)
{
	
	myQT->startTask();
 	
	iVTKWindow::getCurrentWindow()->getParticlesFamily()->show(s);

    CALL_FUNCTION1(View_Part,setChecked,s);
    CALL_FUNCTION1(myQT->TabWidgetPart->View_PartLocal,setChecked,s);
	CALL_FUNCTION1(myQT->actionShowPart,setOn,s);
    
    myQT->TabWidgetPart->updatePart_Widgets();
	if(myQT->Toolbar->hasMouse() && s) myQT->TabWidget->setCurrentPage(DISPLAYPAGE_PART);

    myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_Vect( bool s )
{
	
	myQT->startTask();
	
	iVTKWindow::getCurrentWindow()->getVectorFamily()->show(s);

    CALL_FUNCTION1(View_Vect,setChecked,s);
    CALL_FUNCTION1(myQT->TabWidgetVect->View_VectLocal,setChecked,s);
	CALL_FUNCTION1(myQT->actionShowVect,setOn,s);
    
    myQT->TabWidgetVect->updateVect_Widgets();
	if(myQT->Toolbar->hasMouse() && s) myQT->TabWidget->setCurrentPage(DISPLAYPAGE_VECT);

    myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_Tens( bool s )
{
	
	myQT->startTask();
	
	iVTKWindow::getCurrentWindow()->getTensorFamily()->show(s);

    CALL_FUNCTION1(View_Tens,setChecked,s);
    CALL_FUNCTION1(myQT->TabWidgetTens->View_TensLocal,setChecked,s);
	CALL_FUNCTION1(myQT->actionShowTens,setOn,s);
    
    myQT->TabWidgetTens->updateTens_Widgets();
	if(myQT->Toolbar->hasMouse() && s) myQT->TabWidget->setCurrentPage(DISPLAYPAGE_TENS);

    myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_DesiredFrameRate( int v )
{
    
    iVTKWindow::getCurrentWindow()->setUpdateRate(0.001+v);
	
}


void iQT_View::updateView_Projection( int v )
{
	
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->setParallelProjection(v==0); 
	View_ViewAngleSlider->setEnabled(v==1);
	View_ViewAngleLCD->setEnabled(v==1);
    myQT->render(true);
	myQT->endTask();
	
}


void iQT_View::updateView_ViewAngle( int v )
{
	
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->getRenderer()->GetActiveCamera()->SetViewAngle(v); 
    if(myQT->sliderRenderTracking) myQT->render(true);
	myQT->endTask();
	
}


void iQT_View::updateView_Flyby( bool s )
{
	
    View_Projection->setEnabled(!s);
	myQT->startTask();
    iVTKWindow::getCurrentWindow()->setInteractorStyleFlight(s);
	CALL_FUNCTION1(View_MeasuringBoxButton,setChecked,false);
	if(!s) iVTKWindow::getCurrentWindow()->resetCamera();
    myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_BoundingBoxButton(bool s)
{
	
	myQT->startTask();
    iVTKWindow::getCurrentWindow()->showBox(s);
    myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_Ruler(bool s)
{

	myQT->startTask();
    iVTKWindow::getCurrentWindow()->showRuler(s);
    myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_TimeLabel(bool s)
{
	
	myQT->startTask();
    iVTKWindow::getCurrentWindow()->showLabel(s);
    myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_MeasuringBox( bool s )
{
	
	myQT->startTask();
    iVTKWindow::getCurrentWindow()->showMeasuringBox(s);
	CALL_FUNCTION1(View_Flyby,setOn,false);
    myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_ColorBarsButton( bool s )
{
	
	myQT->startTask();
    iVTKWindow::getCurrentWindow()->showColorBars(s);
    myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_BackgroundColor()
{
	
    QColor c = QColorDialog::getColor(); 
	if(c.isValid())
	{
		CALL_FUNCTION1(View_BackgroundColorLabel,setPaletteBackgroundColor,c);
		myQT->startTask();
		iVTKWindow::getCurrentWindow()->setBackgroundColor(c);
		//
		//  Update marker widgets too
		//
		this->updateView_MarkWidgets();
		myQT->render(true);
		myQT->endTask();
	}
    
}


void iQT_View::updateView_Reset()
{

	myQT->startTask();
	iVTKWindow::getCurrentWindow()->resetCamera();
	myQT->ProgressDialog->finish();
    myQT->render(true);
	myQT->endTask();

}


void iQT_View::updateView_Light_Hbright( int v )
{
	
    float k = iVTKWindow::getCurrentWindow()->getLightKit()->GetKeyLightIntensity();
    iVTKWindow::getCurrentWindow()->getLightKit()->SetKeyToHeadRatio(k/(0.01*v));
    if(myQT->sliderRenderTracking) myQT->render(false);
    
}


void iQT_View::updateView_Light_Fbright( int v )
{
	
    float k = iVTKWindow::getCurrentWindow()->getLightKit()->GetKeyLightIntensity();
    iVTKWindow::getCurrentWindow()->getLightKit()->SetKeyToFillRatio(k/(0.01*v));
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_View::updateView_Light_Kbright( int v )
{
	
    iVTKWindow::getCurrentWindow()->getLightKit()->SetKeyLightIntensity(0.01*v);
    CALL_FUNCTION1(View_Light_HeadBrightness,setValue,round(1.0*v/iVTKWindow::getCurrentWindow()->getLightKit()->GetKeyToHeadRatio()));
    CALL_FUNCTION1(View_Light_FillBrightness,setValue,round(1.0*v/iVTKWindow::getCurrentWindow()->getLightKit()->GetKeyToFillRatio()));
    if(myQT->sliderRenderTracking) myQT->render(false);
    
}

void iQT_View::updateView_Light_Kdirection(float lon, float lat)
{
	
    iVTKWindow::getCurrentWindow()->getLightKit()->SetKeyLightAzimuth(lon);
    iVTKWindow::getCurrentWindow()->getLightKit()->SetFillLightAzimuth(-lon);
    iVTKWindow::getCurrentWindow()->getLightKit()->SetKeyLightElevation(lat);
    iVTKWindow::getCurrentWindow()->getLightKit()->SetFillLightElevation(-1.5*lat);
    myQT->render(false);
	
}

void iQT_View::updateView_Anim_Style(int v)
{
    if(v==4 && iVTKWindow::getCurrentWindow()->isParallelProjection())
	{
		myQT->popupInfo("Camera paths are normally only used with the perspective projection.");
	}
    iVTKWindow::getCurrentWindow()->getAnimator()->setMode(v);
}


void iQT_View::updateView_Anim_FlybySpeedSlider( int v )
{
    iVTKWindow::getCurrentWindow()->getAnimator()->setFlybySpeed(pow(10.0,-3.0+0.1*v));
}


void iQT_View::updateView_Anim_RestoreCamera(bool s)
{
    iVTKWindow::getCurrentWindow()->getAnimator()->setRestoreCamera(s);
}


void iQT_View::updateView_Anim_FramesSlider( int v )
{
	
    int vt = 0;
    int vs = 1;
    for(int i=0; i<v; i++)
	{
		if(i < 40) vs = 1; else if(i < 70) vs = 2; else if(i < 100) vs = 10; else if(i<130) vs = 20; else vs = 100; 
		vt = vt + vs;
    }
    View_Anim_FramesLCD->display(vt);
    iVTKWindow::getCurrentWindow()->getAnimator()->setNframes(vt);
    
}


void iQT_View::updateView_Anim_RotatePhiSlider( int v )
{
	
    double f = 0.1*v;
    View_Anim_RotatePhiLCD->display(f);
    iVTKWindow::getCurrentWindow()->getAnimator()->setDPhi(f);
	
}


void iQT_View::updateView_Anim_RotateThetaSlider( int v )
{
	
    double f = 0.1*v;
    View_Anim_RotateThetaLCD->display(f);
    iVTKWindow::getCurrentWindow()->getAnimator()->setDTheta(f);
    
}


void iQT_View::updateView_Anim_RotateRollSlider( int v )
{
	
    double f = 0.1*v;
    View_Anim_RotateRollLCD->display(f);
    iVTKWindow::getCurrentWindow()->getAnimator()->setDRoll(f);
    
}


void iQT_View::updateView_Anim_ScaleSlider( int v )
{
	
    double f = pow(10.0,0.01*v);
    View_Anim_ScaleLCD->display(f);
    iVTKWindow::getCurrentWindow()->getAnimator()->setDScale(f);
    
}


void iQT_View::updateView_Anim_InheritSettings( bool s )
{
    iVTKWindow::getCurrentWindow()->getAnimator()->setInheritSettings(s);
}


void iQT_View::updateView_Clip_Show(bool s)
{

	myQT->startTask();
	iVTKWindow::getCurrentWindow()->showClipPlane(s);
	myQT->render(true);
	myQT->endTask();
    
}


void iQT_View::updateView_Clip_Position( int v )
{
	int i;
	vtkFloat n[3], x[3];
	
	vtkPlane *p = iVTKWindow::getCurrentWindow()->getClipPlane();
	p->GetNormal(n);
	for(i=0; i<3; i++) x[i] = -n[i]*0.01*v;
	p->SetOrigin(x);
	//
	//  Move the glass plane
	//
	iVTKWindow::getCurrentWindow()->getGlassClipPlane()->SetPosition(x);

	if(myQT->sliderRenderTracking || clipPlaneSliderRenderTracking) myQT->render(false);

}


void iQT_View::updateView_Clip_Direction( float x, float y, float z )
{
	int i;
	vtkFloat x0[3];
	vtkFloat n[3], l;
	
	vtkPlane *p = iVTKWindow::getCurrentWindow()->getClipPlane();
	p->GetNormal(n);
	p->GetOrigin(x0);
	l = sqrt(x0[0]*x0[0]+x0[1]*x0[1]+x0[2]*x0[2]);
	n[0] = x;
	n[1] = y;
	n[2] = z;
	for(i=0; i<3; i++) x0[i] = -n[i]*l;
	p->SetNormal(n);
	p->SetOrigin(x0);
	//
	//  Move the glass plane
	//
	vtkActor *gp = iVTKWindow::getCurrentWindow()->getGlassClipPlane();
	gp->SetPosition(x0);
	vtkFloat ngp[4], rAxis[3];
	gp->GetMapper()->GetInput()->GetPointData()->GetNormals()->GetTuple(0,ngp); ngp[3] = 0;
	gp->GetMatrix()->MultiplyPoint(ngp,ngp);
	vtkMath::Cross(ngp,n,rAxis);
	vtkMath::Normalize(rAxis);
	vtkFloat phi = 180/vtkMath::Pi()*acos(vtkMath::Dot(ngp,n));
	gp->RotateWXYZ(phi,rAxis[0],rAxis[1],rAxis[2]);
		
	if(myQT->sliderRenderTracking || clipPlaneSliderRenderTracking) myQT->render(false);

}


void iQT_View::updateView_Clip_GlassPlaneOff()
{
	
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->showGlassClipPlane(false);
    myQT->render(true);
	myQT->endTask();

}


void iQT_View::updateView_Clip_GlassPlaneOn()
{
	
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->showGlassClipPlane(true);
	myQT->render(false);
	myQT->endTask();

}


void iQT_View::updateView_Clip_RenderWhileTrackingBox( bool s )
{
    clipPlaneSliderRenderTracking = s;
}


void iQT_View::updateView_Anim_XsectionMotionSlider( int v )
{
    iVTKWindow::getCurrentWindow()->getAnimator()->setSlideSpeed(0.01*v);
}


void iQT_View::updateView_Anim_BlendedEffectsSlider( int v )
{
	static bool started = false;
	if(v>0 && !started)
	{
		started = true;
		myQT->popupInfo("During blending IFRIT behavior changes:\n - Visualization window will show only last unblended render scene. The scenes will be blended in the output file.");
	}
    iVTKWindow::getCurrentWindow()->getAnimator()->setNumBlendedFrames(v);
}


void iQT_View::updateView_Anim_TransitionEffectsSlider( int v )
{
	static bool started = false;
	if(v>0 && !started)
	{
		started = true;
		myQT->popupInfo("During the transition IFRIT behavior changes:\n - Two subsequent file sets will be loaded alternately once each frame.\n - Visualization window will show two myQT->render scenes - one for each of the\n   two alternate file sets. These scenes will be blended in the output file.");
	}
    iVTKWindow::getCurrentWindow()->getAnimator()->setNumTransitionFrames(v);
}


void iQT_View::updateView_Anim_LaunchEditorButton()
{

}


void iQT_View::updateView_Ster_EyeAngle(int v)
{
	
	iVTKWindow::getCurrentWindow()->getRenderer()->GetActiveCamera()->SetEyeAngle(0.25*v);
	View_Ster_EyeAngleLCD->display(0.25*v);
	if(myQT->sliderRenderTracking) myQT->render(false);

}


void iQT_View::updateView_Ster_Method(int v)
{
	
	if(v != iVTKWindow::getCurrentWindow()->getWindow()->GetStereoType())
	{
		myQT->startTask();
		iVTKWindow::getCurrentWindow()->getWindow()->SetStereoType(v);
		myQT->render(true);
		myQT->endTask();
	}

}


void iQT_View::updateView_Anim_UseScript(bool s)
{

	iVTKWindow::getCurrentWindow()->getAnimator()->setUseScript(s);

}


void iQT_View::updateView_Anim_CameraPathHandlesSpinBox(int v)
{
	iVTKWindow::getCurrentWindow()->getAnimator()->setNumberOfPathHandles(v);
	//
	//  No need to render here - vtkSplineWidget does it for us
	//
	//	iVTKWindow::getCurrentWindow()->render(false);
}


void iQT_View::updateView_Anim_CameraPathStepsSlider(int v)
{
	int vv = round(pow10(0.1*v));
	myQT->startTask();
	CALL_FUNCTION1(View_Anim_CameraPathStepsLCD,display,vv);
	iVTKWindow::getCurrentWindow()->getAnimator()->setNumberOfPathSteps(vv);
	iVTKWindow::getCurrentWindow()->render(false);
	myQT->endTask();
}


void iQT_View::updateView_Anim_CameraPathLoopCheckBox(bool v)
{
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->getAnimator()->setPathLoop(v);
	iVTKWindow::getCurrentWindow()->render(false);
	myQT->endTask();
}


void iQT_View::updateView_Anim_CameraPathFocalCheckBox(bool v)
{
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->getAnimator()->useFocalPointPath(v);
	iVTKWindow::getCurrentWindow()->render(false);
	myQT->endTask();
}


void iQT_View::updateView_Anim_CameraPathFocalToPointCheckBox(bool v)
{
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->getAnimator()->setFocalPointPathToPoint(v);
	iVTKWindow::getCurrentWindow()->render(false);
	myQT->endTask();
}


void iQT_View::updateView_Anim_CameraPathDemo()
{
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->getAnimator()->playCameraPathDemo();
	myQT->endTask();
}


void iQT_View::updateView_Mark_MaterialSpecularPower(int n)
{
	CURRENT_MARKER->setSpecularPower(0.01*n);
	View_Mark_MaterialSpecularPowerLCD->display(0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
}


void iQT_View::updateView_Mark_MaterialAmbientSlider(int n)
{
	CURRENT_MARKER->setAmbient(0.01*n);
	View_Mark_MaterialAmbientLCD->display(0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
}


void iQT_View::updateView_Mark_MaterialDiffuseSlider(int n)
{
	CURRENT_MARKER->setDiffuse(0.01*n);
	View_Mark_MaterialDiffuseLCD->display(0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
}


void iQT_View::updateView_Mark_MaterialSpecularSlider(int n)
{
	CURRENT_MARKER->setSpecular(0.01*n);
	View_Mark_MaterialSpecularLCD->display(0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
}


void iQT_View::updateView_Mark_CurrentMarkerComboBox(int n)
{
	iVTKWindow::getCurrentWindow()->getMarkerFamily()->setCurrentMemberIndex(n);
	this->updateView_MarkWidgets();
}


void iQT_View::updateView_Mark_AddMarkerPushButton()
{
	int ret;
	myQT->startTask();
	if(View_Mark_CurrentMarkerComboBox->count() == 0) 
	{
		ret = 0; 
		this->enableView_MarkWidgets(true,0);
	}
	else 
	{
		ret = iVTKWindow::getCurrentWindow()->getMarkerFamily()->createMember();
	}

	if(ret == -1)
	{
		myQT->popupInfo("Cannot create a new marker - must be out of memory\n");
		CALL_FUNCTION1(View_Mark_AddMarkerPushButton,setEnabled,false);
	}
	else
	{
		if(!View_Mark_DeleteMarkerPushButton->isEnabled()) CALL_FUNCTION1(View_Mark_DeleteMarkerPushButton,setEnabled,true);
		iVTKWindow::getCurrentWindow()->getMarkerFamily()->getMember(iVTKWindow::getCurrentWindow()->getMarkerFamily()->getMaxMemberIndex())->copyState(CURRENT_MARKER);
		iVTKWindow::getCurrentWindow()->getMarkerFamily()->setCurrentMemberIndex(iVTKWindow::getCurrentWindow()->getMarkerFamily()->getMaxMemberIndex());
		CALL_FUNCTION1(View_Mark_CurrentMarkerComboBox,insertItem,"");
		CALL_FUNCTION1(View_Mark_CurrentMarkerComboBox,setCurrentItem,iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMemberIndex());
		QString s;
		s.setNum(1+iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMemberIndex());
		s = "#" + s + " " + CURRENT_MARKER->getModeAsString();
		CALL_FUNCTION1(View_Mark_CurrentMarkerComboBox,setCurrentText,s);
		CURRENT_MARKER->show(true);
	}
	iVTKWindow::getCurrentWindow()->buildMarkerLegend();
	this->updateView_MarkWidgets();
	if(ret > 0) CALL_FUNCTION0(View_Mark_CaptionLineEdit,selectAll);
	iVTKWindow::getCurrentWindow()->render(true);
	myQT->endTask();
}


void iQT_View::updateView_Mark_DeleteMarkerPushButton()
{
	int c = iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMemberIndex();
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->getMarkerFamily()->getMember(c)->show(false);
	CALL_FUNCTION1(View_Mark_DeleteMarkerPushButton,setEnabled,iVTKWindow::getCurrentWindow()->getMarkerFamily()->deleteMember(c));
	CALL_FUNCTION1(View_Mark_CurrentMarkerComboBox,removeItem,c);
	if(!View_Mark_AddMarkerPushButton->isEnabled()) CALL_FUNCTION1(View_Mark_AddMarkerPushButton,setEnabled,true);
	iVTKWindow::getCurrentWindow()->buildMarkerLegend();
	this->updateView_MarkWidgets();
	iVTKWindow::getCurrentWindow()->render(true);
	myQT->endTask();
}


void iQT_View::updateView_Mark_PlaceMarkerToolButton(bool s)
{
	this->enableView_MarkWidgets(!s,1);
	iVTKWindow::getCurrentWindow()->placeMarker(s);
	if(!s && CURRENT_VECTOR->getSourceType()==1) 
	{
		myQT->startTask();
		CURRENT_VECTOR->updateSource();
		iVTKWindow::getCurrentWindow()->render(true);
		myQT->endTask();
	}
}


void iQT_View::updateView_Mark_MoveCaptionToolButton(bool s)
{
	this->enableView_MarkWidgets(!s,2);
	iVTKWindow::getCurrentWindow()->moveMarkerCaption(s);
}


void iQT_View::updateView_Mark_MarkerStyleButtonGroup(int m)
{
	myQT->startTask();
	CURRENT_MARKER->setMode(m);
	if(View_Mark_CurrentMarkerComboBox->count() > 0)
	{
		QString s;
		s.setNum(1+iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMemberIndex());
		s = "#" + s + " " + CURRENT_MARKER->getModeAsString();
		CALL_FUNCTION1(View_Mark_CurrentMarkerComboBox,setCurrentText,s);
	}
	iVTKWindow::getCurrentWindow()->render(true);
	myQT->endTask();
}


void iQT_View::updateView_Mark_MarkerColorPushButton()
{
    QColor c = QColorDialog::getColor(); 
	if(c.isValid())
	{
		CALL_FUNCTION1(View_Mark_MarkerColorLabel,setPaletteBackgroundColor,c);
		myQT->startTask();
		CURRENT_MARKER->setColor(c);
		myQT->render(true);
		myQT->endTask();
	}
}


void iQT_View::updateView_Mark_CaptionColorPushButton()
{
    QColor c = QColorDialog::getColor(); 
	if(c.isValid())
	{
		CALL_FUNCTION1(View_Mark_CaptionColorLabel,setPaletteBackgroundColor,c);
		myQT->startTask();
		CURRENT_MARKER->setTextColor(c);
		if(CURRENT_MARKER->isColorAutomatic()) 
		{
			CALL_FUNCTION0(View_Mark_CaptionColorLabel,unsetPalette);
			CALL_FUNCTION1(View_Mark_CaptionColorLabel,setText,"Auto"); 
		}
		else 
		{
			CALL_FUNCTION1(View_Mark_CaptionColorLabel,setPaletteBackgroundColor,CURRENT_MARKER->getTextColor());
			CALL_FUNCTION1(View_Mark_CaptionColorLabel,setText,""); 
		}
		myQT->render(true);
		myQT->endTask();
	}
}


void iQT_View::updateView_Mark_ShowLegendCheckBox(bool s)
{
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->buildMarkerLegend();
	iVTKWindow::getCurrentWindow()->showMarkerLegend(s);
	iVTKWindow::getCurrentWindow()->render(true);
	myQT->endTask();
}


void iQT_View::updateView_Mark_ShowCaptionCheckBox(bool s)
{
	myQT->startTask();
	CURRENT_MARKER->setShowCaption(s);
	iVTKWindow::getCurrentWindow()->render(true);
	myQT->endTask();
}


void iQT_View::updateView_Mark_LegendLocationButtonGroup(int n)
{
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->setMarkerLegendPosition(n);
	iVTKWindow::getCurrentWindow()->render(true);
	myQT->endTask();
}


void iQT_View::updateView_Mark_CaptionLineEdit(const QString &s)
{
	CURRENT_MARKER->setCaption(s);
	iVTKWindow::getCurrentWindow()->buildMarkerLegend(iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMemberIndex());
	iVTKWindow::getCurrentWindow()->render(false);
}


void iQT_View::updateView_Mark_MarkerSizeSlider(int v)
{
	float vv = pow10(0.1*v);
	CURRENT_MARKER->setSize(vv);
	if(myQT->sliderRenderTracking) iVTKWindow::getCurrentWindow()->render(false);
}


void iQT_View::updateView_Mark_MakeFocalPointPushButton()
{
	int n = round(2.0/iVTKWindow::getCurrentWindow()->getRenderer()->GetLastRenderTimeInSeconds());
	if(n > 60) n = 60;
	if(n < 10) n = 0;
	iVTKWindow::getCurrentWindow()->getInteractor()->SetNumberOfFlyFrames(n);
	double *x = iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMember()->getPosition();
	iVTKWindow::getCurrentWindow()->getInteractor()->FlyTo(iVTKWindow::getCurrentWindow()->getRenderer(),x[0],x[1],x[2]);
}


void iQT_View::updateView_Mark_MarkerPositionXLineEdit(const QString &s)
{
	this->updateView_Mark_MarkerPosition(0,s);
}


void iQT_View::updateView_Mark_MarkerPositionYLineEdit(const QString &s)
{
	this->updateView_Mark_MarkerPosition(1,s);
}


void iQT_View::updateView_Mark_MarkerPositionZLineEdit(const QString &s)
{
	this->updateView_Mark_MarkerPosition(2,s);
}


void iQT_View::updateView_Mark_MarkerPosition(int i, const QString &s)
{
	double *x = iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMember()->getPosition();
	bool ok;
	double p = s.toDouble(&ok);
	if(ok && i>=0 && i<3)
	{
		double x1[3];
		x1[0] = x[0];
		x1[1] = x[1];
		x1[2] = x[2];
		x1[i] = iVTKWindow::getCurrentWindow()->transformToInternalCoordinate(p);
		iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMember()->setPosition(x1);
		myQT->render(false);
	}
}


void iQT_View::updateView_MarkWidgets()
{
	static int nInstance = -1;
	static iVTK *myVTK = 0;
	int i;
	QString s;
    
	//
	//  update instance lists if VTK or the # of instances changed
	//
	if(iVTKWindow::getCurrentWindow()!=myVTK || nInstance!=iVTKWindow::getCurrentWindow()->getMarkerFamily()->getMaxMemberIndex())
	{
		//
		//  Reset some things if window changed
		//

		myVTK = iVTKWindow::getCurrentWindow();
		nInstance = iVTKWindow::getCurrentWindow()->getMarkerFamily()->getMaxMemberIndex();
		if(nInstance==0 && !iVTKWindow::getCurrentWindow()->getMarkerFamily()->getMember(0)->isVisible()) nInstance = -1;

		if(nInstance >= 0)
		{
			View_Mark_CurrentMarkerComboBox->blockSignals(true);
			View_Mark_CurrentMarkerComboBox->clear();
			for(i=0; i<=nInstance; i++)
			{
				s.setNum(i+1);
				s = "#" + s + " " + iVTKWindow::getCurrentWindow()->getMarkerFamily()->getMember(i)->getModeAsString();
				View_Mark_CurrentMarkerComboBox->insertItem(s,-1);
			}
			View_Mark_CurrentMarkerComboBox->blockSignals(false);
		}

	}

	if(View_Mark_CurrentMarkerComboBox->count() > 0)
	{
		CALL_FUNCTION1(View_Mark_CurrentMarkerComboBox,setCurrentItem,iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMemberIndex());
		this->enableView_MarkWidgets(true,0);
	}
	else
	{
		this->enableView_MarkWidgets(false,0);
	}

	CALL_FUNCTION1(View_Mark_CaptionLineEdit,setText,CURRENT_MARKER->getCaption());

	CALL_FUNCTION1(View_Mark_MarkerSizeSlider,setValue,round(10.0*log10(CURRENT_MARKER->getSize())));

	CALL_FUNCTION1(View_Mark_MarkerColorLabel,setPaletteBackgroundColor,CURRENT_MARKER->getColor());

	if(CURRENT_MARKER->isColorAutomatic()) 
	{
		CALL_FUNCTION0(View_Mark_CaptionColorLabel,unsetPalette);
		CALL_FUNCTION1(View_Mark_CaptionColorLabel,setText,"Auto"); 
	}
	else 
	{
		CALL_FUNCTION1(View_Mark_CaptionColorLabel,setPaletteBackgroundColor,CURRENT_MARKER->getTextColor());
		CALL_FUNCTION1(View_Mark_CaptionColorLabel,setText,""); 
	}

	CALL_FUNCTION1(View_Mark_MarkerStyleButtonGroup,setButton,CURRENT_MARKER->getMode());

	CALL_FUNCTION1(View_Mark_ShowCaptionCheckBox,setChecked,CURRENT_MARKER->isCaptionShown());

	CALL_FUNCTION1(View_Mark_ShowLegendCheckBox,setChecked,iVTKWindow::getCurrentWindow()->isMarkerLegendShown());

	CALL_FUNCTION1(View_Mark_LegendLocationButtonGroup,setButton,iVTKWindow::getCurrentWindow()->getMarkerLegendPosition());
	//
	//  Material panel widgets
	//    
	float f;
    f = CURRENT_MARKER->getDiffuse();
    CALL_FUNCTION1(View_Mark_MaterialDiffuseSlider,setValue,round(100.0*f));
    View_Mark_MaterialDiffuseLCD->display((double)f);
    
    f = CURRENT_MARKER->getAmbient();
    CALL_FUNCTION1(View_Mark_MaterialAmbientSlider,setValue,round(100.0*f));
    View_Mark_MaterialAmbientLCD->display((double)f);
    
    f = CURRENT_MARKER->getSpecular();
    CALL_FUNCTION1(View_Mark_MaterialSpecularSlider,setValue,round(100.0*f));
    View_Mark_MaterialSpecularLCD->display((double)f);
    
    f = CURRENT_MARKER->getSpecularPower();
    CALL_FUNCTION1(View_Mark_MaterialSpecularPowerSlider,setValue,round(f));
    View_Mark_MaterialSpecularPowerLCD->display((double)f);

	myQT->displayCurrentMarkerPosition();

}


void iQT_View::enableView_MarkWidgets(bool s, int mode)
{

	CALL_FUNCTION1(View_Mark_CurrentMarkerComboBox,setEnabled,s);
	CALL_FUNCTION1(View_Mark_CaptionLineEdit,setEnabled,s);
	CALL_FUNCTION1(View_Mark_MarkerSizeSlider,setEnabled,s);
	CALL_FUNCTION1(View_Mark_MakeFocalPointPushButton,setEnabled,s);
	CALL_FUNCTION1(View_Mark_MarkerColorPushButton,setEnabled,s);
	CALL_FUNCTION1(View_Mark_CaptionColorPushButton,setEnabled,s);
	CALL_FUNCTION1(View_Mark_MarkerStyleButtonGroup,setEnabled,s);
	CALL_FUNCTION1(View_Mark_ShowLegendCheckBox,setEnabled,s);
	CALL_FUNCTION1(View_Mark_ShowCaptionCheckBox,setEnabled,s);
	CALL_FUNCTION1(View_Mark_LegendLocationButtonGroup,setEnabled,s);
    CALL_FUNCTION1(View_Mark_MaterialDiffuseSlider,setEnabled,s);
    CALL_FUNCTION1(View_Mark_MaterialAmbientSlider,setEnabled,s);
    CALL_FUNCTION1(View_Mark_MaterialSpecularSlider,setEnabled,s);
    CALL_FUNCTION1(View_Mark_MaterialSpecularPowerSlider,setEnabled,s);
	//
	//  Specials
	//
	CALL_FUNCTION1(View_Mark_DeleteMarkerPushButton,setEnabled,(s&&View_Mark_CurrentMarkerComboBox->count()>0));

	CALL_FUNCTION1(View_Mark_MarkerPositionXLineEdit,setReadOnly,!s);
	CALL_FUNCTION1(View_Mark_MarkerPositionYLineEdit,setReadOnly,!s);
	CALL_FUNCTION1(View_Mark_MarkerPositionZLineEdit,setReadOnly,!s);

	switch (mode)
	{
	case 0:
		{
			CALL_FUNCTION1(View_Mark_AddMarkerPushButton,setEnabled,true);
			CALL_FUNCTION1(View_Mark_PlaceMarkerToolButton,setEnabled,s);
			CALL_FUNCTION1(View_Mark_MoveCaptionToolButton,setEnabled,s);
			break;
		}
	case 1:
		{
			CALL_FUNCTION1(View_Mark_AddMarkerPushButton,setEnabled,s);
			CALL_FUNCTION1(View_Mark_PlaceMarkerToolButton,setEnabled,true);
			CALL_FUNCTION1(View_Mark_MoveCaptionToolButton,setEnabled,s);
			break;
		}
	case 2:
		{
			CALL_FUNCTION1(View_Mark_AddMarkerPushButton,setEnabled,s);
			CALL_FUNCTION1(View_Mark_PlaceMarkerToolButton,setEnabled,s);
			CALL_FUNCTION1(View_Mark_MoveCaptionToolButton,setEnabled,true);
			break;
		}
	}

}


void iQT_View::updateView_Bars_AutomaticCheckBox(bool s)
{
	myQT->startTask();
	iVTKWindow::getCurrentWindow()->getColorBars()->showAutomatic(s);
	this->updateView_BarsWidgets();
	iVTKWindow::getCurrentWindow()->render(true);
	myQT->endTask();
}


void iQT_View::updateView_Bars_LeftShowCheckBox(bool s)
{
	this->updateView_Bars_FullControl(0,1,0,0,0,s);
}


void iQT_View::updateView_Bars_LeftVarButtonGroup(int v)
{
	this->updateView_Bars_FullControl(1,1,v+1,0,0,false);
}


void iQT_View::updateView_Bars_LeftDataChannelComboBox(int d)
{
	this->updateView_Bars_FullControl(2,1,0,d,0,false);
}


void iQT_View::updateView_Bars_LeftPaletteListComboBox(int p)
{
	this->updateView_Bars_FullControl(3,1,0,0,p,false);
}


void iQT_View::updateView_Bars_RightShowCheckBox(bool s)
{
	this->updateView_Bars_FullControl(0,0,0,0,0,s);
}


void iQT_View::updateView_Bars_RightVarButtonGroup(int v)
{
	this->updateView_Bars_FullControl(1,0,v+1,0,0,false);
}


void iQT_View::updateView_Bars_RightDataChannelComboBox(int d)
{
	this->updateView_Bars_FullControl(2,0,0,d,0,false);
}


void iQT_View::updateView_Bars_RightPaletteListComboBox(int p)
{
	this->updateView_Bars_FullControl(3,0,0,0,p,false);
}


void iQT_View::updateView_Bars_FullControl(int mode, int bar, int v, int d, int p, bool s)
{
	int v0, d0, p0;
	bool s0;

	myQT->startTask();

	iVTKWindow::getCurrentWindow()->getColorBars()->getManualBar(bar,v0,d0,p0,s0);
	switch(mode)
	{
	case 0: { s0 = s; break; }
	case 1: { v0 = v; break; }
	case 2: { d0 = d; break; }
	case 3: { p0 = p; break; }
	}
	iVTKWindow::getCurrentWindow()->getColorBars()->setManualBar(bar,v0,d0,p0,s0);

	iVTKWindow::getCurrentWindow()->render(true);
	myQT->endTask();
}


void iQT_View::updateView_Bars_SideOffsetSlider(int v)
{
	iVTKWindow::getCurrentWindow()->getColorBars()->setSideOffset(0.01*v);
	if(myQT->sliderRenderTracking) iVTKWindow::getCurrentWindow()->render(false);
}


void iQT_View::updateView_BarsWidgets()
{
	int v,d,p; bool s;

	s = iVTKWindow::getCurrentWindow()->getColorBars()->isAutomatic();
	CALL_FUNCTION1(View_Bars_AutomaticCheckBox,setChecked,s);
	View_Bars_LeftGroupBox->setEnabled(!s);
	View_Bars_RightGroupBox->setEnabled(!s);

	iVTKWindow::getCurrentWindow()->getColorBars()->getManualBar(0,v,d,p,s);
	CALL_FUNCTION1(View_Bars_RightShowCheckBox,setChecked,s);
	CALL_FUNCTION1(View_Bars_RightVarButtonGroup,setButton,v-1);
	CALL_FUNCTION1(View_Bars_RightDataChannelComboBox,setCurrentItem,d);
	CALL_FUNCTION1(View_Bars_RightPaletteListComboBox,setCurrentItem,p);

	iVTKWindow::getCurrentWindow()->getColorBars()->getManualBar(1,v,d,p,s);
	CALL_FUNCTION1(View_Bars_LeftShowCheckBox,setChecked,s);
	CALL_FUNCTION1(View_Bars_LeftVarButtonGroup,setButton,v-1);
	CALL_FUNCTION1(View_Bars_LeftDataChannelComboBox,setCurrentItem,d);
	CALL_FUNCTION1(View_Bars_LeftPaletteListComboBox,setCurrentItem,p);

	CALL_FUNCTION1(View_Bars_SideOffsetSlider,setValue,round(100.0*iVTKWindow::getCurrentWindow()->getColorBars()->getSideOffset()));
}


void iQT_View::updateView_Anim_TitlePageLoad()
{
	iString filename = QFileDialog::getOpenFileName(iEnvironment::getInstance()->get_IFRIT_DIR(),"Images (*.jpg *.jpeg *.pnm *.bmp *.png *.tif *.tiff)",this,"","Open Title Page File");
	if(filename.isEmpty()) return;
	if(iVTKWindow::getCurrentWindow()->getAnimator()->setTitlePageFile(filename))
	{
		CALL_FUNCTION1(View_Anim_TitlePageFileNameLineEdit,setText,filename);
		QImage q; QPixmap p;
		iVTK::convertImage(iVTKWindow::getCurrentWindow()->getAnimator()->getTitlePageImage(),q);
		p.convertFromImage(q.smoothScale(64,999999,QImage::ScaleMin));
		View_Anim_TitlePageImagePixmapLabel->setPixmap(p);
	}
	else
	{
		CALL_FUNCTION1(View_Anim_TitlePageFileNameLineEdit,setText,"");
		View_Anim_TitlePageImagePixmapLabel->setPixmap(0);
		myQT->popupInfo("Unable to load image from the file.\n The file may be corrupted.");
	}
}


void iQT_View::updateView_Anim_TitlePageRemove()
{
	CALL_FUNCTION1(View_Anim_TitlePageFileNameLineEdit,setText,"");
	iVTKWindow::getCurrentWindow()->getAnimator()->setTitlePageFile(iString(""));
	View_Anim_TitlePageImagePixmapLabel->setPixmap(0);
}


void iQT_View::updateView_Anim_TitlePageShowTime(int n)
{
	iVTKWindow::getCurrentWindow()->getAnimator()->setNumTitlePageFrames(n*30);
}


void iQT_View::updateView_Anim_TitlePageDissolveTime(int n)
{
	iVTKWindow::getCurrentWindow()->getAnimator()->setNumTitlePageBlendedFrames(n*30);
}


void iQT_View::updateView_Anim_LogoLoad()
{
	iString filename = QFileDialog::getOpenFileName(iEnvironment::getInstance()->get_IFRIT_DIR(),"Images (*.jpg *.jpeg *.pnm *.bmp *.png *.tif *.tiff)",this,"","Open Title Page File");
	if(filename.isEmpty()) return;
	if(iVTKWindow::getCurrentWindow()->getAnimator()->setLogoFile(filename))
	{
		CALL_FUNCTION1(View_Anim_LogoFileNameLineEdit,setText,filename);
		QImage q; QPixmap p;
		iVTK::convertImage(iVTKWindow::getCurrentWindow()->getAnimator()->getLogoImage(),q);
		p.convertFromImage(q.smoothScale(64,999999,QImage::ScaleMin));
		View_Anim_LogoImagePixmapLabel->setPixmap(p);
	}
	else
	{
		CALL_FUNCTION1(View_Anim_LogoFileNameLineEdit,setText,"");
		View_Anim_LogoImagePixmapLabel->setPixmap(0);
		myQT->popupInfo("Unable to load image from the file.\n The file may be corrupted.");
	}
}


void iQT_View::updateView_Anim_LogoRemove()
{
	CALL_FUNCTION1(View_Anim_LogoFileNameLineEdit,setText,"");
	iVTKWindow::getCurrentWindow()->getAnimator()->setLogoFile(iString(""));
	View_Anim_LogoImagePixmapLabel->setPixmap(0);
}


void iQT_View::updateView_Anim_LogoOpacityButtonGroup(int n)
{
	iVTKWindow::getCurrentWindow()->getAnimator()->setLogoOpacity(0.25+0.25*n);
}


void iQT_View::updateView_Anim_LogoPositionButtonGroup(int n)
{
	iVTKWindow::getCurrentWindow()->getAnimator()->setLogoPosition(n);
}
