/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ivtk.h"
#include "iglobals.h"

#include "ipalette.h"
#include "ilimits.h"
#include "imath.h"
#include "icolorbars.h"
#include "itextactor.h"
#include "idatareader.h"

#include <vtkScalarBarActor.h>
#include <vtkMath.h>
#include <vtkRenderer.h>
#include <vtkCamera.h>
#include <vtkMatrix4x4.h>
#include <vtkLookupTable.h>


void reportNullPointer(int ec);


iColorBars* iColorBars::New(iVTK *m)
{

	return new iColorBars(m);

}


iColorBars::iColorBars(iVTK *m) : iVTKSource(m)
{

	int i, j;

	fontSize = 14;
	fontType = 0;
	started = false;
	scale = 1.0;

	jmax = 0;
	width = 0.03;
	height = 0.7;
	yOff = 0.2;

	nqueue = 0;
	for(i=0; i<NQUEUE; i++) var[i] = dc[i] = count[i] = pal[i] = 0;
	for(i=0; i<NQUEUE; i++) pri[i] = NQUEUE;

	varSave[0] = varSave[1] = 1;
	dcSave[0] = palSave[0] = dcSave[1] = palSave[1] = 0;

	mat = vtkMatrix4x4::New();
	if(mat == 0) reportNullPointer(8001);

	for(i=0; i<2; i++)
	{
		bar[i] = vtkScalarBarActor::New();
		if(bar[i] == 0) reportNullPointer(8002);
		bar[i]->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
		bar[i]->SetOrientationToVertical();
		bar[i]->SetMaximumNumberOfColors(256);
		bar[i]->SetWidth(width);
		bar[i]->SetHeight(height);
		bar[i]->SetNumberOfLabels(0);
		bar[i]->SetTitle("");
		bar[i]->PickableOff();
	}

	for(i=0; i<2; i++)
	{
		titleActor[i] = iTextActor::New();
		if(titleActor[i] == 0) reportNullPointer(8003);
		titleActor[i]->rotate(90.0);
		titleActor[i]->setVJustification(ITEXT_CENTER);
		titleActor[i]->setBold(true);
	}
	titleActor[0]->setHJustification(ITEXT_RIGHT);
	titleActor[1]->setHJustification(ITEXT_LEFT);
	
	for(i=0; i<2; i++)
	{
		for(j=0; j<NLAB; j++)
		{
			textActorBase[j][i] = iTextActor::New();
			if(textActorBase[j][i] == 0) reportNullPointer(8004);
			textActorPower[j][i] = iTextActor::New();
			if(textActorPower[j][i] == 0) reportNullPointer(8005);
			textActorBase[j][i]->setBold(true);
			textActorPower[j][i]->setBold(true);
		}
	}

	this->setSideOffset(0.08);
	this->setColor(0.0,0.0,0.0);
	this->setFontSize(fontSize);
	this->setFontType(fontType);

}

iColorBars::~iColorBars()
{
	int i, j;

	mat->Delete();

	for(i=0; i<2; i++) bar[i]->Delete();

	for(i=0; i<2; i++)
	{
		titleActor[i]->Delete();
	}

	for(i=0; i<2; i++)
	{
		for(j=0; j<NLAB; j++)
		{
			textActorBase[j][i]->Delete();
			textActorPower[j][i]->Delete();
		}
	}

}


void iColorBars::setSideOffset(float v)
{
	if(v>0.01 && v<0.2)
	{
		xOff[1] = v; 
		xOff[0] = 1.0 - width - xOff[1]; 
		xTextOff[0] = xOff[0] + width + 0.01; xTextOff[1] = xOff[1] - 0.01;
		xTitleOff[0] = xOff[0] - 0.3*width; xTitleOff[1] = xOff[1] + width + 0.0*width;
		titleActor[0]->setPosition(xTitleOff[0],0.5);
		titleActor[1]->setPosition(xTitleOff[1],0.5);
		bar[0]->SetPosition(xOff[0],yOff);
		bar[1]->SetPosition(xOff[1],yOff);
		this->Modified();
	}
}


void iColorBars::setColor(float r, float g, float b)
{
	int i,j;
	for(i=0; i<2; i++)
	{
		titleActor[i]->setColor(r,g,b);
		for(j=0; j<NLAB; j++)
		{
			textActorBase[j][i]->setColor(r,g,b);
			textActorPower[j][i]->setColor(r,g,b);
		}
	}
}

	
void iColorBars::setFontSize(int s)
{
	int i,j;
	textActorBase[0][0]->setFontSize(s);
	if(s == textActorBase[0][0]->getFontSize())
	{
		fontSize = s;
		for(i=0; i<2; i++)
		{
			titleActor[i]->setFontSize(round(1.25*s));
			for(j=0; j<NLAB; j++)
			{
				textActorBase[j][i]->setFontSize(s);
				textActorPower[j][i]->setFontSize(s);
			}
		}
	}
}

	
void iColorBars::setFontType(int s)
{
	int i,j;
	textActorBase[0][0]->setFontType(s);
	if(s == textActorBase[0][0]->getFontType())
	{
		fontType = s;
		for(i=0; i<2; i++)
		{
			titleActor[i]->setFontType(ITEXT_VECTOR);
			for(j=0; j<NLAB; j++)
			{
				textActorBase[j][i]->setFontType(s);
				textActorPower[j][i]->setFontType(s);
			}
		}
	}
}

	
void iColorBars::ReleaseGraphicsResources(vtkWindow *win)
{
	for(int i=0; i<2; i++) 
	{
		bar[i]->ReleaseGraphicsResources(win);
		titleActor[i]->ReleaseGraphicsResources(win);
		for(int j=0; j<NLAB; j++)
		{
			textActorBase[j][i]->ReleaseGraphicsResources(win);
			textActorPower[j][i]->ReleaseGraphicsResources(win);
		}
	}
	this->vtkProp::ReleaseGraphicsResources(win);
}


int iColorBars::RenderOverlay(vtkViewport* viewport)
{

	vtkCamera *cam = ((vtkRenderer *)viewport)->GetActiveCamera();
	if(!started || cam==0 || cam->GetParallelProjection()==0) return 0;

	for(int i=0; i<2; i++)  if(count[i] > 0) 
	{
		
		bar[i]->RenderOverlay(viewport);
		titleActor[i]->RenderOverlay(viewport);

		for(int j=0; j<NLAB; j++)
		{
			textActorBase[j][i]->RenderOverlay(viewport);
			textActorPower[j][i]->RenderOverlay(viewport);
		}

	}
	
	return 1;

}


int iColorBars::RenderOpaqueGeometry(vtkViewport* viewport)
{
	int i, j;
	float w1 = 0.0, h1 = 0.0, w2 = 0.0, h2 = 0.0;
	float dv, voff, vstep;

	vtkCamera *cam = ((vtkRenderer *)viewport)->GetActiveCamera();
	if(cam==0 || cam->GetParallelProjection()==0) return 0;

	iVTK *vtk = iVTK::convertFromViewport(viewport);

	if(!started) started = true;

	if(vtk->getRenderingMagnification() == 1)
	{
		
		for(i=0; i<2; i++) 
		{
			bar[i]->SetPosition(xOff[i],yOff);
			bar[i]->SetWidth(width);
			bar[i]->SetHeight(height);
		}

		for(i=0; i<2; i++) if(count[i] > 0) 
		{
			int nstep;
			//
			//  Log table
			//
			if(myVTK->getReader()->getLimits(dc[i])->getVarStretch(var[i]) == 1)
			{

				int vl = 1 + (int)floor(myVTK->getReader()->getLimits(dc[i])->getVarLow(var[i])-0.01);
				int vh = 0 + (int)floor(myVTK->getReader()->getLimits(dc[i])->getVarHigh(var[i])+0.01);
				dv = myVTK->getReader()->getLimits(dc[i])->getVarHigh(var[i]) - myVTK->getReader()->getLimits(dc[i])->getVarLow(var[i]);
				
				if(dv<=0.0 || vl>myVTK->getReader()->getLimits(dc[i])->getVarHigh(var[i])) continue;
				
				voff = vl - myVTK->getReader()->getLimits(dc[i])->getVarLow(var[i]);
				int ndex = vh - vl;
				if(ndex > 1) nstep = 1 + ndex/NLAB; else nstep = 1;
				
				jmax = 1 + ndex/nstep;
				vstep = nstep;
				
				for(j=0; j<jmax; j++)
				{
					
					int pow = vl + nstep*j;
					float s1[2], s2[2];
					char s[10];
					sprintf(s,"%d",pow);
					
					if(pow == 0)
					{
						textActorBase[j][i]->setText(" 1"); 
						textActorPower[j][i]->setText(""); 
					} 
					else 
					{
						textActorBase[j][i]->setText("10"); 
						textActorPower[j][i]->setText(s);
					}
					
					textActorBase[j][i]->getSize(viewport,s1);
					textActorPower[j][i]->getSize(viewport,s2);
					if(w1 < s1[0]) w1 = s1[0];
					if(h1 < s1[1]) h1 = s1[1];
					if(w2 < s2[0]) w2 = s2[0];
					if(h2 < s2[1]) h2 = s2[1];
					
				}
								
			}
			else
			{

				jmax = 5; voff = 0.0;
				float vl = myVTK->getReader()->getLimits(dc[i])->getVarLow(var[i]);
				float vh = myVTK->getReader()->getLimits(dc[i])->getVarHigh(var[i]);
				if(fabs(vl) < 1.0e-3*fabs(vh)) vl = 0.0;
				if(fabs(vh) < 1.0e-3*fabs(vl)) vh = 0.0;
				dv = vh - vl;
				vstep = dv/(jmax-1);

				for(j=0; j<jmax; j++)
				{
					
					float v = vl + vstep*j;
					float s1[2], s2[2];
					char s[100];
					sprintf(s,"%g",v);
					
					textActorBase[j][i]->setText(s); 
					textActorPower[j][i]->setText(""); 
					
					textActorBase[j][i]->getSize(viewport,s1);
					textActorPower[j][i]->getSize(viewport,s2);
					if(w1 < s1[0]) w1 = s1[0];
					if(h1 < s1[1]) h1 = s1[1];
					if(w2 < s2[0]) w2 = s2[0];
					if(h2 < s2[1]) h2 = s2[1];
					
				}
				
			}

			w1 *= 0.8;
			w2 *= 0.8;
			h1 *= 0.8;
			
			float xpos, ypos;
			xpos = xTextOff[i];
			for(j=0; j<jmax; j++)
			{
				
				ypos = yOff + 0.86*height*(voff+vstep*j)/dv;
				
				if(i == 0) 
				{
					textActorBase[j][i]->setPosition(xpos,ypos-0.5*h1);
					textActorPower[j][i]->setPosition(w1+xpos,ypos+0.5*h1);
				} 
				else 
				{
					textActorBase[j][i]->setPosition(xpos-w1-w2,ypos-0.5*h1);
					textActorPower[j][i]->setPosition(xpos-w2,ypos+0.5*h1);
				}
				
			}
			
			for(j=jmax; j<NLAB; j++)
			{
				textActorBase[j][i]->setText(""); 
				textActorPower[j][i]->setText(""); 
			}

			titleActor[i]->setText((char *)myVTK->getReader()->getLimits(dc[i])->getVarName(var[i]).latin1());
			
			bar[i]->SetLookupTable(iPaletteList::getInstance()->getPalette(pal[i])->getTable());
			
		}
		
	}
	else
	{
		
		int mag = vtk->getRenderingMagnification();
		for(i=0; i<2; i++)
		{
			bar[i]->SetWidth(mag*width);
			bar[i]->SetHeight(mag*height);
		}
		//
		//  Shift positions if under magnification - 
		//
		int winij[2];
		vtk->computePositionShiftsUnderMagnification(winij,viewport);
		
		for(i=0; i<2; i++) bar[i]->SetPosition(mag*xOff[i]-winij[0],mag*yOff-winij[1]);
		
	}
	//
	//  Render everything
	//
	for(i=0; i<2; i++) if(count[i] > 0) 
	{
		bar[i]->RenderOpaqueGeometry(viewport);
		titleActor[i]->RenderOpaqueGeometry(viewport);
		for(j=0; j<NLAB; j++)
		{
			textActorBase[j][i]->RenderOpaqueGeometry(viewport);
			textActorPower[j][i]->RenderOpaqueGeometry(viewport);
		}
	}	
		
	return 1;
	
}


void iColorBars::showBar(int priority, int v, int d, int p, bool show)
{
	int i, j;
	
	if(d<0 || d>MAXDATACHANNEL || v<1 || v>myVTK->getReader()->getLimits(d)->getNvar()) return;
	
	i = -1;
	for(j=0; j<nqueue; j++)
	{
		if(v==var[j] && d==dc[j] && p==pal[j]) { i = j; break; }
	}

	if(i >= 0) 
	{
		if(show) count[i]++; else 
		{
			count[i]--;
			if(count[i] == 0) // delete the entry
			{
				for(j=i; j<nqueue-1; j++)
				{
					pri[j] = pri[j+1];
					var[j] = var[j+1];
					pal[j] = pal[j+1];
					dc[j] = dc[j+1];
					count[j] = count[j+1];
				}
				pri[nqueue-1] = NQUEUE;
				var[nqueue-1] = 0;
				pal[nqueue-1] = 0;
				dc[nqueue-1] = 0;
				count[nqueue-1] = 0;
				nqueue--;
			}
		}
	}
	else 
	{
		if(show)
		{
			//
			//  Insert a new one keeping the list ordered by priority
			//
			if(nqueue < NQUEUE)
			{
				for(i=0; i<nqueue && pri[i]<=priority; i++);
				for(j=i; j<nqueue; j++)
				{
					pri[j+1] = pri[j];
					var[j+1] = var[j];
					pal[j+1] = pal[j];
					dc[j+1] = dc[j];
					count[j+1] = count[j];
				}
				pri[i] = priority;
				var[i] = v;
				dc[i] = d;
				pal[i] = p;
				count[i] = 1;
				nqueue++;
			}
		}
		else
		{
			// cannot delete non-existing entry
		}
	}

	this->Modified();
	
}


void iColorBars::showAutomatic(bool show)
{
	int j;

	if(!show && this->isAutomatic())
	{
		if(nqueue < NQUEUE) nqueue++;
		if(nqueue < NQUEUE) nqueue++;

		for(j=0; j<nqueue-2; j++)
		{
			pri[j+2] = pri[j];
			var[j+2] = var[j];
			pal[j+2] = pal[j];
			dc[j+2] = dc[j];
			count[j+2] = count[j];
		}
		pri[0] = COLORBARS_PRIORITY_MANUAL0;
		var[0] = varSave[0];
		pal[0] = palSave[0];
		dc[0] = dcSave[0];
		pri[1] = COLORBARS_PRIORITY_MANUAL1;
		var[1] = varSave[1];
		pal[1] = palSave[1];
		dc[1] = dcSave[1];
	}

	if(show && !this->isAutomatic())
	{
		varSave[0] = var[0];
		palSave[0] = pal[0];
		dcSave[0] = dc[0];
		varSave[1] = var[1];
		palSave[1] = pal[1];
		dcSave[1] = dc[1];
		for(j=0; j<nqueue-2; j++)
		{
			pri[j] = pri[j+2];
			var[j] = var[j+2];
			pal[j] = pal[j+2];
			dc[j] = dc[j+2];
			count[j] = count[j+2];
		}
		pri[nqueue-2] = NQUEUE;
		var[nqueue-2] = 0;
		pal[nqueue-2] = 0;
		dc[nqueue-2] = 0;
		count[nqueue-2] = 0;
		pri[nqueue-1] = NQUEUE;
		var[nqueue-1] = 0;
		pal[nqueue-1] = 0;
		dc[nqueue-1] = 0;
		count[nqueue-1] = 0;
	}

	this->Modified();

}



void iColorBars::setManualBar(int bar, int v, int d, int p, bool s)
{
	if(bar<0 || bar>1 || this->isAutomatic()) return;
	var[bar] = v;
	dc[bar] = d;
	pal[bar] = p;
	if(s) count[bar] = 1; else count[bar] = 0;
}


void iColorBars::getManualBar(int bar, int &v, int &d, int &p, bool &s)
{
	if(bar<0 || bar>1 || this->isAutomatic())
	{
		v = d = p = 0;
		s = false;
	}
	else
	{
		v = var[bar];
		d = dc[bar];
		p = pal[bar];
		s = (count[bar]>0);
	}
}
