/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

  Implementation of idatagateway.h
  
=========================================================================*/


#include "iglobals.h"
#include "idatagateway.h"

#include "imath.h"
#include "ilimits.h"
#include "iuniformmeshdata.h"

#include <vtkObjectFactory.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>


void reportNullPointer(int ec);


//------------------------------------------------------------------------------
iDataGateway* iDataGateway::New()
{
	return new iDataGateway;
}


// Construct object to extract all of the input data.
iDataGateway::iDataGateway()
{
	
	localData = 0;

	error = 0;

	smooth = 0;
	int i, j, k;
	for(i=0; i<3; i++) for(j=0; j<3; j++) for(k=0; k<3; k++)
	{
		kern[0][i][j][k] = 0.0;
		kern[1][i][j][k] = 0.0;
		kern[2][i][j][k] = 0.0;
		kern[3][i][j][k] = 0.0;
		kern[4][i][j][k] = 1.0/27.0;
	}

	kern[0][1][1][1] = 1.0;

	for(i=0; i<3; i++)
	{
		kern[1][i][1][1] = 1.0/12.0;
		kern[1][1][i][1] = 1.0/12.0;
		kern[1][1][1][i] = 1.0/12.0;
	}
	kern[1][1][1][1] = 0.5;

	for(i=0; i<3; i++)
	{
		kern[2][i][1][1] = 1.0/7.0;
		kern[2][1][i][1] = 1.0/7.0;
		kern[2][1][1][i] = 1.0/7.0;
	}

	for(i=0; i<3; i++) for(j=0; j<3; j++)
	{
		kern[3][1][i][j] = 1.0/19.0;
		kern[3][i][1][j] = 1.0/19.0;
		kern[3][i][j][1] = 1.0/19.0;
	}

	array = vtkFloatArray::New();
	if(array == 0) reportNullPointer(8101);

}


iDataGateway::~iDataGateway()
{

	if(localData != 0) delete[] localData;
	array->Delete();

}

	
void iDataGateway::ExecuteData(vtkDataObject *)
{
	iUniformMeshData *input = (iUniformMeshData *)this->GetInput();
	iUniformMeshData *output = (iUniformMeshData *)this->GetOutput();
	int dims[3];
	vtkFloat pos[3];
	
	if(input == 0) return;

	output->ShallowCopy(input);

	if(smooth == 0)
	{
		error = 0;
		return;
	}

	int n = input->GetNumberOfScalarComponents();
	if(n == 0) return;
	//
	input->GetDimensions(dims);
	output->SetDimensions(dims);
	vtkIdType size = (vtkIdType)dims[0]*dims[1]*dims[2];
	
	input->GetSpacing(pos);
	output->SetSpacing(pos);
	
	input->GetOrigin(pos);
	output->SetOrigin(pos);
	
	output->SetScalarTypeToFloat();
	output->SetNumberOfScalarComponents(n);
	
	float *data = (float *)input->GetScalarPointer();
	
	this->smoothData(data,localData,n,dims);
	if(localData == 0) 
	{
		error = 1;
		array->SetArray(data,n*size,1);
	} 
	else 
	{
		error = 0;
		array->SetArray(localData,n*size,1);
	}
	
	if(input->GetPointData()->GetScalars() != 0) output->GetPointData()->SetScalars(array);
	if(input->GetPointData()->GetVectors() != 0) output->GetPointData()->SetVectors(array);
	if(input->GetPointData()->GetTensors() != 0) output->GetPointData()->SetTensors(array);

	output->Modified();
	
}


void iDataGateway::setSmoothingFactor(int v)
{
	if(v != smooth) 
	{
		v = (v < 0) ? 0 : v;
		v = (v > 4) ? 4 : v;
		smooth = v;
		this->Modified();
	}
}


void iDataGateway::smoothData(float *inData, float* &outData, int nvar, int dims[3])
{

	int ic, jc, kc, im, jm, km, ip, jp, kp;
	int n, nmax;
	vtkIdType loff;

	vtkIdType size = (vtkIdType)dims[0]*dims[1]*dims[2];

	outData = new float[nvar*size];
	if(outData == 0) return;

	nmax = dims[0];
	nmax = (dims[1] > nmax) ? dims[1] : nmax;
	nmax = (dims[2] > nmax) ? dims[2] : nmax;

	for(ic=0; ic<dims[0]; ic++)
	{
		
		ip = ic + 1;
		im = ic - 1;
		if(ip >= dims[0])
		{
			if(nmax == dims[0]) ip = 0; else ip = dims[0] - 1;
		}
		if(im < 0)
		{
			if(nmax == dims[0]) im = dims[0] - 1; else im = 0;
		}
		
		for(jc=0; jc<dims[1]; jc++)
		{
			
			jp = jc + 1;
			jm = jc - 1;
			if(jp >= dims[1])
			{
				if(nmax == dims[1]) jp = 0; else jp = dims[1] - 1;
			}
			if(jm < 0)
			{
				if(nmax == dims[1]) jm = dims[1] - 1; else jm = 0;
			}
			
			for(kc=0; kc<dims[2]; kc++)
			{
				
				kp = kc + 1;
				km = kc - 1;
				if(kp >= dims[2])
				{
					if(nmax == dims[2]) kp = 0; else kp = dims[2] - 1;
				}
				if(km < 0)
				{
					if(nmax == dims[2]) km = dims[2] - 1; else km = 0;
				}
				
				for(n=0; n<nvar; n++)
				{
					
					loff = n*size;
					
					outData[loff+kc+dims[2]*(jc+dims[1]*ic)] =
						kern[smooth][0][0][0]*inData[loff+km+dims[2]*(jm+dims[1]*im)] +
						kern[smooth][0][0][1]*inData[loff+kc+dims[2]*(jm+dims[1]*im)] +
						kern[smooth][0][0][2]*inData[loff+kp+dims[2]*(jm+dims[1]*im)] +
						kern[smooth][0][1][0]*inData[loff+km+dims[2]*(jc+dims[1]*im)] +
						kern[smooth][0][1][1]*inData[loff+kc+dims[2]*(jc+dims[1]*im)] +
						kern[smooth][0][1][2]*inData[loff+kp+dims[2]*(jc+dims[1]*im)] +
						kern[smooth][0][2][0]*inData[loff+km+dims[2]*(jp+dims[1]*im)] +
						kern[smooth][0][2][1]*inData[loff+kc+dims[2]*(jp+dims[1]*im)] +
						kern[smooth][0][2][2]*inData[loff+kp+dims[2]*(jp+dims[1]*im)] +
						kern[smooth][1][0][0]*inData[loff+km+dims[2]*(jm+dims[1]*ic)] +
						kern[smooth][1][0][1]*inData[loff+kc+dims[2]*(jm+dims[1]*ic)] +
						kern[smooth][1][0][2]*inData[loff+kp+dims[2]*(jm+dims[1]*ic)] +
						kern[smooth][1][1][0]*inData[loff+km+dims[2]*(jc+dims[1]*ic)] +
						kern[smooth][1][1][1]*inData[loff+kc+dims[2]*(jc+dims[1]*ic)] +
						kern[smooth][1][1][2]*inData[loff+kp+dims[2]*(jc+dims[1]*ic)] +
						kern[smooth][1][2][0]*inData[loff+km+dims[2]*(jp+dims[1]*ic)] +
						kern[smooth][1][2][1]*inData[loff+kc+dims[2]*(jp+dims[1]*ic)] +
						kern[smooth][1][2][2]*inData[loff+kp+dims[2]*(jp+dims[1]*ic)] +
						kern[smooth][2][0][0]*inData[loff+km+dims[2]*(jm+dims[1]*ip)] +
						kern[smooth][2][0][1]*inData[loff+kc+dims[2]*(jm+dims[1]*ip)] +
						kern[smooth][2][0][2]*inData[loff+kp+dims[2]*(jm+dims[1]*ip)] +
						kern[smooth][2][1][0]*inData[loff+km+dims[2]*(jc+dims[1]*ip)] +
						kern[smooth][2][1][1]*inData[loff+kc+dims[2]*(jc+dims[1]*ip)] +
						kern[smooth][2][1][2]*inData[loff+kp+dims[2]*(jc+dims[1]*ip)] +
						kern[smooth][2][2][0]*inData[loff+km+dims[2]*(jp+dims[1]*ip)] +
						kern[smooth][2][2][1]*inData[loff+kc+dims[2]*(jp+dims[1]*ip)] +
						kern[smooth][2][2][2]*inData[loff+kp+dims[2]*(jp+dims[1]*ip)];
					
				}
				
			}
			
		}
		
	}
	
}


float iDataGateway::getMemorySize()
{
	if(smooth == 0) return 0.0; 
	return this->GetOutput()->GetActualMemorySize();
}







