/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/**********************************************************************
*
*  IFRIT animation script class - maps commands to function calls
*
***********************************************************************/


#ifndef ISCRIPT_H
#define ISCRIPT_H


#include "istring.h"


namespace iScriptType 
{
const int maxSize = 1000;
//
//  Assignment operation type
//
enum Assignment 
{ 
	Absolute = 0, 
	RelativeAdd = 1, 
	RelativeMult = 2, 
	RelativeAddMult = 3, 
	RelativeInvert = 4, 
	RelativeInvertAdd = 5,  
	RelativeInvertMult = 6, 
	RelativeInvertAddMult = 7 
};
//
//  Value Method constants
//
enum ValueMethod
{
	None,
	ByValue,
	ByPointer,
	ByFunction
};
//
//  Value Type constants
//
enum ValueType
{
	IVALUE_NO =  0,
	IVALUE_BL =  1,		//  boolean scalar
	IVALUE_IS =  2,		//  integer scalar
	IVALUE_FS =  4,		//  float scalar
	IVALUE_FP =  8,		//  float pointer
	IVALUE_ST = 16,		//  iString "scalar" (string)
	IVALUE_VA = 32		//  iValue array 
};
//
//  Operation Type constants
//
enum OperationType
{
	IOPERATION_DC_SV,		//  declaration, simple variable
	IOPERATION_FC_LB,		//  loop & branch
	IOPERATION_SF_V0,		//  f(void)
	IOPERATION_SF_V1,		//  f(iValue*,Assignment)
	IOPERATION_AF_V1		//  f(iValue*,iValue*,Assignment)
};
//
//  Entry points
//
enum EntryPoint
{ 
	Entry = 1,				// can only enter this point (if)
	Exit = 2,				// can only exit this point (endif)
	EntryExit = 3			// can do both (else)
};
//
//  Status of the branch gate
//
enum GateStatus
{
	Skip = 0,
	Open = 1,
	Close = 2,
	Flip = 3
};

struct StackEntry
{
	OperationType Type;
	int Class;
	int Line;
};

};

//
//  Define the base class for possible values.
//  All values are accessed transparently via the base class - in a VTK style.
//  Specify a creator New() for each of possible value types (defined below).
//
//   All values are divided into different classes:
//   - IS integer scalar 
//   - FS float scalar
//   - FA float array
//
//   Values can be accessed in 3 ways: as a numeric constant, as a pointer to a value ("local variable"),
//   as a result of calling a function ("global variable").
//

class iScript;

class iValue
{

friend class iScript;

public:

	// BL creator
	static iValue* New(iString n, iScriptType::ValueMethod m, bool v, bool *p = NULL, bool (*f)(void) = NULL);
	// IS creator
	static iValue* New(iString n, iScriptType::ValueMethod m, int v, int *p = NULL, int (*f)(void) = NULL);
	// FS creator
	static iValue* New(iString n, iScriptType::ValueMethod m, float v, float *p = NULL, float (*f)(void) = NULL);
	// FP creator
	static iValue* New(iString n, iScriptType::ValueMethod m, float* v, float* *p = NULL, float* (*f)(void) = NULL);
	// ST creator
	static iValue* New(iString n, iScriptType::ValueMethod m, iString v, iString *p = NULL, iString (*f)(void) = NULL);
	// VA creator
	static iValue* New(iString n, iScriptType::ValueMethod m, int s, iValue* i, iValue **a, iValue* (*f)(int) = NULL);

	// generic New - copy constructor, optionally set value by reference rather than copy it
	static iValue* New(iValue *v, bool ref = false);

	virtual void Delete();

	virtual iString getValueAsText();
	virtual iString getTypeAsText();

	static bool areTypesCompatible(iValue *, iValue *);
	static bool areTypesCompatible(iScriptType::ValueType t, iValue *);

	inline iScriptType::ValueType getType(){ return Type; }
	inline iScriptType::ValueMethod getMethod(){ return Method; }
	inline iString getName(){ return Name; }
	
	inline bool isScalar(){ return (Type==iScriptType::IVALUE_BL || Type==iScriptType::IVALUE_FP || Type==iScriptType::IVALUE_FS || Type==iScriptType::IVALUE_IS); }
	inline bool isArray(){ return (Type == iScriptType::IVALUE_VA); }

	virtual bool setMethod(iScriptType::ValueMethod m) = 0;

	virtual bool assignValue(iScriptType::Assignment r, iValue *) = 0;
	virtual bool assignValue(iScriptType::Assignment, bool){ return false; }
	virtual bool assignValue(iScriptType::Assignment, int){ return false; }
	virtual bool assignValue(iScriptType::Assignment, float){ return false; }

	virtual bool getValue(void *) = 0;
	virtual bool getValue(bool &){ return false; }
	virtual bool getValue(int &){ return false; }
	virtual bool getValue(float &){ return false; }
	virtual bool getValue(double &){ return false; }
	virtual bool getValue(float* &){ return false; }
	virtual bool getValue(iString &){ return false; }
	virtual bool getValue(int, int *){ return false; }
	virtual bool getValue(int, float *){ return false; }

	virtual void Register(){ ReferenceCount++; }
	inline int getReferenceCount(){ return ReferenceCount; }

#ifdef _DEBUG
	static int counterCreate, counterDestroy;
	int counterAtCreation;
	static int cval[999];
#endif

protected:

	int ReferenceCount;
	iString Name;
	iScriptType::ValueType Type;
	iScriptType::ValueMethod Method;

#ifdef _DEBUG
	iValue(iString n, iScriptType::ValueType t, iScriptType::ValueMethod m);
	virtual ~iValue();
#else
	iValue(iString n, iScriptType::ValueType t, iScriptType::ValueMethod m)
	{ 
		ReferenceCount = 1; Name = n, Type = t; Method = m; 
	}
	virtual ~iValue(){}
#endif

};
//
//  Value type BL: 
//    Parameter: bool v, bool *p, bool (*f)(void)
//    Result: v, *p, f()
//
class iValue_BL : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual bool setMethod(iScriptType::ValueMethod m);
	virtual void setNumericValue(bool v){ Value = v; Method = iScriptType::ByValue; }

	virtual bool assignValue(iScriptType::Assignment r, iValue *);
	virtual bool assignValue(iScriptType::Assignment r, bool b);

	virtual bool getValue(void *);
	virtual bool getValue(bool &);

protected:

	iValue_BL(iString n, iScriptType::ValueMethod m, bool v, bool *p, bool (*f)(void)) : iValue(n,iScriptType::IVALUE_BL,m)
	{ 
		Value = v; 
		Pointer = p; 
		Function = f; 
	}

	bool Value;
	bool* Pointer;
	bool (*Function)(void);

};
//
//  Value type IS: 
//    Parameter: int v, int *p, int (*f)(void)
//    Result: v, *p, f()
//
class iValue_IS : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual bool setMethod(iScriptType::ValueMethod m);
	virtual void setNumericValue(int v){ Value = v; Method = iScriptType::ByValue; }

	virtual bool assignValue(iScriptType::Assignment r, iValue *);
	virtual bool assignValue(iScriptType::Assignment r, int i);
	virtual bool assignValue(iScriptType::Assignment r, float f);

	virtual bool getValue(void *);
	virtual bool getValue(bool &);
	virtual bool getValue(int &);
	virtual bool getValue(float &);

protected:

	iValue_IS(iString n, iScriptType::ValueMethod m, int v, int *p, int (*f)(void)) : iValue(n,iScriptType::IVALUE_IS,m)
	{ 
		Value = v; 
		Pointer = p; 
		Function = f; 
	}

	int Value;
	int* Pointer;
	int (*Function)(void);

};
//
//  Value type FS: 
//    Parameter: float v, float *p, float (*f)(void)
//    Result: v, *p, f()
//
class iValue_FS : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual bool setMethod(iScriptType::ValueMethod m);
	virtual void setNumericValue(float v){ Value = v; }

	virtual bool assignValue(iScriptType::Assignment r, iValue *);
	virtual bool assignValue(iScriptType::Assignment r, int i);
	virtual bool assignValue(iScriptType::Assignment r, float f);

	virtual bool getValue(void *);
	virtual bool getValue(bool &);
	virtual bool getValue(int &);
	virtual bool getValue(float &);
	virtual bool getValue(double &);

protected:

	iValue_FS(iString n, iScriptType::ValueMethod m, float v, float *p, float (*f)(void)) : iValue(n,iScriptType::IVALUE_FS,m)
	{ 
		Value = v; 
		Pointer = p; 
		Function = f; 
	}

	float Value;
	float* Pointer;
	float (*Function)(void);

};
//
//  Value type FP: 
//    Parameter: float *v, float **p, float* (*f)(void)
//    Result: v, *p, f()
//
class iValue_FP : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual bool setMethod(iScriptType::ValueMethod m);
	virtual void setNumericValue(float *v){ Value = v; }

	virtual bool assignValue(iScriptType::Assignment r, iValue *);

	virtual bool getValue(void *);
	virtual bool getValue(float* &p);

protected:

	iValue_FP(iString n, iScriptType::ValueMethod m, float* v, float* *p, float* (*f)(void)) : iValue(n,iScriptType::IVALUE_FP,m)
	{ 
		Value = v; 
		Pointer = p; 
		Function = f; 
	}

	float* Value;
	float** Pointer;
	float* (*Function)(void);

};
//
//  Value type ST: 
//    Parameter: iString v, iString *p, iString (*f)(void)
//    Result: v, *p, f()
//
class iValue_ST : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual bool setMethod(iScriptType::ValueMethod m);
	virtual void setNumericValue(iString v){ Value = v; }

	virtual bool assignValue(iScriptType::Assignment r, iValue *);

	virtual bool getValue(void *);
	virtual bool getValue(iString &p);

protected:

	iValue_ST(iString n, iScriptType::ValueMethod m, iString v, iString *p, iString (*f)(void)) : iValue(n,iScriptType::IVALUE_ST,m)
	{ 
		Value = v; 
		Pointer = p; 
		Function = f; 
	}

	iString Value;
	iString* Pointer;
	iString (*Function)(void);

};
//
//  Value type VA: 
//    Parameter: iValue* i, iValue** a, iValue* (*f)(int)
//    Result: a[i], f(i)
//
class iValue_VA : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual void Delete();

	virtual bool setMethod(iScriptType::ValueMethod m);

	inline int getSize(){ return Size; }
	inline iValue* getIndex(){ return Index; }
	virtual iValue* getComponent(iValue *);
	virtual iValue* getComponent(int i);

	void setIndex(iValue *i);

	virtual bool assignValue(iScriptType::Assignment r, iValue *);

	virtual bool getValue(void *);
	virtual bool getValue(bool &);
	virtual bool getValue(int &);
	virtual bool getValue(float &);
	virtual bool getValue(int, int *);
	virtual bool getValue(int, float *);

protected:

	iValue_VA(iString n, iScriptType::ValueMethod m, int s, iValue* i, iValue **a, iValue* (*f)(int)) : iValue(n,iScriptType::IVALUE_VA,m)
	{ 
		Index = i; 
		Array = a; 
		Function = f; 
		Size = s; 
	}

	int Size;
	iValue* Index;
	iValue** Array;
	iValue* (*Function)(int);

};

//
//  Define the base class for possible operations.
//  All operations are accessed transparently via the base class - in a VTK style.
//  Specify a creator New() for each of possible operation types (defined below).
//
//   All operations are divided into different classes:
//   - FC flow control operations 
//   - SF operations calling various scalar functions
//   - AF similar to SF, but have array syntax (e.f. set var[N] = V calls F(N,V))
//

class iOperation 
{

friend class iScript;

public:

	// DC_SV creator
	static iOperation* New(iScript *w, iString p, iString c, int o, iString sp, iScriptType::Assignment r, iValue *v, unsigned char u = 0);
	// FC_LB creator
	static iOperation* New(iScript *w, iString p, iString c, int o, iScriptType::EntryPoint q, iValue *i, iValue *n, int l, unsigned char u = 0);
	// SF_V0 creator
	static iOperation* New(iScript *w, iString p, iString c, int o, void (*f)(iString*), unsigned char u = 0);
	// SF_V1 creator
	static iOperation* New(iScript *w, iString p, iString c, int o, iScriptType::Assignment r, iValue* v, void (*f)(iValue*,iScriptType::Assignment,iString*), unsigned char u = 0);
	// AF_V1 creator
	static iOperation* New(iScript *w, iString p, iString c, int o, iValue* i, iScriptType::Assignment r, iValue* v, void (*f)(iValue*,iValue*,iScriptType::Assignment,iString*), unsigned char u = 0);

	// generic New - copy constructor, optionally set value by reference rather than copy it
	static iOperation* New(iOperation *p);

	virtual void Delete();

protected:

	virtual iValue* getValue(){ return NULL; }
	virtual void setValue(iValue*){};
	virtual bool assignVariableValue(iString n, iScriptType::Assignment r, iValue *v, iValue *i = NULL);
	virtual bool exec() = 0;

	virtual void setErrorMessage(const iString s);

	iString *pErrorMessage;
	iScript *Script;
	iString Command;
	iString Prefix;
	iScriptType::OperationType Type;
	unsigned char UserType;
	int OutputCode;
	int LineInScript;

	iOperation(iScript *w, iString p, iString s, int o, iScriptType::OperationType t, unsigned char u = 0);
	virtual ~iOperation(){}

};
//***************************************************************
//
//	Declaration operations
//
//***************************************************************
//
//  Operation type DC_SV: 
//    Parameter: iValue *V
//    Execution: declare a new variable with value V. Executed at the compile time
//
class iOperation_DC_SV : public iOperation 
{

friend class iScript;
friend class iOperation;

public:

	virtual void Delete();

protected:

	iOperation_DC_SV(iScript *w, iString p, iString s, int o, iString sp, iScriptType::Assignment r, iValue *v, unsigned char u) : iOperation(w,p,s,o,iScriptType::IOPERATION_DC_SV,u)
	{ 
		StatementPrefix = sp;
		Relative = r;
		Value = v; 
	}

	virtual iValue* getValue(){ return Value; }
	virtual bool exec();

	iString StatementPrefix;
	iValue *Value;
	iScriptType::Assignment Relative;

};
//***************************************************************
//
//	Flow Control operations
//
//***************************************************************
//
//  Operation type FC_LB: 
//    Parameter: EntryPoint Q, iValue *I, iValue *C, int C
//    Execution: if Q=Entry, then open the gate if either C is boolen and true, or C is integer and 
//                  C >= Running Valued or close the gate in the opposite case. Flip the gate is C
//                  is NULL.
//               if Q=Exit, close the branch of the previous Q=Entry operation and return control to 
//                  the matching type FC_LB operation of the same class C if it is not done
//    Used as a container for commands like 'do' or 'loop' or 'if'.
//    Use parameter Class to distinguish between different classes of return points,
//    like do, while, for, ...
//
class iOperation_FC_LB : public iOperation 
{

friend class iScript;
friend class iOperation;

public:

	virtual void Delete();

protected:

	iOperation_FC_LB(iScript *w, iString p, iString s, int o, iScriptType::EntryPoint q, iValue* i, iValue *c, int l, unsigned char u) : iOperation(w,p,s,o,iScriptType::IOPERATION_FC_LB,u)
	{ 
		Point = q;
		Gate = iScriptType::Open;
		Index = i;
		Count = c;
		intIndex = intCount = 0; 
		Done = true; 
		Class = l; 
	}

	virtual iValue* getValue(){ return Index; } // the value of this command is the index
	virtual void setValue(iValue* v);
	virtual bool exec();

	iScriptType::EntryPoint Point;
	iScriptType::GateStatus Gate;
	iValue *Index, *Count;
	int intIndex, intCount;
	bool Done;
	int Class;

};


//***************************************************************
//
//	Scalar Function operations
//
//***************************************************************
//
//  Operation type SF_V0: 
//    Parameter: void (*F)(void)
//    Execution: call F()
//
class iOperation_SF_V0 : public iOperation 
{

friend class iScript;
friend class iOperation;

protected:

	iOperation_SF_V0(iScript *w, iString p, iString s, int o, void (*f)(iString*), unsigned char u) : iOperation(w,p,s,o,iScriptType::IOPERATION_SF_V0,u)
	{ 
		Function = f; 
	}

	virtual bool exec();

	void (*Function)(iString*);

};
//
//  Operation type SF_V1: 
//    Parameter: iValue* V, Assignment R, void (*F)(iValue*,Assignment)
//    Execution: call F(V,R)
//
class iOperation_SF_V1 : public iOperation 
{

friend class iScript;
friend class iOperation;

public:

	virtual void Delete();

protected:

	iOperation_SF_V1(iScript *w, iString p, iString s, int o, iScriptType::Assignment r, iValue* v, void (*f)(iValue*,iScriptType::Assignment,iString*), unsigned char u) : iOperation(w,p,s,o,iScriptType::IOPERATION_SF_V1,u)
	{ 
		Value = v;
		Relative = r; 
		Function = f; 
	}

	virtual iValue* getValue(){ return Value; }
	virtual void setValue(iValue* v);
	virtual bool exec();

	iValue* Value;
	iScriptType::Assignment Relative;
	void (*Function)(iValue*,iScriptType::Assignment,iString*);

};
//
//  Operation type AF_V1: 
//    Parameter: iValue* I, iValue* V, Assignment R, void (*F)(iValue*,iValue*,Assignment)
//    Execution: call F(I,V,R) using array syntax
//
class iOperation_AF_V1 : public iOperation 
{

friend class iScript;
friend class iOperation;

public:

	virtual void Delete();

protected:

	iOperation_AF_V1(iScript *w, iString p, iString s, int o, iValue* i, iScriptType::Assignment r, iValue* v, void (*f)(iValue*,iValue*,iScriptType::Assignment,iString*), unsigned char u) : iOperation(w,p,s,o,iScriptType::IOPERATION_AF_V1,u)
	{ 
		Index = i; 
		Value = v;
		Relative = r; 
		Function = f; 
	}

	virtual iValue* getValue(){ return Value; }
	virtual void setValue(iValue* v);
	virtual bool exec();

	iValue* Index;
	iValue* Value;
	iScriptType::Assignment Relative;
	void (*Function)(iValue*,iValue*,iScriptType::Assignment,iString*);

};


//
//*******************************************************
//
//  iScript class declarations
//
//*******************************************************
//
class iExpressionParser;

class iScript 
{
	
	friend class iValue;
	friend class iOperation; // iOperation must have access to the errorMessage string

public:
	
	iScript();
	virtual ~iScript();

	void setText(iString s);
	
	virtual int run(bool compileOnly = false, int firstLine = 0, int lastLine = -1);
	virtual int run(iString s, bool compileOnly = false, int firstLine = 0, int lastLine = -1);

	virtual int runOneLine(bool compileOnly);

	inline int getCompileReturnCode(){ return compileReturnCode; }
	inline int getExecuteReturnCode(){ return executeReturnCode; }

	inline int getNumberOfLines(){ return numLines; }
	inline int getCurrentLine(){ return line; }

	virtual bool isDummyWord(const iString s);
	virtual bool isPrefixWord(const iString s);
	virtual bool isCommandWord(const iString s);
	virtual bool isParameterWord(const iString s);
	virtual bool isVariableWord(const iString s);
	
	virtual bool isReservedWord(const iString s);

	inline int getNumberOfReservedWords(){ return nPrefixWords+nCommandWords+nParameterWords+nDummyWords; }
	virtual iString getReservedWord(int i);
	virtual int getUserTypeForReservedWord(int i);

	virtual const iString getErrorMessage(){ return errorMessage; }
	virtual bool areThereOpenEntryPoints(){ return compilerStackEntryPoint>=0; }

	inline int getNumberOfVariables(){ return nVariable; }
	inline iValue* getVariable(int i){ if(i>=0 && i<nVariable) return pVariable[i]; else return NULL; }
	inline int getLastChangedVariable(){ return lastChangedVariable; }
	virtual void setVariableChanged(iValue *v);

protected:

	//
	//  Main conversion tool 
	//
	virtual iValue* transformToValue(iString s);
	virtual iValue* transformExpressionToValue(iString s, bool acceptArrays = true, bool forceFloat = false);
	virtual iValue* transformSingleTokenToValue(iString s, bool acceptArrays = true, bool forceFloat = false);

	//
	//  Operations on variables
	//
	virtual bool assignVariableValue(iString n, iScriptType::Assignment r, iValue *v, iValue *i = NULL);
	inline bool assignVariableValue(iValue *s, iScriptType::Assignment r, iValue *v, iValue *i = NULL){ return this->assignVariableValue(s->Name,r,v,i); }

	//
	//  Script execution
	//
	virtual void reset();
	virtual int compileOneLine();
	virtual int executeOneLine();

	virtual void setErrorMessage(iString s);
	virtual void skipEmptyLines();

	iString getText(int);

	//
	//  Script creation
	//
	virtual void createAliasWord(const char *, const char *);
	virtual void createDummyWord(const char *);
	virtual void createPrefixWord(const char *);
	virtual void createCommandWord(iOperation*);
	virtual void createParameterWord(iValue *);

	bool (*checkAbort)(int,int,int,int);
	void (*showLine)(int);

	bool caseSensitive, arraysAreFloat;

	iString commentChar, leChar, geChar, eqChar, neChar, andChar, orChar;

private:

	bool active, block;
	int lActivator;

	iString errorMessage;

	int nAliasWords;
	iString pAliasWords[2][iScriptType::maxSize];

	int nPrefixWords;
	iString pPrefixWords[iScriptType::maxSize];

	int nDummyWords;
	iString pDummyWords[iScriptType::maxSize];

	int nCommandWords, nCommand;
	iOperation *pCommandWords[iScriptType::maxSize];

	int nParameterWords;
	iValue *pParameterWords[iScriptType::maxSize];

//	int nValue;
//	iValue *pValue[iScriptType::maxSize];

	int nVariable, lastChangedVariable;
	iValue *pVariable[iScriptType::maxSize];

//	int nValueWords;
//	iValue *pValueWords[iScriptType::maxSize];

	int nCode, lCode, nBuffer;
	iOperation **pCode;

	iScriptType::StackEntry *compilerStack;
	int compilerStackSize, compilerStackEntryPoint;

	iString text;
	int line, numLines, executeReturnCode, compileReturnCode;
	int entryPointIteration, entryPointIterationCount;

	iExpressionParser* parser;

};

#endif // ISCRIPT_H
