/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "imeasuringbox.h"


#include "iactor.h"
#include "ierror.h"
#include "irendertool.h"
#include "itextactor.h"
#include "itextbitmap.h"
#include "itextvector.h"
#include "iviewmodule.h"

#include <vtkCamera.h>
#include <vtkCubeSource.h>
#include <vtkPolyDataMapper.h>
#include <vtkProperty.h>
#include <vtkRenderer.h>

#include <stdio.h>


using namespace iParameter;


iMeasuringBox* iMeasuringBox::New(iViewModule *vm)
{
	return new iMeasuringBox(vm);
}


iMeasuringBox::iMeasuringBox(iViewModule *vm) : iActor(false), iViewModuleComponent(vm)
{
	mStarted = false;
	mSize = 0.25;
	mBaseScale = 1.0;

	mFactor1r = 0.78; mFactor1g = 0.89; mFactor1b = 0.82; mFactor2 = 0.0;

	mCube = vtkCubeSource::New(); IERROR_ASSERT_NULL_POINTER(mCube);
	mFrame = iActor::New(); IERROR_ASSERT_NULL_POINTER(mFrame);

	mCube->SetCenter(0.0,0.0,0.0);
	mCube->SetXLength(2.0);
	mCube->SetYLength(2.0);
	mCube->SetZLength(2.0);

	mFrame->SetInput(mCube->GetOutput());
	mFrame->GetProperty()->SetRepresentationToWireframe();
	mFrame->GetProperty()->SetLineWidth(2);
	mFrame->PickableOff();

	this->SetInput(mCube->GetOutput());
	this->GetProperty()->SetOpacity(0.2);
	this->GetProperty()->SetAmbient(0.5);
	this->GetProperty()->SetDiffuse(0.5);
	this->GetProperty()->SetSpecular(0.7);
	this->GetProperty()->SetSpecularPower(50);

	mText = iTextActor::New(this->GetViewModule()->GetRenderTool()); IERROR_ASSERT_NULL_POINTER(mText);
	mText->SetBold(true);
	mText->SetPosition(0.5,0.03);
	mText->SetHJustification(_TextJustCenter);

	iColor black;
	this->SetColor(black);
}


iMeasuringBox::~iMeasuringBox()
{
	mFrame->Delete();
	mCube->Delete();
	mText->Delete();
}


void iMeasuringBox::SetColor(iColor &c)
{
	mText->SetColor(c); 
	this->GetProperty()->SetColor(mFactor1r+mFactor2*c.ToVTK()[0],mFactor1g+mFactor2*c.ToVTK()[1],mFactor1b+mFactor2*c.ToVTK()[2]);
	mFrame->GetProperty()->SetColor(c.ToVTK());
}

	
void iMeasuringBox::SetOpacity(float o)
{
	if(o > 0.0)
	{
		if(o > 1.0) o = 1.0;
		this->GetProperty()->SetOpacity(o);
	}
}

	
void iMeasuringBox::ReleaseGraphicsResources(vtkWindow *win)
{
	mFrame->ReleaseGraphicsResources(win);
	mText->ReleaseGraphicsResources(win);
	this->iActor::ReleaseGraphicsResources(win);
}


int iMeasuringBox::RenderOverlay(vtkViewport* viewport)
{
	vtkCamera *cam = iPointerCast<vtkRenderer,vtkViewport>(viewport)->GetActiveCamera();
	if(cam==0 || cam->GetParallelProjection()==0) return 0;

	mText->RenderOverlay(viewport);

	return 1;
}


int iMeasuringBox::RenderOpaqueGeometry(vtkViewport* viewport)
{
	vtkCamera *cam = iPointerCast<vtkRenderer,vtkViewport>(viewport)->GetActiveCamera();
	if(cam == 0) return 0;
	
	if(!mStarted)
	{
		mStarted = true;
		mBaseScale = cam->GetParallelScale();
	}
	
	if(this->GetViewModule()->GetRenderTool()->GetRenderingMagnification() == 1)
	{
		float s = mSize*cam->GetParallelScale()/mBaseScale;
		char t[256];
		sprintf(t,"Box size: %6.2g",s);
		
		mText->SetText(t);
		
		this->SetScale(s);
		mFrame->SetUserMatrix(this->GetMatrix());
	}
	
	if(this->GetProperty()->GetOpacity() < 1.0) 
	{ 
	} 
	else 
	{
		iActor::RenderOpaqueGeometry(viewport);
	}
	mText->RenderOpaqueGeometry(viewport);
	mFrame->RenderOpaqueGeometry(viewport);
	
	return 1;
}


int iMeasuringBox::RenderTranslucentGeometry(vtkViewport* viewport)
{
	vtkCamera *cam = iPointerCast<vtkRenderer,vtkViewport>(viewport)->GetActiveCamera();
	if(cam == 0) return 0;
	
	if(!mStarted)
	{
		mStarted = true;
		mBaseScale = cam->GetParallelScale();
	}

	if(this->GetProperty()->GetOpacity() < 1.0) 
	{
		iActor::RenderTranslucentGeometry(viewport);
	}

	return 1;
}


void iMeasuringBox::SetBaseScale(float s)
{
	mStarted = true;
	mBaseScale = s;
}


// Get the length of the diagonal of the bounding box.
float iMeasuringBox::GetLength()
{
	return mFrame->GetLength();
}


