/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iscript.h"


#include "ierror.h"
#include "ierrorstatus.h"
#include "iexpressionparser.h"
#include "imath.h"

#include <stdlib.h>

//
//  Templates
//
#include "iarraytemplate.h"


#ifdef I_CHECK1
int iValue::counterCreate = 0;
int iValue::counterDestroy = 0;
int iValue::cval[999];
#include <iostream>
#endif


void ReportUnallowedPath()
{
#ifdef I_CHECK1
	IERROR_REPORT_ERROR("Bug in iScript: unallowed path.");
#endif
	exit(-1);
}


#define ASSIGN_VALUE(v,rel,rhs) \
{ \
	if(rel & 1) { if(rel & 4) v = (rhs) - v; else v += (rhs); } else { if(rel & 2) v *= (rhs); else v = (rhs); } \
}


#define NotUsed(x)

//
//***************************************************************
//
//  iValue functions
//
//***************************************************************
//


#ifdef I_CHECK1

iValue::iValue(iScript *w, const iString &n, ValueType t, ValueMethod m)
{ 
	ReferenceCount = 1; 
	Script = w; Name = n, Type = t; Method = m; 
	counterAtCreation = counterCreate;
	if(counterCreate < 999) cval[counterCreate] = 1;
	counterCreate++;
}


iValue::~iValue()
{ 
	if(counterAtCreation < 999) cval[counterAtCreation] = 0;
	counterDestroy++;
}

#endif


//
//  Copy constructor
//
iValue* iValue::New(iValue *v0, bool ref)
{
	if(v0 == 0) return 0;

	switch(v0->Type)
	{
	case _Value_BL: 
		{ 
			iValue_BL *v = (iValue_BL *)v0; 
			if(!ref || v->Method!=_ByValue) 
			{
				return iValue::New(v->Script,v->Name,v->Method,v->Value,v->Pointer,v->Function); 
			}
			else
			{
				return iValue::New(v->Script,v->Name,_ByPointer,v->Value,&v->Value,0); 
			}
		}
	case _Value_IS: 
		{ 
			iValue_IS *v = (iValue_IS *)v0; 
			if(!ref || v->Method!=_ByValue) 
			{
				return iValue::New(v->Script,v->Name,v->Method,v->Value,v->Pointer,v->Function); 
			}
			else
			{
				return iValue::New(v->Script,v->Name,_ByPointer,v->Value,&v->Value,0); 
			}
		}
	case _Value_RS: 
		{ 
			iValue_RS *v = (iValue_RS *)v0; 
			if(!ref || v->Method!=_ByValue) 
			{
				return iValue::New(v->Script,v->Name,v->Method,v->Value,v->Pointer,v->Function); 
			}
			else
			{
				return iValue::New(v->Script,v->Name,_ByPointer,v->Value,&v->Value,0); 
			}
		}
	case _Value_RP: 
		{ 
			iValue_RP *v = (iValue_RP *)v0; 
			if(!ref || v->Method!=_ByValue) 
			{
				return iValue::New(v->Script,v->Name,v->Method,v->Value,v->Pointer,v->Function); 
			}
			else
			{
				return iValue::New(v->Script,v->Name,_ByPointer,v->Value,&v->Value,0); 
			}
		}
	case _Value_ST: 
		{ 
			iValue_ST *v = (iValue_ST *)v0; 
			if(!ref || v->Method!=_ByValue) 
			{
				return iValue::New(v->Script,v->Name,v->Method,v->Value,v->Pointer,v->Function,v->Quoted); 
			}
			else
			{
				return iValue::New(v->Script,v->Name,_ByPointer,v->Value,&v->Value,0,v->Quoted); 
			}
		}
	case _Value_VA: 
		{ 
			iValue_VA *v = (iValue_VA *)v0;
			iValue** arr = 0;
			if(v->Size>0 && v->Array!=0)
			{
				arr = new iValue*[v->Size]; IERROR_ASSERT_NULL_POINTER(arr);
				int i;
				for(i=0; i<v->Size; i++) 
				{
					if(v->Array[i] != 0)
					{
						arr[i] = iValue::New(v->Array[i],ref);  // these are not registered - they are owned by iValue_VA
					}
					else
					{
						arr[i] = 0;
					}
				}
			}
			return iValue::New(v->Script,v->Name,v->Method,v->Size,iValue::New(v->Index),arr,v->Function);
		}
	default: { ReportUnallowedPath(); return 0; }
	}
}


//
//  shortcuts for Operation consructors
//
iValue* iValue::NewInt(iScript *w)
{
	return iValue::New(w,"",iValue::_ByValue,0);
}

iValue* iValue::NewBool(iScript *w)
{
	return iValue::New(w,"",iValue::_ByValue,false);
}

iValue* iValue::NewReal(iScript *w)
{
	return iValue::New(w,"",iValue::_ByValue,0.0);
}

iValue* iValue::NewArray(iScript *w, int size)
{
	if(size < 0) size = 0;
	return iValue::New(w,"",iValue::_ByValue,size,(iValue*)0,(iValue**)0);
}

iValue* iValue::NewIndex(iScript *w)
{
	return iValue::New(w,"",iValue::_ByValue,1);
}

iValue* iValue::NewString(iScript *w, bool quoted)
{
	return iValue::New(w,"",iValue::_ByValue,iString(),0,0,quoted);
}


//
//  Check that the types are compatible: the order does matter
//
bool iValue::AreTypesCompatible(iValue *v1, iValue *v2)
{
	if(v1 == 0) return false; else return iValue::AreTypesCompatible(v1->Type,v2);
}


bool iValue::AreTypesCompatible(ValueType t, iValue *v2)
{
	if(v2 == 0) return false;
	if(t == v2->Type) return true; else
	{
		switch(v2->Type)
		{
			case _Value_BL: return false;
			case _Value_IS: return (t == _Value_RS);
			case _Value_RS: return (t == _Value_IS);
			case _Value_RP: return false;
			case _Value_VA: return false;
			default: { ReportUnallowedPath(); return false; }
		}
	}
}


iString iValue::GetValueAsText() const
{
	iString s;

	switch(Type)
	{
	case _Value_ST: 
		{
			if(this->GetValue(s)) return s; else return "undefined";
		}
	case _Value_BL: 
		{
			bool v;
			if(this->GetValue(v)) { if(v) s = "true"; else s = "false"; } else s = "undefined";
			return s;
		}
	case _Value_IS: 
		{
			int v;
			if(this->GetValue(v)) s = iString::FromNumber(v); else s = "undefined";
			return s;
		}
	case _Value_RS: 
		{
			double v;
			if(this->GetValue(v)) s = iString::FromNumber(v); else s = "undefined";
			return s;
		}
	case _Value_RP: 
		{
			double *v;
			if(this->GetValue(&v)) s = iString::FromNumber((void *)v); else s = "undefined";
			return s;
		}
	case _Value_VA: 
		{
			int i;
			for(i=1; i<=((iValue_VA*)this)->GetSize() && i<=10; i++)
			{
				if(((iValue_VA*)this)->GetComponent(i) != 0)
				{
					s += ((iValue_VA*)this)->GetComponent(i)->GetValueAsText();
				}
				else
				{
					s += "undefined";
				}
				s += " ";
			}
			if(((iValue_VA*)this)->GetSize() > 10) s += "..."; 
			return s;
		}
	default: { ReportUnallowedPath(); return ""; }
	}
}


iString iValue::GetTypeAsText() const
{
	switch(Type)
	{
	case _Value_ST:
		{
			return "string";
		}
	case _Value_BL: 
		{
			return "boolean scalar";
		}
	case _Value_IS: 
		{
			return "integer scalar";
		}
	case _Value_RS: 
		{
			return "real scalar";
		}
	case _Value_RP: 
		{
			return "real pointer";
		}
	case _Value_VA: 
		{
			return "array[" + iString::FromNumber(((iValue_VA*)this)->GetSize()) + "]";
		}
	default: { ReportUnallowedPath(); return ""; }
	}
}


void iValue::Delete()
{
	if(--ReferenceCount <= 0) delete this;
}


//
//  BL value
//
iValue* iValue::New(iScript *w, const iString &n, ValueMethod m, bool v, bool *p, bool (*f)(iScript*))
{
	return (iValue*)new iValue_BL(w,n,m,v,p,f);
}


bool iValue_BL::SetMethod(ValueMethod m)
{
	switch (m)
	{
	case _ByValue:    { Method = m; return true; }
	case _ByPointer:  { if(Pointer != 0) { Method = m; return true; } else return false; }
	case _ByFunction: { if(Function != 0) { Method = m; return true; } else return false; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_BL::AssignValue(Assignment r, iValue *v)
{
	bool b;
	if(v->GetValue(b)) return this->AssignValue(r,b); else return false;
}


bool iValue_BL::AssignValue(Assignment NotUsed(r), bool b)
{
	Value = b; 
	return true; 
}


bool iValue_BL::GetValue(void *p) const
{
	if(p == 0) return false;
	switch (Method)
	{
	case _ByValue:    { *(bool *)p = Value; return true; }
	case _ByPointer:  { if(Pointer != 0) { *(bool *)p = *Pointer; return true; } else return false; }
	case _ByFunction: { *(bool *)p = Function(Script); return true; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_BL::GetValue(bool &v) const
{
	return this->GetValue((void *)&v);
}


//
//  IS value
//
iValue* iValue::New(iScript *w, const iString &n, ValueMethod m, int v, int *p, int (*f)(iScript*))
{
	return (iValue*)new iValue_IS(w,n,m,v,p,f);
}


bool iValue_IS::SetMethod(ValueMethod m)
{
	switch (m)
	{
	case _ByValue:    { Method = m; return true; }
	case _ByPointer:  { if(Pointer != 0) { Method = m; return true; } else return false; }
	case _ByFunction: { if(Function != 0) { Method = m; return true; } else return false; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_IS::AssignValue(Assignment r, iValue *v)
{
	int i;
	if(v->GetValue(i)) return this->AssignValue(r,i); else return false;
}


bool iValue_IS::AssignValue(Assignment r, int i)
{
	ASSIGN_VALUE(Value,r,i); 
	return true; 
}


bool iValue_IS::AssignValue(Assignment r, double f)
{
	ASSIGN_VALUE(Value,r,(int)f); 
	return true; 
}


bool iValue_IS::GetValue(void *p) const
{
	if(p == 0) return false;
	switch (Method)
	{
	case _ByValue:    { *(int *)p = Value; return true; }
	case _ByPointer:  { if(Pointer != 0) { *(int *)p = *Pointer; return true; } else return false; }
	case _ByFunction: { *(int *)p = Function(Script); return true; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_IS::GetValue(bool &v) const
{
	int i;
	if(!this->GetValue(i)) return false; else 
	{ 
		if(i==0 || i==1) 
		{
			v = (i==1);
			return true; 
		}
		else return false;
	}
}


bool iValue_IS::GetValue(int &v) const
{
	return this->GetValue((void *)&v);
}


bool iValue_IS::GetValue(float &v) const
{
	int i;
	if(!this->GetValue(i)) return false; else { v = (float)i; return true; }
}


bool iValue_IS::GetValue(double &v) const
{
	int i;
	if(!this->GetValue(i)) return false; else { v = (double)i; return true; }
}


//
//  RS value
//
iValue* iValue::New(iScript *w, const iString &n, ValueMethod m, double v, double *p, double (*f)(iScript*))
{
	return (iValue *)new iValue_RS(w,n,m,v,p,f);
}


bool iValue_RS::SetMethod(ValueMethod m)
{
	switch (m)
	{
	case _ByValue:    { Method = m; return true; }
	case _ByPointer:  { if(Pointer != 0) { Method = m; return true; } else return false; }
	case _ByFunction: { if(Function != 0) { Method = m; return true; } else return false; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_RS::AssignValue(Assignment r, iValue *v)
{
	double f;
	if(v->GetValue(f)) return this->AssignValue(r,f); else return false;
}


bool iValue_RS::AssignValue(Assignment r, int i)
{
	double f = (double)i;
	ASSIGN_VALUE(Value,r,f); 
	return true; 
}


bool iValue_RS::AssignValue(Assignment r, double f)
{
	ASSIGN_VALUE(Value,r,f); 
	return true; 
}


bool iValue_RS::GetValue(void *p) const
{
	if(p == 0) return false;
	switch (Method)
	{
	case _ByValue:    { *(double *)p = Value; return true; }
	case _ByPointer:  { if(Pointer != 0) { *(double *)p = *Pointer; return true; } else return false; }
	case _ByFunction: { *(double *)p = Function(Script); return true; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_RS::GetValue(bool &v) const
{
	int i;
	double f;
	if(!this->GetValue(f)) return false; else 
	{ 
		i = round(f); 
		if(i==0 || i==1) 
		{
			v = (i==1);
			return true; 
		}
		else return false;
	}
}


bool iValue_RS::GetValue(int &v) const
{
	double f;
	if(!this->GetValue(f)) return false; else { v = round(f); return true; }
}


bool iValue_RS::GetValue(double &v) const
{
	return this->GetValue((void *)&v);
}


bool iValue_RS::GetValue(float &v) const
{
	double v1;
	if(this->GetValue((void *)&v1))
	{
		v = (float)v1;
		return true;
	}
	else return false;
}


//
//  FP value
//
iValue* iValue::New(iScript *w, const iString &n, ValueMethod m, double* v, double* *p, double* (*f)(iScript*))
{
	return (iValue *)new iValue_RP(w,n,m,v,p,f);
}


bool iValue_RP::SetMethod(ValueMethod m)
{
	switch (m)
	{
	case _ByValue:    { Method = m; return true; }
	case _ByPointer:  { if(Pointer != 0) { Method = m; return true; } else return false; }
	case _ByFunction: { if(Function != 0) { Method = m; return true; } else return false; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_RP::AssignValue(Assignment r, iValue *v)
{
	if(v->GetType()==_Value_RP && r==_Absolute) 
	{ 
		this->Value = ((iValue_RP *)v)->Value; 
		return true; 
	} 
	else return false;
}


bool iValue_RP::GetValue(void *p) const
{
	if(p == 0) return false;
	switch (Method)
	{
	case _ByValue:    { *(double* *)p = Value; return true; }
	case _ByPointer:  { if(Pointer != 0) { *(double* *)p = *Pointer; return true; } else return false; }
	case _ByFunction: { *(double* *)p = Function(Script); return true; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_RP::GetValue(double* &v) const
{
	return this->GetValue((void *)&v);
}


//
//  ST value
//
iValue* iValue::New(iScript *w, const iString &n, ValueMethod m, const iString &v, iString *p, iString (*f)(iScript*), bool q)
{
	return (iValue *)new iValue_ST(w,n,m,v,p,f,q);
}


bool iValue_ST::SetMethod(ValueMethod m)
{
	switch (m)
	{
	case _ByValue:    { Method = m; return true; }
	case _ByPointer:  { if(Pointer != 0) { Method = m; return true; } else return false; }
	case _ByFunction: { if(Function != 0) { Method = m; return true; } else return false; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_ST::AssignValue(Assignment r, iValue *v)
{
	if(v->GetType()==_Value_ST && r==_Absolute) 
	{ 
		this->Value = ((iValue_ST *)v)->Value; 
		return true; 
	} 
	else return false;
}


bool iValue_ST::GetValue(void *p) const
{
	if(p == 0) return false;
	switch (Method)
	{
	case _ByValue:    { *(iString *)p = Value; return true; }
	case _ByPointer:  { if(Pointer != 0) { *(iString *)p = *Pointer; return true; } else return false; }
	case _ByFunction: { *(iString *)p = Function(Script); return true; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_ST::GetValue(iString &v) const
{
	return this->GetValue((void *)&v);
}

//
//  VA value
//
iValue* iValue::New(iScript *w, const iString &n, ValueMethod m, int s, iValue* i, iValue **a, iValue* (*f)(iScript*,int))
{
	return (iValue *)new iValue_VA(w,n,m,s,i,a,f);
}


bool iValue_VA::SetMethod(ValueMethod m)
{
	switch (m)
	{
	case _ByValue:    { Method = m; return true; }
	case _ByPointer:  { return false; }
	case _ByFunction: { if(Function != 0) { Method = m; return true; } else return false; }
	default: { ReportUnallowedPath(); return false; }
	}
}


void iValue_VA::SetIndex(iValue *i)
{
	if(Index != 0) Index->Delete();
	Index = i;
	Index->Register();
}


iValue* iValue_VA::GetComponent(iValue *v) const
{
	int i;
	if(v->GetType()!=_Value_IS || !v->GetValue(i)) return 0; else return this->GetComponent(i);
}


iValue* iValue_VA::GetComponent(int i) const
{
	if(Size>0 && (i<1 || i>Size)) return 0;

	switch (Method)
	{
	case _ByValue:    { if(Array != 0) return Array[i-1]; else return 0; }
	case _ByPointer:  { return 0; }
	case _ByFunction: { return Function(Script,i-1); }
	default: { ReportUnallowedPath(); return 0; }
	}
}


bool iValue_VA::AssignValue(Assignment r, iValue *v)
{
	int j = 0;
	switch(v->GetType())
	{
	case _Value_VA:
		{
			if(r==_Absolute && this->Array!=0 && ((iValue_VA *)v)->Array!=0 && this->Size==((iValue_VA *)v)->Size)
			{
				for(j=0; j<this->Size; j++)
				{
					if(this->Array[j]!=0 && ((iValue_VA *)v)->Array[j]!=0) this->Array[j]->AssignValue(r,((iValue_VA *)v)->Array[j]); 
				}
				return true;
			}
			else return false;
		}
	case _Value_BL:
	case _Value_IS:
	case _Value_RS:
	case _Value_RP:
		{
			iValue *val;
			if((val=this->GetComponent(Index)) == 0) return false;
			return val->AssignValue(r,v);
		}
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_VA::GetValue(void *p) const
{
	int i;
	//
	//  Get array index
	//
	if(p==0 || Index==0 || Index->GetType()!=_Value_IS || !Index->GetValue((void *)&i)) return false;
	if(Size>0 && (i<=0 || i>Size)) return false;  //check the Size if it is positive

	switch (Method)
	{
	case _ByValue:    { if(Array!=0 && Array[i-1]!=0) { *(iValue **)p = Array[i-1]; return true; } else return false; }
	case _ByPointer:  { return false; }
	case _ByFunction: { *(iValue **)p = Function(Script,i-1); return true; }
	default: { ReportUnallowedPath(); return false; }
	}
}


bool iValue_VA::GetValue(bool &v) const
{
	iValue *p = this->GetComponent(Index);
	if(p != 0) return p->GetValue(v); else return false;
}


bool iValue_VA::GetValue(int &v) const
{
	iValue *p = this->GetComponent(Index);
	if(p != 0) return p->GetValue(v); else return false;
}


bool iValue_VA::GetValue(double &v) const
{
	iValue *p = this->GetComponent(Index);
	if(p != 0) return p->GetValue(v); else return false;
}


bool iValue_VA::GetValue(int n, int *p) const
{
	int i;
	bool ok;
	int *w;
	iValue *v = 0;
	//
	//  Get array index
	//
	if(p==0 || Index==0 || Index->GetType()!=_Value_IS) return false;
	if(Index->GetValue(i))
	{
		if(Size>0 && (i<=0 || i>Size || n>Size)) return false;  //check the Size if it is positive
	}
	//
	//  Check what we got: an array of values, or a pointer array value
	//
	if((v=this->GetComponent(Index)) == 0) return false;

	if(iValue::AreTypesCompatible(_Value_IS,v))
	{
		w = new int[n]; IERROR_ASSERT_NULL_POINTER(w);
		ok = true;
		for(i=0; i<n && ok; i++) 
		{
			v = this->GetComponent(i+1);
			if(v != 0)
			{
				ok = ok & v->GetValue(w[i]);
			}
			else ok = false;
		}
		if(ok) memcpy(p,w,n*sizeof(int));
		delete [] w;
	}
	else ok = false;

	return ok;
}


bool iValue_VA::GetValue(int n, double *p) const
{
	int i;
	bool ok;
	double *w;
	iValue *v = 0;
	//
	//  Get array index
	//
	if(p==0 || Index==0 || Index->GetType()!=_Value_IS) return false;
	if(Index->GetValue(i))
	{
		if(Size>0 && (i<=0 || i>Size || n>Size)) return false;  //check the Size if it is positive
	}
	//
	//  Check what we got: an array of values, or a pointer array value
	//
	if((v=this->GetComponent(Index)) == 0) return false;

	if(iValue::AreTypesCompatible(_Value_RS,v))
	{
		w = new double[n]; IERROR_ASSERT_NULL_POINTER(w);
		ok = true;
		for(i=0; i<n && ok; i++) 
		{
			v = this->GetComponent(i+1);
			if(v != 0)
			{
				ok = ok & v->GetValue(w[i]);
			}
			else ok = false;
		}
		if(ok) memcpy(p,w,n*sizeof(double));
		delete [] w;
	}
	else
	{
		if(v->GetType() == _Value_RP)  // special case
		{
			ok = v->GetValue((void *)&w);
			if(ok) memcpy(p,w,n*sizeof(double));  
		}
		else ok = false;
	}

	return ok;
}


void iValue_VA::Delete()
{
	int i;
	if(ReferenceCount <= 1)
	{
		if(Array != 0)
		{
			for(i=0; i<Size; i++) if(Array[i] != 0) Array[i]->Delete();
		}
		delete [] Array;
		if(Index != 0) Index->Delete();
	}
	iValue::Delete();
}

//
//***************************************************************
//
//  iOperation functions
//
//***************************************************************
//

iOperation* iOperation::New(iOperation *o0)
{
	switch(o0->Type)
	{
	case _Operation_DC_SV: 
		{
			iOperation_DC_SV *o = (iOperation_DC_SV*)o0;
			return iOperation::New(o->Script,o->Prefix,o->Command,o->OutputCode,o->StatementPrefix,o->Relative,iValue::New(o->Value),o->UserType);
		}
	case _Operation_FC_LB: 
		{
			iOperation_FC_LB *o = (iOperation_FC_LB*)o0;
			return iOperation::New(o->Script,o->Prefix,o->Command,o->OutputCode,o->Point,iValue::New(o->Index),iValue::New(o->Count),o->Class,o->UserType);
		}
	case _Operation_SF_V0: 
		{
			iOperation_SF_V0 *o = (iOperation_SF_V0*)o0;
			return iOperation::New(o->Script,o->Prefix,o->Command,o->OutputCode,o->Function,o->UserType);
		}
	case _Operation_SF_V1: 
		{
			iOperation_SF_V1 *o = (iOperation_SF_V1*)o0;
			return iOperation::New(o->Script,o->Prefix,o->Command,o->OutputCode,o->Relative,iValue::New(o->Value),o->Function,o->UserType);
		}
	case _Operation_AF_V1: 
		{
			iOperation_AF_V1 *o = (iOperation_AF_V1*)o0;
			return iOperation::New(o->Script,o->Prefix,o->Command,o->OutputCode,iValue::New(o->Index),o->Relative,iValue::New(o->Value),o->Function,o->UserType);
		}
	default: { ReportUnallowedPath(); return 0; }
	}
}


iOperation::iOperation(iScript *w, const iString &p, const iString &s, int o, OperationType t, unsigned char u)
{ 
	Script = w; 
	Prefix = p; 
	Command = s; 
	OutputCode = o; 
	Type = t; 
	UserType = u; 
	LineInScript = -1;
}


bool iOperation::AssignVariableValue(const iString &n, iValue::Assignment r, iValue *v, iValue *i)
{
	if(Script != 0) return Script->AssignVariableValue(n,r,v,i); else return false;
}


//
// Type DC_SV operation creator & executor
//
iOperation* iOperation::New(iScript *w, const iString &p, const iString &c, int o, const iString &sp, iValue::Assignment r, iValue *v, unsigned char u)
{
	return (iOperation *)new iOperation_DC_SV(w,p,c,o,sp,r,v,u);
}


bool iOperation_DC_SV::Exec()
{
	return true;
}


//
// Type FC_LB operation creator & executor
//
iOperation* iOperation::New(iScript *w, const iString &p, const iString &c, int o, EntryPoint q, iValue* i, iValue *n, int l, unsigned char u)
{
	return (iOperation *)new iOperation_FC_LB(w,p,c,o,q,i,n,l,u);
}


bool iOperation_FC_LB::Exec()
{
	bool ok = false;
	
	intCount = 0;
	
	if(Point & iOperation::_Exit)
	{
		Gate = iOperation::_Open;
		Done = true;
	}

	if(Point & iOperation::_Entry)
	{
		if(Count != 0)
		{
			switch(Count->GetType())
			{
			case iValue::_Value_BL:
				{
					//
					//  This is a branch
					//
					if(!Count->GetValue(ok)) if(Script != 0) Script->GetErrorStatus()->Set("Invalid value type in statement: "+this->Prefix+" "+this->Command);
					if(ok) Gate = iOperation::_Open; else Gate = iOperation::_Close;
					Done = true;
					break;
				}
			case iValue::_Value_IS:
				{
					//
					//  This is a loop
					//
					if(Done) 
					{
						intIndex = 0;
						Done = false;
					}
					
					ok = Count->GetValue(intCount);
					if(!ok) if(Script != 0) Script->GetErrorStatus()->Set("Invalid interation count in statement: "+this->Prefix+" "+this->Command);
					
					intIndex++;
					
					if(Index!=0 && !Index->GetName().IsEmpty()) 
					{
						if(Index->AssignValue(iValue::_Absolute,intIndex)) { if(Script != 0) Script->SetVariableChanged(Index); } else if(Script != 0) Script->GetErrorStatus()->Set("Invalid interation index type in statement: "+this->Prefix+" "+this->Command);
					}
					
					if(intIndex >= intCount) Done = true;
					if(intIndex > intCount) 
					{
						//
						// Never execute this loop
						//
						Gate = iOperation::_Close;
					}
					else
					{
						//
						//
						// Execute this loop
						Gate = iOperation::_Open;
					}
					break;
				}
			default: if(Script != 0) Script->GetErrorStatus()->Set("Invalid value type in statement: "+this->Prefix+" "+this->Command);
			}
		}
		else
		{
			Gate = iOperation::_Flip;
			Done = true;
		}
	}

	return true;
}


void iOperation_FC_LB::SetValue(iValue* v)
{ 
	if(iValue::AreTypesCompatible(Index,v)) 
	{ 
		if(Index != 0) Index->Delete();
		Index = v; 
		Index->Register(); 
	}
}


//
// Type SF_V0 operation creator & executor
//
iOperation* iOperation::New(iScript *w, const iString &p, const iString &c, int o, void (*f)(iScript*), unsigned char u)
{
	return (iOperation *)new iOperation_SF_V0(w,p,c,o,f,u);
}


bool iOperation_SF_V0::Exec()
{
	Function(Script);
	return true;
}


//
// Type SF_V1 operation creator & executor
//
iOperation* iOperation::New(iScript *w, const iString &p, const iString &c, int o, iValue::Assignment r, iValue *v, void (*f)(iScript*,iValue*,iValue::Assignment), unsigned char u)
{
	return (iOperation *)new iOperation_SF_V1(w,p,c,o,r,v,f,u);
}


bool iOperation_SF_V1::Exec()
{
	if(Function != 0) Function(Script,Value,Relative); else if(!this->AssignVariableValue(this->Command,Relative,Value)) if(Script != 0) Script->GetErrorStatus()->Set("Incompatible value type in statement: "+this->Prefix+" "+this->Command);
	return true;
}


void iOperation_SF_V1::SetValue(iValue* v)
{ 
	if(iValue::AreTypesCompatible(Value,v)) 
	{ 
		if(Value != 0) Value->Delete();
		Value = v; 
		Value->Register(); 
	}
}


//
// Type AF_V1 operation creator & executor
//
iOperation* iOperation::New(iScript *w, const iString &p, const iString &c, int o, iValue *i, iValue::Assignment r, iValue *v, void (*f)(iScript*,iValue*,iValue*,iValue::Assignment), unsigned char u)
{
	return (iOperation *)new iOperation_AF_V1(w,p,c,o,i,r,v,f,u);
}


bool iOperation_AF_V1::Exec()
{
	if(Function != 0) Function(Script,Index,Value,Relative); else if(!this->AssignVariableValue(this->Command,Relative,Value,Index)) if(Script != 0) Script->GetErrorStatus()->Set("Incompatible type in statement: "+this->Prefix+" "+this->Command);
	return true;
}


void iOperation_AF_V1::SetValue(iValue* v)
{ 
	if(iValue::AreTypesCompatible(Value,v)) 
	{ 
		if(Value != 0) Value->Delete();
		Value = v; 
		Value->Register(); 
	}
}


void iOperation_DC_SV::Delete()
{
	if(Value != 0) Value->Delete();
	iOperation::Delete();
}


void iOperation_FC_LB::Delete()
{
	if(Index != 0) Index->Delete();
	if(Count != 0) Count->Delete();
	iOperation::Delete();
}


void iOperation_SF_V1::Delete()
{
	if(Value != 0) Value->Delete();
	iOperation::Delete();
}


void iOperation_AF_V1::Delete()
{
	if(Index != 0) Index->Delete();
	if(Value != 0) Value->Delete();
	iOperation::Delete();
}


//
//***************************************************************
//
//  iScript functions
//
//***************************************************************
//
int iScript::mCounter = 0;

iScript::iScript(iScript *parent) : mAliasWords(AliasWord()), mPrefixWords(""), mDummyWords(""), mParameterWords(0), mVariable(0), mCommand(0), mCode(0), mCompilerStack(StackEntry()), mObservers(0)
{
	mParent = parent;
	mAllowChildAccess = false;  //  children can only access my vars if this is true

	mActive = true;

	mCaseSensitive = false;
	mArraysAreReal = false;

	mNumCommandWords = 0;
	mCodeEntryPoint = 0;

	mCompilerStackEntryPoint = -1;

	mLastChangedVariable = -2;  // -2 means no update

	mActive = true;
	mActivator = 0;
	mBlock = false;
	
	mText = "";
	mCurLine = mNextLine = mNumLines = 0;
	mCompileReturnCode = mExecuteReturnCode = -1;
	mEntryPointIteration = -1;
	mEntryPointIterationCount = 0;

	mParser = new iExpressionParser; IERROR_ASSERT_NULL_POINTER(mParser);

	//
	//  Standard parameters
	//
	this->CreateParameterWord(iValue::New(this,"false",iValue::_ByValue,false));
	this->CreateParameterWord(iValue::New(this,"true",iValue::_ByValue,true));

	//
	//  Special characters - can be overwritten in a child class
	//
	mCommentString = "#";
	mLEString = "<=";
	mGEString = ">=";
	mEQString = "==";
	mNEString = "!=";
	mANDString = "&&";
	mORString = "||";

	//
	// Continuation symbols - can be overwritten in a child class 
	// (may be useful for creatting REALLY complex syntax!)
	//
	mHeadOfLineContinuationString = "";
	mTailOfLineContinuationString = "\\";  // standard C-type continuation line
	mAppendSeparator = "";  // this separates strings when they are appended - useful for complex syntax

	mVariableLastSize = mCommandLastSize = 0;

	mCounter++;
}


iScript::~iScript()
{
	int i;

	this->Reset(true);

	for(i=0; i<mParameterWords.Size(); i++) mParameterWords[i]->Delete();
	for(i=0; i<mNumCommandWords; i++) mCommand[i]->Delete();

	delete mParser;

	while(mObservers.Size() > 0) mObservers.RemoveLast()->SetScript(0);

	mCounter--;

#ifdef I_CHECK1
	if(mCounter==0 && iValue::counterCreate != iValue::counterDestroy)
	{
		int i;
		for(i=0; i<999; i++) if (iValue::cval[i] != 0)
		{
			IERROR_REPORT_ERROR("Uncollected values.");
		}
	}
#endif
}


//
//  Observers
//
void iScript::AddObserver(iScriptObserver *obs)
{
	if(obs != 0) mObservers.AddUnique(obs);
}


void iScript::RemoveObserver(iScriptObserver *obs)
{
	if(obs != 0) mObservers.Remove(obs);
}


//
//  Script Creation
//
void iScript::CreateAliasWord(const iString &alias, const iString &word)
{
	AliasWord tmp;
	tmp.Alias = alias;
	tmp.Word = word;
	mAliasWords.Add(tmp);
}


void iScript::CreateDummyWord(const iString &v)
{
	if(!v.IsEmpty()) mDummyWords.Add(v);
}


void iScript::CreatePrefixWord(const iString &v)
{
	if(!v.IsEmpty()) mPrefixWords.Add(v);
}


void iScript::CreateCommandWord(iOperation *v)
{
	if(v != 0)
	{
		mCommand.Add(v);
		mNumCommandWords++;
		//
		//  Check if a new prefix is also created by this command
		//
		if(!v->Prefix.IsEmpty() && mPrefixWords.Find(v->Prefix)<0) this->CreatePrefixWord(v->Prefix);
	}

}


void iScript::CreateParameterWord(iValue *v)
{
	if(v != 0) mParameterWords.Add(v);
}

//
//  Public interface
//
void iScript::SetText(const iString &s)
{
	if(!mCaseSensitive) 
	{
		//
		//  Turn the script into lower case, but not the comments!
		//
		int i1 = 0, i2 = s.Find('"');
		mText.Clear();
		while(i2 != -1)
		{
			if(i2 > i1) mText += s.Part(i1,i2-i1).Lower();
			i1 = i2;
			i2 = s.Find('"',i2+1);
			if(i2 != -1) 
			{
				mText += s.Part(i1,i2-i1+1);  
				i1 = i2 + 1;
				i2 = s.Find('"',i1);
			}
		}
		mText += s.Part(i1).Lower();
	}
	else 
	{
		mText = s;
	}
	if(mText[(int)mText.Length()-1] != '\n') mText += "\n";
	mNumLines = mText.Contains('\n');
	//
	//  Do partial reset
	//
	this->Reset(false);
}
	
	
bool iScript::IsCommandWord(const iString &s) const
{
	for(int i=0; i<mNumCommandWords; i++) if(mCommand[i]->Command == s) return true;
	return false;
}


bool iScript::IsParameterWord(const iString &s) const
{
	for(int i=0; i<mParameterWords.Size(); i++) if(mParameterWords[i]->Name == s) return true;
	return false;
}


bool iScript::IsDummyWord(const iString &s) const
{
	return (mDummyWords.Find(s) != -1);
}


bool iScript::IsPrefixWord(const iString &s) const
{
	return (mPrefixWords.Find(s) != -1);
}


bool iScript::IsReservedWord(const iString &s) const
{
	return this->IsCommandWord(s) || this->IsParameterWord(s) || this->IsPrefixWord(s) || this->IsDummyWord(s);
}


const iString &iScript::GetReservedWord(int i) const
{
	static iString none = "";
	if(i < 0) return none;
	if(i < mPrefixWords.Size()) return mPrefixWords[i];
	i -= mPrefixWords.Size();
	if(i < mNumCommandWords) return mCommand[i]->Command;
	i -= mNumCommandWords;
	if(i < mParameterWords.Size()) return mParameterWords[i]->Name;
	i -= mParameterWords.Size();
	if(i < mDummyWords.Size()) return mDummyWords[i];
	return none;
}


int iScript::GetUserTypeForReservedWord(int i) const
{
	i -= mPrefixWords.Size();
	if(i>=0 && i<mNumCommandWords) return mCommand[i]->UserType; else return 0;
}

//
//  Getting information about variables
//
bool iScript::IsVariableWord(const iString &s) const
{
	for(int i=0; i<mVariable.Size(); i++) if(mVariable[i]->Name == s) return true;
	return false;
}


//
//  Get the line of code
//
iString iScript::GetText(int i)const
{
	if(i>=0 && i<mNumLines) return mText.Section("\n",i,i); else return "";
}

//
//  Remove the pseudocode;
//
void iScript::Reset(bool complete)
{
	if(complete)
	{
		//
		//  Delete all variables and their associated commands
		//
		mVariableLastSize = mCommandLastSize = 0;
	}
	//
	//  Delete the code
	//
	while(mCode.Size() > 0) mCode.RemoveLast()->Delete();
	mCodeEntryPoint = 0;
	//
	//  Delete the stack
	//
	mCompilerStackEntryPoint = -1;
	//
	//  Delete new variables and their respective command too
	//
	while(mVariable.Size() > mVariableLastSize) mVariable.RemoveLast()->Delete();
	while(mCommand.Size() > mCommandLastSize+mNumCommandWords) mCommand.RemoveLast()->Delete();

	mCurLine = mNextLine = 0;
	this->SkipEmptyLines();

	mActive = true;
	mActivator = 0;
	mBlock = false;
}


void iScript::SkipEmptyLines()
{
	iString s;
	//
	//  Skip lines if they are empty
	//
	while(mCurLine<mNumLines)
	{
		s = this->GetText(mCurLine);
		s.ReduceWhiteSpace();
		if(s.IsEmpty() || (s.Length()==1 && s[0]==' ') || s[0]=='#') mCurLine++; else break;
	}
}


bool iScript::AssignVariableValue(const iString &n, iValue::Assignment r, iValue *v, iValue *i)
{
	int j;	
	for(j=0; j<mVariable.Size(); j++) if(n == mVariable[j]->Name)
	{
		//
		//  Is it an array?
		//
		if(i!=0 && mVariable[j]->Type==iValue::_Value_VA) 
		{
			((iValue_VA*)mVariable[j])->SetIndex(i);
		}
		if(mVariable[j]->AssignValue(r,v)) { mLastChangedVariable = j; return true; } 
		break;
	}
	return false;
}


void iScript::SetVariableChanged(iValue *v)
{
	int j;	
	for(j=0; j<mVariable.Size(); j++) if(v->Name == mVariable[j]->Name)
	{
		mLastChangedVariable = j; 
		break;
	}
}


void iScript::SetErrorLocation(int loc, int add)
{
	if(loc >= 0)
	{
		mErrorLocation = loc + (add<0)?0:add;
	}
	else
	{
		mErrorLocation = 0;
	}
}


void iScript::Run(const iString &s, bool compileOnly, bool reset, int firstLine, int lastLine)
{
	this->SetText(s);
	this->Run(compileOnly,reset,firstLine,lastLine);
}


void iScript::Run(bool compileOnly, bool reset, int NotUsed(firstLine), int NotUsed(lastLine))
{
	this->RunStart(reset);
	while(this->RunOneLine(compileOnly) && this->GetErrorStatus()->NoError());
	this->RunStop(compileOnly);
}


void iScript::RunStart(bool reset)
{
	int i;

	this->Reset(reset);
	this->SkipEmptyLines();

	for(i=0; i<mObservers.Size(); i++) mObservers[i]->OnScriptStart();
}


void iScript::RunStop(bool compileOnly)
{
	int i;

	//
	//  Missing closing Operation::Exit?
	//
	if(this->GetErrorStatus()->NoError() && this->AreThereOpenEntryPoints())
	{
		this->GetErrorStatus()->Set("Missing loop or branch closing statement(s)");
	}

	if(!compileOnly && this->GetErrorStatus()->NoError())
	{
		//
		//  Actualy save the variables are corresponding commands
		//
		mVariableLastSize = mVariable.Size();
		mCommandLastSize = mCommand.Size() - mNumCommandWords;
	}

	for(i=0; i<mObservers.Size(); i++) mObservers[i]->OnScriptStop(this->GetErrorStatus()->GetMessage());
}


//
//  Interrupts
//
bool iScript::CheckAbort(int cur, int num, int level) const
{
	int i;
	bool abort = false;
	for(i=0; !abort && i<mObservers.Size(); i++) abort = abort || mObservers[i]->OnScriptCheckAbort(cur,num,level);
	return abort;
}


//
// Compile and execute one line of script. Returns true if the script is not finished, and false
// if it is done.
//
bool iScript::RunOneLine(bool compileOnly)
{
	int i;
	bool thisActive;

	this->GetErrorStatus()->Clear();
	mErrorLocation = -1;

	if(mBlock)
	{
		this->GetErrorStatus()->Set("Attempting to call script recursively.");
		return false; //  do not multi-thread me!!!
	}
	mBlock = true;  

	do
	{
		this->SkipEmptyLines();
		if(mCurLine == mNumLines) // Script is done
		{
			mBlock = false;
			return false;
		}

		mNextLine = mCurLine;
		thisActive = mActive;
		if(thisActive)
		{
			for(i=0; i<mObservers.Size(); i++) mObservers[i]->OnScriptBeginLine(mCurLine,this->GetText(mCurLine));
		}

		mParser->SetIgnoreDivisionByZero(compileOnly);
		this->CompileOneLine();
		if(this->GetErrorStatus()->IsError())
		{
			mBlock = false; 
			return false;
		}

		if(compileOnly) 
		{
			mCodeEntryPoint++; 
			mBlock = false;
			mCurLine = mNextLine;
			return true;
		}
		
		mLastChangedVariable = -2;  // -2 means no update
		
		this->ExecuteOneLine();
		if(this->GetErrorStatus()->IsError())
		{
			mBlock = false;
			return false;
		}
		
		if(mActive || thisActive) if(this->CheckAbort(mEntryPointIteration,mEntryPointIterationCount))
		{
			this->GetErrorStatus()->SetAbort(-1);
			mBlock = false;
			return false;
		}

		mCurLine = mNextLine;
	}
	while(!mActive && mNextLine<mNumLines);

	this->SkipEmptyLines();
	for(i=0; i<mObservers.Size(); i++) mObservers[i]->OnScriptEndLine(mCurLine,this->GetText(mCurLine));

	mBlock = false;
	return true;
}


//
//  execute one command in the script
//
void iScript::ExecuteOneLine()
{
	int l;
	bool checkGate = false;

	if(mCode.Size() == 0) 
	{
		this->GetErrorStatus()->Set("Attempting execute a script which has not been compiled.");
		return;
	}

	if(mCodeEntryPoint == mCode.Size())
	{
		return;
	}

	mEntryPointIteration = -1;
	mEntryPointIterationCount = 0;

	if(mCode[mCodeEntryPoint] == 0) // empty statement
	{
		mCodeEntryPoint++;
		return;
	}

	//
	//  Because the branch can activate or dis-activate the script, it must be executed first
	//
	if(mCode[mCodeEntryPoint]->Type == iOperation::_Operation_FC_LB)
	{
		iOperation_FC_LB *op = (iOperation_FC_LB *)mCode[mCodeEntryPoint];
		op->Exec();

		//
		//  Update the compiler stack if needed. 
		//
		if(mCompilerStackEntryPoint >= mCompilerStack.Last())
		{
			mCompilerStack.Add(StackEntry());
		}

		if(op->Point & iOperation::_Exit)
		{
			//
			// Give an error if incorrect nesting detected.
			//
			if(mCompilerStackEntryPoint<0 || mCompilerStackEntryPoint>=mCompilerStack.Size() || mCompilerStack[mCompilerStackEntryPoint].Class != op->Class || mCompilerStack[mCompilerStackEntryPoint].Type != op->Type)
			{
				this->GetErrorStatus()->Set("Incorrectly nested operator " + op->Command + ".");
				return;
			}
			//
			//  Operation type FC_LB(Point=iOperation::_Exit) passes control to the last operation type FC_LB(Point=Entry) 
			//  of the same class if that operation is not done. Use the stack to find a match.
			//
			l = mCompilerStack[mCompilerStackEntryPoint].Line;
			if(mCode[l] == 0)
			{
				this->GetErrorStatus()->Set("Script code is corrupted.");
				return;
			}
			mCompilerStackEntryPoint--;
			
			if(mActive || l==mActivator) // I am a match for the activator, so execute
			{
				checkGate = true;
				if(!((iOperation_FC_LB *)mCode[l])->Done) 
				{ 
					mCodeEntryPoint = l; 
					mNextLine = mCode[mCodeEntryPoint]->LineInScript;
					mCodeEntryPoint--; // it will be ++ed at the end of this operator
				} 
				else 
				{
					mEntryPointIteration = -1;
					mEntryPointIterationCount = ((iOperation_FC_LB *)mCode[l])->intCount;
					if(this->CheckAbort(mEntryPointIteration,mEntryPointIterationCount))
					{
						this->GetErrorStatus()->SetAbort(-1);
						return;
					}
				}
			}
		}

		if(op->Point & iOperation::_Entry)
		{
			mCompilerStackEntryPoint++;
			mCompilerStack[mCompilerStackEntryPoint].Type = op->Type;
			mCompilerStack[mCompilerStackEntryPoint].Class = op->Class;
			mCompilerStack[mCompilerStackEntryPoint].Line = mCodeEntryPoint;
			mEntryPointIterationCount = op->intCount;
			mEntryPointIteration = op->intIndex;
			if(mActive) checkGate = true;
		}
		
		if(checkGate)
		{
			switch(op->Gate)
			{
			case iOperation::_Open: { mActive  = true; break; }
			case iOperation::_Close: { mActive  = false; mActivator = mCodeEntryPoint; break; }
			case iOperation::_Flip: { mActive  = !mActive; if(!mActive) mActivator = mCodeEntryPoint; break; }
			case iOperation::_Skip: { break; }
			default: ReportUnallowedPath();
			}
		}
		mCodeEntryPoint++;
		return;
	}

	if(!mActive) 
	{
		mCodeEntryPoint++;
		return;
	}

	switch(mCode[mCodeEntryPoint]->Type)
	{
	case iOperation::_Operation_DC_SV:
		{
			mLastChangedVariable = -1;  // -1 means update all - new are declared
		}
	case iOperation::_Operation_SF_V0:
	case iOperation::_Operation_SF_V1:
	case iOperation::_Operation_AF_V1:
		{
			if(mCode[mCodeEntryPoint]->Exec()) mCodeEntryPoint++;
			break;
		}
	default: 
		{
			ReportUnallowedPath();
			this->GetErrorStatus()->Set("Invalid operation.");
		}
	}
}


//
//  Compile the line of the script.
//
void iScript::CompileOneLine()
{
	iString commandText, commandTextOriginal;
	iString pw, cw, ac, pv;
	int i, n, ioff, comind;
	iValue::Assignment valueRelative = iValue::_Absolute;
	bool ok, isArray = false;
	iValue *val = 0, *arrComponent = 0;

	commandTextOriginal = this->GetText(mNextLine++);

	//
	//  If the following lines are just continuations of this one, append them including mAppendSeparator
	//
	int headLen = mHeadOfLineContinuationString.Length();
	int tailLen = mTailOfLineContinuationString.Length();
	while(mNextLine < mNumLines)
	{
		pw = this->GetText(mNextLine);
		//
		//  The head string preceedes the tail one
		//
		if(headLen>0 && pw.Part(0,headLen)==mHeadOfLineContinuationString)
		{
			commandTextOriginal += " " + mAppendSeparator + pw.Part(headLen);
			mNextLine++;
			continue;
		}
		if(tailLen>0 && pw.Part(-tailLen)==mTailOfLineContinuationString)
		{
			commandTextOriginal += mAppendSeparator + pw.Part(0,pw.Length()-tailLen);
			mNextLine++;
			continue;
		}
		break;
	}

	commandText = commandTextOriginal;

	//
	//  Remove comments if they are present (but not inside a string)
	//
	i = commandText.Find(mCommentString);
	if(i>=0 && commandText.Part(0,i).Contains('"')%2==0) commandText = commandText.Part(0,i);

	//
	//  Replace comparison signs with single char equivalents
	//
	char tmp[2];
	tmp[1] = 0;
	tmp[0] = mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_LE); commandText.Replace(mLEString,tmp);
	tmp[0] = mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_GE); commandText.Replace(mGEString,tmp);
	tmp[0] = mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_EQ); commandText.Replace(mEQString,tmp);
	tmp[0] = mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_NE); commandText.Replace(mNEString,tmp);
	tmp[0] = mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_AND); commandText.Replace(mANDString,tmp);
	tmp[0] = mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_OR); commandText.Replace(mORString,tmp);

	//
	//  Remove dummy words - but not when they are parts of other words.
	//
	for(i=0; i<mDummyWords.Size(); i++) 
	{
		while((n=this->FindScriptWord(commandText,mDummyWords[i]))>-1) commandText.Replace(n,mDummyWords[i].Length()," ");
	}

	//
	//  Replace alias words - but not when they are parts of other words.
	//
	for(i=0; i<mAliasWords.Size(); i++) 
	{
		while((n=this->FindScriptWord(commandText,mAliasWords[i].Alias))>-1) commandText.Replace(n,mAliasWords[i].Alias.Length(),mAliasWords[i].Word);
	}

	//
	//  Remove '=' signs - they are just for aestetics. Remove extra white spaces if appeared.
	//
	
	while((i=commandText.Find('='))>=0 && commandText.Part(0,i).Contains('"')%2==0) commandText[i] = ' ';
	commandText.ReduceWhiteSpace();

	//
	//  No command in this line
	//
	if(commandText.IsEmpty() || (commandText.Length()==1 && commandText[0]==' ') || commandText[0]=='#')
	{
		return;
	}

	//
	//  Read the first word - it must be a command or a prefix word
	//
	ioff = 0;
	cw = commandText.Section(" ",ioff,ioff);

	//
	//  Find whether it is a prefix word - if yes, skip it
	//
	for(i=0; i<mPrefixWords.Size(); i++)
	{
		if(mPrefixWords[i] == cw) break;
	}
	if(i < mPrefixWords.Size())
	{
		pw = cw;
		ioff++;
	}
	else pw = "";

TEST:
	//
	//  Find whether it is a command word: read the word first
	//
	cw = commandText.Section(" ",ioff,ioff);

	//
	//  Look for the array component at the end in the form [N] where N is a iValue;
	//
	i = cw.Find('[');
	if(i >= 0)    // a square bracket found
	{
		ac = cw.Part(i);
		cw = cw.Part(0,i);
		if(ac.Length()<3 || ac[0]!='[' || ac[ac.Length()-1]!=']')
		{
			this->GetErrorStatus()->Set("Syntax error in array index.");
			this->SetErrorLocation(i,commandTextOriginal.Find(cw,0));
			return;
		}
		ac = ac.Part(1,ac.Length()-2);
		//
		//  Postpone turning ac into the iValue.
		//
	}

	//
	//  Search the list of command words - enforce the prefix match too.
	//
	for(i=0; i<mCommand.Size(); i++)
	{
		if(mCommand[i]->Command==cw && mCommand[i]->Prefix==pw) break;
	}
	if(i == mCommand.Size()) //  The word is not a valid command
	{
		if(!pw.IsEmpty())
		{
			//
			//  It may also happen that this is a command with no prefix word but with
			//  the command word that matches one of the prefix words. Since we look for
			//  the prefix work first, we misidentified this command - check it by
			//  removing the prefix word.
			//
			pw.Clear();
			ioff--;
			goto TEST;
		}
		this->GetErrorStatus()->Set("Syntax error - invalid command or undefined variable.");
		this->SetErrorLocation(0,commandTextOriginal.Find(cw,0));
		return;
	}
	comind = i;
	ioff++;

	//
	//  Read the value.
	//
	pv = commandText.Section(" ",ioff,ioff);

	//
	//  If the 'value' turned out to be + or - sign, the command was in the form of 
	//	"Set param += 3" - thus the value should be treated as relative. For '-' change the
	//  value sign as well. 
	//
	if(pv=="+" || pv=="-")
	{
		//
		//  First check if this is allowed
		//
		switch(mCommand[comind]->Type)
		{
		case iOperation::_Operation_DC_SV: 
		case iOperation::_Operation_FC_LB: 
		case iOperation::_Operation_SF_V0: 
			{ 
				ok = false; break; 
			}
		case iOperation::_Operation_SF_V1: 
			{ 
				ok = ((iOperation_SF_V1 *)mCommand[comind])->Relative & 1; break; 
			}
		case iOperation::_Operation_AF_V1: 
			{ 
				ok = ((iOperation_AF_V1 *)mCommand[comind])->Relative & 1; break; 
			}
		default:
			{
				ok = false;	break;
			}
		}
		if(!ok)
		{
			this->GetErrorStatus()->Set("Incrementing operation not allowed.");
			this->SetErrorLocation(0,commandTextOriginal.Find(pv,0));
			return;
		}
		//
		//  It is ok, proceed
		//
		valueRelative = iValue::_RelativeAdd;
		if(pv == "-") valueRelative = iValue::_RelativeInvertAdd;
		ioff++;
		pv = commandText.Section(" ",ioff); // read the rest of the line
	}
	//
	//  If the 'value' turned out to be * sign, the command was in the form of 
	//	"Set param *= 3" - thus the value should be treated as relative.
	//
	else if(pv == "*")
	{
		//
		//  First check if this is allowed
		//
		switch(mCommand[comind]->Type)
		{
		case iOperation::_Operation_DC_SV: 
		case iOperation::_Operation_FC_LB: 
		case iOperation::_Operation_SF_V0: 
			{ 
				ok = false; break; 
			}
		case iOperation::_Operation_SF_V1: 
			{ 
				ok = (((iOperation_SF_V1 *)mCommand[comind])->Relative & 2) != 0; break; 
			}
		case iOperation::_Operation_AF_V1: 
			{ 
				ok = (((iOperation_AF_V1 *)mCommand[comind])->Relative & 2) != 0; break; 
			}
		default:
			{
				ok = false;	break;
			}
		}
		if(!ok)
		{
			this->GetErrorStatus()->Set("Incrementing operation not allowed.");
			this->SetErrorLocation(0,commandTextOriginal.Find(pv,0));
			return;
		}
		//
		//  It is ok, proceed
		//
		valueRelative = iValue::_RelativeMult;
		ioff++;
		pv = commandText.Section(" ",ioff); // read the rest of the line
	}
	else pv = commandText.Section(" ",ioff); // read the rest of the line

	//
	//  Create the arrComponent iValue
	//
	if(!ac.IsEmpty())
	{
		arrComponent = this->TransformToValue(ac);
		isArray = (arrComponent != 0);

		if(!isArray)
		{
			this->GetErrorStatus()->Set("Invalid array index.");
			this->SetErrorLocation(commandTextOriginal.Find(cw,0),cw.Find('['));
			return;
		}
	}

	//
	//  Check that the array index corresponds to the array
	//
	switch(mCommand[comind]->Type)
	{
	case iOperation::_Operation_FC_LB: 
	case iOperation::_Operation_SF_V0: 
	case iOperation::_Operation_SF_V1: 
		{ 
			ok = !isArray; 
			if(!ok) this->GetErrorStatus()->Set("Variable is not an array.");
			break; 
		}
	case iOperation::_Operation_AF_V1: 
		{ 
			//
			//  Check that the whole array assignment is allowed
			//
			ok = ((iOperation_AF_V1*)mCommand[comind])->Value==0 || isArray || ((iOperation_AF_V1*)mCommand[comind])->Value->Type == iValue::_Value_VA; 
			if(!ok) this->GetErrorStatus()->Set("Whole array assignment is not allowed."); 
			break; 
		}
	case iOperation::_Operation_DC_SV: 
		{
			ok = true;	break;
		}
	default:
		{
			ok = false;	break;
		}
	}

	if(!ok)
	{
		if(arrComponent != 0) arrComponent->Delete();
		this->SetErrorLocation(commandTextOriginal.Find(cw,0));
		return;
	}

	//
	//  Get the value
	//
	switch(mCommand[comind]->Type)
	{
	case iOperation::_Operation_DC_SV:  // a declaration is executed at the compile time
		{ 
			//
			//  Remove commas if they are present - they are just for aestetics
			//
			commandText.Replace(","," ");
			commandText.ReduceWhiteSpace();

			pv = commandText.Section(" ",ioff,ioff); ioff++;
			if(pv.IsEmpty())
			{
				this->GetErrorStatus()->Set("Variable name must be specified.");
				ok = false; break;
			}

			ok = true;
			while(!pv.IsEmpty())
			{
				//
				// Check that the name is alpha-numeric
				//
				if(!IsScriptWordFirstLetter(pv[0])) ok = false;
				for(i=1; i<(int)pv.Length() && ok; i++) if(!IsScriptWordLetter(pv[i])) ok = false;
				if(!ok)
				{
					this->GetErrorStatus()->Set("Variable name must be alpha-numeric.");
					break;
				}

				//
				// Check that the name is unique
				//
				for(i=0; i<mCommand.Size() && ok; i++) if(pv == mCommand[i]->Command) ok = false;
				if(!ok)
				{
					this->GetErrorStatus()->Set("Variable name is not unique.");
					break;
				}
				
				//
				//  Is the value configured properly?
				//
				if(((iOperation_DC_SV *)mCommand[comind])->Value!=0 && ((iOperation_DC_SV *)mCommand[comind])->Value->Method != iValue::_ByValue)
				{
					this->GetErrorStatus()->Set("Script is incorrectly configured. Declared variables must have Method=iValue::_ByValue.");
					ok = false; break;
				}
				
				iValue *valtmp;
				//
				//   Is it array?
				//
				if(!isArray)
				{
					//
					//  Simple variable 
					//
					valtmp = iValue::New(((iOperation_DC_SV *)mCommand[comind])->Value);
				}
				else
				{
					if(!arrComponent->GetValue(n) || n<1 )
					{
						this->GetErrorStatus()->Set("Incorrect value for the array dimension.");
						ok = false; break;
					}

					//
					//  Create an array of iValues
					//
					iValue **arr = new iValue*[n]; IERROR_ASSERT_NULL_POINTER(arr);
					if(arr == 0)
					{
						this->GetErrorStatus()->Set("Not enough memory to create this array.");
						ok = false; break;
					}
					//
					//  Fill it in 
					//
					for(i=0; i<n && ok; i++)
					{
						arr[i] = iValue::New(((iOperation_DC_SV *)mCommand[comind])->Value);
						if(arr[i] == 0) ok = false;
					}
					//
					//  If unsuccessfull, destroy it.
					//
					if(!ok)
					{
						for(i=0; i<n; i++) if(arr[i] != 0) arr[i]->Delete();
						delete [] arr;  
						this->GetErrorStatus()->Set("Not enough memory to create this array.");
						break;
					}
					
					//
					//  Create an array-valued variable; iValue_VA takes ownership of everything
					//
					valtmp = iValue::New(this,"",iValue::_ByValue,n,iValue::New(this,"",iValue::_ByValue,1),arr);

				}

				if(valtmp != 0) 
				{
					//
					//  Assign the name to the variable
					//
					valtmp->Name = pv;
					//
					//  Add it to the variable list 
					//
					mVariable.Add(valtmp);
					//
					//  Find all command words with Command == $variable and the same value type and turn then into new statements
					//
					for(i=0; i<mNumCommandWords; i++)
					{
						if(mCommand[i]->Command=="$variable" && mCommand[i]->GetValue()!=0 && mCommand[i]->GetValue()->Type==valtmp->Type)
						{
							iOperation *tmp = iOperation::New(mCommand[i]);
							tmp->Command = pv;
							tmp->SetValue(valtmp);
							mCommand.Add(tmp);
						}
					}
				}
				else ok = false;

				pv = commandText.Section(" ",ioff,ioff); ioff++;
			}

			if(arrComponent != 0)
			{
				arrComponent->Delete(); // not needed any more
				arrComponent = 0;
			}

			break; 
		}
	case iOperation::_Operation_SF_V0: 
		{ 
			ok = true; break; 
		}
	case iOperation::_Operation_FC_LB: 
		{
			//
			//  If there is no count, there should be no value
			//
			if(((iOperation_FC_LB *)mCommand[comind])->Count == 0)
			{
				ok = pv.IsEmpty();
			}
			else
			{
				val = this->TransformToValue(pv);
				ok = (val != 0);
			}
			break;
		}
	case iOperation::_Operation_SF_V1:
		{
			//
			//  A special case is an assignment of the unquoted string - this allows to make very complex commands
			//  by outsourcing complete parsing to a function. Quote the string if it is unquoted.
			//
			if(((iOperation_SF_V1*)mCommand[comind])->Value!=0 && ((iOperation_SF_V1*)mCommand[comind])->Value->Type==iValue::_Value_ST && !((iValue_ST*)((iOperation_SF_V1*)mCommand[comind])->Value)->Quoted)
			{
				//
				//  Replace double quotes in the value string with single quotes (you task is to insure that there is no mix-up)
				//
				pv.Replace("\"","'");
				pv = "\"" + pv + "\"";
			}
			//
			//  No break here, we go on
			//
		}
	case iOperation::_Operation_AF_V1: 
		{ 
			//
			//  Convert to the iValue type
			//
			val = this->TransformToValue(pv);
			ok = (val != 0);
			break; 
		}
	default:
		{
			this->GetErrorStatus()->Set("Invalid value.");
			ok = false;	break;
		}
	}

	if(!ok)
	{
		if(arrComponent != 0) arrComponent->Delete();
		if(this->GetErrorStatus()->NoError()) this->GetErrorStatus()->Set("Unknown error.");
		this->SetErrorLocation(commandTextOriginal.Find(pv,0));
		return;
	}

	//
	//  Check that the value is of correct type
	//
	switch(mCommand[comind]->Type)
	{
	case iOperation::_Operation_FC_LB: 
		{ 
			if(((iOperation_FC_LB*)mCommand[comind])->Count != 0)
			{
				ok = iValue::AreTypesCompatible(((iOperation_FC_LB*)mCommand[comind])->Count,val);
			}
			break; 
		}
	case iOperation::_Operation_SF_V1: 
		{ 
			if(((iOperation_SF_V1*)mCommand[comind])->Value != 0)
			{
				ok = iValue::AreTypesCompatible(((iOperation_SF_V1*)mCommand[comind])->Value,val);
			}
			break; 
		}
	case iOperation::_Operation_AF_V1: 
		{ 
			if(((iOperation_AF_V1*)mCommand[comind])->Value != 0)
			{
				//
				//  Decide whether this is a full array assignment, or a single component.
				//  The possibilities:
				//  arr = arr			rhs = array
				//  arr[i] = arr[i]		rhs = scalar, but mCommandWords[comind] is declared with array iValue
				//  arr[i] = sca		rhs = scalar, but mCommandWords[comind] is declared with scalar iValue
				//
				if(arrComponent == 0)
				{
					ok = iValue::AreTypesCompatible(((iOperation_AF_V1*)mCommand[comind])->Value,val);
					arrComponent = iValue::New(((iOperation_AF_V1*)mCommand[comind])->Index);  
					arrComponent->AssignValue(iValue::_Absolute,1);  // cannot have zero component in the script command
				}
				else
				{
					if(((iOperation_AF_V1*)mCommand[comind])->Value->Type == iValue::_Value_VA)
					{
						//
						//  Full array assignment
						//
						iValue *valtmp = ((iValue_VA*)((iOperation_AF_V1*)mCommand[comind])->Value)->GetComponent(arrComponent);
						if(valtmp != 0) ok = iValue::AreTypesCompatible(valtmp,val); else ok = false;
					}
					else 
					{
						//
						//  Single component assignment to an array that cannot be assigned as a whole
						//  (was declared with a single-valued iValue)
						//
						ok = iValue::AreTypesCompatible(((iOperation_AF_V1*)mCommand[comind])->Value,val);
					}
				}
			}
			break; 
		}
	default:
		{
			ok = true; break;
		}
	}
	
	if(!ok)
	{
		if(arrComponent != 0) arrComponent->Delete();
		if(val != 0) val->Delete();
		this->GetErrorStatus()->Set("Invalid value type.");
		this->SetErrorLocation(commandTextOriginal.Find(pv,0));
		return;
	}

	//
	//  If the value is an array, check that it has the correct number of components
	//
	if(val!=0 && val->Type==iValue::_Value_VA)
	{
		int n = ((iValue_VA *)val)->Size;
		int s;

		switch(mCommand[comind]->Type)
		{
		case iOperation::_Operation_SF_V1: 
			{
				if(((iOperation_SF_V1*)mCommand[comind])->Value != 0)
				{
					s = ((iValue_VA *) ((iOperation_SF_V1*)mCommand[comind])->Value)->Size;
					ok = (n==0 || s==0 || n==s);
				}
				break; 
			}
		case iOperation::_Operation_AF_V1: 
			{
				if(((iOperation_AF_V1*)mCommand[comind])->Value != 0)
				{
					s = ((iValue_VA *) ((iOperation_AF_V1*)mCommand[comind])->Value)->Size;
					ok = (n==0 || s==0 || n==s);
				}
				break; 
			}
		default:
			{
				ok = true; break;
			}
		}
		
		if(!ok)
		{
			if(arrComponent != 0) arrComponent->Delete();
			if(val != 0) val->Delete();
			this->GetErrorStatus()->Set("Array must have " + iString::FromNumber(n) + " components.");
			this->SetErrorLocation(commandTextOriginal.Find(pv,0));
			return;
		}
	}

	//
	//  save the command in pseudo code if this is the first time we reached this line
	//
	if(mCode[mCodeEntryPoint] == 0)
	{
		iOperation *tmp = 0;
		switch(mCommand[comind]->Type)
		{
		case iOperation::_Operation_DC_SV: 
			{ 
				tmp = iOperation::New(this,mCommand[comind]->Prefix,mCommand[comind]->Command,mCommand[comind]->OutputCode,((iOperation_DC_SV *)mCommand[comind])->StatementPrefix,((iOperation_DC_SV *)mCommand[comind])->Relative,iValue::New(((iOperation_DC_SV *)mCommand[comind])->Value)); 
				break; 
			}
		case iOperation::_Operation_FC_LB: 
			{ 
				if(((iOperation_FC_LB *)mCommand[comind])->Index != 0) ((iOperation_FC_LB *)mCommand[comind])->Index->Register();
				tmp = iOperation::New(this,mCommand[comind]->Prefix,mCommand[comind]->Command,mCommand[comind]->OutputCode,((iOperation_FC_LB *)mCommand[comind])->Point,((iOperation_FC_LB *)mCommand[comind])->Index,val,((iOperation_FC_LB *)mCommand[comind])->Class); 
				break; 
			}
		case iOperation::_Operation_SF_V0: 
			{ 
				tmp = iOperation::New(this,mCommand[comind]->Prefix,mCommand[comind]->Command,mCommand[comind]->OutputCode,((iOperation_SF_V0 *)mCommand[comind])->Function); 
				break; 
			}
		case iOperation::_Operation_SF_V1: 
			{ 
				tmp = iOperation::New(this,mCommand[comind]->Prefix,mCommand[comind]->Command,mCommand[comind]->OutputCode,valueRelative,val,((iOperation_SF_V1 *)mCommand[comind])->Function); 
				break; 
			}
		case iOperation::_Operation_AF_V1: 
			{ 
				tmp = iOperation::New(this,mCommand[comind]->Prefix,mCommand[comind]->Command,mCommand[comind]->OutputCode,arrComponent,valueRelative,val,((iOperation_AF_V1 *)mCommand[comind])->Function); 
				break; 
			}
		default: 
			{
				this->GetErrorStatus()->Set("Syntax error - statement is not understood.");
				return;
			}
		}
		tmp->LineInScript = mNextLine - 1;
		mCode.Add(tmp);
	}
	else
	{
		//
		//  We have been here before - if this is an assignment, then assign the value
		//
		
		switch(mCommand[comind]->Type)
		{
		case iOperation::_Operation_DC_SV: 
		case iOperation::_Operation_SF_V0: 
			{ 
				break; // nothing to do here, just execute
			}
		case iOperation::_Operation_FC_LB: 
			{ 
				if(((iOperation_FC_LB *)mCode[mCodeEntryPoint])->Count != 0) ((iOperation_FC_LB *)mCode[mCodeEntryPoint])->Count->AssignValue(iValue::_Absolute,val); // assign a value if it exists
				break; 
			}
		case iOperation::_Operation_SF_V1: 
			{ 
				((iOperation_SF_V1 *)mCode[mCodeEntryPoint])->Relative = valueRelative;
				((iOperation_SF_V1 *)mCode[mCodeEntryPoint])->Value->AssignValue(iValue::_Absolute,val); // this is an assignment to the rhs, not to the value itself
				break; 
			}
		case iOperation::_Operation_AF_V1: 
			{ 
				((iOperation_AF_V1 *)mCode[mCodeEntryPoint])->Relative = valueRelative;
				((iOperation_AF_V1 *)mCode[mCodeEntryPoint])->Index->AssignValue(iValue::_Absolute,arrComponent); 
				((iOperation_AF_V1 *)mCode[mCodeEntryPoint])->Value->AssignValue(iValue::_Absolute,val); // this is an assignment to the rhs, not to the value itself
				break; 
			}
		default: 
			{
				this->GetErrorStatus()->Set("Syntax error - statement is not understood.");
				return;
			}
		}
		//
		//  Delete temporary values
		//
		if(arrComponent != 0) arrComponent->Delete();
		if(val != 0) val->Delete();
	}
}


iValue* iScript::TransformToValue(const iString &pv0)
{
	if(this->GetErrorStatus()->IsError())
	{
		//
		// do not clean errors accumulated earlier
		//
		return 0;
	}

	iValue *val = this->TransformSingleTokenToValue(pv0);

	if(val != 0) 
	{
		//
		//  This is a single token
		//
		return val; 
	}
	else
	{
		//
		//  Parse the expression to check that it is valid.
		//
		val = this->TransformExpressionToValue(pv0);

		if(val != 0)
		{
			//
			//  This is an expression single token
			//
			this->GetErrorStatus()->Clear();  // clear any error status set by TransformSingleToken
			return val; 
		}
		else
		{
			//
			//  Allow a child to parse its own specialized tokens
			//
			return this->TransformUnknownTokenToValue(pv0);
		}
	}
}


iValue* iScript::TransformSingleTokenToValue(const iString &pv0, bool acceptArrays, bool forceReal)
{
	bool ok;
	int vi;
	double vr;
	iValue *v, *val;

	iString pv = pv0;
	pv.ReduceWhiteSpace();

	//
	// A string
	//
	if(pv[0]=='"' && pv[(int)pv.Length()-1]=='"')
	{
		return iValue::New(this,"",iValue::_ByValue,pv.Part(1,pv.Length()-2));
	}

	//
	// A numeric constant
	//
	if((pv[0]>='0'&&pv[0]<='9') || pv[0]=='-' || pv[0]=='+')
	{
		//
		//  Is it an integer number?
		//
		if(!forceReal)
		{
			vi = pv.ToInt(ok);
			if(ok) return iValue::New(this,"",iValue::_ByValue,vi);
		}

		//
		//  Is it a double number?
		//
		vr = pv.ToDouble(ok);
		if(ok) return iValue::New(this,"",iValue::_ByValue,vr);

		//
		//  No? - it may be an expression 
		//
		this->GetErrorStatus()->Set("Invalid numeric value.");
		return 0;
	}

	//
	// An array
	//
	if(pv[0]=='(' && acceptArrays)
	{
		if(pv[pv.Length()-1] != ')') // no closing parenthisis
		{
			this->GetErrorStatus()->Set("Incorrect array expression.");
			return 0;  
		}

		pv = pv.Part(1,pv.Length()-2);

		//
		//  Measure the number of components
		//
		int i, n = pv.Contains(',') + 1;
		iString s;

		//
		//  Create an array of iValues
		//
		iValue **arr = new iValue*[n]; IERROR_ASSERT_NULL_POINTER(arr);

		//
		//  Fill it in
		//
		ok = true;
		for(i=0; i<n; i++)
		{
			s = pv.Section(",",i,i);
			arr[i] = this->TransformSingleTokenToValue(s,false,forceReal | mArraysAreReal);
			if(arr[i]==0 || (i>0 && !iValue::AreTypesCompatible(arr[i],arr[0]))) ok = false;
		}

		//
		//  If unsuccessfull, destroy it.
		//
		if(!ok)
		{
			for(i=0; i<n; i++) if(arr[i] != 0) arr[i]->Delete();
			delete [] arr; // delete the array, not the values - they will be deleted by iScript
			this->GetErrorStatus()->Set("Incorrect array expression.");
			return 0;
		}

		return iValue::New(this,"",iValue::_ByValue,n,iValue::New(this,"",iValue::_ByValue,1),arr);
	}

	//
	//  A variable
	//
	if(IsScriptWordFirstLetter(pv[0]))
	{
		int i, j;
		iString ac;
		iValue *ind = 0;
		//
		//  Look for the array component at the end in the form [N] where N is a iValue;
		//
		i = pv.Find('[');
		if(i >= 0)    // a square bracket found
		{
			ac = pv.Part(i);
			pv = pv.Part(0,i);
			if(ac.Length()<3 || ac[0]!='[' || ac[ac.Length()-1]!=']')
			{
				this->GetErrorStatus()->Set("Syntax error in the array index.");
				return 0;
			}
			ac = ac.Part(1,ac.Length()-2);
			
			ind = this->TransformSingleTokenToValue(ac,false,false);
			if(ind==0 || ind->GetType()!=iValue::_Value_IS)
			{
				if(ind != 0) ind->Delete();
				this->GetErrorStatus()->Set("Incorrect value of the array index.");
				return 0;
			}
		}

		//
		//  Search for a parameter or a variable with this name: search parameters first, then variables
		//
		iArray<iValue*> *p;
		int iv;

		for(j=0; j<2; j++)
		{
			if(j == 0) 
			{
				p = &mParameterWords;
			}
			else
			{
				p = &mVariable;
			}

			for(i=0; i<(*p).Size(); i++) if((v=(*p)[i]) != 0)
			{
				if(pv == v->Name)
				{
					if(ind != 0)  // this is an array
					{
						if(!v->IsArray())
						{
							ind->Delete();
							this->GetErrorStatus()->Set("Variable " + pv + " is not an array.");
							return 0;
						}
						if(!ind->GetValue(iv))
						{
							ind->Delete();
							this->GetErrorStatus()->Set("Variable " + ac + " has no value.");
							return 0;
						}
						ind->Delete();  // index is not needed any more

						int size = ((iValue_VA*)v)->Size;
						if(size>0 && (iv<=0 || iv>size))
						{
							this->GetErrorStatus()->Set("Array index value is outside the array range.");
							return 0;
						}
						val = ((iValue_VA*)v)->GetComponent(iv);
					}
					else val = v;

					if(val != 0) val->Register();  // register this variable to whatever wants it.
					return val;
				}
			}
		}

		//
		//  If we are embedded, ask the parent to identify the variable
		//
		if(mParent!=0 && mParent->mAllowChildAccess)
		{
			iValue *tmp = mParent->TransformSingleTokenToValue(pv);
			mParent->GetErrorStatus()->Clear();  // we are not executing the parent
			if(tmp != 0) return tmp;
		}

		//
		//  The word is not found - not a valid variable
		//
		if(ind != 0) ind->Delete();
		this->GetErrorStatus()->Set("Undefined variable " + pv + ".");
		return 0;
	}

	//
	//  Not a number, array, or a variable
	//
	this->GetErrorStatus()->Set("Incorrect expression syntax.");
	return 0;
}


iValue* iScript::TransformExpressionToValue(const iString &pv0, bool NotUsed(acceptArrays), bool NotUsed(forceReal))
{
	int i, j, j1, j2, n, nVar0, iVar0 = 0;
	double f, fa[3];
	double d, da[3];
	iValue *v;
	iArray<iValue*> *p;
	iString pv, s;

	pv = pv0;
	nVar0 = mVariable.Size();

	//
	//  pre-parsing: identify all vector components v[2], and turn them into tmp variables
	//
	while((i=pv.Find('[')) >= 0)
	{
		for(j1=i-1; j1>0 && IsScriptWordLetter(pv[j1]); j1--);
		if(!IsScriptWordFirstLetter(pv[j1])) j1++;

		j2 = pv.Find(']',i);
		if(j2 == -1)
		{
			this->GetErrorStatus()->Set("No closing ] found.");
			return 0;
		}

		v = this->TransformSingleTokenToValue(pv.Part(j1,j2-j1+1));
		if(v == 0)
		{
			this->GetErrorStatus()->Set(pv.Part(j1,j2-j1+1) + " is not a valid array component.");
			return 0;
		}
		v->Name = "$$" + iString::FromNumber(iVar0++) + "$$";
		mVariable.Add(v);
		pv.Replace(j1,j2-j1+1,v->Name);
	}

	//
	//  pre-parsing: identify all vector constants (1,1,1) and turn them into tmp variables
	//
	i = 0;
	while((i=pv.Find('(',i)) >= 0)
	{
		j1 = i;
		j2 = pv.Find(')',i);
		if(j2 == -1)
		{
			this->GetErrorStatus()->Set("No closing ) found.");
			return 0;
		}
		if(pv.Part(j1,j2-j1+1).Contains('(')>1 || pv.Part(j1,j2-j1+1).Contains(',')!=2) 
		{
			//
			// Not an array expression, search further;
			//
		}
		else
		{
			v = this->TransformSingleTokenToValue(pv.Part(j1,j2-j1+1));
			if(v == 0)
			{
				this->GetErrorStatus()->Set(pv.Part(j1,j2-j1+1) + " is not an array expression.");
				return 0;
			}
			v->Name = "$$" + iString::FromNumber(iVar0++) + "$$";
			mVariable.Add(v);
			pv.Replace(j1,j2-j1+1,v->Name);
		}
		i++;
	}

	//
	//  Set the function for the VTK parser
	//
	mParser->RemoveAllVariables();
	mParser->SetFunction(pv.ToCharPointer());

	//
	//  Assign variables, including the parent it is defined
	//
	int jmax = 2;
	if(mParent!=0 && mParent->mAllowChildAccess) jmax += 2;
	for(j=0; j<jmax; j++)
	{
		switch(j)
		{
		case 0:
			{
				p = &mParameterWords;
				break;
			}
		case 1:
			{
				p = &mVariable;
				break;
			}
		case 2:
			{
				p = &mParent->mParameterWords;
				break;
			}
		case 3:
			{
				p = &mParent->mVariable;
				break;
			}
		default:
			{
				p = 0;
			}
		}
		
		for(i=0; i<(*p).Size(); i++) if((v=(*p)[i])!=0 && (j1=FindScriptWord(pv,v->Name))>=0)
		{
			switch(v->Type)
			{
			case iValue::_Value_IS:
			case iValue::_Value_RS:
				{
					if(!v->GetValue(f))
					{
						this->GetErrorStatus()->Set("The variable " + v->Name + " has no value.");
						return 0;
					}
					mParser->SetScalarVariableValue(v->Name.ToCharPointer(),(double)f);
					break;
				}
			case iValue::_Value_RP:
			case iValue::_Value_VA:
				{
					if(((iValue_VA*)v)->Size!=3 && ((iValue_VA*)v)->Size!=0)
					{
						this->GetErrorStatus()->Set("Only 3-component arrays (vectors) can be used in expressions.");
						return 0;
					}
					if(!v->GetValue(3,fa))
					{
						this->GetErrorStatus()->Set("The array variable " + v->Name + " has no value.");
						return 0;
					}
					da[0] = fa[0];
					da[1] = fa[1];
					da[2] = fa[2];
					mParser->SetVectorVariableValue(v->Name.ToCharPointer(),da);
					break;
				}
			default: ;
			}
		}
	}

	if(mParser->GetScalarResult(d))
	{
		//
		//  Is this a boolean expression?
		//
		if(pv.Find('!')>=0 || pv.Find('>')>=0 || pv.Find('<')>=0 || pv.Find(mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_LE))>=0 || pv.Find(mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_GE))>=0 || pv.Find(mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_EQ))>=0 || pv.Find(mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_NE))>=0 || pv.Find(mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_AND))>=0 || pv.Find(mParser->GetElementaryOperatorByteEncoding(VTK_PARSER_OR))>=0) 
		{
			//
			//  Convert to boolean if possible
			//
			if(fabs(d)<1.0e-12 || fabs(d-1.0)<1.0e-12) 
			{
				v = iValue::New(this,"",iValue::_ByValue,(round(d)==1));
			}
			else
			{
				this->GetErrorStatus()->Set("Invalid boolean expression.");
				v = 0;
			}
		}
		else v = iValue::New(this,"",iValue::_ByValue,(double)d);
	}
	else if(mParser->GetVectorResult(da))
	{
		//
		//  Create an array of iValues
		//
		n = 3;
		iValue **arr = new iValue*[n]; IERROR_ASSERT_NULL_POINTER(arr);
		
		//
		//  Fill it in
		//
		for(i=0; i<n; i++)
		{
			arr[i] = iValue::New(this,"",iValue::_ByValue,(double)da[i]);
		}
		
		v = iValue::New(this,"",iValue::_ByValue,n,iValue::New(this,"",iValue::_ByValue,1),arr);
	}
	else
	{
		this->GetErrorStatus()->Set(mParser->GetErrorMessage());
		v = 0;
	}

	//
	// delete temporary variables
	//
	while(nVar0 < mVariable.Size()) mVariable.RemoveLast()->Delete();  

	if(v != 0) this->GetErrorStatus()->Clear();  // clean errors accumulated earlier
	return v;
}


iValue* iScript::TransformUnknownTokenToValue(const iString& /*token*/, bool /*acceptArrays*/, bool /*forceReal*/)
{
	//
	//  Default implementation: no unknown tokens are allowed
	//
	return 0;
}


int iScript::FindScriptWord(const iString &context, const iString &word, int index) const
{
	int i = index;
	int l = word.Length();

	while((i=context.Find(word,i))>-1)
	{
		//
		// Is this a full word?
		//
		if(i>0 && IsScriptWordLetter(context[i-1])) { i+=l; continue; }
		if(i+l<context.Length() && IsScriptWordLetter(context[i+l])) { i+=l; continue; }
		//
		//  Is it behind a quote - meaning, it is a string?
		//
		if(context.Part(0,i).Contains('"')%2 == 1) { i+=l; continue; }
		break;
	}
	return i;
}


//
//  Observer functions
//
iScriptObserver::iScriptObserver(iScript *s)
{
	mScript = s;
	if(mScript != 0) mScript->AddObserver(this);
}


iScriptObserver::~iScriptObserver()
{
	if(mScript != 0) mScript->RemoveObserver(this);
}


bool iScriptObserver::OnScriptCheckAbort(int cur, int num, int level)
{
	if(mScript != 0) return this->OnScriptCheckAbortBody(cur,num,level); else return false;
}


void iScriptObserver::OnScriptStart()
{
	if(mScript != 0) this->OnScriptStartBody();
}


void iScriptObserver::OnScriptStop(const iString &error)
{
	if(mScript != 0) this->OnScriptStopBody(error);
}


void iScriptObserver::OnScriptBeginLine(int line, const iString &text)
{
	if(mScript != 0) this->OnScriptBeginLineBody(line,text);
}


void iScriptObserver::OnScriptEndLine(int line, const iString &text)
{
	if(mScript != 0) this->OnScriptEndLineBody(line,text);
}


void iScriptObserver::SetScript(iScript *s)
{
	if(mScript != s)
	{
		if(mScript != 0) mScript->RemoveObserver(this);
		mScript = s;
		if(mScript != 0) mScript->AddObserver(this);
	}
}
