/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ishell.h"


#include "iconsole.h"
#include "icontrolmodule.h"
#include "idirectory.h"
#include "ieventobserver.h"
#include "ifile.h"
#include "iimagefactory.h"
#include "iobjecthelp.h"
#include "ioutputwindow.h"
#include "iparallelmanager.h"
#include "ishellfactory.h"
#include "ivtk.h"

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


const iString iShell::mOptionPrefix = "-";


void iShell::RunApplication(const iString &t, int argc, char **argv)
{
	iShell *tmp = iShell::New(t,argc,argv);
	if(tmp != 0)
	{
		tmp->Run();
		tmp->Delete();
	}
}


iShell::iShell(int, char **) : mOptions(Option())
{
	mRunning = false;
	mControlModule = 0; // control module must be created after the constructor is finished
	mOutputWindow = 0; // output window must be created after the constructor is finished
	mType = "undefined";
	mNumProcs = 0;

	//
	//  Define some basic command-line options
	//
	this->AddCommandLineOption("h",false,"show this help");
	this->AddCommandLineOption("help",false,"show this help");
	this->AddCommandLineOption("-help",false,"show this help");
	this->AddCommandLineOption("i",true,"load the state from a file with name <arg>");
	this->AddCommandLineOption("np",true,"set the number of processors to use to <arg>");

	//
	//  Set environment variables
	//  Is the base-dir environment variable set?
	//
	char *e = getenv("IFRIT_DIR");
	if(e != 0)
	{
		mEnvBaseDir = iString(e);
		if(mEnvBaseDir.IsEmpty()) mEnvBaseDir = this->GetCurrentDir();
		if(!mEnvBaseDir.EndsWith(this->GetDirSeparator())) mEnvBaseDir += this->GetDirSeparator();
	}
	else
	{
		//
		//  Get the home directory, if it exists
		//
		e = getenv("APPDATA"); // windows?
		if(e != 0)
		{
			mEnvBaseDir = iString(e) + this->GetDirSeparator() + "IfrIT" + this->GetDirSeparator();
		}
		else // unix?
		{
			e = getenv("HOME");
			mEnvBaseDir = iString(e) + this->GetDirSeparator() + ".ifrit" + this->GetDirSeparator();
		}
		//
		//  Is it readable?
		//
		iDirectory dir;
		if(!dir.Open(mEnvBaseDir))
		{
			//
			//  try to create
			//
			if(!dir.Make(mEnvBaseDir))
			{
				mEnvBaseDir = this->GetCurrentDir() + this->GetDirSeparator();
			}
		}
	}

	//    
	//  Read other environment here    
	//    
	e = getenv("IFRIT_DATA_DIR");               if(e != 0) mEnvDataDir = iString(e) + this->GetDirSeparator(); else mEnvDataDir = this->GetCurrentDir() + this->GetDirSeparator();
    e = getenv("IFRIT_IMAGE_DIR");              if(e != 0) mEnvImageDir = iString(e) + this->GetDirSeparator(); else mEnvImageDir = this->GetCurrentDir() + this->GetDirSeparator();
    e = getenv("IFRIT_SCRIPT_DIR");             if(e != 0) mEnvScriptDir = iString(e) + this->GetDirSeparator(); else mEnvScriptDir = mEnvBaseDir;
    e = getenv("IFRIT_PALETTE_DIR");            if(e != 0) mEnvPaletteDir = iString(e) + this->GetDirSeparator(); else mEnvPaletteDir = mEnvBaseDir;
}


iShell::~iShell()
{
	//
	//  This ensures that control module is deleted after the shell is deleted.
	//
//	mControlModule->Delete();
}


iShell* iShell::New(const iString &t, int argc, char **argv)
{
	//
	//  Also use options to specify shell - they overwrite the default choice
	//
	iString st = t;
	int firstOption = 1;
	if(argc>1 && iString(argv[1]).Part(0,mOptionPrefix.Length())==mOptionPrefix)
	{
		iString list, help, s = iString(argv[1]).Part(mOptionPrefix.Length());
		iShellFactory::GetSupportedShells(list,help);
		if(list.Contains(s+"/") == 1)
		{
			st = s;
			firstOption = 2;
		}
	}
	//
	//  If there is no default and no option, ask ShellFactory to query the user
	//
	if(st.IsEmpty())
	{
		cerr << "No shell has been specified. Use a \"-<shell>\" option to specify a two-letter shell abbreviation <shell>." << endl;
		exit(0);
	}
	//
	//  Create shell
	//
	iShell *tmp = iShellFactory::CreateShell(st,argc,argv);
	if(tmp == 0) 
	{
		cerr << "Shell '" << st.ToCharPointer() << "' is not included in this installation." << endl;
		return 0;
	}
	//
	//  Check that sizes of basic types are correct
	//
	if(sizeof(int)!=4 || sizeof(float)!=4 || sizeof(double)!=8)
	{
		tmp->OutputStartupMessage("IFrIT has not been ported to this machine.");
		exit(0);
	}
	//
	//  Create console: need the fully created "tmp->this" pointer
	//
	iConsole::CreateInstance(tmp);
	if(iConsole::Instance() == 0)
	{
		tmp->OutputStartupMessage("Unable to create Console class.");
		exit(1);
	}
	tmp->mFirstOption = firstOption;
	//
	// Make sure Image factory is up to date
	//
	iImageFactory::FindIcon("genie1.png");
	//
	//  Create the output window: need the fully created "tmp->this" pointer
	//
	tmp->mOutputWindow = iOutputWindow::New(tmp); 
	if(tmp->mOutputWindow == 0)
	{
		tmp->OutputStartupMessage("Unable to create OutputWindow class.");
		exit(1);
	}
	//
	//  Parse command line options
	//
	tmp->DefineSpecificCommandLineOptions();
	tmp->ParseCommandLineOptions(argc,argv);
	tmp->PrepareForConstruction();
	//
	//  This ensures that "tmp->this" pointer is correct before a control module is created.
	//
	tmp->mControlModule = iControlModule::New(tmp); IERROR_ASSERT_NULL_POINTER(tmp->mControlModule);
	if(tmp->mNumProcs > 0) tmp->mControlModule->GetParallelManager()->ResetMaxNumberOfProcessors(tmp->mNumProcs);
	//
	//  Really construct the shell: control module needs to be fully initialized before a shell 
	//  is really constructed.
	//
	tmp->ConstructorBody(); 
	//
	//   Load widget info into ObjectKeyHelp objects.
	//
	tmp->LoadObjectKeyHelp();
	//
	//  Load the state file if needed
	//
	if(tmp->mStateFileName.IsEmpty())
	{
		iString fname = tmp->mControlModule->GetFileName(_EnvironmentBase,"ifrit.ini");
		if(iFile::IsLoadable(fname)) tmp->mStateFileName = fname;
	}
	if(!tmp->mStateFileName.IsEmpty())
	{
		tmp->LoadStateFile(tmp->mStateFileName);
	}
	//
	//  Start the shell
	//
	tmp->Start();

	return tmp;
}


void iShell::Delete()
{
	//
	//  We are just about to start deleting objects. First thing we do is to block all
	//  observers since we do not know at each moment in the deleting process which objects
	//  are still present and which have been already deleted.
	//
	iEventObserver::BlockEventObservers(true);
	//
	//  Check whether the error log was created by iConsole:
	//
	if(iConsole::Instance()->IsLogCreated())
	{
		iString ws = "A log of all error messages was created in the file:\n" + iConsole::Instance()->GetLogFileName() + 
			"\n Please, consider e-mailing this file to ifrit.bugs@gmail.com";
		iConsole::Instance()->DisplayCriticalInformationMessage(ws.ToCharPointer());
	}

	//
	//  Go on rampage!
	//
	this->DestructorBody();

	//
	//  This ensures that control module is deleted before the shell is deleted.
	//
	mControlModule->Delete();

	delete this;
}


void iShell::Run()
{
	mRunning = true;
	this->RunBody();
	mRunning = false;
}


void iShell::LoadObjectKeyHelp()
{
	const iObjectKey *key;

	iObjectKeyRegistry::InitTraversal();
	while((key=iObjectKeyRegistry::GetNextKey()) != 0)
	{
		//
		//  We are accesing the class members directly since key->GetHelp() is const.
		//
//		this->FindWidgetForKey(key,key->mHelp->mWidgetName,key->mHelp->mWidgetDescription);
	}
}


void iShell::AddCommandLineOption(const iString &root, bool hasValue, const iString &help)
{
	Option tmp;
	tmp.Name = root;
	tmp.NeedsValue = hasValue;
	tmp.Help = help;
	mOptions.Add(tmp);
}


void iShell::ParseCommandLineOptions(int argc, char **argv)
{
	int i, j;
	Option o;
	bool done;

	i = mFirstOption;
	while(i < argc)
	{
		done = false;
		if(iString(argv[i]).Part(0,mOptionPrefix.Length()) == mOptionPrefix)
		{
			o.Name = argv[i] + mOptionPrefix.Length();
			for(j=0; !done && j<mOptions.Size(); j++) if(o.Name == mOptions[j].Name)
			{
				if(mOptions[j].NeedsValue)
				{
					i++;
					if(i < argc) o.Value = argv[i]; else
					{
						this->OutputStartupMessage("Option "+o.Name+" requires an argument.\nIFrIT will now exit.");
						exit(1);
					}
				}
				if(!AnalyseOneCommandLineOption(o))
				{
					this->OutputStartupMessage("Error has occured in analysing option "+o.Name+"\nIFrIT will now exit.");
					exit(1);
				}
				else done = true;
			}
			if(done) i++;
		}

		//
		//  Not an option
		//
		if(!done && i==argc-1)
		{
			//
			//  This must be the name of the default data directory
			//
			iDirectory d;
			if(d.Open(argv[i]))
			{
				d.Close();
				mEnvDataDir = argv[i];
				i++;
				done = true;
			}
		}

		//
		//  Unrecognizable argument
		//
		if(!done)
		{
			this->OutputStartupMessage("Command-line argument "+iString(argv[i])+" is not supported.\nIFrIT will now exit.");
			exit(1);
		}
	}
}


bool iShell::AnalyseOneCommandLineOption(const struct Option &o)
{
	if(o.Name=="h" || o.Name=="help" || o.Name=="-help")
	{
		iString sl, sh, s = "Format: ifrit [shell] [option] ... [data_dir_name]\n";
		s += "Shell: -<id>,  where a two-letter shell id is one of the following:\n";
		iShellFactory::GetSupportedShells(sl,sh);
		int i, c = sl.Contains('/');
		for(i=0; i<c; i++)
		{
			s += "\t" + sl.Section("/",i,i) + "\t" + sh.Section("/",i,i) + "\n";
		}
		s += "Options:\n";
		for(i=0; i<mOptions.Size(); i++)
		{
			s += "\t-" + mOptions[i].Name + " ";
			if(mOptions[i].NeedsValue) s += "<arg> "; else s += "      ";
			s += "\t" + mOptions[i].Help + "\n";
		}
		this->OutputStartupMessage(s);
		exit(0);
	}

	if(o.Name == "np")
	{
		bool ok;
		int n = o.Value.ToInt(ok);
		if(!ok || n<0 || n>16384) return false;
		mNumProcs = n;
		return true;
	}

	if(o.Name == "i")
	{
		if(iFile::IsLoadable(o.Value))
		{
			mStateFileName = o.Value;
			return true;
		}
		else
		{
			this->OutputStartupMessage("File "+o.Value+" is not accessible.\nIFrIT will now exit.");
			exit(1);
		}
	}
	
	return this->AnalyseOneExtendedCommandLineOption(o);
}


//
//  Environment querying
//
const iString& iShell::GetEnvironment(int key) const
{
	static const iString none;

	switch(key)
	{
	case _EnvironmentBase:
		{
			return mEnvBaseDir;
		}
	case _EnvironmentData:
		{
			return mEnvDataDir;
		}
	case _EnvironmentImage:
		{
			return mEnvImageDir;
		}
	case _EnvironmentScript:
		{
			return mEnvScriptDir;
		}
	case _EnvironmentPalette:
		{
			return mEnvPaletteDir;
		}
	default:
		{
			return none;
		}
	}
}


const iString& iShell::GetDirSeparator() const
{
	static const iString sep = "/";
	return sep;
}


const iString& iShell::GetCurrentDir() const
{
	static const iString cur = ".";
	return cur;
}

