/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iuniformgriddatasubject.h"


#include "ibuffer.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ieventobserver.h"
#include "iexpressionparser.h"
#include "ifile.h"
#include "iparallel.h"
#include "iparallelfft.h"
#include "iparallelmanager.h"
#include "iparallelworker.h"
#include "isystem.h"
#include "itune.h"
#include "iviewmodule.h"
#include "ivtk.h"

#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkStructuredPoints.h>
#ifndef IVTK_4
#include <vtkInformation.h>
#endif

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"


IDATASUBJECT_DEFINE_TYPE(iUniformScalarsDataSubject,UniformScalars,us);
//
//  Inherited keys
//
IDATASUBJECT_DEFINE_INHERITED_KEYS(iUniformScalarsDataSubject);


IDATASUBJECT_DEFINE_TYPE(iUniformVectorsDataSubject,UniformVectors,uv);
//
//  Inherited keys
//
IDATASUBJECT_DEFINE_INHERITED_KEYS(iUniformVectorsDataSubject);


IDATASUBJECT_DEFINE_TYPE(iUniformTensorsDataSubject,UniformTensors,ut);
//
//  Inherited keys
//
IDATASUBJECT_DEFINE_INHERITED_KEYS(iUniformTensorsDataSubject);


//
//  ********************************************************************
//
//  Generic GridDataSubject class
//
//  ********************************************************************
//
//  Helper class for parallel executions
//
class iGridDataHelper : protected iParallelWorker
{

public:

	iGridDataHelper(iGridDataSubject *subject);
	~iGridDataHelper();

	void TransformToPointData(int mode, bool arrayIsVertical, int numComponents, float *&cellData, int cellDims[3]);
	void ShiftData(int numCom, int dim, int dn, int dataDims[3], float *data, float *buf);

protected:

	virtual int ExecuteStep(int step, iParallel::ProcessorInfo &p);

	void CopyArray(int numCom, int dims[3], float *src, bool srcVertical, float *dst, bool dstVertical);
	void TransformToPointData1(int dims[3], float *data, float *dnyq, float ps, float dp);
	void TransformToPointData2(bool per[3], int cellDims[3], long cellOffset[3], long cellComponentOffset, float *cellData, float *data, float ps, float dp);
	void TransformToPointData3(int cellToPointMode, int numCom, bool per[3], int cellDims[3], long cellOffset[3], long cellComponentOffset, float *cellData, int pointDims[3], long pointOffset[3], long pointComponentOffset, float *pointData, float ps, float dp);

	void ExecuteCopyArray(iParallel::ProcessorInfo &p);
	void ExecuteTransformToPointData1(iParallel::ProcessorInfo &p);
	void ExecuteTransformToPointData2(iParallel::ProcessorInfo &p);
	void ExecuteTransformToPointData3(iParallel::ProcessorInfo &p);
	void ExecuteShiftData(iParallel::ProcessorInfo &p);

	iGridDataSubject *mSubject;
	iParallelFFT *mFft;

	int mNc, mMode;
	int mN1, mN2, mN3;
	long *mOffset, mComponentOffset;
	int *mDims2;
	long *mOffset2, mComponentOffset2;
	bool *mPeriodic;
	float *mArr1, *mArr2;
	double *mDarr;

	float mProgStart, mProgStep;

	int mItmp1, mItmp2;
	bool mBtmp1, mBtmp2;
	long mLtmp1, mLtmp2;
	float mFtmp1;
	
	float *mFfMin, *mFfMax;
};


//
//  Main class
//
iGridDataSubject::iGridDataSubject(iDataReader *r, const iDataType &type) : iDataSubject(r,type), mMaxDimension(1670000)
{
	mCellToPointMode = 2;
	mVoxelLocation = 0;
	mNumVars = mNumArrays = mNumComponents = 0;
	mDims[0] = mDims[1] = mDims[2] = 0;

	mBufferSize = 0;
	mBuffer = 0;

	mScaledDim = -1;

	mInternalData = mData = vtkStructuredPoints::New(); IERROR_ASSERT_NULL_POINTER(mInternalData);
	mExternalData =         vtkStructuredPoints::New(); IERROR_ASSERT_NULL_POINTER(mExternalData);

	mHelper = new iGridDataHelper(this); IERROR_ASSERT_NULL_POINTER(mHelper);
}


iGridDataSubject::~iGridDataSubject()
{
	delete mHelper;
}


void iGridDataSubject::ReadFileBody(const iString &suffix, const iString &fname)
{
	//
	//  is the suffix valid?
	//
	if(suffix.Lower()!="txt" && suffix.Lower()!="bin")
	{
		this->GetErrorStatus()->Set("Incorrect file suffix.");
		return;
	}

	//
	//  Read the data
	//
	if(suffix.Lower() == "txt") this->ReadTxtFile(fname); else this->ReadBinFile(fname);
}


void iGridDataSubject::ReadBinFile(const iString &fname)
{
	int i;
	vtkIdType l, ntot;
	float vmin, vmax, *d;
	iFile F(fname);
	bool err;

	if(!F.Open(iFile::_Read,iFile::_Binary))
	{
		this->GetErrorStatus()->Set("File does not exist.");
		return;
	}

	mObserver->Started(iProgressEventObserver::_Reading);
	mObserver->SetProgress(0.0);

	//
	// auto-detect data endiness
	//
	if(!this->DetectFileStructureFromFirstRecord(F,3*sizeof(int)))
	{ 
		F.Close(); 
		mObserver->Finished();
		this->GetErrorStatus()->Set("Corrupted header.");
		return;
	}

	//
	//  Read the header
	//
	err = false;
	int ndim[3];
	if(!this->ReadFortranRecord(F,Buffer(ndim,3),0.0,0.0)) err = true;

    if(ndim[0]<1 || ndim[0]>mMaxDimension || ndim[1]<1 || ndim[1]>mMaxDimension || ndim[2]<1 || ndim[2]>mMaxDimension) err = true;

	if(err) 
	{ 
		F.Close(); 
		mObserver->Finished();
		this->GetErrorStatus()->Set("Corrupted header.");
		return;
	}

	ntot = (vtkIdType)ndim[0]*ndim[1]*ndim[2];
	//
	//  Measure the file size by skipping blocks with size of one variable
	//
	int nvar, nrec;
	int mar = F.SetMarker();
	for(nrec=0; nrec<999 && this->SkipFortranRecord(F,sizeof(float)*ntot); nrec++);
	F.ReturnToMarker(mar,true);

	nvar = nrec;
	this->SetIndicies(nvar,nrec);
	//
	//  If no records, exit
	//
	if(nvar < 1)
	{ 
		F.Close(); 
		mObserver->Finished(); 
		this->GetErrorStatus()->Set("Corrupted data.");
		return; 
	}
	mObserver->SetProgress(0.01);
	//
	//  parameters for the Progress Bar
	//
	float prog1 = 0.99/nrec;
	//
	//  Allocate memory; erase the old mBuffer if needed.
	//
	long newBufferSize = (long)ntot*(long)nvar;
	if(newBufferSize>mBufferSize || newBufferSize<mBufferSize/2) // do not use too much extra memory
	{
		if(mBuffer != 0) delete [] mBuffer;
		mBuffer = new float[newBufferSize];
		mBufferSize = newBufferSize;
		if(mBuffer == 0)
		{
			F.Close();
			mObserver->Finished();
			this->GetErrorStatus()->Set("Not enough memory to load the data.");
			return;
		}
	}
	//
	//  Read the data from the file.
	//
	d = new float[ntot]; 
	if(d == 0)
	{
		this->EraseBuffer();
		F.Close();
		mObserver->Finished();
		this->GetErrorStatus()->Set("Not enough memory to load the data.");
		return;
	}

	for(i=0; !err && i<nrec; i++) 
	{
		err = !this->ReadFortranRecord(F,Buffer(d,ntot),0.01+prog1*i,prog1);
		if(err || mObserver->IsAborted()) break;

		this->AssignBinData(i,ntot,d);
		vmin = vmax = d[0];
		for(l=1; l<ntot; l++)
		{
			if(vmin > d[l]) vmin = d[l];
			if(vmax < d[l]) vmax = d[l];
			mVarMin[i] = vmin;
			mVarMax[i] = vmax;
		}
	}

	delete [] d;

	mObserver->SetProgress(1.0);
	mObserver->Finished();
	F.Close();

	if(err || mObserver->IsAborted())
	{
		this->EraseBuffer();
		if(err) this->GetErrorStatus()->Set("Corrupted data."); else this->GetErrorStatus()->SetAbort();
		return;
	}

	mNumVars = nvar;
	for(i=0; i<3; i++) mDims[i] = ndim[i];
}


void iGridDataSubject::ReadTxtFile(const iString &fname)
{
	int i, n1, n2, n3, ret;
	iString s;
	long ntot, l;
	float f[999];
	iFile F(fname);
	bool err;

	if(!F.Open(iFile::_Read,iFile::_Text))
	{
		this->GetErrorStatus()->Set("File does not exist.");
		return;
	}

	mObserver->Started(iProgressEventObserver::_Reading);
	mObserver->SetProgress(0.0);
	//
	//  Read the header
	//
	err = false;
	if(!F.ReadLine(s)) err = true;

	ret = sscanf(s.ToCharPointer(),"%d %d %d",&n1,&n2,&n3);
	if(ret != 3) err = true;

	if(n1<1 || n1>mMaxDimension || n2<1 || n2>mMaxDimension || n3<1 || n3>mMaxDimension) err = true;

	if(err)
	{ 
		F.Close();
		mObserver->Finished();
		this->GetErrorStatus()->Set("Corrupted header.");
		return;
	}

	ntot = (vtkIdType)n1*n2*n3;
	//
	//  Find out the number of variables
	//
	if(!F.ReadLine(s)) err = true;
	ret = sscanf(s.ToCharPointer(),"%g %g %g %g %g %g %g %g %g %g",&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
	if(ret<1 || ret>10) err = true;
	int nvar = 0, nrec = ret;

	nvar = nrec;
	this->SetIndicies(nvar,nrec);
	//
	//  If no records or an error, exit
	//
	if(nvar<1 || err)
	{ 
		F.Close();
		mObserver->Finished();
		this->GetErrorStatus()->Set("Corrupted data.");
		return;
	}

	mObserver->SetProgress(0.01);
	//
	//  Allocate memory; erase the old mBuffer if needed.
	//
	long newBufferSize = (long)ntot*(long)nvar;
	if(newBufferSize>mBufferSize || newBufferSize<mBufferSize/2) // do not use too much extra memory
	{
		if(mBuffer != 0) delete [] mBuffer;
		mBuffer = new float[newBufferSize];
		mBufferSize = newBufferSize;
		if(mBuffer == 0)
		{
			F.Close();
			mObserver->Finished();
			this->GetErrorStatus()->Set("Not enough memory to create the data.");
			return;
		}
	}
	//
	//  Save the first line
	//
	this->AssignTxtData(nvar,ntot,0,f);
	for(i=0; i<nvar; i++)
	{
		mVarMin[i] = f[i];
		mVarMax[i] = f[i];
	}

	//
	//  Read the data from the file.
	//
	for(l=1; !err && l<ntot; l++) 
	{
		if(!F.ReadLine(s)) err = true;
		ret = sscanf(s.ToCharPointer(),"%g %g %g %g %g %g %g %g %g %g",&f[0],&f[1],&f[2],&f[3],&f[4],&f[5],&f[6],&f[7],&f[8],&f[9]);
		if(ret != nrec) err = true;
		if((100*l)/ntot < (100*(l+1))/ntot) 
		{
			mObserver->SetProgress(0.01+(float)l/ntot);
			if(mObserver->IsAborted())
			{
				F.Close();
				mObserver->Finished();
				this->GetErrorStatus()->SetAbort();
				return;
			}
		}

		this->AssignTxtData(nvar,ntot,l,f);
		for(i=0; i<nvar; i++)
		{
			if(mVarMin[i] > f[i]) mVarMin[i] = f[i];
			if(mVarMax[i] < f[i]) mVarMax[i] = f[i];
		}
	}	

	mObserver->SetProgress(1.0);
	mObserver->Finished();
	F.Close();

	if(err || mObserver->IsAborted())
	{
		this->EraseBuffer();
		if(err) this->GetErrorStatus()->Set("Corrupted data."); else this->GetErrorStatus()->SetAbort();
		return;
	}

	mNumVars = nvar;
	mDims[0] = n1;
	mDims[1] = n2;
	mDims[2] = n3;
}


void iGridDataSubject::EraseBuffer()
{
	delete [] mBuffer; 
	mBuffer = 0; 
	mBufferSize = 0;
}


void iGridDataSubject::FinalizeBody()
{
	//
	//  Now we need to decide whether the data are CellData or PointData.
	//  We use voxelLocation for that: if it is 0, we have point data and we
	//  don't need to do anything. If it is 1, we have cell data and we need to
	//  interpolate to points with the appropriate boundary conditions.
	//
	if(mBuffer!=0 && mDims[0]>0 && mDims[1]>0 && mDims[2]>0)
	{
		mNumArrays = mNumComponents = 0;
		this->GetLimits()->ResetVars();
		this->SetArraysAndComponents();
#ifdef I_CHECK1
		if(mNumVars != mNumArrays*mNumComponents)
		{
			IERROR_REPORT_ERROR("Bug in iGridDataSubject.");
		}
#endif
		
		if(this->GetVoxelLocation() != 0)
		{
			if(mNumArrays>1 && mNumComponents!=1)
			{
				IERROR_REPORT_ERROR("This array configuration is not supported.");
			}
			else
			{
				mHelper->TransformToPointData(mCellToPointMode,mNumArrays==1,mNumVars,mBuffer,mDims);
			}
		}

		mData->Initialize();
		mData->SetDimensions(mDims);
		mData->SetScalarTypeToFloat();
		mData->SetNumberOfScalarComponents(mNumVars);

		this->ComputeSpacing();
		this->AttachBuffer();
	}
}


void iGridDataSubject::Polish()
{
#ifndef IVTK_4
	//
	//  Extra class needed under VTK 5. What a mess!
	//
	vtkImageData *d = dynamic_cast<vtkImageData*>(mExternalData);
	if(d != 0)
	{
		vtkInformation *info = d->GetPipelineInformation();
		if(info == 0)
		{
			info = vtkInformation::New(); IERROR_ASSERT_NULL_POINTER(info);
			d->SetPipelineInformation(info);
		}

		double spa[3], org[3];
		d->GetOrigin(org);
		d->GetSpacing(spa);

		info->Set(vtkDataObject::ORIGIN(),org,3);
		info->Set(vtkDataObject::SPACING(),spa,3);
	}
#endif
}


void iGridDataSubject::SetScaledDimension(int v)
{
	if(v>=-2 && v<=2)
	{
		mScaledDim = v;
		int i;
		for(i=0; i<3; i++) this->ShiftDataBody(i,-mShift[i]);
		this->ComputeSpacing();
		for(i=0; i<3; i++) this->ShiftDataBody(i,mShift[i]);
		if(mTwoCopies)
		{
			mExternalData->DeepCopy(mInternalData);
		}
		else
		{
			mExternalData->ShallowCopy(mInternalData);
		}
		this->Polish();
	}
}


void iGridDataSubject::ComputeSpacing()
{
	if(mDims[0]<=0 || mDims[1]<=0 || mDims[2]<=0) return;

	int nmax, nmin;
	int i, imax = 0, imin = 0;
	nmax = nmin = mDims[0];
	for(i=1; i<3; i++)
	{
		if(nmax < mDims[i]) 
		{
			nmax = mDims[i];
			imax = i;
		}
		if(nmin > mDims[i]) 
		{
			nmin = mDims[i];
			imin = i;
		}
	}

	for(i=0; i<3; i++) this->SetDirectionPeriodic(i,mDims[i]==nmax);

	float s;
	switch (mScaledDim)
	{
	case -1: { s = 2.0/(nmin-1); break; }
	case  0: { s = 2.0/(mDims[0]-1); break; }
	case  1: { s = 2.0/(mDims[1]-1); break; }
	case  2: { s = 2.0/(mDims[2]-1); break; }
	case -2: 
	default: { s = 2.0/(nmax-1); break; }
	}

	double org[3], spa[3];
	for(i=0; i<3; i++) 
	{	
		org[i] = -1.0;
		spa[i] = s;
	}

	mData->SetOrigin(org);
	mData->SetSpacing(spa);
}


void iGridDataSubject::ShiftDataBody(int d, double dx)
{
	if(d<0 || d>2 || !this->IsThereData()) return;

	if(this->IsDirectionPeriodic(d))
	{
		int n, dimt[3];
		this->GetCellDims(dimt);

		int dn = round(dx*dimt[d]);
		if(dn==0 || abs(dn)>=dimt[d]) return;

		vtkIdType size = (vtkIdType)mDims[0]*mDims[1]*mDims[2]*mNumComponents;
		float *buf = new float[mNumComponents*mDims[d]]; if(buf == 0) return;
		for(n=0; n<mNumArrays; n++)
		{
			mHelper->ShiftData(1,d,dn,mDims,mBuffer+size*n,buf);
		}
		delete[] buf;

		double sp[3];
		mData->GetOrigin(sp);
		sp[d] += 2.0*(dx-float(dn)/dimt[d]);
		mData->SetOrigin(sp);
	}
	else
	{
		double sp[3];
		mData->GetOrigin(sp);
		sp[d] += 2.0*dx;
		mData->SetOrigin(sp);
	}

	mData->Modified();
}


void iGridDataSubject::SetCellToPointMode(int m)
{ 
	if(m>=0 && m<=5) mCellToPointMode = m;
}


void iGridDataSubject::SetVoxelLocation(int m)
{ 
	if(m>=0 && m<=1) mVoxelLocation = m;
}


void iGridDataSubject::GetCellDims(int dimCell[3]) const
{
	int n;
	for(n=0; n<3; n++) if(this->GetVoxelLocation() == 0) dimCell[n] = mDims[n] - 1; else dimCell[n] = mDims[n];
}


void iGridDataSubject::DataSubjectPackStateBody(iString &) const
{
}


void iGridDataSubject::DataSubjectUnPackStateBody(const iString &)
{
}


//
//  Helper class
//
iGridDataHelper::iGridDataHelper(iGridDataSubject *subject) : iParallelWorker(subject->GetViewModule()->GetParallelManager())
{
	mSubject = subject;
	mFft = iParallelFFT::New(this->GetManager()); IERROR_ASSERT_NULL_POINTER(mFft);
}


iGridDataHelper::~iGridDataHelper()
{
	mFft->Delete();
}


void iGridDataHelper::TransformToPointData(int mode, bool arrayIsVertical, int numComponents, float *&cellData, int cellDims[3])
{
	int i, n;
	long cellOffset[3], pointOffset[3], cellComponentOffset, pointComponentOffset;
	
	int pointDims[3];
	long pointSize, cellSize;
	float *pointData;

	for(i=0; i<3; i++) pointDims[i] = cellDims[i] + 1; 
	pointSize = (long)pointDims[0]*pointDims[1]*pointDims[2];
	cellSize = (long)cellDims[0]*cellDims[1]*cellDims[2];

	long arraySize = pointSize;
	if(mode == 0)
	{
		//
		//  Point data array will be used as auxilary storage for FFT.
		//  Make sure it has enough size (at least (n1+2)*n2*n3)
		//
		long fftSize = ((long)cellDims[0]+2L)*cellDims[1]*cellDims[2];
		if(fftSize > arraySize) arraySize = fftSize;
	}

	pointData = new float[numComponents*arraySize];
	if(pointData == 0) return;

	cellOffset[0] = 1;
	cellOffset[1] = cellDims[0];
	cellOffset[2] = cellDims[0]*cellDims[1];
	pointOffset[0] = 1;
	pointOffset[1] = pointDims[0];
	pointOffset[2] = pointDims[0]*pointDims[1];
	cellComponentOffset = cellSize;
	pointComponentOffset = pointSize;

	if(arrayIsVertical)
	{
		for(i=0; i<3; i++) 
		{
			cellOffset[i] *= numComponents;
			pointOffset[i] *= numComponents;
		}
		cellComponentOffset = pointComponentOffset = 1;
	}

	mSubject->GetObserver()->Started(iProgressEventObserver::_Projecting);
	mSubject->GetObserver()->SetProgress(0.0);

	float progMax = cellDims[2]*numComponents;
	float dprog, prog = 0.0;

	bool per[3];
	for(i=0; i<3; i++) per[i] = mSubject->IsDirectionPeriodic(i);

	//
	//  Fourier transform method(s)
	//
	if(mode == 0)
	{
		//
		//  Use point data array as auxilary storage, and then put everything back into cellData
		//
		this->CopyArray(numComponents,cellDims,cellData,arrayIsVertical,pointData,false);

		int n1 = cellDims[0];
		int n2 = cellDims[1];
		int n3 = cellDims[2];
		int n12;
		if(n1%2 == 0) n12 = n1/2; else n12 = (n1-1)/2;

		float *data, *dnyq = pointData + numComponents*cellSize;	// there should be enough space at the end of pointData
																	// array for the Nyquist frequency
		progMax += numComponents*(2*n3+1.5*2*(n2+2*n3));
		dprog = 1.5*(n2+2*n3)/progMax;

		for(n=0; n<numComponents; n++)
		{
			data = pointData + n*cellSize;

			mFft->Transform(true,n1,n2,n3,data,dnyq,mSubject->GetObserver(),prog/progMax,dprog);
			prog += dprog*progMax;

			this->TransformToPointData1(cellDims,data,dnyq,prog/progMax,n3/progMax);
			prog += n3;

			mFft->Transform(false,n1,n2,n3,data,dnyq,mSubject->GetObserver(),prog/progMax,dprog);
			prog += dprog*progMax;

			//
			//  Limit with the cell data
			//
			this->TransformToPointData2(per,cellDims,cellOffset,n*cellComponentOffset,cellData,data,prog/progMax,n3/progMax);
			prog += n3;
		}
		//
		//  Put back into cell data
		//
		this->CopyArray(numComponents,cellDims,pointData,false,cellData,arrayIsVertical);
	}
	//
	//  Projection
	//
	this->TransformToPointData3(mode,numComponents,per,cellDims,cellOffset,cellComponentOffset,cellData,pointDims,pointOffset,pointComponentOffset,pointData,prog/progMax,1.0-prog/progMax);
	//
	//  Replace cell data with point data
	//
	for(i=0; i<3; i++) cellDims[i] = pointDims[i];
	if(mSubject->OwnsData()) delete [] cellData; // don't delete if we do not own the data
	cellData = pointData;

	mSubject->GetObserver()->SetProgress(1.0);
	mSubject->GetObserver()->Finished();
}


void iGridDataHelper::CopyArray(int numCom, int dims[3], float *src, bool srcVertical, float *dst, bool dstVertical)
{
	mNc = numCom;
	mN1 = dims[0];
	mN2 = dims[1];
	mN3 = dims[2];
	mArr1 = src;
	mArr2 = dst;
	mBtmp1 = srcVertical;
	mBtmp2 = dstVertical;

	this->ParallelExecute(1);
}


void iGridDataHelper::TransformToPointData1(int dims[3], float *data, float *dnyq, float ps, float dp)
{
	mN1 = dims[0];
	mN2 = dims[1];
	mN3 = dims[2];
	mArr1 = data;
	mArr2 = dnyq;
	mProgStart = ps;
	mProgStep = dp;

	this->ParallelExecute(2);
}


void iGridDataHelper::TransformToPointData2(bool per[3], int cellDims[3], long cellOffset[3], long cellComponentOffset, float *cellData, float *data, float ps, float dp)
{
	mN1 = cellDims[0];
	mN2 = cellDims[1];
	mN3 = cellDims[2];
	mOffset = cellOffset;
	mComponentOffset = cellComponentOffset;
	mPeriodic = per;
	mArr1 = cellData;
	mArr2 = data;
	mProgStart = ps;
	mProgStep = dp;

	this->ParallelExecute(3);
}


void iGridDataHelper::TransformToPointData3(int cellToPointMode, int numCom, bool per[3], int cellDims[3], long cellOffset[3], long cellComponentOffset, float *cellData, int pointDims[3], long pointOffset[3], long pointComponentOffset, float *pointData, float ps, float dp)
{
	mMode = cellToPointMode;
	mNc = numCom;
	mN1 = cellDims[0];
	mN2 = cellDims[1];
	mN3 = cellDims[2];
	mOffset = cellOffset;
	mComponentOffset = cellComponentOffset;
	mPeriodic = per;
	mArr1 = cellData;
	mArr2 = pointData;
	mProgStart = ps;
	mProgStep = dp;
	mDims2 = pointDims;
	mOffset2 = pointOffset;
	mComponentOffset2 = pointComponentOffset;

	this->ParallelExecute(4);
}


void iGridDataHelper::ShiftData(int numCom, int dim, int dn, int dataDims[3], float *data, float *buf)
{
	mNc = numCom;
	mDims2 = dataDims;
	mItmp2 = dn;
	mItmp1 = dim;
	mArr1 = data;
	mArr2 = buf;

	this->ParallelExecute(5);
}


int iGridDataHelper::ExecuteStep(int step, iParallel::ProcessorInfo &p)
{
	switch(step)
	{
	case 1:
		{
			this->ExecuteCopyArray(p);
			return 0;
		}
	case 2:
		{
			this->ExecuteTransformToPointData1(p);
			return 0;
		}
	case 3:
		{
			this->ExecuteTransformToPointData2(p);
			return 0;
		}
	case 4:
		{
			this->ExecuteTransformToPointData3(p);
			return 0;
		}
	case 5:
		{
			this->ExecuteShiftData(p);
			return 0;
		}
	default: 
		{
#ifdef I_CHECK1
			IERROR_REPORT_BUG;
#endif
			return 2;
		}
	}
}


void iGridDataHelper::ExecuteCopyArray(iParallel::ProcessorInfo &p)
{
	long size = (long)mN1*(long)mN2*(long)mN3;
	long ntot = mNc*size; 

	if(mBtmp1 == mBtmp2)
	{
		long nbeg, nend, nlen;
		iParallel::SplitRange(p,ntot,nbeg,nend,nlen);
		memcpy(mArr2+nbeg,mArr1+nbeg,sizeof(float)*nlen);
	}
	else
	{
		int i, j, k, n, kbeg, kend, kstp;
		iParallel::SplitRange(p,mN3,kbeg,kend,kstp);
		long ls, ld, ll;

		if(mBtmp1)
		{
			for(n=0; n<mNc; n++)
			{
				for(k=kbeg; k<kend; k++)
				{
					for(j=0; j<mN2; j++)
					{
						ll = mN1*(j+mN2*k);
						ls = n + mNc*ll;
						ld = ll + n*size;
						for(i=0; i<mN1; i++)
						{
							mArr2[i+ld] = mArr1[mNc*i+ls];
						}
					}
				}
			}
		}
		else
		{
			for(k=kbeg; k<kend; k++)
			{
				for(j=0; j<mN2; j++)
				{
					for(i=0; i<mN1; i++)
					{
						ls = i + mN1*(j+mN2*k);
						ld = mNc*ls;
						for(n=0; n<mNc; n++)
						{
							mArr2[n+ld] = mArr1[ls+n*size];
						}
					}
				}
			}
		}
	}
}

//
//  Fourier shift
//
void iGridDataHelper::ExecuteTransformToPointData1(iParallel::ProcessorInfo &p)
{
	int i, j, k, jj, kk;
	long lf;
	int kbeg, kend, kstp;
	iParallel::SplitRange(p,mN3,kbeg,kend,kstp);

	int n12;
	if(mN1%2 == 0) n12 = mN1/2; else n12 = (mN1-1)/2;
	float progScale = mProgStep/kstp;

	float akx, aky, akz, cc, ss, dr, di;
	for(k=kbeg; k<kend; k++) 
	{
		if(this->IsMaster(p)) mSubject->GetObserver()->SetProgress(mProgStart+progScale*(k-kbeg));
		if(mSubject->GetObserver()->IsAborted()) return;
		kk = k; if(k >= mN3/2) kk -= mN3;
		akz = 6.2831854/mN3*kk;
		for(j=0; j<mN2; j++)
		{
			jj = j; if(j >= mN2/2) jj -= mN2;
			aky = 6.2831854/mN2*jj;
			lf = mN1*(j+mN2*k);
			for(i=0; i<n12; i++)
			{
				akx = 6.2831854/mN1*i;
				cc = cos(0.5*(akx+aky+akz));
				ss = sin(0.5*(akx+aky+akz));
				dr = mArr1[2*i+0+lf];
				di = mArr1[2*i+1+lf]; 
				mArr1[2*i+0+lf] = dr*cc + di*ss;
				mArr1[2*i+1+lf] = di*cc - dr*ss; 
			}
			//
			//  Nyquist frequency
			//
			lf = 2*(j+mN2*k);
			{
				akx = 3.1415927;
				cc = cos(0.5*(akx+aky+akz));
				ss = sin(0.5*(akx+aky+akz));
				dr = mArr2[0+lf];
				di = mArr2[1+lf]; 
				mArr2[0+lf] = dr*cc + di*ss;
				mArr2[1+lf] = di*cc - dr*ss; 
			}
		}
	}
}

//
//  Limit with the cell data
//
void iGridDataHelper::ExecuteTransformToPointData2(iParallel::ProcessorInfo &p)
{
	int i, j, k, m;
	long lim, ljm, lkm, lip, ljp, lkp;  // cell offSets
	float vmin, vmax, vv, v[8];
	long lf;
	int kbeg, kend, kstp;
	iParallel::SplitRange(p,mN3,kbeg,kend,kstp);
	
	float progScale = mProgStep/kstp;

	for(k=kbeg; k<kend; k++) 
	{
		if(mSubject->GetObserver() != 0)
		{
			if(this->IsMaster(p)) mSubject->GetObserver()->SetProgress(mProgStart+progScale*(k-kbeg));
			if(mSubject->GetObserver()->IsAborted()) return;
		}
		lkm = k - 1;
		lkp = k;
		if(lkm < 0) 
		{
			if(mPeriodic[2]) lkm += mN3; else lkm = lkp;
		}
		if(lkp >= mN3) 
		{
			if(mPeriodic[2]) lkp -= mN3; else lkp = lkm;
		}
		lkm *= mOffset[2];
		lkp *= mOffset[2];

		lkm += mComponentOffset;
		lkp += mComponentOffset;
		
		for(j=0; j<mN2; j++)
		{
			ljm = j - 1;
			ljp = j;
			if(ljm < 0) 
			{
				if(mPeriodic[1]) ljm += mN2; else ljm = ljp;
			}
			if(ljp >= mN2) 
			{
				if(mPeriodic[1]) ljp -= mN2; else ljp = ljm;
			}
			ljm *= mOffset[1];
			ljp *= mOffset[1];
			
			lf = mN1*(j+mN2*k);
			
			for(i=0; i<mN1; i++)
			{
				lim = i - 1;
				lip = i;
				if(lim < 0) 
				{
					if(mPeriodic[0]) lim += mN1; else lim = lip;
				}
				if(lip >= mN1) 
				{
					if(mPeriodic[0]) lip -= mN1; else lip = lim;
				}
				lim *= mOffset[0];
				lip *= mOffset[0];
				//
				//  8 neighbors
				//
				v[0] = mArr1[lim+ljm+lkm];
				v[1] = mArr1[lip+ljm+lkm];
				v[2] = mArr1[lim+ljp+lkm];
				v[3] = mArr1[lip+ljp+lkm];
				v[4] = mArr1[lim+ljm+lkp];
				v[5] = mArr1[lip+ljm+lkp];
				v[6] = mArr1[lim+ljp+lkp];
				v[7] = mArr1[lip+ljp+lkp];
				
				vmin = vmax = v[0];
				for(m=1; m<8; m++)
				{
					if(vmin > v[m]) vmin = v[m];
					if(vmax < v[m]) vmax = v[m];
				}
				
				vv = mArr2[i+lf];
				if(vv < vmin) vv = vmin;
				if(vv > vmax) vv = vmax;
				mArr2[i+lf] = vv; 
			}
		}
	}
}

//
//  Projection
//
void iGridDataHelper::ExecuteTransformToPointData3(iParallel::ProcessorInfo &p)
{
	int i, j, k, n, ii, jj;
	long lim, ljm, lkm, lip, ljp, lkp;  // cell offSets
	float vv, v[8];
	long lf;
	int kbeg, kend, kstp;
	iParallel::SplitRange(p,mDims2[2],kbeg,kend,kstp);
	
	float progScale = mProgStep/kstp;
	
	for(k=kbeg; k<kend; k++)
	{
		if(mSubject->GetObserver() != 0)
		{
			if(this->IsMaster(p)) mSubject->GetObserver()->SetProgress(mProgStart+progScale*(k-kbeg));
			if(mSubject->GetObserver()->IsAborted()) return;
		}
		lkm = k - 1;
		lkp = k;
		if(lkm < 0) 
		{
			if(mPeriodic[2]) lkm += mN3; else lkm = lkp;
		}
		if(lkp >= mN3) 
		{
			if(mPeriodic[2]) lkp -= mN3; else lkp = lkm;
		}
		lkm *= mOffset[2];
		lkp *= mOffset[2];
		
		for(j=0; j<mDims2[1]; j++)
		{
			ljm = j - 1;
			ljp = j;
			if(ljm < 0) 
			{
				if(mPeriodic[1]) ljm += mN2; else ljm = ljp;
			}
			if(ljp >= mN2) 
			{
				if(mPeriodic[1]) ljp -= mN2; else ljp = ljm;
			}
			ljm *= mOffset[1];
			ljp *= mOffset[1];
			
			for(i=0; i<mDims2[0]; i++)
			{
				lim = i - 1;
				lip = i;
				if(lim < 0) 
				{
					if(mPeriodic[0]) lim += mN1; else lim = lip;
				}
				if(lip >= mN1) 
				{
					if(mPeriodic[0]) lip -= mN1; else lip = lim;
				}
				lim *= mOffset[0];
				lip *= mOffset[0];
				
				lf = mOffset2[0]*i + mOffset2[1]*j + mOffset2[2]*k;
				
				for(n=0; n<mNc; n++) 
				{
					//
					//  8 neighbors
					//
					v[0] = mArr1[lim+ljm+lkm+n*mComponentOffset];
					v[1] = mArr1[lip+ljm+lkm+n*mComponentOffset];
					v[2] = mArr1[lim+ljp+lkm+n*mComponentOffset];
					v[3] = mArr1[lip+ljp+lkm+n*mComponentOffset];
					v[4] = mArr1[lim+ljm+lkp+n*mComponentOffset];
					v[5] = mArr1[lip+ljm+lkp+n*mComponentOffset];
					v[6] = mArr1[lim+ljp+lkp+n*mComponentOffset];
					v[7] = mArr1[lip+ljp+lkp+n*mComponentOffset];
					//
					//  Order data for some of modes
					//
					if(mMode>1 && mMode<5)
					{
						for(jj=1; jj<8; jj++) 
						{ 
							vv = v[jj];
							ii = jj - 1;
							while(ii>0 && v[ii] > vv) 
							{ 
								v[ii+1] = v[ii];
								ii--;
							}
							v[ii+1] = vv; 
						}
					}
					//
					//  Do projection
					//
					switch (mMode) 
					{
					case 0:  // use Fourier transform - at this moment the data are already shifted.
						{
							vv = v[7];
							break;
						}
					case 2:  // median
						{
							vv = v[3];
							break;
						}
					case 3:  // min
						{
							vv = v[0];
							break;
						}
					case 4:  // max
						{
							vv = v[7];
							break;
						}
					case 5:  // corner projection
						{
							vv = v[0];
							break;
						}
						
					case 1:  // 8-point average
					default: // is the default one
						{
							vv = 0.125*(v[0]+v[1]+v[2]+v[3]+v[4]+v[5]+v[6]+v[7]);
							break;
						}
					}
					mArr2[lf+n*mComponentOffset2] = vv;
				}
			}
		}
	}
}

//
//  Shift an nc-component vertical array (treat the mesh array as nvar 1-component vertical arrays)
//
void iGridDataHelper::ExecuteShiftData(iParallel::ProcessorInfo &p)
{
	int n, l1, l2, l, lold, add1;
	int d1 = 0, d2 = 0;
	
	int d = mItmp1;
	int dn = mItmp2;

	switch(d) 
	{
	case 0: { d1 = 1; d2 = 2; break; }
	case 1: { d1 = 0; d2 = 2; break; }
	case 2: { d1 = 0; d2 = 1; break; }
	}
	
	int ddim = mDims2[d];
	long off0 = 0L, off1 = 0L, add2;
	
	int kbeg, kend, kstp;
	iParallel::SplitRange(p,mDims2[d1],kbeg,kend,kstp);

	for(l1=kbeg; l1<kend; l1++)
	{
		if(mSubject->GetObserver() != 0)
		{
			if(this->IsMaster(p)) mSubject->GetObserver()->SetProgress((d+(float)(l1-kbeg)/(kend-kbeg))/3.0);
			if(mSubject->GetObserver()->IsAborted()) return;
		}
		for(l2=0; l2<mDims2[d2]; l2++)
		{
			switch(d) 
			{
			case 0: { off0 = mNc*(long)mDims2[0]*(l1+mDims2[1]*l2); off1 = mNc;	break; }
			case 1: { off0 = mNc*(l1+(long)mDims2[0]*mDims2[1]*l2); off1 = mNc*mDims2[0]; break; }
			case 2: { off0 = mNc*(l1+(long)mDims2[0]*l2); off1 = mNc*(long)mDims2[0]*mDims2[1]; break; }
			}
			
			for(l=0; l<ddim-1; l++) 
			{
				lold = l - dn;
				while(lold < 0) lold += ddim;
				while(lold >= ddim) lold -= ddim;
				
				add1 = mNc*l;
				add2 = off0 + off1*lold;

				iTune::CopyFloatPointers(mArr2+add1,mArr1+add2,mNc);
			}
			add1 = mNc*(ddim-1);
			for(n=0; n<mNc; n++) mArr2[n+add1] = mArr2[n];
			
			for(l=0; l<ddim; l++) 
			{
				add1 = mNc*l;
				add2 = off0 + off1*l;
				
				iTune::CopyFloatPointers(mArr1+add2,mArr2+add1,mNc);
			}
		}
	}
}


//
//  ********************************************************************
//
//  iUniformVectorsDataSubject class
//
//  ********************************************************************
//
iUniformVectorsDataSubject::iUniformVectorsDataSubject(iDataReader *r) : iGridDataSubject(r,iDataType::UniformVectors())
{
}


void iUniformVectorsDataSubject::SetIndicies(int &nvar, int &ncomp)
{
	nvar = 3;
	ncomp = 3;
}


void iUniformVectorsDataSubject::AssignBinData(int n, vtkIdType ntot, float *d)
{
	vtkIdType l;
	for(l=0; l<ntot; l++) mBuffer[n+3*l] = d[l];
}


void iUniformVectorsDataSubject::AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f)
{
	int n;
	for(n=0; n<3; n++) mBuffer[n+3*ind] = f[n];
}


void iUniformVectorsDataSubject::SetArraysAndComponents()
{
	mNumArrays = 1;
	mNumComponents = 3;
	this->GetLimits()->AddVar(0);
	if(mAdjustableLimits)
	{
		this->GetLimits()->SetMin(0,0.0f);
		this->GetLimits()->SetMax(0,sqrt(mVarMax[0]*mVarMax[0]+mVarMax[1]*mVarMax[1]+mVarMax[2]*mVarMax[2]));
	}
}


void iUniformVectorsDataSubject::AttachBuffer()
{
	//
	//  Attach the actual data
	//
	vtkIdType size = (vtkIdType)mDims[0]*mDims[1]*mDims[2]*3;
	vtkFloatArray *array = vtkFloatArray::New(); IERROR_ASSERT_NULL_POINTER(array);
	array->SetNumberOfComponents(3);
	array->SetArray(mBuffer,size,1);
	array->SetName("Vectors");
	mData->GetPointData()->SetVectors(array);
	array->Delete();
}


iDataLimits* iUniformVectorsDataSubject::CreateLimits() const
{
	return new iDataLimits(this,1,"Vector");
}


//
//  ********************************************************************
//
//  iUniformTensorsDataSubject class
//
//  ********************************************************************
//
iUniformTensorsDataSubject::iUniformTensorsDataSubject(iDataReader *r) : iGridDataSubject(r,iDataType::UniformTensors())
{
}


void iUniformTensorsDataSubject::SetIndicies(int &nvar, int &ncomp)
{
	nvar = 9;
	ncomp = 6;
}


void iUniformTensorsDataSubject::AssignBinData(int n, vtkIdType ntot, float *d)
{
	int noff1, noff2;
	vtkIdType l;

	switch (n)
	{
	case 0: { noff1 = 0; noff2 = -1; break; }
	case 1: { noff1 = 1; noff2 = 3; break; }
	case 2: { noff1 = 2; noff2 = 6; break; }
	case 3: { noff1 = 4; noff2 = -1; break; }
	case 4: { noff1 = 5; noff2 = 7; break; }
	case 5: { noff1 = 8; noff2 = -1; break; }
	default: 
		{
			noff1 = 0; noff2 = -1;
			IERROR_REPORT_ERROR("Bug detected.");
		}
	}

	for(l=0; l<ntot; l++) mBuffer[noff1+9*l] = d[l];
	if(noff2 > 0) for(l=0; l<ntot; l++) mBuffer[noff2+9*l] = d[l];
}


void iUniformTensorsDataSubject::AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f)
{
	//
	//  Fill in 3x3 tensor with 6 components
	//
	mBuffer[0+9*ind] = f[0];
	mBuffer[1+9*ind] = f[1];
	mBuffer[2+9*ind] = f[2];
	mBuffer[3+9*ind] = f[1];
	mBuffer[4+9*ind] = f[3];
	mBuffer[5+9*ind] = f[4];
	mBuffer[6+9*ind] = f[2];
	mBuffer[7+9*ind] = f[4];
	mBuffer[8+9*ind] = f[5];
}


void iUniformTensorsDataSubject::SetArraysAndComponents()
{
	mNumArrays = 1;
	mNumComponents = 9;
	this->GetLimits()->AddVar(0);
	if(mAdjustableLimits)
	{
		this->GetLimits()->SetMin(0,0.0f);
		this->GetLimits()->SetMax(0,sqrt(mVarMax[0]*mVarMax[0]+mVarMax[1]*mVarMax[1]+mVarMax[2]*mVarMax[2]+mVarMax[3]*mVarMax[3]+mVarMax[4]*mVarMax[4]+mVarMax[5]*mVarMax[5]+mVarMax[6]*mVarMax[6]+mVarMax[7]*mVarMax[7]+mVarMax[8]*mVarMax[8]));
	}
}


void iUniformTensorsDataSubject::AttachBuffer()
{
	//
	//  Attach the actual data
	//
	vtkIdType size = (vtkIdType)mDims[0]*mDims[1]*mDims[2]*9;
	vtkFloatArray *array = vtkFloatArray::New(); IERROR_ASSERT_NULL_POINTER(array);
	array->SetNumberOfComponents(9);
	array->SetArray(mBuffer,size,1);
	array->SetName("Tensors");
	mData->GetPointData()->SetTensors(array);
	array->Delete();
}


iDataLimits* iUniformTensorsDataSubject::CreateLimits() const
{
	return new iDataLimits(this,1,"Tensor");
}


//
//  ********************************************************************
//
//  iUniformScalarsDataSubject class
//
//  ********************************************************************
//
//  Helper class for parallel executions
//
class iUniformScalarsDataHelper : protected iParallelWorker
{

public:

	iUniformScalarsDataHelper(iUniformScalarsDataSubject *subject);

	void OperateOnData1(int nvar, int dims[3], iExpressionParser *calc, bool doVector, float ps, float dp);
	void OperateOnData2(int nvar, int dims[3], float *fMin, float *fMax, bool &overflow, float ps, float dp);

protected:

	virtual int ExecuteStep(int step, iParallel::ProcessorInfo &p);

	void ExecuteOperateOnData1(iParallel::ProcessorInfo &p);
	void ExecuteOperateOnData2(iParallel::ProcessorInfo &p);

	iUniformScalarsDataSubject *mSubject;
	iProgressEventObserver *mObserver;
	iExpressionParser *mCalculator;

	int mNumVars, mNumProcs, *mDims;

	iBuffer<float> mMin, mMax;
	float mProgStart, mProgStep;

	bool mDoVector, mOverflow;
};


//
//  Main class
//
iUniformScalarsDataSubject::iUniformScalarsDataSubject(iDataReader *r, iUniformVectorsDataSubject *vs) : iGridDataSubject(r,iDataType::UniformScalars())
{
	mVectorFieldSubject = vs;

	mCalculatorOutputVariable = 0;
//	mCalculatorFunction = "";

	mCalculator = new iExpressionParser; IERROR_ASSERT_NULL_POINTER(mCalculator);
	mHelper2 = new iUniformScalarsDataHelper(this); IERROR_ASSERT_NULL_POINTER(mHelper2);
}


iUniformScalarsDataSubject::~iUniformScalarsDataSubject()
{
	delete mHelper2;
	delete mCalculator;
}


void iUniformScalarsDataSubject::SetIndicies(int &nvar, int &ncomp)
{
	//
	//  Try to set all the records from the file. DataLimits will either expand to accommodate 
	//  all of them or limit the allowed number to the number of listed records.
	//
	this->GetLimits()->BlockNotifications(true);
	this->GetLimits()->AssignVars(nvar);
	this->GetLimits()->BlockNotifications(false);
	ncomp = nvar = this->GetLimits()->GetNumVars();
}


void iUniformScalarsDataSubject::AssignBinData(int n, vtkIdType ntot, float *d)
{
	memcpy((void *)(mBuffer+n*ntot),(void *)d,ntot*sizeof(float));
}


void iUniformScalarsDataSubject::AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f)
{
	int n;

	for(n=0; n<nvar; n++) mBuffer[ind+n*ntot] = f[n];
}


void iUniformScalarsDataSubject::AttachBuffer()
{
	//
	//  Attach the actual data
	//
	int i;
	vtkIdType size = (vtkIdType)mDims[0]*mDims[1]*mDims[2];
	vtkFloatArray *array;
	iString name = "Scalars";
	for(i=0; i<mNumVars; i++)
	{
		array = vtkFloatArray::New(); IERROR_ASSERT_NULL_POINTER(array);
		array->SetNumberOfComponents(1);
		array->SetArray(mBuffer+i*size,size,1);
		array->SetName((name+iString::FromNumber(i)).ToCharPointer());
		mData->GetPointData()->AddArray(array);
		array->Delete();
	}

	this->OperateOnData();
}


void iUniformScalarsDataSubject::SetArraysAndComponents()
{
	mNumArrays = mNumVars;
	mNumComponents = 1;
	this->GetLimits()->AssignVars(mNumArrays);
	//
	//  Limits are set after the data are operated upon
	//
}


iDataLimits* iUniformScalarsDataSubject::CreateLimits() const
{
	return new iDataLimits(this,3,"Variable");
}


void iUniformScalarsDataSubject::SetCalculatorOutputVariable(int n)
{
	if(n>=0 && n<this->GetLimits()->GetNumVars()) 
	{
		mCalculatorOutputVariable = n;
		this->GetLimits()->RestoreNames();
	}
}


void iUniformScalarsDataSubject::SetCalculatorFunction(const iString &s)
{
	mCalculatorFunction = s;
	if(s.IsEmpty()) this->GetLimits()->RestoreNames();
}


void iUniformScalarsDataSubject::OperateOnData()
{
	int n;
	float fMin = iMath::_LargeFloat, fMax = -iMath::_LargeFloat, f = 0.0;

	int nvar = this->GetLimits()->GetNumVars();

	iDataLimits *slim = this->GetLimits();
	iDataLimits *vlim = mVectorFieldSubject->GetLimits();
	slim->BlockNotifications(true);

	//
	//  Do the array function first
	//
	if(mCalculatorOutputVariable>=0 && mCalculatorOutputVariable<nvar && !mCalculatorFunction.IsEmpty())
	{
		this->GetObserver()->Started(iProgressEventObserver::_Operating);
		this->GetObserver()->SetProgress(0.0);

		mCalculator->SetFunction(mCalculatorFunction);

		bool doVector = false;
		if(mVectorFieldSubject->IsThereData() && this->GetDim(0)==mVectorFieldSubject->GetDim(0) && this->GetDim(1)==mVectorFieldSubject->GetDim(1) && this->GetDim(2)==mVectorFieldSubject->GetDim(2)) doVector = true;

		mHelper2->OperateOnData1(nvar,mDims,mCalculator,doVector,0.0,1.0);
	}

	this->GetObserver()->Started(iProgressEventObserver::_Formatting);
	this->GetObserver()->SetProgress(0.0);

	//
	//  Format everything in parallel
	//
	mVarMin.Extend(nvar);
	mVarMax.Extend(nvar);
	mHelper2->OperateOnData2(nvar,mDims,mVarMin,mVarMax,mOverflow,0.0,1.0);

#ifdef I_CHECK2
	//
	//  Format everything serially
	//
	vtkIdType lMin, lMax, size = (vtkIdType)mDims[0]*mDims[1]*mDims[2];
	for(n=0; n<nvar; n++)
	{
		fMin = iMath::_LargeFloat;
		fMax = -iMath::_LargeFloat;
		long l;
		float *p = mBuffer + n*size;
		for(l=0; l<size; l++)
		{
			f = p[l];
			if(f > fMax)
			{
				lMax = l;
				fMax = f;
			}
			if(f < fMin)
			{
				lMin = l;
				fMin = f;
			}
		}
		if(fabs(mVarMin[n]-fMin)>iMath::_FloatRes || fabs(mVarMax[n]-fMax)>iMath::_FloatRes)
		{
			IERROR_REPORT_BUG;
		}
	}
#endif

	this->GetObserver()->SetProgress(1.0);
	this->GetObserver()->Finished();

	slim->BlockNotifications(false);

	//
	//  Correct limits in log stretching
	//
	if(mAdjustableLimits)
	{
		for(n=0; n<nvar; n++)
		{
			fMin = mVarMin[n];
			fMax = mVarMax[n];
			if(slim->GetStretch(n) == 1)
			{
				fMin = 0.1*floor(10.0*fMin);
				fMax = 0.1*floor(1+10.0*fMax);
			}
			slim->SetMin(n,fMin);
			slim->SetMax(n,fMax);
		}
	}
}


//
//  Helper class
//
iUniformScalarsDataHelper::iUniformScalarsDataHelper(iUniformScalarsDataSubject *subject) : iParallelWorker(subject->GetViewModule()->GetParallelManager())
{
	mSubject = subject;
	mObserver = subject->GetObserver();
}


void iUniformScalarsDataHelper::OperateOnData1(int nvar, int dims[3], iExpressionParser *calc, bool doVector, float ps, float dp)
{
	mNumVars = nvar;
	mDims = dims;
	mCalculator = calc;
	mDoVector = doVector;

	mProgStart = ps;
	mProgStep = dp;

	this->ParallelExecute(1);
}


void iUniformScalarsDataHelper::OperateOnData2(int nvar, int dims[3], float *fMin, float *fMax, bool &overflow, float ps, float dp)
{
	int i, n;

	mNumVars = nvar;
	mDims = dims;
	mNumProcs = this->GetManager()->GetNumberOfProcessors();
	mMin.Extend(mNumVars*mNumProcs);
	mMax.Extend(mNumVars*mNumProcs);
	
	mProgStart = ps;
	mProgStep = dp;
	mOverflow = false;

	this->ParallelExecute(2);

	for(n=0; n<nvar; n++)
	{
		fMin[n] = mMin[n];
		fMax[n] = mMax[n];
		for(i=1; i<mNumProcs; i++)
		{
			if(fMin[n] > mMin[n+nvar*i]) fMin[n] = mMin[n+nvar*i];
			if(fMax[n] < mMax[n+nvar*i]) fMax[n] = mMax[n+nvar*i];
		}
	}

	overflow = mOverflow;
}


int iUniformScalarsDataHelper::ExecuteStep(int step, iParallel::ProcessorInfo &p)
{
	switch(step)
	{
	case 1:
		{
			this->ExecuteOperateOnData1(p);
			return 0;
		}
	case 2:
		{
			if(mNumProcs != p.NumProcs) return 3;
			this->ExecuteOperateOnData2(p);
			return 0;
		}
	default: 
		{
#ifdef I_CHECK1
			IERROR_REPORT_BUG;
#endif
			return 2;
		}
	}
}


//
//  Operate on mesh data
//
void iUniformScalarsDataHelper::ExecuteOperateOnData1(iParallel::ProcessorInfo &p)
{
	char arName[5];
	double res;
	
	arName[0] = 'V';
	arName[1] = 'a';
	arName[2] = 'r';
	arName[4] = 0;

	float *arr1 = mSubject->GetDataPointer();
	float *arr2 = mSubject->mVectorFieldSubject->GetDataPointer();

	if(arr1==0 || (mDoVector && arr2==0)) return;

	long size = (long)mDims[0]*mDims[1]*mDims[2];
	long l, loff = mSubject->GetCalculatorOutputVariable()*size;
	int n;
	long kbeg, kend, kstp;
	iParallel::SplitRange(p,size,kbeg,kend,kstp);

	float progScale = mProgStep/kstp;

	for(l=kbeg; l<kend; l++) 
	{
		if(l%1000==0 && mObserver!=0)
		{
			if(this->IsMaster(p)) mObserver->SetProgress(mProgStart+progScale*(l-kbeg));
			if(mObserver->IsAborted()) return;
		}
		for(n=0; n<mNumVars; n++)
		{
			arName[3] = (char)((int)'1'+n);
			mCalculator->SetScalarVariableValue(arName,arr1[n*size+l]);
		}
		if(mDoVector)
		{
			mCalculator->SetVectorVariableValue("Vector",arr2[0+3*l],arr2[1+3*l],arr2[2+3*l]);
		} 
		else 
		{
			mCalculator->SetVectorVariableValue("Vector",0.0,0.0,0.0);
		}
		
		if(mCalculator->GetScalarResult(res)) arr1[loff+l] = res;
	}
}


void iUniformScalarsDataHelper::ExecuteOperateOnData2(iParallel::ProcessorInfo &p)
{
	float fMin, fMax;
	long size = (long)mDims[0]*mDims[1]*mDims[2];
	long kbeg, kend, kstp;
	iParallel::SplitRange(p,size,kbeg,kend,kstp);

	//
	//  Format everything
	//
	iDataLimits *lim = mSubject->GetLimits();

	long l;
	int n;

	float *arr1, *arr0 = mSubject->GetDataPointer();
	if(arr0 == 0) return;

	float progScale = mProgStep/(mNumVars*kstp);

	for(n=0; n<mNumVars; n++)
	{
		fMin = iMath::_LargeFloat;
		fMax = -iMath::_LargeFloat;
		arr1 = arr0 + n*size;

		for(l=kbeg; l<kend; l++)
		{
			if(l%10000==0 && mObserver!=0)
			{
				if(this->IsMaster(p)) mObserver->SetProgress(mProgStart+progScale*(l-kbeg+n*kstp));
				if(mObserver->IsAborted()) return;
			}
			if(arr1[l] < -iMath::_LargeFloat)
			{
				mOverflow = true;
				arr1[l] = -iMath::_LargeFloat;
			}
			if(arr1[l] > iMath::_LargeFloat)
			{
				mOverflow = true;
				arr1[l] = iMath::_LargeFloat;
			}
			if(arr1[l] > fMax) fMax = arr1[l];
			if(arr1[l] < fMin) fMin = arr1[l];
		}
		
		if(mNumProcs == p.NumProcs)
		{
			mMin[n+mNumVars*p.ThisProc] = fMin;
			mMax[n+mNumVars*p.ThisProc] = fMax;
		}
	}
}


//
//  Two helper functions
//
void iGridDataSubject::GetCICInterpolation(bool periodic[3], int dims[3], double org[3], double spa[3], float pos[3], int ijk1[3], int ijk2[3], double d1[3], double d2[3])
{
	int i;
	for(i=0; i<3; i++)
	{
		d1[i] = (pos[i]-org[i])/spa[i];
		ijk1[i] = (int)floor(d1[i]);
		d2[i] = d1[i] - ijk1[i];
		d1[i] = 1.0 - d2[i];
		
		if(periodic[i])
		{
			while(ijk1[i] < 0) ijk1[i] += dims[i];  // in case it is extended 
			while(ijk1[i] >= dims[i]) ijk1[i] -= dims[i];
		}
		else
		{
			if(ijk1[i] < 0) ijk1[i] = 0;  
			if(ijk1[i] >= dims[i]) ijk1[i] = dims[i] - 1;
		}

		ijk2[i] = ijk1[i] + 1;

		if(periodic[i])
		{
			while(ijk2[i] >= dims[i]) ijk2[i] -= dims[i]; 
		}
		else
		{
			if(ijk2[i] >= dims[i]) ijk2[i] = dims[i] - 1;
		}
	}
}


void iGridDataSubject::GetCICInterpolation(bool periodic[3], int dims[3], double org[3], double spa[3], double pos[3], int ijk1[3], int ijk2[3], double d1[3], double d2[3])
{
	int i;
	for(i=0; i<3; i++)
	{
		d1[i] = (pos[i]-org[i])/spa[i];
		ijk1[i] = (int)floor(d1[i]);
		d2[i] = d1[i] - ijk1[i];
		d1[i] = 1.0 - d2[i];
		
		if(periodic[i])
		{
			while(ijk1[i] < 0) ijk1[i] += dims[i];  // in case it is extended 
			while(ijk1[i] >= dims[i]) ijk1[i] -= dims[i];
		}
		else
		{
			if(ijk1[i] < 0) ijk1[i] = 0;  
			if(ijk1[i] >= dims[i]) ijk1[i] = dims[i] - 1;
		}

		ijk2[i] = ijk1[i] + 1;

		if(periodic[i])
		{
			while(ijk2[i] >= dims[i]) ijk2[i] -= dims[i]; 
		}
		else
		{
			if(ijk2[i] >= dims[i]) ijk2[i] = dims[i] - 1;
		}
	}
}
