/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iviewmodule.h"


#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "imarkerviewobject.h"
#include "iviewmodule.h"
#include "iviewobjectfamily.h"
#include "iviewobjectobserver.h"
#include "iviewobjectreplicator.h"

#include "iarraytemplate.h"


#include "iparameter.h"
using namespace iParameter;


IVIEWOBJECT_DEFINE_TYPE(iViewObject,ViewObject,-vo);  // helper type

IOBJECT_DEFINE_KEY(iViewObject,ClipPlane,cp,Bool,1);
IOBJECT_DEFINE_KEY(iViewObject,DataPresent,dp,Bool,1);
IOBJECT_DEFINE_KEY(iViewObject,ActiveDataType,dt,Int,1);
IOBJECT_DEFINE_KEY(iViewObject,Ambient,ma,Float,1);
IOBJECT_DEFINE_KEY(iViewObject,Diffuse,md,Float,1);
IOBJECT_DEFINE_KEY(iViewObject,SpecularPower,mp,Float,1);
IOBJECT_DEFINE_KEY(iViewObject,Specular,ms,Float,1);
IOBJECT_DEFINE_KEY(iViewObject,Shading,mw,Bool,1);
IOBJECT_DEFINE_KEY(iViewObject,NumReplicas,nr,Int,6);
IOBJECT_DEFINE_KEY(iViewObject,Ready,r,Bool,1);
IOBJECT_DEFINE_KEY(iViewObject,Visible,vis,Bool,1);


iViewObject::iViewObject(iViewModule *vm, const iDataInfo &info, unsigned int flags) : iDataConsumer(vm,info), iReplicated(0,false), mPosition(vm), mAlwaysShaded((flags&_ViewObjectAlwaysShaded)!=0U), mNoReplicating((flags&_ViewObjectNoReplicating)!=0U)
{
	mName = "iViewObject";
	mObjectType = _ObjectTypeUndefined;

	mInit = true;
	
	mClipPlaneOn = false;

	mShading = mAlwaysShaded;
	mAmbient = 0.3f;
	mDiffuse = 0.4f;
	mSpecular = 0.2f;
	mSpecularPower = 50.0;

	mAttachedMarker = 0;
	mAttachedMarkerIndex = -1;

	mObjectObserver = iViewObjectObserver::New(this); IERROR_ASSERT_NULL_POINTER(mObjectObserver);
	if(mNoReplicating)
	{
		mReplicator = 0;
	}
	else
	{
		mReplicator = iViewObjectReplicator::New(this); IERROR_ASSERT_NULL_POINTER(mReplicator);
	}

	mDefaultColor = iColor(0,150,200);
}


iViewObject::~iViewObject()
{
	if(mAttachedMarker != 0) mAttachedMarker->DetachViewObject(this);
	mObjectObserver->Delete();
	if(mReplicator != 0) mReplicator->Delete();
}


void iViewObject::Delete()
{
	this->ShowColorBars(false);
	iDeletable::Delete();
}


bool iViewObject::IsA(const char *cn) const
{
	return (strcmp(mName,cn) == 0);
}


void iViewObject::SetActiveDataTypeIndex(int v)
{
	if(v != mActiveDataTypeIndex)
	{
		this->ShowColorBars(false);
		iDataConsumer::SetActiveDataTypeIndex(v);
		this->ShowColorBars(true);
		this->ClearCache();
	}
}


iDataLimits* iViewObject::GetLimits(const iDataType &type) const
{
	return this->GetViewModule()->GetReader()->GetLimits(type);
}


void iViewObject::SetShading(bool s)
{
	if(!mAlwaysShaded)
	{
		mShading = s;
		this->UpdateMaterialProperties();
		this->ClearCache();
	}
}


void iViewObject::SetAmbient(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	mAmbient = v;
	this->UpdateMaterialProperties();
	this->ClearCache();
}

void iViewObject::SetDiffuse(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	mDiffuse = v;
	this->UpdateMaterialProperties();
	this->ClearCache();
}


void iViewObject::SetSpecular(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	mSpecular = v;
	this->UpdateMaterialProperties();
	this->ClearCache();
}


void iViewObject::SetSpecularPower(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>100.0) ? 100.0 : v;
	mSpecularPower = v;
	this->UpdateMaterialProperties();
	this->ClearCache();
}


void iViewObject::UpdateAfterMarkerDelete(iMarkerViewObject *m)
{
	if(m==0 || mAttachedMarker==0) return;

	if(mAttachedMarker == m)
	{
		mAttachedMarker = 0;
		mAttachedMarkerIndex = -1;
	}
	else 
	{
		int i;
		for(i=0; i<=this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex(); i++)
		{
			if(mAttachedMarker == this->GetViewModule()->GetMarkerFamily()->GetMember(i))
			{
				mAttachedMarkerIndex = i;
				return;
			}
		}
#ifdef I_CHECK1
		IERROR_REPORT_BUG;
#endif
		mAttachedMarker = 0;
		mAttachedMarkerIndex = -1;
	}
}


void iViewObject::PackStateBody(iString &s) const
{
	this->PackValue(s,KeyActiveDataType(),mActiveDataTypeIndex);
	this->PackValue(s,KeyClipPlane(),mClipPlaneOn);
	if(!mAlwaysShaded)
	{
		this->PackValue(s,KeyShading(),mShading);
	}
	this->PackValue(s,KeyAmbient(),mAmbient);
	this->PackValue(s,KeyDiffuse(),mDiffuse);
	this->PackValue(s,KeySpecular(),mSpecular);
	this->PackValue(s,KeySpecularPower(),mSpecularPower);
	if(!mNoReplicating)
	{
		this->PackValue(s,KeyNumReplicas(),mNumReplicas,6);
	}
	this->PackValue(s,KeyVisible(),this->IsVisible());
	this->PackValue(s,KeyDataPresent(),this->IsDataPresent());
	this->PackValue(s,KeyReady(),this->IsReady());

	this->ViewObjectPackStateBody(s);
}


void iViewObject::UnPackStateBody(const iString &s)
{
	int i, d[6]; bool b; float f;

	if(this->UnPackValue(s,KeyActiveDataType(),i)) this->SetActiveDataTypeIndex(i);
	if(this->UnPackValue(s,KeyClipPlane(),b)) this->ShowClipPlane(b);
	if(!mAlwaysShaded)
	{
		if(this->UnPackValue(s,KeyShading(),b)) this->SetShading(b);
	}
	if(this->UnPackValue(s,KeyAmbient(),f)) this->SetAmbient(f);
	if(this->UnPackValue(s,KeyDiffuse(),f)) this->SetDiffuse(f);
	if(this->UnPackValue(s,KeySpecular(),f)) this->SetSpecular(f);
	if(this->UnPackValue(s,KeySpecularPower(),f)) this->SetSpecularPower(f);

	if(!mNoReplicating)
	{
		for(i=0; i<6; i++) d[i] = mNumReplicas[i];
		if(this->UnPackValue(s,KeyNumReplicas(),d,6)) this->SetNumReplicas(d);
	}

	this->ViewObjectUnPackStateBody(s);
}


void iViewObject::SyncWithDataBody(int info, void *data)
{
	this->ClearCache();

	if(this->IsVisible())
	{
		//
		//  If we are visible, but data disappeared, hide us.
		//
		if(!this->IsReady()) this->ShowBody(false);
	}
	else
	{
		//
		//  If we are not visible, automatically switch to the first data type that has data
		//
		if(!this->IsDataPresent())
		{
			int i;
			for(i=0; i<mPrimaryDataInfo->Count(); i++)
			{
				if(this->GetViewModule()->GetReader()->IsThereData(mPrimaryDataInfo->Type(i)))
				{
					this->SetActiveDataType(mPrimaryDataInfo->Type(i));
					break;
				}
			}
		}
	}
	this->ViewObjectSyncWithData(info,data);
}


void iViewObject::Show(bool s)
{
	if(s && mInit)
	{
		this->FinishInitialization();
		mInit = false;
	}
	if(!mInit)
	{
		this->SyncWithData(0,0);
		this->ShowBody(s && this->CanBeShown());
		this->ClearCache();
	}
}


bool iViewObject::IsReady() const
{
	return this->CanBeShown() && this->IsDataPresent();
}


void iViewObject::UpdateDataType()
{
	// Nothing to update here
}


void iViewObject::UpdateReplicas()
{
	mReplicator->Update();
	this->ClearCache();
}

