/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
// A small Set of common functionality for all IFrIT viewization objects
//

#ifndef IVIEWOBJECT_H
#define IVIEWOBJECT_H


#include "iobject.h"
#include "idataconsumer.h"
#include "ireplicated.h"
#include "ideletable.h"


#include "iarray.h"
#include "iposition.h"


class iDataLimits;
class iDataReader;
class iMarkerViewObject;
class iViewObjectObserver;
class iViewObjectReplicator;


//
//  Useful macro to declare all keys that have to be present in children too
//
#define IVIEWOBJECT_DECLARE_INHERITED_KEYS \
	static const iObjectKey& KeyActiveDataType(); \
	static const iObjectKey& KeyClipPlane(); \
	static const iObjectKey& KeyShading(); \
	static const iObjectKey& KeyAmbient(); \
	static const iObjectKey& KeyDiffuse(); \
	static const iObjectKey& KeySpecular(); \
	static const iObjectKey& KeySpecularPower(); \
	static const iObjectKey& KeyNumReplicas(); \
	static const iObjectKey& KeyVisible(); \
	static const iObjectKey& KeyDataPresent(); \
	static const iObjectKey& KeyReady()


class iViewObject : public iObject, public iDataConsumer, public iReplicated, public iDeletable
{
	
	friend class iControlModule;
	friend class iViewObjectComponent;

public:
	
	virtual void Delete();
	static const iObjectType& Type();

	IVIEWOBJECT_DECLARE_INHERITED_KEYS;

	inline bool GetClipPlane() const { return mClipPlaneOn; }
		
	void SetShading(bool);
	inline bool GetShading() const { return mShading; }
	
	void SetAmbient(float v);
	inline float GetAmbient() const { return mAmbient; }

	void SetDiffuse(float v);
	inline float GetDiffuse() const { return mDiffuse; }

	void SetSpecular(float v);
	inline float GetSpecular() const { return mSpecular; }

	void SetSpecularPower(float v);
	inline float GetSpecularPower() const { return mSpecularPower; }
	
	virtual void UpdateMaterialProperties() = 0;

	virtual void ShowClipPlane(bool s) = 0;
	virtual void ShowColorBars(bool s) = 0;

	virtual bool IsInitialized() const { return !mInit; }
	virtual bool IsVisible() const = 0;
	virtual const iObjectType& RealType();
	bool IsReady() const;

	void Show(bool s);
	virtual void Reset() = 0;

	virtual float GetMemorySize() const = 0;

	bool IsA(const char *cn) const;
	inline int GetObjectType() const { return mObjectType; }

	inline const iPosition& GetPosition() const { return mPosition; }
	inline iViewObjectReplicator* GetReplicator() const { return mReplicator; }

	virtual void UpdateAfterMarkerMove(){}
	virtual void UpdateAfterMarkerAdd(iMarkerViewObject *){}
	virtual void UpdateAfterMarkerDelete(iMarkerViewObject *);

	virtual void SetActiveDataTypeIndex(int v);

	virtual bool CanBeShown() const = 0;

protected:
	
	iViewObject(iViewModule *vm, const iDataInfo &info, unsigned int flags);
	virtual ~iViewObject();

	virtual void ViewObjectPackStateBody(iString &s) const = 0;
	virtual void ViewObjectUnPackStateBody(const iString &s) = 0;

	virtual void SyncWithDataBody(int info, void *data);
    virtual void ViewObjectSyncWithData(int info, void *data) = 0;
	virtual void UpdateDataType();
	virtual void UpdateReplicas();

	inline iDataLimits* GetLimits() const { return this->GetLimits(this->GetActiveDataType()); }
	iDataLimits* GetLimits(const iDataType &type) const;

	virtual void ShowBody(bool s) = 0;
	virtual void FinishInitialization() = 0;

	iViewObjectObserver *mObjectObserver;
	iViewObjectReplicator *mReplicator;
	const char *mName;
	int mObjectType, mAttachedMarkerIndex;
	iPosition mPosition;

	bool mInit;
	bool mClipPlaneOn;
	const bool mAlwaysShaded, mNoReplicating;

	iMarkerViewObject *mAttachedMarker;
	//
	//  Material properties
	//
	bool mShading;
	float mAmbient;
	float mDiffuse;
	float mSpecular;
	float mSpecularPower;

	iColor mDefaultColor;

private:

	virtual void PackStateBody(iString &s) const;
	virtual void UnPackStateBody(const iString &s);
};


//
//  Useful macros to declare all members that have to be overwritten in children
//
#define IVIEWOBJECT_DEFINE_TYPE(_type_,_fname_,_sname_) \
	IOBJECT_DEFINE_TYPE(_type_,_fname_,_sname_,iObjectType::_Visual); \
	const iObjectType& _type_::RealType(){ return _type_::Type(); }

#define IVIEWOBJECT_DECLARE_INHERITED_MEMBERS \
	virtual void UpdateMaterialProperties(); \
	virtual void Reset(); \
	virtual void ShowClipPlane(bool s); \
	virtual void ShowColorBars(bool s); \
	virtual float GetMemorySize() const; \
	virtual bool IsVisible() const; \
	virtual bool CanBeShown() const; \
	protected: \
	virtual void ViewObjectPackStateBody(iString &s) const; \
	virtual void ViewObjectUnPackStateBody(const iString &s); \
	virtual void ViewObjectSyncWithData(int info, void *data); \
	virtual void ShowBody(bool s); \
	virtual void FinishInitialization(); \
	public: \
	virtual const iObjectType& RealType()

#define IVIEWOBJECT_DEFINE_INHERITED_KEYS_COMMON(_type_) \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,ActiveDataType,dt,Int,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,ClipPlane,cp,Bool,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,Ambient,ma,Float,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,Diffuse,md,Float,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,Specular,ms,Float,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,SpecularPower,mp,Float,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,Visible,vis,Bool,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,DataPresent,dp,Bool,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,Ready,r,Bool,1)

#define IVIEWOBJECT_DEFINE_INHERITED_KEYS_SHADING(_type_) \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,Shading,mh,Bool,1)

#define IVIEWOBJECT_DEFINE_INHERITED_KEYS_REPLICATING(_type_) \
	IOBJECT_DEFINE_INHERITED_KEY(iViewObject,_type_,NumReplicas,nr,Int,6)

#endif // IVIEWOBJECT_H

