/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggframerangemapping.h"


#include "icontrolmodule.h"
#include "ierror.h"
#include "imath.h"
#include "ihistogram.h"
#include "iobjecthelp.h"
#include "irangecollection.h"
#include "irangemapping.h"
#include "ishell.h"

#include "iggframehistogramareaextra.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeybutton.h"
#include "iggwidgetotherbutton.h"

#include "ibgwidgetareasubject.h"
#include "ibgwidgetentrysubject.h"

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;

#include <stdlib.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


namespace iggFrameRangeMapping_Private
{
	//
	//  Helper classes
	//
	class RenderButton;
	class HistogramSlider;

	class Area : public iggWidgetHistogramArea
	{

		friend class AddRangeButton;
		friend class RemoveRangeButton;
		friend class ResetButton;
		friend class RenderButton;

	public:

		Area(const iObjectKey &key, iggFrameRangeMapping *parent) : iggWidgetHistogramArea(true,true,parent), mKey(key)
		{
			mParent = parent;
			mCurRange = 0;
			mRenderButton = 0;
			mHistogramSlider = 0;

			mNeedsBaloonHelp = false;
		}

		void SetRenderButton(RenderButton *b)
		{
			mRenderButton = b;
		}

		void SetHistogramSlider(HistogramSlider *b)
		{
			mHistogramSlider = b;
		}

	protected:

		virtual bool UpdateWidgetBody();

		virtual void DrawForegroundBody()
		{
			static const int lineWidth = 3;
			static const iColor regColor = iColor(0,0,0);
			static const iColor curColor = iColor(255,0,0);
			float dx;
			int i, i1, ixmax, iymax, ix1, iy1, ix2;

			if(!mParent->IsActive()) return;

			mCurRange = mParent->GetCurrentRange();

			iRangeMapping *rm = 0;
			if(this->GetShell()->GetControlModule()->QueryValue(mKey,rm) && rm!=0) 
			{
				iRangeCollection *rc = rm->GetRangeCollection();
				
				ixmax = mSubject->Width() - 1;
				iymax = mSubject->Height() - 1;

				float aMin, aMax;
				int npic = rc->GetN();
				float attMin = rc->GetGlobalMin();
				float attMax = rc->GetGlobalMax();

				dx = ixmax/(attMax-attMin);

				for(i=0; i<=npic; i++)
				{
					if(i == npic)
					{
						i1 = mCurRange;
					}
					else
					{
						i1 = i;
					}
					aMin = rc->GetMin(i1);
					aMax = rc->GetMax(i1);
					
					ix1 = round(dx*(aMin-attMin));
					if(ix1>=0 && ix1<lineWidth/2) ix1 = lineWidth/2;
					ix2 = round(dx*(aMax-attMin));
					if(ix2<=ixmax && ix2>ixmax-lineWidth/2) ix2 = ixmax-lineWidth/2;
					iy1 = round(iymax*(0.95-0.4*i1/npic));
					
					if(i == npic)
					{
						mSubject->DrawLine(ix1,0,ix1,iymax,curColor,lineWidth);
						mSubject->DrawLine(ix2,0,ix2,iymax,curColor,lineWidth);
						mSubject->DrawLine(ix1,iy1,ix2,iy1,curColor,lineWidth);
					}
					else
					{
						mSubject->DrawLine(ix1,0,ix1,iymax,regColor,lineWidth);
						mSubject->DrawLine(ix2,0,ix2,iymax,regColor,lineWidth);
						mSubject->DrawLine(ix1,iy1,ix2,iy1,regColor,lineWidth);
					}
				}					
 			}
			else
			{
				IERROR_REPORT_ERROR("Unable to query RangeMapping.");
			}
		}
		
		void OnAddRange()
		{
			mParent->OnAddRange();
			mSubject->RequestPainting(ibgWidgetDrawAreaSubject::_Foreground);
		}

		void OnRemoveRange();

		void OnReset();

		void OnRender();

		void OnMousePress(int x, int y, int b)
		{
			this->OnMouseEvent(x,y,b,true);
		}

		void OnMouseMove(int x, int y, int b)
		{
			this->OnMouseEvent(x,y,b,false);
		}

		void OnMouseEvent(int x, int y, int b, bool click); // defined below

		iggFrameRangeMapping *mParent;
		int mCurRange;
		RenderButton *mRenderButton;
		HistogramSlider *mHistogramSlider;
		const iObjectKey &mKey;
	};	


	//
	//  HistogramSlider
	//
	class HistogramSlider : public iggWidget
	{

	public:

		HistogramSlider(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetEntrySubject(this,true,0,"",0);
			mSubject->SetRange(-10,10);
			mSubject->SetValue(0);

			mNeedsBaloonHelp = false;
		}

		float GetValue() const
		{
			return (float)pow10(0.1*mSubject->GetValue());
		}

	protected:

		virtual bool UpdateWidgetBody()
		{
			return true;
		}

		ibgWidgetEntrySubject *mSubject;
	};


	//
	//  AddRangeButton
	//
	class AddRangeButton : public iggWidgetSimpleButton
	{

	public:

		AddRangeButton(Area *area, iggFrame *parent) : iggWidgetSimpleButton("Add group",parent)
		{
			mArea = area;
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void Execute()
		{
			mArea->OnAddRange();
		}

		Area *mArea;
	};


	//
	//  RemoveRangeButton
	//
	class RemoveRangeButton : public iggWidgetSimpleButton
	{

	public:

		RemoveRangeButton(Area *area, iggFrame *parent) : iggWidgetSimpleButton("Remove group",parent)
		{
			mArea = area;
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void Execute()
		{
			mArea->OnRemoveRange();
		}

		Area *mArea;
	};

	//
	//  ResetButton
	//
	class ResetButton : public iggWidgetSimpleButton
	{

	public:

		ResetButton(Area *area, iggFrame *parent) : iggWidgetSimpleButton("Reset",parent)
		{
			mArea = area;
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void Execute()
		{
			mArea->OnReset();
		}

		Area *mArea;
	};


	//
	//  RenderButton
	//
	class RenderButton : public iggWidgetSimpleButton
	{

	public:

		RenderButton(Area *area, iggFrame *parent) : iggWidgetSimpleButton("Render",parent)
		{
			mArea = area;
			mNeedsBaloonHelp = false;

			this->Enable(false);
		}

	protected:

		virtual void Execute()
		{
			mArea->OnRender();
		}

		Area *mArea;
	};


	//
	//  Area functions
	//
	bool Area::UpdateWidgetBody()
	{
		iRangeMapping *rm = 0;
		
		if(this->GetShell()->GetControlModule()->QueryValue(mKey,rm) && rm!=0) 
		{
			//
			//  Must update histogram - who knows what changed anywhere else
			//
			int nBins = round(rm->GetEstimatedNumberOfBins()*mHistogramSlider->GetValue());
			if(nBins < 2) nBins = 2;
			this->SetHistogram(rm->GetHistogram(nBins),0);
			return iggWidgetHistogramArea::UpdateWidgetBody();
		}
		else return this->UpdateFailed();
	}


	void Area::OnMouseEvent(int x, int, int b, bool click)
	{
		static const int lineWidth = 3;
		static int tol = 15;
	    float dx;
	    int i, idx, idxmin, indmin, ixmax;
	    int ix1, ix2;
		static bool isMax = false;

		if(!mParent->IsActive()) return;

		mCurRange = mParent->GetCurrentRange();

		iRangeMapping *rm = 0;
		if(this->GetShell()->GetControlModule()->QueryValue(mKey,rm) && rm!=0) 
		{
			iRangeCollection *rc = rm->GetRangeCollection();
				
			ixmax = mSubject->Width() - 1;

			float aMin, aMax, a;
			int npic = rc->GetN();
			float attMin = rc->GetGlobalMin();
			float attMax = rc->GetGlobalMax();

			int oldCurRange = mCurRange;

			dx = ixmax/(attMax-attMin);
			
			if(click)
			{
				idxmin = ixmax;
				indmin = -1;
				for(i=0; i<npic; i++)
				{
					aMin = rc->GetMin(i);
					aMax = rc->GetMax(i);

					ix1 = round(dx*(aMin-attMin));
					if(ix1>=0 && ix1<lineWidth/2) ix1 = lineWidth/2;
					ix2 = round(dx*(aMax-attMin));
					if(ix2<=ixmax && ix2>ixmax-lineWidth/2) ix2 = ixmax-lineWidth/2;
					
					idx = abs(x-ix1);
					if(idxmin > idx)
					{
						indmin = i;
						idxmin = idx;
						isMax = false;
					}
					idx = abs(x-ix2);
					if(idxmin > idx)
					{
						indmin = i;
						idxmin = idx;
						isMax = true;
					}
				}	

				if(click && indmin>=0 && idxmin<tol)
				{
					mCurRange = indmin;
				}
			}					
			
			a = x/dx + attMin;
			aMin = rc->GetMin(mCurRange);
			aMax = rc->GetMax(mCurRange);
			if(isMax) aMax = a; else aMin = a;
			rc->SetRange(mCurRange,aMin,aMax);
			
			if(mCurRange != oldCurRange)
			{
				mParent->SetCurrentRange(mCurRange);
			}
			
			mSubject->RequestPainting(ibgWidgetDrawAreaSubject::_Foreground);

			if(b == _MouseRightButton) this->OnRender(); else if(!mRenderButton->IsEnabled()) mRenderButton->Enable(true);
		}
	}

	void Area::OnRemoveRange()
	{
		mParent->OnRemoveRange();
		mSubject->RequestPainting(ibgWidgetDrawAreaSubject::_Foreground);
		if(!mRenderButton->IsEnabled()) mRenderButton->Enable(true);
	}
	
	void Area::OnReset()
	{
		mParent->OnReset();
		mSubject->RequestPainting(ibgWidgetDrawAreaSubject::_Foreground);
		if(!mRenderButton->IsEnabled()) mRenderButton->Enable(true);
	}

	void Area::OnRender()
	{
		iRangeMapping *rm = 0;
		if(this->GetShell()->GetControlModule()->QueryValue(mKey,rm) && rm!=0) 
		{
			rm->Modified();
		}
		this->GetShell()->GetControlModule()->Render();
		mRenderButton->Enable(false);
	}
};


using namespace iggFrameRangeMapping_Private;


iggFrameRangeMapping::iggFrameRangeMapping(const iObjectKey &key, const iObjectKey &tilekey, iggFrame *parent) : iggFrame(parent,4)
{
	Area *a = new Area(key,this);
	this->AddLine(a,4);
	this->AddLine(new iggFrameHistogramAreaExtra(a,this),4);

	iggWidgetKeyCheckBox *tb = new iggWidgetKeyCheckBox("Tile",tilekey,this);
	tb->AddDependent(a);
	this->AddLine(tb,3);

	RenderButton *rb = new RenderButton(a,this);
	this->AddLine(new AddRangeButton(a,this),new RemoveRangeButton(a,this));
	this->AddLine(new ResetButton(a,this),rb);

	iggFrame *hf = new iggFrame(this,3);
	hf->AddLine(new iggWidgetTextArea("Lower",hf),new iggWidgetTextArea("Optimal",hf),new iggWidgetTextArea("Higher",hf));
	this->AddLine(hf,3);
	
	HistogramSlider *hs = new HistogramSlider(this);
	hs->AddDependent(a);
	this->AddLine(hs,3);
	this->AddLine(new iggWidgetTextArea("Resolution of the histogram",this),3);

	this->SetRowStretch(0,10);
	this->SetColStretch(3,3);
	
	a->SetRenderButton(rb);
	a->SetHistogramSlider(hs);

//	mNeedsBaloonHelp = false;
	this->SetBaloonHelp("Interactively controls the collection of ranges. Press Shift+F1 for more help.","The range collection can be controlled interactively by clicking on vertical line that represents the range boundary and dragging it around with the mouse. Four buttons below the interactive area allow you to add or remove a new range (group), to reset the collection to a single range, and to render the scene. The latter button is important because while you are dragging a range boundary with the left mouse button pressed, the scene is actually not being rendered, even if the range collection is being changed - this is done to maintain the interactive performance. Because the scene may be complex and slow to render, the interactive area will appear \"jumpy\" if the scene is rendered all the time. If you would like to render the scene continously, drag the range boundary with the right mouse button pressed.");
}

