/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icontrolscript.h"


#include "ianimator.h"
#include "ianimatorscript.h"
#include "icolor.h"
#include "icontrolmodule.h"
#include "idatareader.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "iobjecthelp.h"
#include "ishellfactory.h"
#include "istring.h"
#include "iviewmodule.h"

#include <vtkIOStream.h>

//
//  macros
//
#include "ibasescriptmacro.h"


#define curScript		iPointerCast<iControlScript,iScript>(me)
#define curRender		iPointerCast<iControlScript,iScript>(me)->mRender
#define curControl		iPointerCast<iControlScript,iScript>(me)->GetControlModule()
#define curVisual		iPointerCast<iControlScript,iScript>(me)->GetControlModule()->GetViewModule()


namespace iControlScript_Private
{
	template<class T>
	T* QueryKeyData(T, const iObjectKey *tmp, iControlScript *me, int d)
	{
		T* data = new T[d];
		if(data != 0) 
		{
			if(me->GetControlModule()->QueryValue(*tmp,data,d)) return data; else
			{
				delete [] data;
				return 0;
			}
		}
		else return 0;
	}

	template<class T>
	iValue* CreateValueFromKeyData(T *data, iControlScript *me, int d)
	{
		int i;
		iValue *val;

		if(data == 0) 
		{
			me->GetErrorStatus()->Set("The value of this property is not currently available.");
			return 0;
		}

		if(d == 1)
		{
			//
			//  Return the single value
			//
			val = iValue::New(me,"",iValue::_ByValue,data[0]);
		}
		else
		{
			//
			//  Create an array of iValues
			//
			iValue **arr = new iValue*[d]; 
			if(arr == 0)
			{
				delete [] data;
				me->GetErrorStatus()->Set("Not enough memory.");
				return 0;
			}
			//
			//  Fill it in
			//
			bool ok = true;
			for(i=0; i<d; i++)
			{
				arr[i] = iValue::New(me,"",iValue::_ByValue,data[i]);
				if(arr[i]==0 || (i>0 && !iValue::AreTypesCompatible(arr[i],arr[0]))) ok = false;
			}
			//
			//  If unsuccessfull, destroy it.
			//
			if(!ok)
			{
				for(i=0; i<d; i++) if(arr[i] != 0) arr[i]->Delete();
				delete [] arr; 
				me->GetErrorStatus()->Set("Not enough memory.");
				return 0;
			}
			val = iValue::New(me,"",iValue::_ByValue,d,iValue::New(me,"",iValue::_ByValue,1),arr);
		}
		delete [] data;
		if(val == 0)
		{
			me->GetErrorStatus()->Set("Not enough memory.");
		}
		else
		{
			me->GetErrorStatus()->Clear();  // clean errors accumulated earlier
		}
		return val;
	}
};


using namespace iControlScript_Private;
using namespace iParameter;


iControlScript* iControlScript::New(iControlModule *cm, iScript *parent)
{
	return iShellFactory::CreateControlScript(cm,parent);
}


iControlScript::iControlScript(iControlModule *cm, iScript *parent) : iBaseScript(parent), iControlModuleComponent(cm)
{
	//
	//  Alias words (shortcuts)
	//
	this->CreateAliasWord(".e","exec"); 
	this->CreateAliasWord(".s","show"); 
	this->CreateAliasWord(".q","hide"); 
	this->CreateAliasWord(".h","help"); 
	this->CreateAliasWord(".lo","list objects"); 
	this->CreateAliasWord(".lp","list properties"); 
	this->CreateAliasWord(".ho","help object"); 
	this->CreateAliasWord(".hp","help property"); 
	this->CreateAliasWord(".p","print"); 
	this->CreateAliasWord(".c","create"); 
	this->CreateAliasWord(".d","delete"); 
	this->CreateAliasWord(".r","render"); 
	this->CreateAliasWord(".u","current-window"); 
	this->CreateAliasWord(".a","animate"); 
	this->CreateAliasWord(".ss","save-state"); 
	this->CreateAliasWord(".ls","load-state"); 
	this->CreateAliasWord(".eo","exec-all-objects"); 
	this->CreateAliasWord(".ew","exec-all-windows"); 

	//
	//  Parameter words
	//
	this->CreateParameterWord(iValue::New(this,"off",iValue::_ByValue,false));
	this->CreateParameterWord(iValue::New(this,"on",iValue::_ByValue,true));

	//
	//  Command words
	//

	//
	//  Statements
	//
	this->CreateCommandWord(iOperation::New(this,"","exec",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::Exec,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","show",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::Show,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","hide",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::Hide,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","print",1,iValue::_Absolute,0,iControlScript::Print,(unsigned char)1));  // prints anything
	this->CreateCommandWord(iOperation::New(this,"","create",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::Create,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","remove",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::Remove,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","current-window",1,iValue::_Absolute,iValue::NewInt(this),iControlScript::Current,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","animate",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::Animate,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","exec-all-objects",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::ExecAllObjects,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","exec-all-windows",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::ExecAllWindows,(unsigned char)1));

	this->CreateCommandWord(iOperation::New(this,"","save-state",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::SaveState,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","load-state",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::LoadState,(unsigned char)1));

	this->CreateCommandWord(iOperation::New(this,"embed","animator-script",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::EmbedAnimatorScript,(unsigned char)1));

	//
	//  Help statements
	//
	this->CreateCommandWord(iOperation::New(this,"","help",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::Help,(unsigned char)2));

	this->CreateCommandWord(iOperation::New(this,"list","objects",1,iControlScript::ListObjects,(unsigned char)2));
	this->CreateCommandWord(iOperation::New(this,"list","properties",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::ListProperties,(unsigned char)2));
	this->CreateCommandWord(iOperation::New(this,"help","object",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::HelpObject,(unsigned char)2));
	this->CreateCommandWord(iOperation::New(this,"help","property",1,iValue::_Absolute,iValue::NewString(this,false),iControlScript::HelpProperty,(unsigned char)2));

	//
	//  Genuine assignment operations to pre-defined variables
	//
	this->CreateCommandWord(iOperation::New(this,"","render",1,iValue::_Absolute,iValue::NewBool(this),iControlScript::Render,(unsigned char)1));

	mRender = true;
}


iControlScript::~iControlScript()
{
}


//
//  Turn an unknown token into a string
//
iValue* iControlScript::TransformUnknownTokenToValue(const iString &v, bool /*acceptArrays*/, bool /*forceFloat*/)
{
	if(v.Find(iObjectKey::PrefixSeparator()) < 0) return 0; // is not even in a key format

	const iObjectKey *key = iObjectKeyRegistry::FindKey(v,true);
	if(key != 0)
	{
		int d = key->Dimension();

		if(d <= 0)
		{
			iString s;
			this->GetControlModule()->QueryValueAsString(*key,s);
			this->GetErrorStatus()->Clear();  // clean errors accumulated earlier
			return iValue::New(this,"",iValue::_ByValue,s);
		}

		switch(key->Argument())
		{
		case iObjectKey::_Int:
			{
				int *data = QueryKeyData(0,key,this,d);
				return CreateValueFromKeyData(data,this,d);
			}
		case iObjectKey::_Bool:
			{
				bool *data = QueryKeyData(false,key,this,d);
				return CreateValueFromKeyData(data,this,d);
			}
		case iObjectKey::_Float:
			{
				double *ddata = 0;
				float *data = QueryKeyData(0.0f,key,this,d);
				if(data != 0)
				{
					//
					//  Transform to doubles
					//
					ddata = new double[d];
					if(ddata != 0)
					{
						int i;
						for(i=0; i<d; i++) ddata[i] = data[i];
					}
					delete [] data;
				}
				return CreateValueFromKeyData(ddata,this,d);
			}
		case iObjectKey::_Double:
			{
				double *data = QueryKeyData(0.0,key,this,d);
				return CreateValueFromKeyData(data,this,d);
			}
		case iObjectKey::_Color:
			{
				int *idata = 0;
				iColor *data = QueryKeyData(iColor(),key,this,d);
				if(data != 0)
				{
					//
					//  Transform to ints
					//
					idata = new int[d];
					if(idata != 0)
					{
						int i;
						for(i=0; i<d; i++) idata[i] = data[i].Red() + 256*(data[i].Green()+256*data[i].Blue());
					}
					delete [] data;
				}
				return CreateValueFromKeyData(idata,this,d);
			}
		case iObjectKey::_String:
			{
				iString *data = QueryKeyData(iString(),key,this,d);
				return CreateValueFromKeyData(data,this,d);
			}
		default:
			{
				iString s;
				this->GetControlModule()->QueryValueAsString(*key,s);
				return iValue::New(this,"",iValue::_ByValue,s);
			}
		}
	}

	if(v.Part(v.Length()-iObjectKey::PrefixSeparator().Length()) != iObjectKey::PrefixSeparator())
	{
		this->GetErrorStatus()->Set("Invalid object property.");
		return 0;
	}

	//
	//  List the complete contents of this type
	//
	const iObjectType *type = iObjectTypeRegistry::FindType(v.Section(iObjectKey::PrefixSeparator(),0,0));
	if(type != 0)
	{
		int i, lmax = 0;
		iString s, s1, out;
		const iObjectKey *ptr;
	
		iObjectKeyRegistry::InitTraversal(type);
		while((ptr=iObjectKeyRegistry::GetNextKey()) != 0) if(ptr->UnprefixedFullName().Part(0,9) != "___old___")
		{
			if(lmax < ptr->UnprefixedFullName().Length()) lmax = ptr->UnprefixedFullName().Length();
		}

		for(i=0; i<lmax; i++) s += "$";

		out  = "\1<b>" + type->FullName() + "</b>:<br><hr>";

		iObjectKeyRegistry::InitTraversal(type);
		while((ptr=iObjectKeyRegistry::GetNextKey()) != 0) if(ptr->UnprefixedFullName().Part(0,9) != "___old___")
		{
			out += "<b>" + ptr->UnprefixedFullName() + "</b>" + s.Part(0,lmax-ptr->UnprefixedFullName().Length()) + ":$$";
			this->GetControlModule()->QueryValueAsString(*ptr,s1);
			out += s1 + "<br>";
		}
		out.Replace("$","&nbsp;");
		this->GetErrorStatus()->Clear();  // clean errors accumulated earlier
		return iValue::New(this,"",iValue::_ByValue,out);
	}
	
	this->GetErrorStatus()->Set("Invalid object property.");
	return 0;
}

//
//  Exec a ControlModule command
//
void iControlScript::Exec(iScript *me, iValue *v0, iValue::Assignment)
{
	ExecBody(me,v0,_ObjectModeCurrent|_ModuleModeCurrent|_RenderModeAuto);
}


void iControlScript::ExecAllObjects(iScript *me, iValue *v0, iValue::Assignment)
{
	ExecBody(me,v0,_ObjectModeAll|_ModuleModeCurrent|_RenderModeAuto);
}


void iControlScript::ExecAllWindows(iScript *me, iValue *v0, iValue::Assignment)
{
	ExecBody(me,v0,_ObjectModeAll|_ModuleModeAll|_RenderModeAuto);
}


//
//  Actually do Exec 
//
void iControlScript::ExecBody(iScript *me, iValue *v0, int mode)
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	//
	//  Mini-parser: replaces expressions in between of special symbols {} with their values.
	//
	iValue *val;
	iString tmp;
	int i1, i2;

	i2 = v.Find('{');
	while(i2 > 0 )
	{
		i1 = i2;
		i2 = v.Find('}',i1+1);
		if(i2 == -1)
		{
			me->GetErrorStatus()->Set("Closing } is missing after position " + iString::FromNumber(i1) + ".");
			return;
		}
		tmp = v.Part(i1+1,i2-i1-1);
		val = curScript->TransformToValue(tmp);
		if(val == 0)
		{
			me->GetErrorStatus()->Set(curScript->GetErrorStatus()->GetMessage() + " in position " + iString::FromNumber(i1));
			return;
		}
		v.Replace(i1,i2-i1+1,val->GetValueAsText());
		val->Delete();
		i2 = v.Find('{',i1);
	}

	//
	//  Execute the command
	//
	if(!curControl->Execute(v,curRender,mode))
	{
		me->GetErrorStatus()->Set("Incorrect command syntax.");
	}
}

//
//  Show/hide a ViewSubject
//
void iControlScript::Show(iScript *me, iValue *v0, iValue::Assignment)
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	const iObjectType *tmp = iObjectTypeRegistry::FindType(v);
	if(tmp != 0)
	{
		curControl->Show(*tmp,true);
	}
	else
	{
		me->GetErrorStatus()->Set("Invalid object name.");
	}
}


void iControlScript::Hide(iScript *me, iValue *v0, iValue::Assignment)
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	const iObjectType *tmp = iObjectTypeRegistry::FindType(v);
	if(tmp != 0)
	{
		curControl->Show(*tmp,false);
	}
	else
	{
		me->GetErrorStatus()->Set("Invalid object name.");
	}
}

//
//  Print anything
//
void iControlScript::Print(iScript *me, iValue *v0, iValue::Assignment)
{
	if(curScript->mParent != 0) return;
	if(v0->GetValueAsText()[0] == '\1') // special case
	{
		curScript->OutputText(v0->GetValueAsText().Part(1));
	}
	else
	{
		curScript->OutputText("<b>"+v0->GetName()+"</b>("+v0->GetTypeAsText()+"): "+v0->GetValueAsText());
	}
}

//
//  Create/delete an object
//
void iControlScript::Create(iScript *me, iValue *v0, iValue::Assignment)
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	const iObjectType *tmp = iObjectTypeRegistry::FindType(v.Section(" ",0,0));
	if(tmp != 0)
	{
		int vmtype = _ModuleTypeNew;

		if(*tmp == iViewModule::Type())
		{
			v = v.Section(" ",1);
			v.ReduceWhiteSpace();
			if(v == "/copy") vmtype = _ModuleTypeCopy;
			if(v == "/clone") vmtype = _ModuleTypeClone;
		}

		if(!curControl->CreateObject(*tmp,curRender,-1,vmtype))
		{
			me->GetErrorStatus()->Set("Object cannot be created - perhaps, there is not enough memory.");
		}
	}
	else
	{
		me->GetErrorStatus()->Set("Invalid object name.");
	}
}


void iControlScript::Remove(iScript *me, iValue *v0, iValue::Assignment)
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	const iObjectType *tmp = iObjectTypeRegistry::FindType(v);
	if(tmp != 0)
	{
		if(!curControl->DeleteObject(*tmp,curRender))
		{
			me->GetErrorStatus()->Set("It is not allowed to delete this object.");
		}
	}
	else
	{
		me->GetErrorStatus()->Set("Invalid object name.");
	}
}

//
//  Save/restore state
//
void iControlScript::SaveState(iScript *me, iValue *v0, iValue::Assignment)
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	if(!curControl->SaveStateToFile(v))
	{
		me->GetErrorStatus()->Set("Saving state to a file failed.");
	}
}


void iControlScript::LoadState(iScript *me, iValue *v0, iValue::Assignment)
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	if(!curControl->LoadStateFromFile(v))
	{
		me->GetErrorStatus()->Set("Loading state from a file failed.");
	}
}
//
//  Create/delete an object
//
void iControlScript::Current(iScript *me, iValue *v0, iValue::Assignment)
{
	int v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	curControl->SetCurrentViewModuleIndex(v);
}


void iControlScript::Animate(iScript *me, iValue *v0, iValue::Assignment)
{
	if(curScript->mParent != 0) return;

	if(!curVisual->GetReader()->IsFileAnimatable())
	{
		me->GetErrorStatus()->Set("Please, load an animatable file for the Animator to work.");
		return;
	}

	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	if(v.IsEmpty())
	{
		curVisual->GetAnimator()->SetUseScript(false);
	}
	else
	{
		if(v[0] == '+') v = curControl->GetFileName(_EnvironmentBase,v.Part(1));

		iFile f(v);
		if(!f.Open(iFile::_Read,iFile::_Text))
		{
			me->GetErrorStatus()->Set("Animation script file is not found.");
			return;
		}

		iString st, tmp;
		while(f.ReadLine(tmp)) st += tmp + "\n";
		if(st.IsEmpty())
		{
			me->GetErrorStatus()->Set("Animation script file is empty or unreadable.");
			return;
		}

		f.Close();

		curVisual->GetAnimator()->GetScript()->SetText(st);
		curVisual->GetAnimator()->SetUseScript(true);
	}

	me->GetErrorStatus()->Monitor(curVisual->GetAnimator());
	curVisual->GetAnimator()->Animate();
}


void iControlScript::EmbedAnimatorScript(iScript *me, iValue *v0, iValue::Assignment)
{
	if(curScript->mParent != 0) return;

	if(!curVisual->GetReader()->IsFileAnimatable())
	{
		me->GetErrorStatus()->Set("Please, load an animatable file for the Animator to work.");
		return;
	}

	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	//
	//  Replace %% with \n
	//
	v.Replace("%%","\n");

	me->GetErrorStatus()->Monitor(curVisual->GetAnimator());
	curVisual->GetAnimator()->GetScript()->SetText(v);
	curVisual->GetAnimator()->SetUseScript(true);
	curScript->mAllowChildAccess = true;
	curVisual->GetAnimator()->Animate();
	curScript->mAllowChildAccess = false;
}

//
//  Set default render mode
//
void iControlScript::Render(iScript *me, iValue *v0, iValue::Assignment)
{
	bool v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	curRender = v;
}

//
//  Help functions
//
void iControlScript::Help(iScript *me, iValue* v0, iValue::Assignment)
{
	if(curScript->mParent != 0) return;

	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	if(v == "exec")
	{
		curScript->OutputText(curScript->GetHelpString(_Exec));
		return;
	}
	if(v == "print")
	{
		curScript->OutputText(curScript->GetHelpString(_Print));
		return;
	}
	if(v=="create" || v=="delete")
	{
		curScript->OutputText(curScript->GetHelpString(_CreateDelete));
		return;
	}
	if(v == "current")
	{
		curScript->OutputText(curScript->GetHelpString(_Current));
		return;
	}
	if(v == "control")
	{
		curScript->OutputText(curScript->GetHelpString(_Control));
		return;
	}

	curScript->OutputText(curScript->GetHelpString(_Root));
}


void iControlScript::HelpObject(iScript *me, iValue* v0, iValue::Assignment)
{
	if(curScript->mParent != 0) return;

	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	const iObjectType *tmp = iObjectTypeRegistry::FindType(v);
	if(tmp != 0)
	{
		if(tmp->GetHelp() != 0) curScript->OutputText(tmp->GetHelp()->GetHTML()); else curScript->OutputText("<font color=#FF0000>No help is available for this item.</font>");
	}
	else
	{
		me->GetErrorStatus()->Set("Invalid object name.");
	}
}


void iControlScript::HelpProperty(iScript *me, iValue* v0, iValue::Assignment)
{
	if(curScript->mParent != 0) return;

	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	const iObjectKey *tmp = iObjectKeyRegistry::FindKey(v);
	if(tmp != 0)
	{
		if(tmp->GetHelp() != 0) curScript->OutputText(tmp->GetHelp()->GetHTML()); else curScript->OutputText("<font color=#FF0000>No help is available for this item.</font>");
	}
	else
	{
		me->GetErrorStatus()->Set("Invalid object property.");
	}
}


void iControlScript::ListObjects(iScript *me)
{
	if(curScript->mParent != 0) return;

	const int lmax = 10;
	int i;
	iString out, s;
	const iObjectType *ptr;

	for(i=0; i<lmax; i++) s += "$";

	out = "To get help for a specific object, use: <code><b>help object</b> <i>&lt;object-name&gt;</i></code><br>Available objects:<br>";
	out += "<i>Short form" + s.Part(0,lmax-10);
	out += "$$Long form</i><br><hr>";

	iObjectTypeRegistry::InitTraversal();
	while((ptr=iObjectTypeRegistry::GetNextType()) != 0)
	{
		out += "<br>$$" + ptr->ShortName() + s.Part(0,lmax-2-ptr->ShortName().Length()) + "$$" + ptr->FullName();
	}
	out += "</b><br>";
	out.Replace("$","&nbsp;");

	curScript->OutputText(out);
}


void iControlScript::ListProperties(iScript *me, iValue* v0, iValue::Assignment)
{
	if(curScript->mParent != 0) return;

	const int lmax1 = 12, lmax2 = 10;
	int i;
	iString out, v, s;
	const iObjectKey *ptr;

	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	const iObjectType *tmp = iObjectTypeRegistry::FindType(v);
	if(tmp != 0)
	{
		for(i=0; i<lmax1+lmax2; i++) s += "$";

		out = "To get help for a specific property, use: <code><b>help property</b> &lt;<i>object-name</i>&gt;<b>:</b><i>property-name</i></code><br>Available properties:<br>";
		out += "$$<i>Type[Size]" + s.Part(0,lmax1-12);
		out += "$$Short form" + s.Part(0,lmax2-10);
		out += "$$Long form</i><br><hr>";

		iObjectKeyRegistry::InitTraversal(tmp);
		while((ptr=iObjectKeyRegistry::GetNextKey()) != 0) if(ptr->UnprefixedFullName().Part(0,9) != "___old___")
		{
			switch(ptr->Argument())
			{
			case iObjectKey::_OffsetInt:
			case iObjectKey::_Int:
				{
					v = "int";
					break;
				}
			case iObjectKey::_Bool:
				{
					v = "bool";
					break;
				}
			case iObjectKey::_Float:
				{
					v = "float";
					break;
				}
			case iObjectKey::_Double:
				{
					v = "double";
					break;
				}
			case iObjectKey::_Color:
				{
					v = "color";
					break;
				}
			case iObjectKey::_String:
				{
					v = "string";
					break;
				}
			case iObjectKey::_Any:
				{
					v = "any";
					break;
				}
			}
			out += "<b>" + s.Part(0,6-v.Length()) + v + "</b>[";
			if(ptr->Dimension() > 0)
			{
				v = iString::FromNumber(ptr->Dimension());
			}
			else
			{
				v = "*";
			}
			out += v + "]" + s.Part(0,lmax1-8-v.Length()) + "$$<b>" + ptr->UnprefixedShortName() + s.Part(0,lmax2-ptr->UnprefixedShortName().Length()) + "$$" + ptr->UnprefixedFullName() + "</b><br>";
		}
		out += "Use 0/1 for <code><b>false</b></code>/<code><b>true</b></code> for bool values and <code>int.int.int</code> for an RGB color (like 255.0.0 for red). Strings must not be included in quotes and contain slash characters (/). Use @@ if you need a slash character.<br>";
		out.Replace("$","&nbsp;");
		curScript->OutputText(out);
	}
	else
	{
		me->GetErrorStatus()->Set("Invalid object name.");
	}
}

//
//  This is whether the actual help text lives
//
const iString& iControlScript::GetHelpString(_HelpId id) const
{
	static const iString none = "No help available for this item";
	static const iString help[6] = 
	{
	//
		"Control Script understands the same flow control operators as the AnimatorScript. Type <code><b>help control</b></code> for the full description of the flow control. IFrIT consists of a collection of <b>objects</b>; each <b>object</b> is controlled by a set of <b>properties</b>.<br>"
		"Available commands:<br><blockquote>"
		"    <code><b>help</b> [<i>topic</i>]</code><br></blockquote>"
		"gives help on a specific topic. The short form of <code><b>help</b></code> is <code><b>.h</b></code>.<br><blockquote>"
		"    <code><b>list objects</b></code><br></blockquote>"
		"lists all available objects (short form <code><b>.lo</b></code>). Type <code><b>help object</b> &lt;<i>object-name</i>&gt;</code> (short form <code><b>.ho</b></code>) for help on a specific object.<br><blockquote>"
		"    <code><b>list properties</b> &lt;<i>object-name</i>&gt;</code><br></blockquote>"
		"lists all properties for an object specified by <code>&lt;<i>object-name</i>&gt;</code> (short form <code><b>.lp</b></code>). Type <code><b>help property</b> &lt;<i>object-name</i>&gt;<b>:</b>&lt;<i>property-name</i>&gt;</code> (short form <code><b>.hp</b></code>) to get help on a specific property.<br><blockquote>"
		"    <code><b>exec</b> &lt;<i>request</i>&gt;</code><br></blockquote>"
		"executes a request (short form <code><b>.e</b></code>). Type <code><b>help exec</b></code> for more info.<br><blockquote>"
		"    <code><b>show</b> &lt;<i>object-name</i>&gt;</code><br>"
		"    <code><b>hide</b> &lt;<i>object-name</i>&gt;</code><br></blockquote>"
		"shows/hides an object <code>&lt;<i>object-name</i>&gt;</code> in the scene (short forms <code><b>.s</b></code>/<code><b>.q</b></code>).<br><blockquote>"
		"    <code><b>print</b> &lt;<i>query</i>&gt;</code><br></blockquote>"
		"prints values of variables and properties (short form <code><b>.p</b></code>). Type <code><b>help print</b></code> for more info.<br><blockquote>"
		"    <code><b>create</b> &lt;<i>object-name</i>&gt;</code><br>"
		"    <code><b>delete</b> &lt;<i>object-name</i>&gt;</code><br></blockquote>"
		"creates a new instance/deletes the current instance of an object <code>&lt;<i>object-name</i>&gt;</code> (short forms <code><b>.c</b></code> and <code><b>.d</b></code>). Type <code><b>help create</b></code> for more info.<br><blockquote>"
		"    <code><b>[set] current-window</b> &lt;<i>id</i>&gt;</code><br></blockquote>"
		"sets the viewization window specified by <code>&lt;<i>id</i>&gt;</code> as current (short form <code><b>.u</b></code>). Type <code><b>help current</b></code> for more info.<br><blockquote>"
		"    <code><b>save-state</b> [<i>file-name</i>]</code><br>"
		"    <code><b>load-state</b> [<i>file-name</i>]</code><br></blockquote>"
		"saves/loads the complete state of IFrIT from a file <code>&lt;<i>file-name</i>&gt;</code> (short forms <code><b>.ss</b></code> and <code><b>.ls</b></code>). If the file name is not specified, the default is used.<br><blockquote>"
		"    <code><b>animate</b> [<i>script-file-name</i>]</code><br></blockquote>"
		"animate the scene using the AnimatorScript from file <code>&lt;<i>script-file-name</i>&gt;</code> if it is specified, or using the current Animator settings otherwise (short form <code><b>.a</b></code>).<br><blockquote>"
		"    <code><b>render</b> &lt;<i>state</i>&gt;</code><br></blockquote>"
		"toggles whether the scene is rendered after each command (short form <code><b>.r</b></code>). <code>&lt;<i>state</i>&gt;</code> takes only two values: <code><b>on</b></code> or <code><b>off</b></code>.<br><blockquote>"
		"    <code><b>&lt;</b> &lt;<i>script-file-name</i>&gt;</code><br></blockquote>"
		"inserts the contents of <code>&lt;<i>script-file-name</i>&gt;</code> (which must contain valid ControlScript commands) into the current script as if they were typed one after another on the command line. If <code>&lt;<i>script-file-name</i>&gt;</code> starts with the plus sign (+), the name of the IFrIT base directory will be prepended to the name of the file.<br>",
//
            "Format:<br><blockquote>"
			"    <code><b>exec</b> &lt;<i>request</i>&gt; [<i>request</i> ...]</code><br></blockquote>"
			"This command executes one or more control module requests for the current object. A request has the following form:<br><blockquote>"
			"    <code>&lt;<i>object-name</i>&gt;<b>:</b>&lt;<i>property-name</i>&gt;<b>/</b>&lt;<i>value</i>&gt;[<b>/</b><i>value</i>...]</code><br></blockquote>"
			"Here <code>&lt;<i>object-name</i>&gt;</code> is the string that encapsulates the object to which this request is addressed to (use <code><b>list objects</b></code> to get the list of all available objects), <code>&lt;<i>property-name</i>&gt;</code> is the property that serves the request (use <code><b>list properties</b> &lt;<i>object-name</i>&gt;</code> to get the list of all properties for the object <code>&lt;<i>object-name</i>&gt;</code>), and <code>&lt;<i>value</i>&gt;</code> is the value to be assigned to the property (can be an array). For example, the following command:<br><blockquote>"
			"    <code><b>exec Camera:ParallelProjection/0</b></code><br></blockquote>"
			"will set the projection in the viewization window to perspective. The same command can be typed in a short form as:<br><blockquote>"
			"    <code><b>.e c:pp/0</b></code><br></blockquote>"
			"where <code><b>.e</b></code> is the short form for <code><b>exec</b></code>. Two special forms of the <code><b>exec</b></code> command: <code><b>exec-all-objects</b></code> (short form <code><b>.eo</b></code>) and <code><b>exec-all-windows</b></code> (short form <code><b>.ew</b></code>) will execute the request for all instances of the object in the current viewization window and for all instances in all windows respectively.<br>",
//
			"Format:<br><blockquote>"
			"    <code><b>print</b> &lt;<i>query</i>&gt;</code><br></blockquote>"
			"This command prints the value of <code>&lt;<i>query</i>&gt;</code>, which can be a script expression or a property in the form <code>&lt;<i>object-name</i>&gt;<b>:</b>&lt;<i>property-name</i>&gt;</code>. Use <code><b>list objects</b></code> to get the list of all available objects and <code><b>list properties</b> &lt;<i>object-name</i>&gt;</code> to get the list of all available properties for a given object <code>&lt;<i>object-name</i>&gt;</code>.<br>",
//
		"Format:<br><blockquote>"
		"    <code><b>create</b> &lt;<i>object-name</i>&gt;</code><br>"
		"    <code><b>delete</b> &lt;<i>object-name</i>&gt;</code><br></blockquote>"
		"These commands create or remove a new instance of a viewization object <code>&lt;<i>object-name</i>&gt;</code>. The valid values for <code>&lt;<i>object-name</i>&gt;</code> are <code><b>Surface</b></code>, <code><b>CrossSection</b></code>, <code><b>ParticlesGroup</b></code>, and <code><b>View</b></code> (other objects cannot have multiple instances). For view modules, an additional option <code><b>/copy</b></code> or <code><b>/clone</b></code> is permitted, in order to create a copy or a clone of the current view module.<br>",
//
			"Format:<br><blockquote>"
			"    <code><b>[set] current-window =</b> &lt;<i>id</>&gt;</code><br></blockquote>"
			"This command sets the viewization window #<code>&lt;<i>id</i>&gt;</code> as current. Here <code>&lt;<i>id</>&gt;</code> is an integer number from 1 to the number of windows available. For example, the following 2 commands create a new viewization window (module) as a clone of the furst one, and make the second window current:<br><blockquote>"
			"    <code><b>create View /clone</b></code><br>"
			"    <code><b>current-window = 2</b></code><br></blockquote>",
//
		"Variable declaration:<br><blockquote>"
		"    <code><b>var</b> &lt;<i>type</i>&gt;[<b>[</b><i>dim</i><b>]</b>] &lt;<i>name1</i>&gt; [<b>,</b> <i>name2</i> ...]</code><br></blockquote>"
		"declares a variable(s) of type <code>&lt;<i>type</i>&gt;</code> and name(s) <code>&lt;<i>name1</i>&gt;</code> (<code>&lt;<i>name2</i>&gt;</code> ...) with the optional dimension <code>&lt;<i>dim</i>&gt;</code>. More than one variable of the same type can be declared in a single statement. <code>&lt;<i>type<i/>&gt;</code> is either <code><b>int</b></code>, <code><b>real</b></code>, or <code><b>bool</b></code>. For example, the following statement declares two integer arrays <code><i>p</i></code> and <code><i>q</i></code> with 5 elements each:<br><blockquote>"
		"    <code><b>var int[5] p, q</b></code><br></blockquote>"
		"Declared variables can be assigned values and can be used in expressions:<br><blockquote>"
		"    <code><b>q[1] = 3</b></code><br>"
		"    <code><b>p += (1,2,3,4,5)</b></code><br>"
		"    <code><b>q[2] *= q[1]*sin(q[3]^2)</b></code><br></blockquote><br>"
		"Loop statements:<br><blockquote>"
		"    <code><b>loop</b> &lt;<i>count</i>&gt;</code><br>"
		"    <code>...</code><br>"
		"    <code><b>end</b></code><br></blockquote>"
		"or<br><blockquote>"
		"    <code><b>for</b> &lt;<i>var</i>&gt; <b>to</b> &lt;<i>count</i>&gt;</code><br>"
		"    <code>...</code><br>"
		"    <code><b>end</b></code><br></blockquote>"
		"repeat the statements within the body of the loop <code>&lt;<i>count</i>&gt;</code> times. The second form also uses the variable <code>&lt;<i>var</i>&gt;</code> (which must be declared) as a loop index, taking consequitive values from 1 to <code>&lt;<i>count</i>&gt;</code>.<br><br>"
		"Conditional statement:<br><blockquote>"
		"    <code><b>if</b> &lt;<i>boolean-expression</i>&gt; <b>then</b></code><br>"
		"    <code>...</code><br>"
		"    <code><b>else</b></code><br>"
		"    <code>...</code><br>"
		"    <code><b>endif</b></code><br></blockquote>"
		"behaves as expected. Use usual C-style comparison operators to compare numbers: <code><b>&lt;</b></code> <code><b>&gt;</b></code> <code><b>&lt;=</b></code> <code><b>&gt;=</b></code> <code><b>&&</b></code> <code><b>||</b></code> <code><b>!=</b></code>.<br><br>"
		"The text of the AnimatorScript can be embedded into the body of the ControlScript using the embedding statement:<br><blockquote>"
		"    <code><b>embed animator-script</b></code><br>"
		"    <code><b>&gt; </b><i>first-animator-script-statement</i></code><br>"
		"    <code><b>&gt; </b><i>second-animator-script-statement</i></code><br>"
		"    <code><b>&gt; </b>...</code><br>"
		"    <code><b>&gt; </b><i>last-animator-script-statement</i></code><br></blockquote><br>"
	};

	if(id>=0 && id<8) return help[id]; else return none;
}
