/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


//
//  iRenderTool is a combination of a RenderWindow and a Renderer.
//  In dual-window stereo mode it pops up a separate view for the right eye.
//
#ifndef IRENDERTOOL_H
#define IRENDERTOOL_H


#include "iviewmodulecomponent.h"
#include "ideletable.h"


#include "iarray.h"

class iActor;
class iCamera;
class iColor;
class iDualWindowObserver;
class iLightKit;
class iMagnifier;
class iRendererObserver;
class iString;
class iStereoImage;
class iStereoImageArray;

class vtkCommand;
class vtkProp;
class vtkPropAssembly;
class vtkRenderer;
class vtkRenderWindow;
class vtkRenderWindowCollection;
class vtkRenderWindowInteractor;


class iRenderTool : public iViewModuleComponent, public iDeletable
{
	
	friend class iCamera;
	friend class iDualWindowObserver;
	friend class iExtensionFactory;
	friend class iRendererObserver;

public:

	static iRenderTool* New(iViewModule *vm);

	virtual int GetNumberOfActiveViews() const;

	void SetRenderWindowPosition(int x, int y);
	const int* GetRenderWindowPosition() const;
	virtual void SetRenderWindowSize(int w, int h);
	const int* GetRenderWindowSize() const;

	inline iMagnifier* GetMagnifier() const { return mMagnifier; }
	inline vtkRenderer* GetRenderer() const { return mMainRen; }
	inline vtkRenderWindow* GetRenderWindow() const { return mMainWin; }
	inline vtkRenderWindowInteractor* GetInteractor() const { return mMainInt; }
	vtkRenderWindowCollection* GetRenderWindowCollection();
	inline iCamera* GetCamera() const { return mMainCam; }

	int GetRenderingMagnification() const;

	void Render();
	virtual void AddObserver(unsigned long event, vtkCommand *command);    
	virtual void UpdateWindowName(const iString &base);

	iLightKit* GetLights() const { return mLights; }
	virtual void SetBackground(iColor &color); // this is non-const due to vtkProperty::SetColor() function signature 
	virtual float GetLastRenderTimeInSeconds() const;
	void GetAspectRatio(double ar[2]) const;

	virtual void AddObject(vtkProp* p);
	virtual void RemoveObject(vtkProp* p);

	virtual void SetAntialiasing(bool s);
	inline bool GetAntialiasing() const { return mAntialiasing; }

	virtual void SetStereoType(int s);
	inline int GetStereoType() const { return mStereoType; }

	virtual void SetStereo(bool s);
	bool GetStereo() const;

	virtual void ShowStereoAlignmentMarkers(bool s); 

	virtual void RenderImages(int mag, iStereoImageArray &images);
	void RenderStereoImage(int mag, iStereoImage &image);

protected:
	
	iRenderTool(iViewModule *vm, vtkRenderer *ren = 0, iMagnifier *mag = 0);
	virtual ~iRenderTool();

	virtual iRenderTool* CreateInstance(vtkRenderer *ren) const;
	vtkRenderer* CreateRenderer() const;
	void ShowDualWindow(bool s);

	//
	//  Functions used by iCamera class
	//
	virtual void ResetCamera();
	void ResetCameraClippingRange();
	void SetAdjustCameraClippingRangeAutomatically(bool s);
	inline bool GetAdjustCameraClippingRangeAutomatically() const { return mAutoClippingRange; }
	void SetCameraClippingRange(double cr[2]);
	inline void GetCameraClippingRange(double cr[2]) const { cr[0] = mClippingRange[0]; cr[1] = mClippingRange[1]; }

	virtual void UpdateWindowCollection();
	void WindowsModified();

	//
	//  Rendering pipeline
	//
	vtkRenderWindow *mMainWin, *mDualWin;
	vtkRenderer *mMainRen, *mDualRen;
	vtkRenderWindowInteractor *mMainInt;
	iDualWindowObserver *mDualWindowObserver;
	iRendererObserver *mRendererObserver;
	vtkRenderWindowCollection *mWindowCollection;
	iCamera *mMainCam;
	vtkPropAssembly *mStereoAlignmentMarkersActor;

	//
	//  Observers registry
	//
	struct Observer
	{
		unsigned long Event;
		vtkCommand *Command;
		Observer(){ Event = 0L; Command = 0; }
	};
	iArray<Observer> mObservers;

private:

	//
	//  Other members
	//
	iLightKit *mLights;
	iMagnifier *mMagnifier;
	int mStereoType;
	double mClippingRange[2];
	bool mAntialiasing, mStereoAlignmentMarkersOn, mAutoClippingRange, mWindowCollectionUpToDate;
};

#endif // IRENDERVIEW_H

