/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icaption.h"


#include "iactor.h"
#include "ierror.h"
#include "ioverlayhelper.h"

#include <vtkActor2D.h>
#include <vtkCamera.h>
#include <vtkCellArray.h>
#include <vtkMath.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper2D.h>
#include <vtkProperty.h>
#include <vtkProperty2D.h>
#include <vtkRenderer.h>
#include <vtkTextMapper.h>
#include <vtkTextProperty.h>


iCaption* iCaption::New(iRenderTool *rt)
{
	IERROR_ASSERT(rt);
	return new iCaption(rt);
}


iCaption::iCaption(iRenderTool *rt) : mOverlayHelper(rt)
{
	this->BorderOff(); // we redefine border and leader - they do not scale under magnification
	this->LeaderOff();
	this->ThreeDimensionalLeaderOn();
	this->GetPositionCoordinate()->SetReferenceCoordinate(0);
	this->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
	this->GetPosition2Coordinate()->SetCoordinateSystemToNormalizedViewport();
	this->SetPosition(0.02,0.88);
	this->SetPosition2(0.20,0.10);
	this->GetCaptionTextProperty()->SetJustificationToCentered(); 
	this->GetCaptionTextProperty()->SetVerticalJustificationToCentered();

	this->PickableOff();

	mBorderPolyData = vtkPolyData::New(); IERROR_ASSERT(mBorderPolyData);
	vtkPoints *pts = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT(pts);
	pts->SetNumberOfPoints(4);
	mBorderPolyData->SetPoints(pts);
	pts->Delete();
	vtkCellArray *border = vtkCellArray::New(); IERROR_ASSERT(border);
	border->InsertNextCell(5);
	border->InsertCellPoint(0);
	border->InsertCellPoint(1);
	border->InsertCellPoint(2);
	border->InsertCellPoint(3);
	border->InsertCellPoint(0);
	mBorderPolyData->SetLines(border);
	border->Delete();
	
	mBorderMapper = vtkPolyDataMapper2D::New(); IERROR_ASSERT(mBorderMapper);
	mBorderMapper->SetInput(mBorderPolyData);
	mBorderActor = vtkActor2D::New(); IERROR_ASSERT(mBorderActor);
	mBorderActor->SetMapper(mBorderMapper);
	
	// This is the leader (line) from the attachment point to the caption
	mLeaderPolyData = vtkPolyData::New(); IERROR_ASSERT(mLeaderPolyData);
	pts = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT(pts);
	pts->SetNumberOfPoints(2);
	mLeaderPolyData->SetPoints(pts);
	pts->Delete();
	vtkCellArray *leader = vtkCellArray::New(); IERROR_ASSERT(leader);
	leader->InsertNextCell(2); 
	leader->InsertCellPoint(0);
	leader->InsertCellPoint(1); //at the attachment point
	mLeaderPolyData->SetLines(leader);
	leader->Delete();
	
	mLeaderMapper2D = vtkPolyDataMapper2D::New(); IERROR_ASSERT(mLeaderMapper2D);
	mLeaderMapper2D->SetInput(mLeaderPolyData);
	mLeaderActor2D = vtkActor2D::New(); IERROR_ASSERT(mLeaderActor2D);
	mLeaderActor2D->SetMapper(mLeaderMapper2D);

	mLeaderActor3D = iActor::New(); IERROR_ASSERT(mLeaderActor3D);
	mLeaderActor3D->SetInput(mLeaderPolyData);

	mTmpMapper = vtkTextMapper::New(); IERROR_ASSERT(mTmpMapper);

	mLeader3D = true; // 2D leader does not work under magnification
}


iCaption::~iCaption()
{
	mBorderPolyData->Delete();
	mBorderMapper->Delete();
	mBorderActor->Delete();
	
	mLeaderPolyData->Delete();
	mLeaderMapper2D->Delete();
	mLeaderActor2D->Delete();
	mLeaderActor3D->Delete();
	
	mTmpMapper->Delete();
}


void iCaption::ReleaseGraphicsResources(vtkWindow *win)
{
	this->vtkCaptionActor2D::ReleaseGraphicsResources(win);
	mBorderActor->ReleaseGraphicsResources(win); 
	mLeaderActor2D->ReleaseGraphicsResources(win); 
	mLeaderActor3D->ReleaseGraphicsResources(win); 
}


int iCaption::RenderOverlay(vtkViewport* viewport)
{
	int ret = 0;
	ret += this->vtkCaptionActor2D::RenderOverlay(viewport);
	ret += mBorderActor->RenderOverlay(viewport);
    if(mLeader3D) ret += mLeaderActor3D->RenderOverlay(viewport); else ret += mLeaderActor2D->RenderOverlay(viewport);
	return ret;
}


int iCaption::RenderOpaqueGeometry(vtkViewport* viewport)
{
	vtkCamera *cam = mOverlayHelper->GetCamera(viewport);
	if(cam == 0) return 0;
	
	int ret = 0;
	//
	//  Maintain the font size
	//
	double *x1, *x2;
	x1 = this->GetPosition();
	x2 = this->GetPosition2();
	int mag = mOverlayHelper->GetRenderingMagnification();
	int *size = viewport->GetSize();

	float s = (mOverlayHelper->GetFontSize(viewport)+2*Padding)/fabs(size[1]*x2[1]);

	if(mag == 1)
	{
		mUnmagx1[0] = x1[0];
		mUnmagx1[1] = x1[1];
		mUnmagx2[0] = s*x2[0];
		mUnmagx2[1] = s*x2[1];
	}
	else
	{
		int winij[2];
		mOverlayHelper->ComputePositionShiftsUnderMagnification(winij);
		
		this->SetPosition(mag*mUnmagx1[0]-winij[0],mag*mUnmagx1[1]-winij[1]);
		Padding *= mag;
	}
	
	mTmpMapper->GetTextProperty()->SetFontSize(mOverlayHelper->GetFontSize(viewport,mag));
	mTmpMapper->SetInput(Caption);
	int w = mTmpMapper->GetWidth(viewport);
	if(w+2*Padding >= size[0]*mUnmagx2[0]) mUnmagx2[0] = 10.0*floor(1+0.1*(w+2*Padding))/size[0];
	
	this->SetPosition2(mUnmagx2[0]*mag,mUnmagx2[1]*mag);

	this->GetProperty()->SetColor(this->GetOverlayHelper()->GetColor(viewport).ToVTK());
	this->GetCaptionTextProperty()->SetColor(this->GetOverlayHelper()->GetColor(viewport).ToVTK());
	this->GetOverlayHelper()->UpdateTextProperty(this->GetCaptionTextProperty());
	ret = this->vtkCaptionActor2D::RenderOpaqueGeometry(viewport);
	
	//
	//  Scale lines
	//
	mBorderActor->GetProperty()->SetLineWidth(mag);
	if(mLeader3D) mLeaderActor3D->GetProperty()->SetLineWidth(mag); else mLeaderActor2D->GetProperty()->SetLineWidth(mag);
	//
	// This piece is copied from vtkCaptionActor2D
	//
	{
		//
		// compute coordinates and Set point values
		//
		int *x1, *x2, *x3;
		double p1[4], p2[4], p3[4];
		x1 = this->AttachmentPointCoordinate->GetComputedDisplayValue(viewport);
		x2 = this->PositionCoordinate->GetComputedDisplayValue(viewport);
		x3 = this->Position2Coordinate->GetComputedDisplayValue(viewport);
		p1[0] = (float)x1[0]; p1[1] = (float)x1[1]; p1[2] = 0.0;
		p2[0] = (float)x2[0]; p2[1] = (float)x2[1]; p2[2] = p1[2];
		p3[0] = (float)x3[0]; p3[1] = (float)x3[1]; p3[2] = p1[2];
		
		// Define the border
		vtkPoints *pts = mBorderPolyData->GetPoints();
		pts->SetPoint(0, p2);
		pts->SetPoint(1, p3[0],p2[1],p1[2]);
		pts->SetPoint(2, p3[0],p3[1],p1[2]);
		pts->SetPoint(3, p2[0],p3[1],p1[2]);
		
		if(mag == 1)
		{
			// Define the leader. Have to find the closest point from the
			// border to the attachment point. We look at the four vertices
			// and four edge centers.
			double d2, minD2, pt[3], minPt[3];
			minD2 = VTK_LARGE_FLOAT;
			
			pt[0] = p2[0]; pt[1] = p2[1]; pt[2] = minPt[2] = 0.0;
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[0] = (p2[0]+p3[0])/2.0;
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[0] = p3[0];
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[1] = (p2[1]+p3[1])/2.0;
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[1] = p3[1];
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[0] = (p2[0]+p3[0])/2.0;
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[0] = p2[0];
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			pt[1] = (p2[1]+p3[1])/2.0;
			if ( (d2 = vtkMath::Distance2BetweenPoints(p1,pt)) < minD2 )
			{
				minD2 = d2;
				minPt[0] = pt[0]; minPt[1] = pt[1]; 
			}
			
			// Set the leader coordinates in appropriate coordinate system
			// The pipeline is connected differently depending on the dimension
			// and availability of a leader head.
			pts = mLeaderPolyData->GetPoints();
			
			if(mLeader3D)
			{
				viewport->SetDisplayPoint(minPt[0],minPt[1],0.0);
				viewport->DisplayToWorld();
				viewport->GetWorldPoint(p1);
				
				double *cpos = cam->GetPosition();
				double *clip = cam->GetClippingRange();
				
				p3[0] = p1[0] - cpos[0];
				p3[1] = p1[1] - cpos[1];
				p3[2] = p1[2] - cpos[2];
				
				cam->GetDirectionOfProjection(p2);
				vtkMath::Normalize(p2);
				float s = clip[0]/vtkMath::Dot(p2,p3);
				p1[0] = cpos[0] + s*p3[0];
				p1[1] = cpos[1] + s*p3[1];
				p1[2] = cpos[2] + s*p3[2];
				
				pts->SetPoint(0,this->AttachmentPointCoordinate->GetComputedWorldValue(viewport));
				pts->SetPoint(1,p1);
				pts->Modified();
				
				mLeaderPolyData->Update();
			}
			else
			{
				pts->SetPoint(0,p1);
				pts->SetPoint(1,minPt[0],minPt[1],0.0);
			}
		}
		else
		{
			//
			//  Shift positions if under magnification - 
			//
			int winij[2];
			mOverlayHelper->ComputePositionShiftsUnderMagnification(winij);

			pts = mLeaderPolyData->GetPoints();
		}

		//
		// assign properties
		//
		mBorderActor->GetProperty()->SetColor(this->GetOverlayHelper()->GetColor(viewport).ToVTK());
		if(mLeader3D) mLeaderActor3D->GetProperty()->SetColor(this->GetOverlayHelper()->GetColor(viewport).ToVTK()); else mLeaderActor2D->GetProperty()->SetColor(this->GetOverlayHelper()->GetColor(viewport).ToVTK());
		
	}
	
	ret += mBorderActor->RenderOpaqueGeometry(viewport);
	if(mLeader3D) ret += mLeaderActor3D->RenderOpaqueGeometry(viewport); else ret += mLeaderActor2D->RenderOpaqueGeometry(viewport); 
	
	//
	//  Reset to the unmagnified state
	//
	if(mag > 1)
	{
		this->SetPosition(mUnmagx1[0],mUnmagx1[1]);
		this->SetPosition2(mUnmagx2[0],mUnmagx2[1]);
		Padding /= mag;
	}
	
	return ret;
}

