/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ioutputchannel.h"


#include "ifile.h"
#include "iparameter.h"
#include "ishell.h"
#include "ivtk.h"

#include <vtkMutexLock.h>
#include <vtkOutputWindow.h>
#include <vtkTimerLog.h>


iOutputChannel* iOutputChannel::mInstance = 0;


namespace iOutputChannel_Private
{
	//
	//  Converts a Gregorian date to a Julian day.
	//  This algorithm is taken from Communications of the ACM, Vol 6, No 8.
	//
	unsigned int DateToJd(int y, int m, int d)
	{
		unsigned int c, ya;
		if ( y <= 99 )
			y += 1900;
		if ( m > 2 ) {
			m -= 3;
		} else {
			m += 9;
			y--;
		}
		c = y;
		c /= 100;
		ya = y - 100*c;
		return 1721119 + d + (146097*c)/4 + (1461*ya)/4 + (153*m+2)/5;
	}
	
	//
	//  Converts a Julian day to a Gregorian date.
	//	This algorithm is taken from Communications of the ACM, Vol 6, No 8.
	//
	void JdToDate(unsigned int jd, int &y, int &m, int &d)
	{
		unsigned int x;
		unsigned int j = jd - 1721119;
		y = (j*4 - 1)/146097;
		j = j*4 - 146097*y - 1;
		x = j/4;
		j = (x*4 + 3) / 1461;
		y = 100*y + j;
		x = (x*4) + 3 - 1461*j;
		x = (x + 4)/4;
		m = (5*x - 3)/153;
		x = 5*x - 3 - 153*m;
		d = (x + 5)/5;
		if ( m < 10 ) {
			m += 3;
		} else {
			m -= 9;
			y++;
		}
	}

	//
	//  Converts Universal Coordinated time to a Julian day
	//
	unsigned int UctToJd(double uct)
	{
		return 2440588 + int(uct/86400.0);
	}

	//
	//  Converts a Julian day to Universal Coordinated time
	//
	double JdToUct(unsigned int jd)
	{
		return 86400.0*(jd-2440588);
	}

	//
	//  Converts Universal Coordinated time to Gregorian date/time
	//
	void UctToDateTime(double uct, int &year, int &month, int &day, int &hour, int &minute, int &second)
	{
		unsigned int jd = UctToJd(uct);
		JdToDate(jd,year,month,day);
		unsigned int s = round(uct-JdToUct(jd));
		hour = s/3600;
		minute = (s-3600*hour)/60;
		second = s - 3600*hour - 60*minute;
	}

	//
	//  Take over VTK display function to prevent it from displaying extra windows
	//
	class OutputWindow : public vtkOutputWindow
	{
		vtkTypeMacro(OutputWindow,vtkOutputWindow);
		static OutputWindow* New(iOutputChannel *c = 0)
		{
			if(c == 0) return 0; else return new OutputWindow(c);
		}

		//
		//  Don't overwrite those, they ARE thread-safe
		//
		virtual void DisplayText(const char* message)
		{
			if(mMutex!=0 && mOutput!=0 && !mInDisplay)
			{
				mMutex->Lock();
				mInDisplay = true;
				mOutput->Display(iConsole::_Info,message,0,0);
				mInDisplay = false;
				mMutex->Unlock();
			}
		}

		virtual void DisplayErrorText(const char* message)
		{
			if(mMutex!=0 && mOutput!=0 && !mInDisplay)
			{
				const char* fm = this->FormattedErrorMessage(message);
				if(fm != 0)
				{
					mMutex->Lock();
					mInDisplay = true;
					mOutput->Display(iConsole::_LowError,fm,0,0);
					mInDisplay = false;
					mMutex->Unlock();
				}
			}
		}

		virtual void DisplayWarningText(const char* message)
		{
			if(mMutex!=0 && mOutput!=0 && !mInDisplay)
			{
				mMutex->Lock();
				mInDisplay = true;
				mOutput->Display(iConsole::_Warning,message,0,0);
				mInDisplay = false;
				mMutex->Unlock();
			}
		}

		virtual void DisplayGenericWarningText(const char* message)
		{
			this->DisplayWarningText(message);
		}

		virtual void DisplayDebugText(const char* message)
		{
#ifdef I_DEBUG
			this->DisplayText(message);
#endif
		}

	protected:

		virtual ~OutputWindow()
		{
			if(mMutex != 0) mMutex->Delete();
		}

	private:

		OutputWindow(iOutputChannel *c)
		{
			mInDisplay = false;
			mOutput = c;
			mMutex = vtkMutexLock::New();
		}

		const char* FormattedErrorMessage(const char* message) const
		{
			//
			//  Block some of the VTK error messages - not all are useful
			//
			const int nBlockError = 4;
			const char *classBlockError[] = { 
				"vtkDecimate",
				"vtkPolyDataNormals",
				"vtkWindowedSincPolyDataFilter",
				"vtkStreamingDemandDrivenPipeline"
			};
			const char *messgBlockError[] = {
				"No data to decimate!",
				"No data to generate normals for!",
				"No data to smooth!",
				"The update extent specified in the information for output port"
			};

			int i;
			for(i=0; i<nBlockError; i++)
			{
				if(strstr(message,classBlockError[i])!=0 && strstr(message,messgBlockError[i])!=0) return 0;
			}

			//
			//  Cut extra parts of VTK error messages
			//
			static const char *marker = "MESSAGE:";
			const char *outMessage = strstr(message,marker);
			if(outMessage == 0)
			{
				outMessage = message;
			}
			else
			{
				outMessage += strlen(marker);
			}

			return outMessage;
		}

		const char* FormattedMessage(const char* message) const
		{
		}

		bool mInDisplay;
		vtkMutexLock *mMutex;
		iOutputChannel *mOutput;
	};
};
	

using namespace iOutputChannel_Private;


iOutputChannel* iOutputChannel::GetInstance()
{
	if(mInstance == 0)
	{
		mInstance = new iOutputChannel(0);
	}
	return mInstance;
}


void iOutputChannel::SetInstance(iOutputChannel *c)
{
	if(c != 0)
	{
		if(mInstance != 0) mInstance->Delete();
		mInstance = c;
		OutputWindow *ow = OutputWindow::New(c);
		vtkOutputWindow::SetInstance(ow);
		ow->Delete();
	}
}


iOutputChannel::iOutputChannel(iShell *s)
{
	mInDisplay = mLoggingBlocked = mLogCreated = mInUpdateLog = false;

	if(s != 0)
	{
		//
		//  Attach the log file
		//
		mLogFileName = s->GetEnvironment(iParameter::_EnvironmentBase) + "ifrit.err";
	}
}


iOutputChannel::~iOutputChannel()
{
}


void iOutputChannel::UpdateLog(const iString &message, const char* file, int line)
{
	if(mLogFileName.IsEmpty() || mLoggingBlocked || mInUpdateLog) return;

	mInUpdateLog = true;

	iFile F(mLogFileName);
	if(F.Open(iFile::_Append,iFile::_Text) || F.Open(iFile::_Write,iFile::_Text))
	{
		iString ws = message + "  File: " + file + "  Line: " + iString::FromNumber(line);
	
		if(!mLogCreated)
		{
#ifdef IVTK_4
			double s = vtkTimerLog::GetCurrentTime();
#else
			double s = vtkTimerLog::GetUniversalTime();
#endif
			int year, month, day, hour, minute, second;
			UctToDateTime(s,year,month,day,hour,minute,second);

			F.WriteLine(" ");
			F.WriteLine("New run: "+iString::FromNumber(year)+"/"+iString::FromNumber(month)+"/"+iString::FromNumber(day)+"/"+iString::FromNumber(hour)+"/"+iString::FromNumber(minute)+"/"+iString::FromNumber(second));
			F.WriteLine("---------------------------");
		}

		if(F.WriteLine(ws)) mLogCreated = true;
		F.Close();
	}

	mInUpdateLog = false;
}


void iOutputChannel::Display(iConsole::MessageType type, const iString &message, const char *file, int line)
{
	if(mInDisplay) return;
	mInDisplay = true;

#ifdef I_DEBUG
	if(type == iConsole::_Info)
	{
		iConsole::PrintDebugMessage(message);
	}
	else
	{
		if(message.Part(0,12) != "Missing key:")
		{
			const char *ptr = message.ToCharPointer();
			int ooo = 0;
		}
	}
#endif

	this->DisplayBody(type,message,file,line);

	//
	//  Update log
	//
	switch(type)
	{
	case iConsole::_HighError:
//	case iConsole::_LowError:
		{
			this->UpdateLog(message,file,line);
			break;
		}
	case iConsole::_FatalError:
		{
			this->UpdateLog(message,file,line);
			exit(1);
			break;
		}
	}

	mInDisplay = false;
}


void iOutputChannel::DisplayBody(iConsole::MessageType type, const iString &message, const char *file, int line)
{
	//
	//  Send everything to the standard vtkOutputWindow
	//
	vtkOutputWindow::GetInstance()->DisplayText(this->FormPlainTextMessage(type,message,file,line).ToCharPointer());
}


iString iOutputChannel::FormPlainTextMessage(iConsole::MessageType type, const iString &message, const char *file, int line)
{
	iString text;

	switch(type)
	{
	case iConsole::_Info:
		{
			text = message + "\n";
			break;
		}
	case iConsole::_Warning:
		{
			text = "!IMPORTANT: " + message + "\n";
			break;
		}
	case iConsole::_LowError:
	case iConsole::_HighError:
		{
			text = "! NON-FATAL ERROR: " + message + "\n";
			if(file != 0)
			{
				text += "             FILE: " + iString(file) + "\n";
				text += "             LINE: " + iString::FromNumber(line) + "\n";
			}
			break;
		}
	case iConsole::_FatalError:
		{
			text = "!!! FATAL ERROR: " + message + "\n";
			if(file != 0)
			{
				text += "             FILE: " + iString(file) + "\n";
				text += "             LINE: " + iString::FromNumber(line) + "\n";
			}
			text += "IFrIT will now abort.\n";
			break;
		}
	}

	return text;
}


void iOutputChannel::BlockLogging(bool s)
{
	mLoggingBlocked = s;
}


void iOutputChannel::NotifyIfLogCreated()
{
	if(mLogCreated)
	{
		iString ws = "A log of all error messages was created in the file:\n" + mLogFileName + "\n Please, consider e-mailing this file to ifrit.bugs@gmail.com";
		this->Display(iConsole::_Warning,ws,0,0);
	}
}

