/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//**********************************************************************
//
//  IFrIT script class - maps commands to function calls
//
//**********************************************************************


#ifndef ISCRIPT_H
#define ISCRIPT_H


#include <vtkObjectBase.h>


#include "iarray.h"
#include "istring.h"

#include <vtkSetGet.h>

//
//  Define the base class for possible values.
//  All values are accessed transparently via the base class - in a VTK style.
//  Specify a creator New() for each of possible value types (defined below).
//  Values can be accessed in 3 ways: as a numeric constant, as a pointer to a value ("local variable"),
//  as a result of calling a function ("global variable").
//

class iScript;

class iValue
{

friend class iScript;

public:

	virtual void Delete();

	//
	//  Assignment operation type
	//
	enum Assignment 
	{ 
		_Absolute = 0, 
		_RelativeAdd = 1, 
		_RelativeMult = 2, 
		_RelativeAddMult = 3, 
		_RelativeInvert = 4, 
		_RelativeInvertAdd = 5,  
		_RelativeInvertMult = 6, 
		_RelativeInvertAddMult = 7 
	};
	//
	//  Value Method constants
	//
	enum ValueMethod
	{
		_None,
		_ByValue,
		_ByPointer,
		_ByFunction
	};
	//
	//  Value Type constants
	//
	enum ValueType
	{
		_Value_NO =  0,
		_Value_BL =  1,		//  boolean scalar
		_Value_IS =  2,		//  integer scalar
		_Value_RS =  4,		//  real scalar
		_Value_RP =  8,		//  real pointer
		_Value_ST = 16,		//  iString "scalar" (string)
		_Value_VA = 32		//  iValue array 
	};

	// BL creator
	static iValue* New(iScript *w, const iString &n, ValueMethod m, bool v, bool *p = 0, bool (*f)(iScript* ) = 0);
	// IS creator
	static iValue* New(iScript *w, const iString &n, ValueMethod m, int v, int *p = 0, int (*f)(iScript* ) = 0);
	// FS creator
	static iValue* New(iScript *w, const iString &n, ValueMethod m, double v, double *p = 0, double (*f)(iScript* ) = 0);
	// FP creator
	static iValue* New(iScript *w, const iString &n, ValueMethod m, double* v, double* *p = 0, double* (*f)(iScript* ) = 0);
	// ST creator
	static iValue* New(iScript *w, const iString &n, ValueMethod m, const iString &v, iString *p = 0, iString (*f)(iScript* ) = 0, bool q = true);
	// VA creator
	static iValue* New(iScript *w, const iString &n, ValueMethod m, int s, iValue* i, iValue **a, iValue* (*f)(iScript*,int) = 0);

	// generic New - copy constructor, optionally Set value by reference rather than copy it
	static iValue* New(iValue *v, bool ref = false);

	// shortcuts for Operation constructors
	static iValue* NewInt(iScript *w);
	static iValue* NewBool(iScript *w);
	static iValue* NewReal(iScript *w);
	static iValue* NewArray(iScript *w, int size = 0);
	static iValue* NewIndex(iScript *w);
	static iValue* NewString(iScript *w, bool quoted = true);

	virtual iString GetValueAsText() const;
	virtual iString GetTypeAsText() const;

	static bool AreTypesCompatible(iValue *v1, iValue *v2);
	static bool AreTypesCompatible(ValueType t, iValue *v);

	inline ValueType GetType() const { return Type; }
	inline ValueMethod GetMethod() const { return Method; }
	inline const iString& GetName() const { return Name; }
	
	inline bool IsScalar() const { return (Type==_Value_BL || Type==_Value_RP || Type==_Value_RS || Type==_Value_IS); }
	inline bool IsArray() const { return (Type == _Value_VA); }

	virtual bool SetMethod(ValueMethod m) = 0;

	virtual bool AssignValue(Assignment r, iValue *v) = 0;
	virtual bool AssignValue(Assignment, bool){ return false; }
	virtual bool AssignValue(Assignment, int){ return false; }
	virtual bool AssignValue(Assignment, double){ return false; }

	virtual bool GetValue(void *v) const  = 0;
	virtual bool GetValue(bool &) const { return false; }
	virtual bool GetValue(int &) const { return false; }
	virtual bool GetValue(float &) const { return false; }
	virtual bool GetValue(double &) const { return false; }
	virtual bool GetValue(double* &) const { return false; }
	virtual bool GetValue(iString &) const { return false; }
	virtual bool GetValue(int, int * ) const { return false; }
	virtual bool GetValue(int, double * ) const { return false; }

	virtual void Register(){ ReferenceCount++; }
	inline int GetReferenceCount() const { return ReferenceCount; }

#ifdef I_CHECK1
	static int counterCreate, counterDestroy;
	int counterAtCreation;
	static int cval[999];
#endif

protected:

	iValue(iScript *w, const iString &n, ValueType t, ValueMethod m);
	virtual ~iValue();

	int ReferenceCount;
	iScript *Script;
	iString Name;
	ValueType Type;
	ValueMethod Method;
};
//
//  Value type BL: 
//    Parameter: bool v, bool *p, bool (*f)(void)
//    Result: v, *p, f()
//
class iValue_BL : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual bool SetMethod(ValueMethod m);
	virtual void SetNumericValue(bool v){ Value = v; Method = _ByValue; }

	virtual bool AssignValue(Assignment r, iValue *v);
	virtual bool AssignValue(Assignment r, bool b);

	virtual bool GetValue(void *p) const;
	virtual bool GetValue(bool &) const;

protected:

	iValue_BL(iScript *w, const iString &n, ValueMethod m, bool v, bool *p, bool (*f)(iScript* )) : iValue(w,n,_Value_BL,m)
	{ 
		Value = v; 
		Pointer = p; 
		Function = f; 
	}

	bool Value;
	bool* Pointer;
	bool (*Function)(iScript* );

};
//
//  Value type IS: 
//    Parameter: int v, int *p, int (*f)(void)
//    Result: v, *p, f()
//
class iValue_IS : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual bool SetMethod(ValueMethod m);
	virtual void SetNumericValue(int v){ Value = v; Method = _ByValue; }

	virtual bool AssignValue(Assignment r, iValue *v);
	virtual bool AssignValue(Assignment r, int i);
	virtual bool AssignValue(Assignment r, double f);

	virtual bool GetValue(void *p) const;
	virtual bool GetValue(bool &) const;
	virtual bool GetValue(int &) const;
	virtual bool GetValue(float &) const;
	virtual bool GetValue(double &) const;

protected:

	iValue_IS(iScript *w, const iString &n, ValueMethod m, int v, int *p, int (*f)(iScript* )) : iValue(w,n,_Value_IS,m)
	{ 
		Value = v; 
		Pointer = p; 
		Function = f; 
	}

	int Value;
	int* Pointer;
	int (*Function)(iScript* );

};
//
//  Value type RS: 
//    Parameter: double v, double *p, double (*f)(void)
//    Result: v, *p, f()
//
class iValue_RS : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual bool SetMethod(ValueMethod m);
	virtual void SetNumericValue(double v){ Value = v; }

	virtual bool AssignValue(Assignment r, iValue *v);
	virtual bool AssignValue(Assignment r, int i);
	virtual bool AssignValue(Assignment r, double f);

	virtual bool GetValue(void *p) const;
	virtual bool GetValue(bool &) const;
	virtual bool GetValue(int &) const;
	virtual bool GetValue(float &) const;
	virtual bool GetValue(double &) const;

protected:

	iValue_RS(iScript *w, const iString &n, ValueMethod m, double v, double *p, double (*f)(iScript* )) : iValue(w,n,_Value_RS,m)
	{ 
		Value = v; 
		Pointer = p; 
		Function = f; 
	}

	double Value;
	double* Pointer;
	double (*Function)(iScript* );
};
//
//  Value type RP: 
//    Parameter: double *v, double **p, double* (*f)(void)
//    Result: v, *p, f()
//
class iValue_RP : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual bool SetMethod(ValueMethod m);
	virtual void SetNumericValue(double *v){ Value = v; }

	virtual bool AssignValue(Assignment r, iValue *v);

	virtual bool GetValue(void *p) const;
	virtual bool GetValue(double* &p) const;

protected:

	iValue_RP(iScript *w, const iString &n, ValueMethod m, double* v, double* *p, double* (*f)(iScript* )) : iValue(w,n,_Value_RP,m)
	{ 
		Value = v; 
		Pointer = p; 
		Function = f; 
	}

	double* Value;
	double** Pointer;
	double* (*Function)(iScript* );
};
//
//  Value type ST: 
//    Parameter: iString v, iString *p, iString (*f)(void)
//    Result: v, *p, f()
//
class iValue_ST : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual bool SetMethod(ValueMethod m);
	virtual void SetNumericValue(const iString &v){ Value = v; }

	virtual bool AssignValue(Assignment r, iValue *v);

	virtual bool GetValue(void *p) const;
	virtual bool GetValue(iString &p) const;

	bool IsQuoted() const { return Quoted; }

protected:

	iValue_ST(iScript *w, const iString &n, ValueMethod m, const iString &v, iString *p, iString (*f)(iScript* ), bool q) : iValue(w,n,_Value_ST,m)
	{ 
		Value = v; 
		Pointer = p; 
		Function = f; 
		Quoted = q;
	}

	iString Value;
	iString* Pointer;
	iString (*Function)(iScript* );
	bool Quoted;
};
//
//  Value type VA: 
//    Parameter: iValue* i, iValue** a, iValue* (*f)(int)
//    Result: a[i], f(i)
//
class iValue_VA : public iValue
{

friend class iScript;
friend class iValue;

public:

	virtual void Delete();

	virtual bool SetMethod(ValueMethod m);

	inline int GetSize() const { return Size; }
	inline iValue* GetIndex() const { return Index; }
	virtual iValue* GetComponent(iValue *v) const;
	virtual iValue* GetComponent(int i) const;

	void SetIndex(iValue *i);

	virtual bool AssignValue(Assignment r, iValue *v);

	virtual bool GetValue(void *p) const;
	virtual bool GetValue(bool &) const;
	virtual bool GetValue(int &) const;
	virtual bool GetValue(double &) const;
	virtual bool GetValue(int, int *v) const;
	virtual bool GetValue(int, double *v) const;

protected:

	iValue_VA(iScript *w, const iString &n, ValueMethod m, int s, iValue* i, iValue **a, iValue* (*f)(iScript*,int)) : iValue(w,n,_Value_VA,m)
	{ 
		Index = i; 
		Array = a; 
		Function = f; 
		Size = s; 
	}

	int Size;
	iValue* Index;
	iValue** Array;
	iValue* (*Function)(iScript*,int);
};

//
//  Define the base class for possible operations.
//  All operations are accessed transparently via the base class - in a VTK style.
//  Specify a creator New() for each of possible operation types (defined below).
//
//   All operations are divided into different classes:
//   - FC flow control operations 
//   - SF operations calling various scalar functions
//   - AF similar to SF, but have array syntax (e.f. Set var[N] = V calls F(N,V))
//

class iOperation
{

friend class iScript;

public:

	virtual void Delete(){ delete this; }

	//
	//  Operation Type constants
	//
	enum OperationType
	{
		_Operation_DC_SV,		//  declaration, simple variable
		_Operation_FC_LB,		//  loop & branch
		_Operation_SF_V0,		//  f(void)
		_Operation_SF_V1,		//  f(iValue*,Assignment)
		_Operation_AF_V1		//  f(iValue*,iValue*,Assignment)
	};
	//
	//  Entry points
	//
	enum EntryPoint
	{ 
		_Entry = 1,				// can only enter this point (if)
		_Exit = 2,				// can only exit this point (endif)
		_EntryExit = 3			// can do both (else)
	};
	//
	//  Status of the branch gate
	//
	enum GateStatus
	{
		_Skip = 0,
		_Open = 1,
		_Close = 2,
		_Flip = 3
	};

	// DC_SV creator
	static iOperation* New(iScript *w, const iString &p, const iString &c, int o, const iString &sp, iValue::Assignment r, iValue *v, unsigned char u = 0);
	// FC_LB creator
	static iOperation* New(iScript *w, const iString &p, const iString &c, int o, EntryPoint q, iValue *i, iValue *n, int l, unsigned char u = 0);
	// SF_V0 creator
	static iOperation* New(iScript *w, const iString &p, const iString &c, int o, void (*f)(iScript* ), unsigned char u = 0);
	// SF_V1 creator
	static iOperation* New(iScript *w, const iString &p, const iString &c, int o, iValue::Assignment r, iValue* v, void (*f)(iScript*,iValue*,iValue::Assignment), unsigned char u = 0);
	// AF_V1 creator
	static iOperation* New(iScript *w, const iString &p, const iString &c, int o, iValue* i, iValue::Assignment r, iValue* v, void (*f)(iScript*,iValue*,iValue*,iValue::Assignment), unsigned char u = 0);

	// generic New - copy constructor, optionally Set value by reference rather than copy it
	static iOperation* New(iOperation *p);

protected:

	virtual iValue* GetValue() const { return 0; }
	virtual void SetValue(iValue* ){}
	virtual bool AssignVariableValue(const iString &n, iValue::Assignment r, iValue *v, iValue *i = 0);
	virtual bool Exec() = 0;

	iScript *Script;
	iString Command;
	iString Prefix;
	OperationType Type;
	unsigned char UserType;
	int OutputCode;
	int LineInScript;

	iOperation(iScript *w, const iString &p, const iString &s, int o, OperationType t, unsigned char u = 0);
	virtual ~iOperation();
};
//***************************************************************
//
//	Declaration operations
//
//***************************************************************
//
//  Operation type DC_SV: 
//    Parameter: iValue *V
//    Execution: declare a new variable with value V. Executed at the compile time
//
class iOperation_DC_SV : public iOperation 
{

friend class iScript;
friend class iOperation;

public:

	virtual void Delete();

protected:

	iOperation_DC_SV(iScript *w, const iString &p, const iString &s, int o, const iString &sp, iValue::Assignment r, iValue *v, unsigned char u) : iOperation(w,p,s,o,_Operation_DC_SV,u)
	{ 
		StatementPrefix = sp;
		Relative = r;
		Value = v; 
	}

	virtual iValue* GetValue() const { return Value; }
	virtual bool Exec();

	iString StatementPrefix;
	iValue *Value;
	iValue::Assignment Relative;
};
//***************************************************************
//
//	Flow Control operations
//
//***************************************************************
//
//  Operation type FC_LB: 
//    Parameter: EntryPoint Q, iValue *I, iValue *C, int C
//    Execution: if Q=Entry, then open the gate if either C is boolen and true, or C is integer and 
//                  C >= Running Valued or close the gate in the opposite case. Flip the gate is C
//                  is 0.
//               if Q=Exit, close the branch of the previous Q=Entry operation and return control to 
//                  the matching type FC_LB operation of the same class C if it is not done
//    Used as a container for commands like 'do' or 'loop' or 'if'.
//    Use parameter Class to distinguish between different classes of return points,
//    like do, while, for, ...
//
class iOperation_FC_LB : public iOperation 
{

friend class iScript;
friend class iOperation;

public:

	virtual void Delete();

protected:

	iOperation_FC_LB(iScript *w, const iString &p, const iString &s, int o, EntryPoint q, iValue* i, iValue *c, int l, unsigned char u) : iOperation(w,p,s,o,_Operation_FC_LB,u)
	{ 
		Point = q;
		Gate = _Open;
		Index = i;
		Count = c;
		intIndex = intCount = 0; 
		Done = true; 
		Class = l; 
	}

	virtual iValue* GetValue() const { return Index; } // the value of this command is the index
	virtual void SetValue(iValue* v);
	virtual bool Exec();

	EntryPoint Point;
	GateStatus Gate;
	iValue *Index, *Count;
	int intIndex, intCount;
	bool Done;
	int Class;
};


//***************************************************************
//
//	Scalar Function operations
//
//***************************************************************
//
//  Operation type SF_V0: 
//    Parameter: void (*F)(void)
//    Execution: call F()
//
class iOperation_SF_V0 : public iOperation 
{

friend class iScript;
friend class iOperation;

protected:

	iOperation_SF_V0(iScript *w, const iString &p, const iString &s, int o, void (*f)(iScript* ), unsigned char u) : iOperation(w,p,s,o,_Operation_SF_V0,u)
	{ 
		Function = f; 
	}

	virtual bool Exec();

	void (*Function)(iScript* );
};
//
//  Operation type SF_V1: 
//    Parameter: iValue* V, Assignment R, void (*F)(iValue*,Assignment)
//    Execution: call F(V,R)
//
class iOperation_SF_V1 : public iOperation 
{

friend class iScript;
friend class iOperation;

public:

	virtual void Delete();

protected:

	iOperation_SF_V1(iScript *w, const iString &p, const iString &s, int o, iValue::Assignment r, iValue* v, void (*f)(iScript*,iValue*,iValue::Assignment), unsigned char u) : iOperation(w,p,s,o,_Operation_SF_V1,u)
	{ 
		Value = v;
		Relative = r; 
		Function = f; 
	}

	virtual iValue* GetValue() const { return Value; }
	virtual void SetValue(iValue* v);
	virtual bool Exec();

	iValue* Value;
	iValue::Assignment Relative;
	void (*Function)(iScript*,iValue*,iValue::Assignment);
};
//
//  Operation type AF_V1: 
//    Parameter: iValue* I, iValue* V, Assignment R, void (*F)(iValue*,iValue*,Assignment)
//    Execution: call F(I,V,R) using array syntax
//
class iOperation_AF_V1 : public iOperation 
{

friend class iScript;
friend class iOperation;

public:

	virtual void Delete();

protected:

	iOperation_AF_V1(iScript *w, const iString &p, const iString &s, int o, iValue* i, iValue::Assignment r, iValue* v, void (*f)(iScript*,iValue*,iValue*,iValue::Assignment), unsigned char u) : iOperation(w,p,s,o,_Operation_AF_V1,u)
	{ 
		Index = i; 
		Value = v;
		Relative = r; 
		Function = f; 
	}

	virtual iValue* GetValue() const { return Value; }
	virtual void SetValue(iValue* v);
	virtual bool Exec();

	iValue* Index;
	iValue* Value;
	iValue::Assignment Relative;
	void (*Function)(iScript*,iValue*,iValue*,iValue::Assignment);
};


//
//*******************************************************
//
//  iScript class declarations
//
//*******************************************************
//
#include "ipointermacro.h"

class iErrorStatus;
class iExpressionParser;
class iScriptObserver;


class iScript : public vtkObjectBase
{
	
	friend class iValue;
	friend class iOperation; // iOperation must have access to the errorMessage string
	friend class iScriptObserver;

	IPOINTER_AS_USER(ErrorStatus);

public:
	
	//
	//  Compiler stack
	//
	struct StackEntry
	{
		iOperation::OperationType Type;
		int Class;
		int Line;
	};
	//
	//  Aliasing
	//
	struct AliasWord
	{
		iString Word;
		iString Alias;
	};
	
	vtkTypeMacro(iScript,vtkObjectBase);

	void SetText(const iString &s);
	inline const iString& GetText() const { return mText; }
	
	virtual void Run(bool compileOnly = false, bool reset = true, int firstLine = 0, int lastLine = -1);
	virtual void Run(const iString &s, bool compileOnly = false, bool reset = true, int firstLine = 0, int lastLine = -1);

	inline int GetNumberOfLines() const { return mNumLines; }
	inline int GetCurrentLine() const { return mCurLine; }

	virtual bool IsDummyWord(const iString &s) const;
	virtual bool IsPrefixWord(const iString &s) const;
	virtual bool IsCommandWord(const iString &s) const;
	virtual bool IsParameterWord(const iString &s) const;
	virtual bool IsVariableWord(const iString &s) const;
	
	virtual bool IsReservedWord(const iString &s) const;

	inline int GetNumberOfReservedWords() const { return mPrefixWords.Size()+mNumCommandWords+mParameterWords.Size()+mDummyWords.Size(); }
	virtual const iString& GetReservedWord(int i) const;
	virtual int GetUserTypeForReservedWord(int i) const;

	virtual bool AreThereOpenEntryPoints() const { return mCompilerStackEntryPoint>=0; }

	inline int GetNumberOfVariables() const { return mVariable.Size(); }
	inline iValue* GetVariable(int i) const { return mVariable[i]; }
	inline int GetLastChangedVariable() const { return mLastChangedVariable; }
	virtual void SetVariableChanged(iValue *v);

	inline int GetErrorLocation() const { return mErrorLocation; }

	//
	//  Internal interrupt
	//
	bool CheckAbort(int cur, int num, int level = 0) const;

protected:

	iScript(iScript *parent = 0);
	virtual ~iScript();

	//
	//  Main conversion tool 
	//
	virtual iValue* TransformToValue(const iString &s);
	virtual iValue* TransformExpressionToValue(const iString &s, bool acceptArrays = true, bool forceReal = false);
	virtual iValue* TransformSingleTokenToValue(const iString &s, bool acceptArrays = true, bool forceReal = false);
	virtual iValue* TransformUnknownTokenToValue(const iString &s, bool acceptArrays = true, bool forceReal = false);

	//
	//  Operations on variables
	//
	bool AssignVariableValue(const iString &n, iValue::Assignment r, iValue *v, iValue *i = 0);
	inline bool AssignVariableValue(iValue *s, iValue::Assignment r, iValue *v, iValue *i = 0){ return this->AssignVariableValue(s->Name,r,v,i); }

	//
	//  Line-by-line driver components
	//
	void RunStart(bool reset);
	bool RunOneLine(bool compileOnly);
	void RunStop(bool compileOnly);

	//
	//  Script execution
	//
	void Reset(bool complete);
	void CompileOneLine();
	void ExecuteOneLine();

	//
	//  Other helpers
	//
	void SetErrorLocation(int loc, int add = 0);
	void SkipEmptyLines();
	iString GetText(int) const;
	int FindScriptWord(const iString &context, const iString &word, int index = 0) const;
	static bool IsScriptWordLetter(char c);
	static bool IsScriptWordFirstLetter(char c);

	//
	//  Script creation
	//
	void CreateAliasWord(const iString&, const iString&);
	void CreateDummyWord(const iString&);
	void CreatePrefixWord(const iString&);
	void CreateCommandWord(iOperation *op);
	void CreateParameterWord(iValue *val);

	bool mCaseSensitive, mArraysAreReal;
	bool mAllowChildAccess;

	iString mCommentString, mLEString, mGEString, mEQString, mNEString, mANDString, mORString;
	iString mHeadOfLineContinuationString, mTailOfLineContinuationString, mAppendSeparator;

	static int mCounter;

	iScript *mParent;  // cannot be const

private:

	//
	//  Observers
	//
	void AddObserver(iScriptObserver *obs);
	void RemoveObserver(iScriptObserver *obs);

	bool mActive, mBlock;
	int mActivator, mErrorLocation;

	iArray<AliasWord> mAliasWords;
	iSearchableArray<iString> mPrefixWords;
	iSearchableArray<iString> mDummyWords;
	iArray<iValue*> mParameterWords;

	int mLastChangedVariable;
	iArray<iValue*> mVariable;

	int mNumCommandWords;
	iArray<iOperation*> mCommand;

	int mCodeEntryPoint;
	iArray<iOperation*> mCode;

	iArray<StackEntry> mCompilerStack;
	int mCompilerStackEntryPoint;

	iString mText;
	int mCurLine, mNextLine, mNumLines, mExecuteReturnCode, mCompileReturnCode, mVariableLastSize, mCommandLastSize;
	int mEntryPointIteration, mEntryPointIterationCount;

	iExpressionParser* mParser;

	iSearchableArray<iScriptObserver*> mObservers;
};


inline bool iScript::IsScriptWordLetter(char c)
{
	//
	//  Script words may contain letters, numbers, and '_'.
	//
	return ((c>='0'&&c<='9') || (c>='A'&&c<='Z') || (c>='a'&&c<='z') || c=='_');
}


inline bool iScript::IsScriptWordFirstLetter(char c)
{
	//
	//  Script words must start with a letter or '_'.
	//
	return ((c>='A'&&c<='Z') || (c>='a'&&c<='z') || c=='_');
}


//
//  Inheritable observer class
//
class iScriptObserver
{
	
public:

	void SetScript(iScript *parent);

	void OnScriptStart();
	void OnScriptStop(const iString &error);
	void OnScriptBeginLine(int line, const iString &text);
	void OnScriptEndLine(int line, const iString &text);
	bool OnScriptCheckAbort(int cur, int num, int level);

protected:

	iScriptObserver(iScript *s);
	virtual ~iScriptObserver();

	virtual void OnScriptStartBody() = 0;
	virtual void OnScriptStopBody(const iString &error) = 0;
	virtual void OnScriptBeginLineBody(int line, const iString &text) = 0;
	virtual void OnScriptEndLineBody(int line, const iString &text) = 0;
	virtual bool OnScriptCheckAbortBody(int cur, int num, int level) = 0;

private:

	iScript *mScript;
};

#endif // ISCRIPT_H
