/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggwidgetkeyslider.h"


#include "icontrolmodule.h"
#include "idatastretch.h"
#include "idatasubject.h"
#include "imath.h"
#include "ishell.h"

#include "iggframe.h"

#include "ibgwidgetentrysubject.h"

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;
using namespace iParameter;

//
//  Template
//
#include "iarraytemplate.h"
#include "iggwidgetkeyhandlertemplate.h"
#include "iggwidgetkeyslidertemplate.h"


//
//******************************************
//
//  Helper class slider
//
//******************************************
//
bool iggWidgetSlider::mIsEditableByDefault = false;


iggWidgetSlider::iggWidgetSlider(int /*numdig*/, const iString& /*label*/, ibgWidgetEntrySubject *subject, iggWidget *owner) : iggWidget(owner->GetParent())
{
	mSubjectOwner = false;
	mSubject = subject;
	this->AttachSubject(subject);

	mSubject->SetEditable(mIsEditableByDefault);
	List().AddUnique(this);

	mWasLaidOut = true;
	mNeedsBaloonHelp = false;
}


iggWidgetSlider::~iggWidgetSlider()
{
	List().Remove(this);
}


void iggWidgetSlider::UpdateWidgetBody()
{
}


void iggWidgetSlider::SetAllEditable(bool s)
{
	int i;
	iPointerArray<iggWidgetSlider> &list = List();

	for(i=0; i<list.Size(); i++)
	{
		list[i]->mSubject->SetEditable(s);
	}

	mIsEditableByDefault = s;
}


iPointerArray<iggWidgetSlider>& iggWidgetSlider::List()
{
	static iPointerArray<iggWidgetSlider> list(100);
	return list;
}

//
//******************************************
//
//  Int slider
//
//******************************************
//
iggWidgetKeyIntSlider::iggWidgetKeyIntSlider(int min, int max, int numdig, const iString &label, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeySlider<int>(numdig,label,key,rm,parent,index,indkey)
{
	int smin, smax;

	this->ConvertToSliderInt(min,smin);
	this->ConvertToSliderInt(max,smax);

	if(smin < smax)
	{
		this->SetRange(smin,smax);
	}
	else IERROR_LOW("Invalid slider range.");
}


void iggWidgetKeyIntSlider::ConvertToSliderInt(int val, int &sint) const
{
	sint = val;
}


void iggWidgetKeyIntSlider::ConvertFromSliderInt(int sint, int &val) const
{
	val = sint;
}


//
//******************************************
//
//  LargeInt slider
//
//******************************************
//
iggWidgetKeyLargeIntSlider::iggWidgetKeyLargeIntSlider(int min, int max, int numdig, const iString &label, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyIntSlider(min,max,numdig,label,key,rm,parent,index,indkey)
{
}


void iggWidgetKeyLargeIntSlider::ConvertToSliderInt(int val, int &sint) const
{
	const int ndex = 10 + 15 + 10;
	const int ddex[] = { 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 22, 24, 26, 28, 30, 32, 34, 36, 38, 40, 42, 44, 46, 48, 50, 55, 60, 65, 70, 75, 80, 85, 90, 95 };

	if(val < 10)
	{
		sint = val;
		return;
	}
	int i, dex = int(log10((double)val));
	sint = 10 + ndex*(dex-1);
	val /= round(pow10(double(dex-1)));
	for(i=0; i<ndex && ddex[i]<val; i++);
	sint += i;
}


void iggWidgetKeyLargeIntSlider::ConvertFromSliderInt(int sint, int &val) const
{
	const int ndex = 10 + 15 + 10;
	const int ddex[] = { 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 22, 24, 26, 28, 30, 32, 34, 36, 38, 40, 42, 44, 46, 48, 50, 55, 60, 65, 70, 75, 80, 85, 90, 95 };

	if(sint < 10)
	{
		val = sint;
		return;
	}
	sint -= 10;
	int dex = sint/ndex;
	sint -= ndex*dex;
	val = round(pow10(double(dex))*ddex[sint]);

#ifdef I_CHECK2
	int sint2;
	this->ConvertToSliderInt(val,sint2);
	if(sint+10+ndex*dex != sint2)
	{
		this->ConvertToSliderInt(val,sint2);
		IERROR_FATAL("Bug in iggWidgetKeyLargeIntSlider::ConvertFromSliderIn");
	}
#endif
}


//
//******************************************
//
//  template Number slider
//
//******************************************
//
template<class T>
iggWidgetKeyGenericNumberSlider<T>::iggWidgetKeyGenericNumberSlider(T min, T max, int res, int stretch, T tiny, int numdig, const iString &label, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeySlider<T>(numdig,label,key,rm,parent,index,indkey), mTiny(tiny)
{
	this->mResolution = res;
	if(this->mResolution < 1) this->mResolution = 1;
	this->mStretch = stretch;
	this->mMin = min;
	this->mMax = max;

	this->SetRange(0,mResolution);
}


template<class T>
void iggWidgetKeyGenericNumberSlider<T>::ConvertToSliderInt(T val, int &sint) const
{
	val = iDataStretch::ApplyStretch(val,this->mStretch,false);
	T min = iDataStretch::ApplyStretch(this->mMin,this->mStretch,false);
	T max = iDataStretch::ApplyStretch(this->mMax,this->mStretch,true);
	T stp = (max-min)/this->mResolution;
	if(fabs(stp) < this->mTiny) stp = 1.0;

	if(val < min) val = min;
	if(val > max) val = max;

	sint = round((val-min)/stp);
	if(sint < 0) sint = 0;
	if(sint > this->mResolution) sint = this->mResolution; 
}


template<class T>
void iggWidgetKeyGenericNumberSlider<T>::ConvertFromSliderInt(int sint, T &val) const
{
	T min = iDataStretch::ApplyStretch(this->mMin,this->mStretch,false);
	T max = iDataStretch::ApplyStretch(this->mMax,this->mStretch,true);

	if(sint < 0) sint = 0;
	if(sint > this->mResolution) sint = this->mResolution; 

	val = min + (max-min)*sint/this->mResolution;
	val = iDataStretch::ResetStretch(val,this->mStretch);
}


template<class T>
void iggWidgetKeyGenericNumberSlider<T>::AdjustLowerLimitBody(bool shrink)
{
	if(this->mStretch > 0)
	{
		//
		//  Only adjust log-scaled slider
		//
		T min = iDataStretch::ApplyStretch(this->mMin,this->mStretch,false);
		T max = iDataStretch::ApplyStretch(this->mMax,this->mStretch,true);
		T step = (max-min)/this->mResolution;
		if(shrink)
		{
			this->mResolution /= 2;
			if(this->mResolution < 3) this->mResolution = 3;
		}
		else
		{
			if(this->mResolution < 1000) this->mResolution *= 2;
		}
		min = max - step*this->mResolution;
		this->mMin = iDataStretch::ResetStretch(min,this->mStretch);
		this->SetRange(0,this->mResolution);
	}
}


//
//******************************************
//
//  Float slider
//
//******************************************
//
iggWidgetKeyFloatSlider::iggWidgetKeyFloatSlider(float min, float max, int res, int stretch, int numdig, const iString &label, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyGenericNumberSlider<float>(min,max,res,stretch,iMath::_FloatMin,numdig,label,key,rm,parent,index,indkey)
{
}


void iggWidgetKeyFloatSlider::AdjustLowerLimit(bool shrink)
{
	this->AdjustLowerLimitBody(shrink);
}


//
//******************************************
//
//  Double slider
//
//******************************************
//
iggWidgetKeyDoubleSlider::iggWidgetKeyDoubleSlider(double min, double max, int res, int stretch, int numdig, const iString &label, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyGenericNumberSlider<double>(min,max,res,stretch,iMath::_DoubleMin,numdig,label,key,rm,parent,index,indkey)
{
}


//
//******************************************
//
//  Relative float slider
//
//******************************************
//
iggWidgetKeyRelativeFloatSlider::iggWidgetKeyRelativeFloatSlider(const iObjectKey &relKey, float min, float max, int res, int stretch, int numdig, const iString &label, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyGenericNumberSlider<float>(min,max,res,stretch,iMath::_FloatMin,numdig,label,key,rm,parent,index,indkey), mRelativeKey(relKey)
{
	if(this->mRelativeKey.Dimension() > 0)
	{
		this->mRelativeValue = new float[this->mRelativeKey.Dimension()]; 
	}
	else if(this->mIndex >= 0)
	{
		this->mRelativeValue = new float[this->mIndex+1];
	}
	else this->mRelativeValue = 0;
}


iggWidgetKeyRelativeFloatSlider::~iggWidgetKeyRelativeFloatSlider()
{
	if(this->mRelativeValue != 0) delete [] this->mRelativeValue;
}


void iggWidgetKeyRelativeFloatSlider::ConvertToSliderInt(float val, int &sint) const
{
	bool ok;

	const iObjectKey &relativeKey = this->Translate(this->mRelativeKey);

	if(this->mIndex >= 0)
	{
		ok = this->GetShell()->GetControlModule()->QueryValue(relativeKey,this->mIndex,this->mRelativeValue[this->mIndex],this->mRelativeKey.Dimension());
	}
	else 
	{
		ok = this->GetShell()->GetControlModule()->QueryValue(relativeKey,this->mRelativeValue,this->mRelativeKey.Dimension());
	}

	if(ok && mRelativeValue[(mIndex>0)?mIndex:0]>0.0)
	{
		val /= mRelativeValue[(mIndex>0)?mIndex:0];
	}
	iggWidgetKeyGenericNumberSlider<float>::ConvertToSliderInt(val,sint);
}


void iggWidgetKeyRelativeFloatSlider::ConvertFromSliderInt(int sint, float &val) const
{
	bool ok;

	if(this->mIndex >= 0)
	{
		ok = this->GetShell()->GetControlModule()->QueryValue(this->mRelativeKey,this->mIndex,this->mRelativeValue[this->mIndex],this->mRelativeKey.Dimension());
	}
	else 
	{
		ok = this->GetShell()->GetControlModule()->QueryValue(this->mRelativeKey,this->mRelativeValue,this->mRelativeKey.Dimension());
	}

	iggWidgetKeyGenericNumberSlider<float>::ConvertFromSliderInt(sint,val);
    if(ok && mRelativeValue[(mIndex>0)?mIndex:0]>0.0)
	{
		val *= mRelativeValue[(mIndex>0)?mIndex:0];
	}
}


//
//******************************************
//
//  Position slider
//
//******************************************
//
iggWidgetKeyPositionSlider::iggWidgetKeyPositionSlider(int numdig, const iString &label, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeySlider<double>(numdig,label,key,rm,parent,index,indkey)
{
	mResolution = 100;

	this->SetRange(-mResolution,mResolution);
	mSubject->SetEditable(true);

	iggWidgetKeyHandlerBase::PositionList().AddUnique(this);
}


iggWidgetKeyPositionSlider::~iggWidgetKeyPositionSlider()
{
	iggWidgetKeyHandlerBase::PositionList().Remove(this);
}


void iggWidgetKeyPositionSlider::ConvertToSliderInt(double val, int &sint) const
{
	sint = round(mResolution*val);
	if(sint < -mResolution) sint = -mResolution;
	if(sint >  mResolution) sint =  mResolution; 
}


void iggWidgetKeyPositionSlider::ConvertFromSliderInt(int sint, double &val) const
{
	if(sint < -mResolution) sint = -mResolution;
	if(sint >  mResolution) sint =  mResolution; 
	val = (double)sint/mResolution;
}


double iggWidgetKeyPositionSlider::GetDisplayedValue(double val) const
{
	iDistance d(this->GetShell()->GetControlModule()->GetViewModule(),false);
	d = val;
	return d.BoxDistance();
}


double iggWidgetKeyPositionSlider::GetInternalValue(double dv) const
{
	iDistance d(this->GetShell()->GetControlModule()->GetViewModule(),false);
	d.SetBoxDistance(dv);
	return d;
}

//
//******************************************
//
//  Size slider
//
//******************************************
//
iggWidgetKeySizeSlider::iggWidgetKeySizeSlider(double min, int numdig, const iString &label, const iObjectKey &key, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyGenericNumberSlider<double>((min<iMath::_DoubleMin)?iMath::_DoubleMin:min,1.0,100,(min>iMath::_DoubleMin)?1:0,iMath::_DoubleMin,numdig,label,key,rm,parent,index,indkey)
{
	iggWidgetKeyHandlerBase::PositionList().AddUnique(this);
}


iggWidgetKeySizeSlider::~iggWidgetKeySizeSlider()
{
	iggWidgetKeyHandlerBase::PositionList().Remove(this);
}


double iggWidgetKeySizeSlider::GetDisplayedValue(double val) const
{
	iDistance d(this->GetShell()->GetControlModule()->GetViewModule(),true);
	d = val;
	return d.BoxDistance();
}


double iggWidgetKeySizeSlider::GetInternalValue(double dv) const
{
	iDistance d(this->GetShell()->GetControlModule()->GetViewModule(),true);
	d.SetBoxDistance(dv);
	return d;
}


void iggWidgetKeySizeSlider::AdjustLowerLimit(bool shrink)
{
	this->AdjustLowerLimitBody(shrink);
}


//
//******************************************
//
//  VariableLimits slider
//
//******************************************
//
iggWidgetKeyVariableLimitsSlider::iggWidgetKeyVariableLimitsSlider(const iggDataTypeProvider *provider, int numdig, const iString &label, const iObjectKey &key, const iObjectKey *varkey, const iObjectKey *stretchkey, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyFloatSlider(0.0,1.0,100,0,numdig,label,key,rm,parent,index,indkey), mVarKey(varkey), mStretchKey(stretchkey)
//iggWidgetKeyVariableLimitsSlider::iggWidgetKeyVariableLimitsSlider(const iggDataTypeProvider *provider, int numdig, const iString &label, const iObjectKey &key, const iObjectKey *varkey, const iObjectKey &varmaxkey, const iObjectKey &minkey, const iObjectKey &maxkey, const iObjectKey &stepkey, const iObjectKey *stretchkey, int rm, iggFrame *parent, int index, const iObjectKey *indkey) : iggWidgetKeyFloatSlider(0.0,1.0,100,false,numdig,label,key,rm,parent,index,indkey), mVarMaxKey(varmaxkey), mVarKey(varkey), mMinKey(minkey), mMaxKey(maxkey), mStepKey(stepkey), mStretchKey(stretchkey)
{
	mSelfChecking = true;
	iggWidgetKeyHandlerBase::VariableLimitsList().AddUnique(this);

	mStep = 0.0f;

	this->SetExecuteFlags(_ObjectOptionOne|_ModuleOptionOne|_RenderOptionClones);

	this->AttachDataTypeProvider(provider);
}


iggWidgetKeyVariableLimitsSlider::~iggWidgetKeyVariableLimitsSlider()
{
	iggWidgetKeyHandlerBase::VariableLimitsList().Remove(this);
}


void iggWidgetKeyVariableLimitsSlider::UpdateWidgetBody()
{
	float min, max;
	int var, num;

	//
	//  Adjust to the parent page if it is set
	//
	const iObjectKey &minKey = this->Translate(iDataSubject::KeyMin());
	const iObjectKey &maxKey = this->Translate(iDataSubject::KeyMax());
	const iObjectKey &numKey = this->Translate(iDataSubject::KeyNumVars());
	const iObjectKey &stretchKey = this->Translate(iDataSubject::KeyStretch());

	iControlModule *cm = this->GetShell()->GetControlModule();

	if(!cm->QueryValue(numKey,num) || num<0) return;

	if(mVarKey != 0)
	{
		if(!cm->QueryValue(*mVarKey,var)) return;
	}
	else
	{
		if(this->mIndexKey != 0)
		{
			//
			//  Determine the index
			//
		    if(!this->GetShell()->GetControlModule()->QueryValue(*this->mIndexKey,this->mIndex))
			{
				return;
			}
		}
		var = mIndex;
	}

	if(var<0 || var>=num)
	{
		//
		//  Inactive value
		//
		this->Enable(false);
		return;
	}
	else this->Enable(true);

	if(mStretchKey != 0)
	{
		if(!cm->QueryValue(*mStretchKey,mStretch)) return;
	}
	else
	{
		if(!cm->QueryValue(stretchKey,var,mStretch,num,-1,-1)) return;
	}

	if(cm->QueryValue(minKey,var,min,num) && cm->QueryValue(maxKey,var,max,num))
	{
		mMin = min;
		mMax = max;
		mResolution = this->ComputeResolution();
		this->SetRange(0,mResolution);
		this->iggWidgetKeyFloatSlider::UpdateWidgetBody();
	}
}


int iggWidgetKeyVariableLimitsSlider::ComputeResolution()
{
	float step;
	float min = iDataStretch::ApplyStretch(mMin,mStretch,false);
	float max = iDataStretch::ApplyStretch(mMax,mStretch,true);

	if(mStep > 1.0e-36f)
	{
		step = mStep;
	}
	else
	{
		switch(mStretch)
		{
		case 1:
			{
				if(max > min+1.0f) step = 0.1f; else if(max > min+0.1f) step = 0.01f; else step = 0.01f*(max-min);
				break;
			}
		default:
			{
				step = 0.01f*(max-min);
				break;
			}
		}
		if(step < 1.0e-36f) step = 1.0e-36f;
	}

	int res = round((max-min)/step);
	if(res < 1) res = 1;
	return res;
}


void iggWidgetKeyVariableLimitsSlider::SetStep(float step)
{
	mStep = step;
}

