/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "isurfaceviewsubject.h"


#include "icommondatadistributors.h"
#include "icolorbars.h"
#include "idatahelper.h"
#include "idatalimits.h"
#include "ierror.h"
#include "imarker.h"
#include "imarkerfamily.h"
#include "ipicker.h"
#include "ireplicatedactor.h"
#include "isurfacepipeline.h"
#include "iviewmodule.h"
#include "iviewsubjectobserver.h"
#include "iviewsubjectparallelpipeline.h"

#include <vtkImageData.h>
#include <vtkPolyData.h>

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWSUBJECT_DEFINE_TYPE(iSurfaceViewSubject,Surface,s);

IOBJECT_DEFINE_KEY(iSurfaceViewSubject,IsoSurfaceMethod,im,Int,1);
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,IsoSurfaceLevel,l,Float,1);
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,Method,m,Int,1);
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,NormalsFlipped,nf,Bool,1);
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,IsoSurfaceOptimization,op,Bool,1);
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,PlaneDirection,pd,Float,3);
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,PaintVar,pv,OffsetInt,2);
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,IsoSurfaceReduction,rd,Int,1);
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,AlternativeIsoSurfaceReductionMethod,rda,Bool,1);
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,IsoSurfaceSmoothing,sm,Int,1);
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,IsoSurfaceVar,v,OffsetInt,1);

IOBJECT_DEFINE_DISTANCE_KEY(iSurfaceViewSubject,Size,s);
//
//  Inherited keys
//
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COMMON(iSurfaceViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_REPLICATING(iSurfaceViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_POSITION(iSurfaceViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COLOR(iSurfaceViewSubject,2);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_OPACITY(iSurfaceViewSubject,1);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_PALETTE(iSurfaceViewSubject,2);

#ifdef IOBJECT_BACKWARD_COMPATIBLE
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,AttachProbeToMarker,-am,OffsetInt,1);  //  for sphere and plane
IOBJECT_DEFINE_KEY(iSurfaceViewSubject,ProbePositionToPickedPoint,-p2p,Bool,1); // action key
#endif


//
// iSurfaceViewSubject class
//
iSurfaceViewSubject::iSurfaceViewSubject(iViewModule *vm, const iDataType &type, const iString &name) : iSolidViewSubject(vm,type,name,2,_ViewSubjectHasPosition|_ViewSubjectSameOpacity), mSize(vm,true)
{
	mObjectType = _ObjectTypeSurface;

	mLev = 0.0f;
	mVar = 0;
#if defined(IVTK_4) && defined(VTK_USE_PATENTED)
	//
	//  vtkMarchingContourFilter has an obscure bug undet VTK 5 and in multi-threaded mode.
	//
	mIsoSurfaceMethod = 1;
#else
	mIsoSurfaceMethod = 0;
#endif

	//
	//  mPaintVar[0,1] = -1 means no painting on the exterior,interior.
	//
	mPaintVarExt = -1;
	mPaintVarInt = -1;

	//
	//  Pay tribute to IFrITness
	//
	int nvar = this->GetLimits()->GetNumVars();
	float lmin = this->GetLimits()->GetMin(this->GetIsoSurfaceVar());
	float lmax = this->GetLimits()->GetMax(this->GetIsoSurfaceVar());
	if(this->GetIsoSurfaceVar()==0 && (this->GetLimits()->GetName(0)=="Neutral fraction" || this->GetLimits()->GetName(0)=="HI fraction"))
	{
		mLev = 0.5; 
		mLev = (mLev < lmin) ? lmin : mLev;
		mLev = (mLev > lmax) ? lmax : mLev;
		if(nvar > 1)
		{
			mPaintVarExt = 1;
			mActors[0]->SyncWithLimits(this->GetLimits(),1);
			mActors[0]->SetScalarVisibility(true);
		}
	} 
	else mLev = lmax; 
	
	mReduction = 1;
	mSmoothing = 0;
	mNormalsFlipped = false;
	mOptimization = false;
	mAlternativeReductionMethod = false;

	mShading = true;
	this->UpdateMaterialProperties();
	mMethod = _SurfaceMethodIsoSurface;
	mSize = 0.5;
	mDir[0] = 0.0;
	mDir[1] = 0.0;
	mDir[2] = 1.0;

	mAttachedMarkerIndex = -1;
	mAttachedMarker = 0;

	//
	//  Add observer to keep information about this object
	//
	iViewSubjectObserver::IsoSurfaceData d;
	d.Var = mVar;
	d.Level = mLev;
	mObjectObserver->SetIsoSurfaceData(d);
}


iSurfaceViewSubject::~iSurfaceViewSubject()
{
}


void iSurfaceViewSubject::ConfigureBody()
{
	//
	//  Create pipeline (must be created after the object is fully created)
	//	
	this->AddMainPipeline(1);
	mActors[0]->SetInput(this->Pipeline()->GetOutput(0));
	mActors[1]->SetInput(this->Pipeline()->GetOutput(1));
}


void iSurfaceViewSubject::ResetPipeline()
{
}


void iSurfaceViewSubject::ShowColorBarsBody(bool show)
{
	if(this->GetPaintVar(0) >= 0) this->GetViewModule()->GetColorBars()->ShowBar(_ColorBarsPrioritySurfaceExterior,this->GetPaintVar(0),this->GetDataType(),mPalettes[0],show);
	if(this->GetPaintVar(1) >= 0) this->GetViewModule()->GetColorBars()->ShowBar(_ColorBarsPrioritySurfaceInterior,this->GetPaintVar(1),this->GetDataType(),mPalettes[1],show);
}


void iSurfaceViewSubject::SetIsoSurfaceMethod(int m)
{
	if(m>=0 && m<=1)
	{
		mIsoSurfaceMethod = m;
		this->Pipeline()->UpdateContents(iSurfacePipeline::_IsoSurfaceMethod);
		this->ClearCache();
	}
}


void iSurfaceViewSubject::SetIsoSurfaceVar(int v)
{ 
	if(v>=0 && v<this->GetLimits()->GetNumVars())
	{
		mVar = v;
		this->Pipeline()->UpdateContents(iSurfacePipeline::_IsoSurfaceVar);
		this->SetIsoSurfaceLevel(this->GetLimits()->GetMax(this->GetIsoSurfaceVar()));
		this->SetNormalsFlipped(mNormalsFlipped);
		this->SyncWithData(this->Request(mVar));
		this->ClearCache();
		//
		//  Update observer
		//
		iViewSubjectObserver::IsoSurfaceData d;
		d.Var = mVar;
		d.Level = mLev;
		mObjectObserver->SetIsoSurfaceData(d);
	}
}


void iSurfaceViewSubject::SetIsoSurfaceLevel(float l)
{
	if(this->GetIsoSurfaceVar() < this->GetLimits()->GetNumVars())
	{
		float lmin = this->GetLimits()->GetMin(this->GetIsoSurfaceVar());
		float lmax = this->GetLimits()->GetMax(this->GetIsoSurfaceVar());
		l = (l < lmin) ? lmin : l;
		l = (l > lmax) ? lmax : l;
		mLev = l;
		this->Pipeline()->UpdateContents(iSurfacePipeline::_IsoSurfaceLevel);
		this->ClearCache();
		//
		//  Update observer
		//
		iViewSubjectObserver::IsoSurfaceData d;
		d.Var = mVar;
		d.Level = mLev;
		mObjectObserver->SetIsoSurfaceData(d);
	}
}


void iSurfaceViewSubject::SetPaintVar(int s, int v)
{
	if(s<0 || s>1) return;

	if(v<-1 || v>=this->GetLimits()->GetNumVars()) v = -1;

	this->ShowColorBars(false);
	if(s == 0) mPaintVarExt = v; else mPaintVarInt = v;
	this->ShowColorBars(true);
	this->Pipeline()->UpdateContents(iSurfacePipeline::_VarPaint,s);
	if(s == 0)
	{
		if(v>=0 && v<this->GetLimits()->GetNumVars()) 
		{
			mActors[0]->SyncWithLimits(this->GetLimits(),v);
			mActors[0]->SetScalarVisibility(true);
			mActors[0]->ColorByArrayComponent(0,v);
		} 
		else 
		{
			mActors[0]->SetScalarVisibility(false);
		}
	} 
	else 
	{
		if(v>=0 && v<this->GetLimits()->GetNumVars()) 
		{
			mActors[1]->SyncWithLimits(this->GetLimits(),v);
			mActors[1]->SetScalarVisibility(true);
			mActors[1]->ColorByArrayComponent(0,v);
			mActors[1]->VisibilityOn();
		} 
		else 
		{
			mActors[1]->SetScalarVisibility(false);
			mActors[1]->VisibilityOff();
		}
	}
	this->ClearCache();
}


void iSurfaceViewSubject::SetIsoSurfaceReduction(int s)
{
	mReduction = s; 
	this->Pipeline()->UpdateContents(iSurfacePipeline::_IsoSurfacePipeline);
	this->ClearCache();
} 


void iSurfaceViewSubject::SetAlternativeIsoSurfaceReductionMethod(bool s)
{
	mAlternativeReductionMethod = s; 
	this->Pipeline()->UpdateContents(iSurfacePipeline::_IsoSurfacePipeline);
	this->ClearCache();
} 


void iSurfaceViewSubject::SetIsoSurfaceOptimization(bool s)
{
	mOptimization = s; 
	this->Pipeline()->UpdateContents(iSurfacePipeline::_IsoSurfacePipeline);
	this->ClearCache();
} 


void iSurfaceViewSubject::SetIsoSurfaceSmoothing(int p)
{
	mSmoothing = p;
	this->Pipeline()->UpdateContents(iSurfacePipeline::_IsoSurfacePipeline);
	this->ClearCache();
}


void iSurfaceViewSubject::SetNormalsFlipped(bool s)
{ 
	mNormalsFlipped = s;
	this->Pipeline()->UpdateContents(iSurfacePipeline::_NormalsFlipped);
	this->ClearCache();
}


void iSurfaceViewSubject::ShowBody(bool show)
{
	if(show)
	{
		if(this->GetPaintVar(0) >= 0) 
		{
			mActors[0]->SyncWithLimits(this->GetLimits(),this->GetPaintVar(0));
		}
		if(this->GetPaintVar(1) >= 0) 
		{
			mActors[0]->SyncWithLimits(this->GetLimits(),this->GetPaintVar(1));
			mActors[1]->VisibilityOn();
		}
		else mActors[1]->VisibilityOff();
		mActors[0]->VisibilityOn();
		this->ShowColorBars(true);
	} 
	else 
	{
		this->ShowColorBars(false);
		mActors[0]->VisibilityOff();
		mActors[1]->VisibilityOff();
	}
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iSurfaceViewSubject::SolidViewSubjectPackStateBody(iString &s) const
{
	int iv[2];

	this->PackValue(s,KeyNormalsFlipped(),mNormalsFlipped);
	this->PackValue(s,KeyIsoSurfaceReduction(),mReduction);
	this->PackValue(s,KeyAlternativeIsoSurfaceReductionMethod(),mAlternativeReductionMethod);
	this->PackValue(s,KeyIsoSurfaceOptimization(),mOptimization);

	this->PackValue(s,KeyIsoSurfaceMethod(),mIsoSurfaceMethod);
	this->PackValue(s,KeyIsoSurfaceVar(),this->GetIsoSurfaceVar());
	iv[0] = this->GetPaintVar(0);
	iv[1] = this->GetPaintVar(1);
	this->PackValue(s,KeyPaintVar(),iv,2);
	this->PackValue(s,KeyIsoSurfaceSmoothing(),mSmoothing);

	this->PackValue(s,KeyIsoSurfaceLevel(),this->GetIsoSurfaceLevel());

	this->PackValue(s,KeyMethod(),mMethod);
	this->PackValue(s,KeyPlaneDirection(),mDir,3);

	this->PackValueDistance(s,KeySize(),mSize);
}


void iSurfaceViewSubject::SolidViewSubjectUnPackStateBody(const iString &s)
{
	int i, iv[2]; bool b; float f; float fv[3];

	if(this->UnPackValue(s,KeyNormalsFlipped(),b)) this->SetNormalsFlipped(b);
	if(this->UnPackValue(s,KeyIsoSurfaceReduction(),i)) this->SetIsoSurfaceReduction(i);
	if(this->UnPackValue(s,KeyAlternativeIsoSurfaceReductionMethod(),b)) this->SetAlternativeIsoSurfaceReductionMethod(b);
	if(this->UnPackValue(s,KeyIsoSurfaceOptimization(),b)) this->SetIsoSurfaceOptimization(b);

	if(this->UnPackValue(s,KeyIsoSurfaceMethod(),i)) this->SetIsoSurfaceMethod(i);
	if(this->UnPackValue(s,KeyIsoSurfaceVar(),i)) this->SetIsoSurfaceVar(i);
	
	iv[0] = this->GetPaintVar(0);
	iv[1] = this->GetPaintVar(1);
	if(this->UnPackValue(s,KeyPaintVar(),iv,2))
	{
		this->SetPaintVar(0,iv[0]);
		this->SetPaintVar(1,iv[1]);
	}

	if(this->UnPackValue(s,KeyIsoSurfaceSmoothing(),i)) this->SetIsoSurfaceSmoothing(i);
	if(this->UnPackValue(s,KeyMethod(),i)) this->SetMethod(i);

	if(this->UnPackValue(s,KeyIsoSurfaceLevel(),f)) this->SetIsoSurfaceLevel(f);

	for(i=0; i<3; i++) fv[i] = mDir[i];
	if(this->UnPackValue(s,KeyPlaneDirection(),fv,3)) this->SetPlaneDirection(fv);

	if(this->UnPackValueDistance(s,KeySize(),mSize)) this->SetSize(mSize);

#ifdef IOBJECT_BACKWARD_COMPATIBLE
	iObject::ReportMissingKeys(false);
	if(this->UnPackValue(s,KeyAttachProbeToMarker(),i)) this->AttachToMarker(i);
	if(this->UnPackValue(s,KeyProbePositionToPickedPoint(),b) && b) this->SetPosition(_LocationPickedPoint); 
	iObject::ReportMissingKeys(true);
#endif
}


void iSurfaceViewSubject::SetMethod(int m)
{
	if(m>=0 && m<3)
	{
		mMethod = m;
		this->SetShading(mMethod != _SurfaceMethodPlane);
		this->Pipeline()->UpdateContents(iSurfacePipeline::_Method);
		this->ClearCache();
	}
}


void iSurfaceViewSubject::SetSize(const iDistance &s)
{
	if(s < 1.0)
	{
		mSize = s;
		this->Pipeline()->UpdateContents(iSurfacePipeline::_FixedSurfaceSize);
		this->ClearCache();
	}
}


void iSurfaceViewSubject::UpdatePosition(const iPosition &)
{
	mPosition.PeriodicWrap();
	this->Pipeline()->UpdateContents(iSurfacePipeline::_FixedSurfacePosition);
}


void iSurfaceViewSubject::SetPlaneDirection(const float n[3])
{
	int i;
	for(i=0; i<3; i++) mDir[i] = n[i];
	this->Pipeline()->UpdateContents(iSurfacePipeline::_PlaneDirection);
	this->ClearCache();
}


bool iSurfaceViewSubject::CanBeShown() const
{
	iDataHelper *h = this->CreateDataHelper(this->GetData());
	bool ret = (h->IsThereScalarData() && this->GetIsoSurfaceVar()<this->GetLimits()->GetNumVars());
	delete h;
	return ret;
}


void iSurfaceViewSubject::ViewSubjectSyncWithData(const iDataSyncRequest &r)
{
	int i;

	this->Pipeline()->SetNthInput(0,this->GetData());

	if(r.Var()==-1 || r.Var()==this->GetIsoSurfaceVar()) this->SetIsoSurfaceLevel(this->GetIsoSurfaceLevel());

	for(i=0; i<2; i++)
	{
		if(r.Var()==-1 || r.Var()==this->GetPaintVar(i)) this->SetPaintVar(i,this->GetPaintVar(i));
	}
}


iViewSubjectPipeline* iSurfaceViewSubject::CreatePipeline(int)
{
	return new iSurfacePipeline(this);
}


void iSurfaceViewSubject::ConfigureMainPipeline(iViewSubjectPipeline *p, int)
{
	iViewSubjectParallelPipeline *pp = iRequiredCast<iViewSubjectParallelPipeline>(INFO,p);

	iImageDataDistributor *idd = new iImageDataDistributor(pp->GetDataManager());
	pp->GetDataManager()->AddDistributor(idd);
	pp->GetDataManager()->AddCollector(new iPolyDataCollector(pp->GetDataManager(),idd));
}


