/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggmainwindow.h"


#include "ianimator.h"
#include "ibasicdatasubjects.h"
#include "icontrolmodule.h"
#include "icrosssectionviewsubject.h"
#include "idata.h"
#include "idataexplorer.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "iedition.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "ifileloader.h"
#include "ihelpfactory.h"
#include "iimagefactory.h"
#include "iparallelmanager.h"
#include "iparticlesviewsubject.h"
#include "ishell.h"
#include "isurfaceviewsubject.h"
#include "isystem.h"
#include "itensorfieldviewsubject.h"
#include "itextactor.h"
#include "ivectorfieldviewsubject.h"
#include "iversion.h"
#include "iviewmodule.h"
#include "ivolumeviewsubject.h"

#include "iggdialoganimatingprogress.h"
#include "iggdialogauto.h"
#include "iggdialogcommandline.h"
#include "iggdialogdataexplorer.h"
#include "iggdialogeventrecorder.h"
#include "iggdialogfilesetexplorer.h"
#include "iggdialoghelp.h"
#include "iggdialogimagecomposer.h"
#include "iggdialogloadfile.h"
#include "iggdialogpaletteeditor.h"
#include "iggdialogparallelcontroller.h"
#include "iggdialogperformancemeter.h"
#include "iggdialogpickerwindow.h"
#include "iggdialogrenderingprogress.h"
#include "iggdialogscriptdebugger.h"
#include "iggextensionwindow.h"
#include "iggframetopparent.h"
#include "iggrenderwindow.h"
#include "iggpagecrosssection.h"
#include "iggpagedata.h"
#include "iggpageparticles.h"
#include "iggpagesurface.h"
#include "iggpagetensorfield.h"
#include "iggpagevectorfield.h"
#include "iggpageview.h"
#include "iggpagevolume.h"
#include "iggshell.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeyhandler.h"
#include "iggwidgetkeylineedit.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetmisc.h"
#include "iggwidgetotherbutton.h"
#include "iggwidgetprogressbar.h"
#include "iggwidgetrendermodebutton.h"
#include "iggwidgettext.h"

#include "ibgdialogsubject.h"
#include "ibgextensionwindowsubject.h"
#include "ibgframesubject.h"
#include "ibgmainwindowsubject.h"
#include "ibgmenuwindowsubject.h"
#include "ibgrenderwindowsubject.h"
#include "ibgwidgetareasubject.h"
#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetselectionboxsubject.h"
#include "ibgwindowhelper.h"

#include <vtkPointData.h>
#include <vtkPolyData.h>
#include <vtkRenderWindowCollection.h>
#include <vtkStructuredPoints.h>
#include <vtkVersion.h>

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;
using namespace iParameter;

//
//  templates
//
#include "iarraytemplate.h"


IOBJECT_DEFINE_TYPE(iggMainWindow,MainWindow,-mw,iObjectType::_Helper);  // helper type

IOBJECT_DEFINE_KEY(iggMainWindow,AttachWindows,aw,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,Docked,d,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,ExtensionGeometry,eg,Int,4);
IOBJECT_DEFINE_KEY(iggMainWindow,WindowUnderFocusCurrent,fc,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,Geometry,g,Int,4);
IOBJECT_DEFINE_KEY(iggMainWindow,InteractorHelp,ih,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,IsIdiosyncratic,ii,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,OptionsAreGlobal,og,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,AllowPrePopulateToolBar,pp,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,SlidersAreEditable,se,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,Theme,t,String,1);
IOBJECT_DEFINE_KEY(iggMainWindow,TabMode,tm,Int,1);
IOBJECT_DEFINE_KEY(iggMainWindow,ToolTips,tt,Bool,1);


namespace iggMainWindow_Private
{
	//
	//  Menu ids
	//
	const int _MenuFileOpenUniformScalars = 1;
	const int _MenuFileOpenBasicParticles = 2;
	const int _MenuFileOpenUniformVectors = 3;
	const int _MenuFileOpenUniformTensors = 4;
	const int _MenuFileSaveState = 90;
	const int _MenuFileSaveStateAs = 91;
	const int _MenuFileLoadStateFrom = 92;
	const int _MenuFileExit = 99;
	const int _MenuFileMax = 100;

	const int _MenuDialogScriptDebugger = 101;
	const int _MenuDialogPaletteEditor = 102;
	const int _MenuDialogPickerWindow = 103;
	const int _MenuDialogEventRecorder = 104;
	const int _MenuDialogParallelController = 105;
	const int _MenuDialogFileSetExplorer = 106;
	const int _MenuDialogDataExplorer = 107;
	const int _MenuDialogImageComposer = 108;
	const int _MenuDialogCommandLine = 109;
	const int _MenuDialogPerformanceMeter = 110;
	const int _MenuDialogAutoBlock = 120;
	const int _MenuDialogAutoBegin = 121;
	const int _MenuDialogAutoEnd = 200;
	const int _MenuDialogMax = 200;

	const int _MenuStyleDockWindow = 201;
	const int _MenuStyleInteractorHelp = 202;
	const int _MenuStyleToolTips = 203;
	const int _MenuStyleTabText = 211;
	const int _MenuStyleTabIcon = 212;
	const int _MenuStyleTabBoth = 213;
	const int _MenuStyleRenderImmediate = 221;
	const int _MenuStyleRenderDelayed = 222;
	const int _MenuStyleRenderResetAll = 223;
	const int _MenuStyleSlidersEditable = 224;
	const int _MenuStyleIsIdiosyncratic = 225;
	const int _MenuStyleAttachWindows = 226;
	const int _MenuStyleDetailLoadedDataInfo = 227;
	const int _MenuStyleAutoRender = 228;
	const int _MenuStyleTheme = 230;
	const int _MenuStyleMax = 300;

	const int _MenuOptionAntialiasing = 301;
	const int _MenuOptionIfritBox = 311;
	const int _MenuOptionClassicBox = 312;
	const int _MenuOptionHairBox = 313;
	const int _MenuOptionAxesBox = 314;
	const int _MenuOptionFontTypeVector = 322;
	const int _MenuOptionFontTypeArial = 323;
	const int _MenuOptionFontTypeCourier = 324;
	const int _MenuOptionFontTypeTimes = 325;
	const int _MenuOptionFontSizem5 = 331;
	const int _MenuOptionFontSizem4 = 332;
	const int _MenuOptionFontSizem3 = 333;
	const int _MenuOptionFontSizem2 = 334;
	const int _MenuOptionFontSizem1 = 335;
	const int _MenuOptionFontSizec0 = 336;
	const int _MenuOptionFontSizep1 = 337;
	const int _MenuOptionFontSizep2 = 338;
	const int _MenuOptionFontSizep3 = 339;
	const int _MenuOptionFontSizep4 = 340;
	const int _MenuOptionFontSizep5 = 341;
	const int _MenuOptionImageFormatPNG = 351;
	const int _MenuOptionImageFormatJPG = 352;
	const int _MenuOptionImageFormatPPM = 353;
	const int _MenuOptionImageFormatBMP = 354;
	const int _MenuOptionImageFormatTIF = 355;
	const int _MenuOptionImageFormatEPS = 356;
	const int _MenuOptionImageZoom001 = 361;
	const int _MenuOptionImageZoom002 = 362;
	const int _MenuOptionImageZoom003 = 363;
	const int _MenuOptionImageZoom004 = 364;
	const int _MenuOptionImageZoom005 = 365;
	const int _MenuOptionImageZoom006 = 366;
	const int _MenuOptionImageZoom008 = 367;
	const int _MenuOptionImageZoom010 = 368;
	const int _MenuOptionImageZoom015 = 369;
	const int _MenuOptionImageZoom020 = 370;
	const int _MenuOptionImageZoom030 = 371;
	const int _MenuOptionImageZoom040 = 372;
	const int _MenuOptionImageZoom050 = 373;
	const int _MenuOptionImageZoom060 = 374;
	const int _MenuOptionImageZoom080 = 375;
	const int _MenuOptionImageZoom100 = 376;
	const int _MenuOptionPostScriptPaperFormatA0 = 401;
	const int _MenuOptionPostScriptPaperFormatA1 = 402;
	const int _MenuOptionPostScriptPaperFormatA2 = 403;
	const int _MenuOptionPostScriptPaperFormatA3 = 404;
	const int _MenuOptionPostScriptPaperFormatA4 = 405;
	const int _MenuOptionPostScriptPaperFormatA5 = 406;
	const int _MenuOptionPostScriptPaperFormatA6 = 407;
	const int _MenuOptionPostScriptPaperFormatA7 = 408;
	const int _MenuOptionPostScriptPaperFormatA8 = 409;
	const int _MenuOptionPostScriptPaperFormatL1 = 410;
	const int _MenuOptionPostScriptPaperFormatL2 = 411;
	const int _MenuOptionPostScriptOrientationPortrait = 421;
	const int _MenuOptionPostScriptOrientationLandscape = 422;
	const int _MenuOptionSettingsGlobal = 430;
	const int _MenuOptionMax = 500;

	const int _MenuHelpContents = 501;
	const int _MenuHelpAbout = 502;
	const int _MenuHelpMax = 599;

	const int _ToolBarOpenWindowsPage = 801;
	const int _ToolBarMinimizeWindowsPage = 802;
	const int _ToolBarMaximizeWindowsPage = 803;

	const int _ToolBarShowSurface = 901;
	const int _ToolBarShowCrossSection = 902;
	const int _ToolBarShowVolume = 903;
	const int _ToolBarShowParticles = 904;
	const int _ToolBarShowVectorField = 905;
	const int _ToolBarShowTensorField = 906;
	const int _ToolBarMax = 999;

	class InteractorHelpTextView : public iggWidgetTextBrowser
	{

	private:

		struct Entry
		{
			const char* Name;
			const char* Body;
			Entry(const char *n, const char *b) : Name(n), Body(b) {}
		};

	public:

		InteractorHelpTextView(iggFrame *parent) : iggWidgetTextBrowser(true,false,parent)
		{
			mShownData = 0;
		}

		virtual void Show(bool s)
		{
			this->UpdateWidget();
			iggWidgetTextBrowser::Show(s);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			static Entry data0[] = 
			{ 
				Entry("U key","dump the current view into an image."),
					Entry("F key","toggle the Full Screen mode on and off."),
					Entry("","(use Style->Show interactor help menu to turn this help off)")
			};

			static Entry dataD[] = 
			{ 
				Entry(   "Trackball/Joystick mode",""),
					Entry("","-------------------------"),
					Entry("Left button","rotate the camera around its focal point."),
					Entry("[Ctrl]+Left button","spin the camera around the Z-axis (axis perpendicular to the screen)."),
					Entry("Middle button","pan (move sideways) the camera."),
					Entry("Right button","zoom in/out."),
					Entry("3 key","toggle into and out of stereo mode."),
					Entry("P key","perform a pick operation."),
					Entry("R key","reset the camera view along the current view direction."),
					Entry("S key","show all solid objects as surfaces."),
					Entry("W key","show all solid objects as wireframe (faster).")
			};

			static Entry dataF[] = 
			{ 
				Entry(   "Fly-by mode",""),
					Entry("","-------------"),
					Entry("Left button","forward motion."),
					Entry("Right button","reverse motion."),
					Entry("[Shift] key","accelerator in mouse and key modes."),
					Entry("[Ctrl]+[Shift] keys","causes sidestep instead of steering."),
					Entry("+/- keys","increase/deacrease normal speed.")
			};

			static Entry dataK[] = 
			{ 
				Entry(   "Keyboard mode",""),
					Entry("","---------------"),
					Entry("Arrow keys (or HJKL keys)","rotate the camera around its focal point."),
					Entry("[Ctrl]+Arrow keys (or [Ctrl]+HJKL keys)","spin the camera around the Z-axis (axis perpendicular to the screen)."),
					Entry("[Shift]+Arrow keys (or [Shift]+HJKL keys)","pan (move sideways) the camera."),
					Entry("+/- keys","zoom in/out."),
					Entry("A/Z keys","speed up/slow down the interaction."),
					Entry("3 key","toggle into and out of stereo mode."),
					Entry("P key","perform a pick operation."),
					Entry("R key","reset the camera view along the current view direction."),
					Entry("S key","show all solid objects as surfaces."),
					Entry("W key","show all solid objects as wireframe (faster).")
			};

			static Entry dataM[] = 
			{
				Entry(   "Measuring box mode",""),
					Entry("","--------------------"),
					Entry("Left button","rotate the measuring box around its origin."),
					Entry("Middle button","translate the measuring box in/out of screen."),
					Entry("Right button","translate the measuring box along scene axes."),
					Entry("A/Z keys","scale the measuring box down/up."),
					Entry("S/X keys","adjust box opacity."),
					Entry("C key","shift the camera focal point to the box center.")
			};

			static Entry dataC[] = 
			{ 
				Entry(   "Camera path mode",""),
					Entry("","------------------"),
					Entry("Left button","move a handle or the whole path."),
					Entry("Middle button","move the path (if you seem unable to grab the line, change the number of steps slightly)."),
					Entry("[Ctrl]+Middle button","spin the path."),
					Entry("Right button","scale the path.")
			};

			int i, n = 0;
			Entry *data = 0;
			bool b;
			if(this->GetShell()->GetControlModule()->QueryValue(iViewModule::KeyInteractorStyle(),i))
			{
				switch(i)
				{
				case _InteractorStyleTrackball:
				case _InteractorStyleJoystick:
					{
						data = dataD;
						n = sizeof(dataD)/sizeof(Entry);
						break;
					}
				case _InteractorStyleFlight:
					{
						data = dataF;
						n = sizeof(dataF)/sizeof(Entry);
						break;
					}
				case _InteractorStyleKeyboard:
					{
						data = dataK;
						n = sizeof(dataK)/sizeof(Entry);
						break;
					}
				}
			}
			if(this->GetShell()->GetControlModule()->QueryValue(iViewModule::KeyMeasuringBox(),b) && b)
			{
				data = dataM;
				n = sizeof(dataM)/sizeof(Entry);
			}
			if(this->GetShell()->GetControlModule()->QueryValue(iAnimator::KeyStyle(),i) && i==4)
			{
				data = dataC;
				n = sizeof(dataC)/sizeof(Entry);
			}
			if(n == 0)
			{
#ifdef I_CHECK1
				IERROR_LOW("Should not be here");
#endif
			}

			if(data != mShownData)
			{
				this->Clear();
				for(i=0; i<n; i++)
				{
					this->AppendTextLine(data[i].Name,data[i].Body,iColor(0,0,255));
				}
				n = sizeof(data0)/sizeof(Entry);
				for(i=0; i<n; i++)
				{
					this->AppendTextLine(data0[i].Name,data0[i].Body,iColor(0,0,255));
				}
				mShownData = data;
			}
			this->iggWidgetTextBrowser::UpdateWidgetBody();
		}

		Entry* mShownData;
	};


	class BusyIndicator : public iggWidgetImageFlipper
	{

	public:

		BusyIndicator(iggFrame *parent) : iggWidgetImageFlipper(parent)
		{
			mSubject->AddImage(*iImageFactory::FindIcon("light_green.png"),true);
			mSubject->AddImage(*iImageFactory::FindIcon("light_red.png"),true);

			this->SetBaloonHelp("Ready/busy indicator","A green light indicates that IFrIT is ready for user input. The red light comes on during the time IFrIT is busy doing some work. User input is blocked while the red light is on.");
		}

		void SetBusy(bool s)
		{
			if(s) mSubject->ShowImage(1); else mSubject->ShowImage(0);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
		}
	};


	class VisitedFileList : public iggWidget
	{

	public:

		VisitedFileList(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,"File:");

			this->SetBaloonHelp("List of visited files","This drop-down box shows the list of recently visited files. A file from the list can be reloaded by simply selecting it.");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			int i, k;
			const iArray<iDataReader::VisitedFile> &list(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetVisitedFilesList());

			k = 0;
			do
			{
				mSubject->Clear();
				for(i=list.MaxIndex(); i>=0; i--)
				{
					mSubject->InsertItem(iString((k>0)?"...":"")+list[i].Name.Part(k));
				}
				if(k == 0) k += 4; else k++;
			}
			while(!mSubject->DoesContentFit());
		}

		virtual void OnInt1Body(int v)
		{
			const iArray<iDataReader::VisitedFile> &list(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetVisitedFilesList());

			if(v>=0 && v<list.Size())
			{
				this->GetMainWindow()->LoadData(list[list.MaxIndex()-v].Loader->GetDataType(0),list[list.MaxIndex()-v].Name);
			}
		}

		ibgWidgetComboBoxSubject *mSubject;
	};


	class ImageButton : public iggWidgetSimpleButton
	{

	public:

		ImageButton(iggFrame *parent) : iggWidgetSimpleButton("Image",parent)
		{
			this->SetBaloonHelp("Dump an image","Dump the scene in the current visualization window as an image. Image format and zoom factor are set in the Options menu.");
		}

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			this->GetShell()->GetControlModule()->GetViewModule()->DumpImages();
			this->GetMainWindow()->Block(false);
		}
	};


	class RenderButton : public iggWidgetSimpleButton
	{

	public:

		RenderButton(iggFrame *parent) : iggWidgetSimpleButton("Render",parent)
		{
			this->SetBaloonHelp("Render the scene","Render the current scene in the current visualization window.");
		}

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			this->GetShell()->GetControlModule()->Render();
			this->GetMainWindow()->Block(false);
		}
	};


	class Book : public iggFrameBook
	{
		
	public:
		
		Book(iggFrame *parent) : iggFrameBook(parent,true)
		{
		}

		void PolishPages()
		{
			int i;
			for(i=0; i<this->mPages.Size(); i++)
			{
				iDynamicCast<iggPageMain,iggFrame>(INFO,this->GetPage(i))->Polish();
			}
		}

	protected:

		virtual void OnInt1Body(int i)
		{
			if(this->GetMainWindow()->GetExtensionWindow() != 0) this->GetMainWindow()->GetExtensionWindow()->OpenBookPageByIndex(i);
		}
	};


	class DialogAbout : public iggDialog
	{

	public:

		DialogAbout(iggMainWindow *parent) : iggDialog(parent,_DialogModal,0,"About",0,3,"Ok")
		{
			int i, n;
			iString s;

			mFrame->AddLine(new iggWidgetTextArea("%b%+IFrIT - Ionization FRont Interactive Tool",mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("Version "+iVersion::GetVersion(),mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea(iEdition::GetEditionName(),mFrame),3);

			mFlipper = new iggWidgetLogoFlipper(mFrame);
			mFrame->AddLine(0,mFlipper);

			mFrame->AddLine(new iggWidgetTextArea(iString("VTK version ")+vtkVersion::GetVTKVersion(),mFrame),3);

			mFrame->AddLine(new iggWidgetTextArea("%bThis installation includes the following extensions:",mFrame),3);
			s = iVersion::GetIncludedExtensions();
			n = s.Contains(';');
			for(i=0; i<n; i++) mFrame->AddLine(new iggWidgetTextArea(s.Section(";",i,i),mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("%bThis installation includes the following shells:",mFrame),3);
			s = iVersion::GetIncludedShells();
			n = s.Contains(';');
			for(i=0; i<n; i++) mFrame->AddLine(new iggWidgetTextArea(s.Section(";",i,i),mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("",mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("IFrIT is created and maintained by Nick Gnedin",mFrame),3);
		}

		virtual void Show(bool s)
		{
			if(s) mFlipper->Start(); else mFlipper->Abort();
			iggDialog::Show(s);
		}

	protected:

		iggWidgetLogoFlipper *mFlipper;
	};


	class DialogDocking : public iggDialog
	{

	public:

		DialogDocking(iggMainWindow *parent) : iggDialog(parent,_DialogBlocking|_DialogNoTitleBar,0,"Docking...",0,3,0)
		{
			mFlipper = new iggWidgetLogoFlipper(mFrame);
			mFrame->AddLine(0,mFlipper);
			mFrame->AddLine(new iggWidgetTextArea("%b%+Rearranging windows...",mFrame),3);
		}

		virtual void Show(bool s)
		{
			if(s) mFlipper->Start(); else mFlipper->Abort();
			iggDialog::Show(s);
		}

	protected:

		iggWidgetLogoFlipper *mFlipper;
	};


	//
	//  A small dialog with three line edit to set the axes labels
	//
	class AxesLabelsDialog : public iggDialogAuto
	{

	public:

		AxesLabelsDialog(iggMainWindow *parent) : iggDialogAuto(parent,"Axes Labels",3)
		{
			mFrame->AddLine(new iggWidgetKeyTextLineEdit(false,"X: Label",iViewModule::KeyAxesBoxLabels(),_RenderModeUseGlobal,mFrame,0),new iggWidgetKeyFloatLineEdit("Min",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,0),new iggWidgetKeyFloatLineEdit("Max",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,1));
			mFrame->AddLine(new iggWidgetKeyTextLineEdit(false,"Y: Label",iViewModule::KeyAxesBoxLabels(),_RenderModeUseGlobal,mFrame,1),new iggWidgetKeyFloatLineEdit("Min",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,2),new iggWidgetKeyFloatLineEdit("Max",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,3));
			mFrame->AddLine(new iggWidgetKeyTextLineEdit(false,"Z: Label",iViewModule::KeyAxesBoxLabels(),_RenderModeUseGlobal,mFrame,2),new iggWidgetKeyFloatLineEdit("Min",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,4),new iggWidgetKeyFloatLineEdit("Max",iViewModule::KeyAxesBoxRanges(),_RenderModeUseGlobal,mFrame,5));
		}
	};

#ifdef I_DEBUG
	class DebugHelperEmphasizeAllWidgetsCheckBox : public iggWidget
	{

	public:

		DebugHelperEmphasizeAllWidgetsCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,_ButtonTypeCheckBox,"Show all layouts",1);
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void UpdateWidgetBody()
		{
		}

		void OnVoid1Body()
		{
			this->GetMainWindow()->Block(true);
			iggWidget::EmphasizeLayouts(mSubject->IsDown());
			this->GetMainWindow()->Block(false);
		}

		ibgWidgetButtonSubject *mSubject;
	};

	class DebugHelperEmphasizeUnderCursorCheckBox : public iggWidget
	{

	public:

		DebugHelperEmphasizeUnderCursorCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,_ButtonTypeCheckBox,"Show under cursor",1);
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void UpdateWidgetBody()
		{
		}

		void OnVoid1Body()
		{
			ibgMainWindowSubject::SetEmphasizeUnderCursor(mSubject->IsDown());
		}

		ibgWidgetButtonSubject *mSubject;
	};

	class DebugHelperFlipAllPagesButton : public iggWidgetSimpleButton
	{

	public:

		DebugHelperFlipAllPagesButton(iggFrame *sf, iggFrame *parent) : iggWidgetSimpleButton("Flip all pages",parent)
		{
			mStartFrame = sf;
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			if(mStartFrame != 0) mStartFrame->FlipThroughAllChildren();
			this->GetMainWindow()->Block(false);
		}

		iggFrame *mStartFrame;
	};

	class DebugHelperCreateUGButton : public iggWidgetSimpleButton
	{

	public:

		DebugHelperCreateUGButton(iggFrame *parent) : iggWidgetSimpleButton("Create User Guide(s)",parent)
		{
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			iHelpFactory::CreateUserGuide(iHelpFactory::_Publish);
			this->GetMainWindow()->Block(false);
		}
	};

	class DebugHelperParallelRadioBox : public iggWidget
	{

	public:

		DebugHelperParallelRadioBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetRadioBoxSubject(this,1,"Parallel execution");
			mSubject->InsertItem("Parallel");
			mSubject->InsertItem("Proc 1");
			mSubject->InsertItem("Proc 2");
			mSubject->InsertItem("Procs 1+2");
			mSubject->InsertItem("No stiches");
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void UpdateWidgetBody()
		{
			if(iParallelManager::DebugSwitch>0 && iParallelManager::DebugSwitch<mSubject->Count()) mSubject->SetValue(iParallelManager::DebugSwitch); else mSubject->SetValue(0);
		}

		void OnInt1Body(int v)
		{
			this->GetMainWindow()->Block(true);
			iParallelManager::DebugSwitch = v;
			this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->ResetPipeline();
			this->GetShell()->GetControlModule()->GetViewModule()->Render();
			this->GetMainWindow()->Block(false);
		}

		ibgWidgetRadioBoxSubject *mSubject;
	};

	class DebugHelperDialog : public iggDialog
	{

	public:

		DebugHelperDialog(iggFrame *sf, iggMainWindow *parent) : iggDialog(parent,0U,0,"Debug Helper",0,1)
		{
			mFrame->AddLine(new DebugHelperParallelRadioBox(mFrame));
			mFrame->AddLine(new DebugHelperEmphasizeAllWidgetsCheckBox(mFrame));
			mFrame->AddLine(new DebugHelperEmphasizeUnderCursorCheckBox(mFrame));
			mFrame->AddLine(new DebugHelperFlipAllPagesButton(sf,mFrame));
			mFrame->AddLine(new DebugHelperCreateUGButton(mFrame));
		}
	};
#endif
};


using namespace iggMainWindow_Private;


//
//  **************************************************************************************
//
//      INITIALIZATION
//
//  **************************************************************************************
//
iggMainWindow::iggMainWindow(iShell *shell) : iObject("MainWindow"), iggMenuWindow(shell), mReloadDataInfo(new iDataInfo), mEraseDataInfo(new iDataInfo)
{
	mInitialized = mDocked = mDoNotOfferToReload = mDetailLoadedDataInfo = mAllowPrePopulateToolBar = false;
	mInteractorHelp = mToolTips = mOptionsAreGlobal = mWindowUnderFocusCurrent = mAutoRender = true;
	mIdiosyncratic = mAttachWindows = true;

	mProgressBarMode = 0;
	mCurrentTheme = -1;
	mBlockLevel = 0;

	mStateFileName = this->GetShell()->GetEnvironment(_EnvironmentBase) + "ifrit.ini";

	mInitialGeometry[0] = mInitialGeometry[1] = mInitialGeometry[2] = mInitialGeometry[3] = 0;

	mInMove = false;
	mPrevPos[0] = mPrevPos[1] = -1;

	mMainWindow = this;

	IERROR_ASSERT(mReloadDataInfo);
	IERROR_ASSERT(mEraseDataInfo);
}


void iggMainWindow::StartInitialization()
{
	//
	//  MainSubject must be created first
	//
	mMainSubject = iggSubjectFactory::CreateMainWindowSubject(this,3);

	//
	//  Have we created MenuSubject as well?
	//
	if(this->GetSubject() == 0)
	{
		mTwoSubjectsAreTheSame = false;
		this->AttachSubject(iggSubjectFactory::CreateMenuWindowSubject(this,iImageFactory::FindIcon("genie1gui.png"),"Ionization Front Interactive Tool"),3);
	}
	else
	{
		mTwoSubjectsAreTheSame = true;
	}

	//
	//  Top frames
	//
	mBusyIndicatorFrame = new iggFrameTopParent(this->GetShell());
	mVisitedFileListFrame = new iggFrameTopParent(this->GetShell());

	mMainSubject->SetTopParentSubjects(mBusyIndicatorFrame,mVisitedFileListFrame);

	IERROR_ASSERT(mBusyIndicatorFrame->GetSubject());
	IERROR_ASSERT(mVisitedFileListFrame->GetSubject());

	//
	//  Other components
	//
	mProgressBar = new iggWidgetProgressBar(mGlobalFrame,true);

	//
	//  Status bar
	//
	mBusyIndicator = new BusyIndicator(mBusyIndicatorFrame);
	mBusyIndicatorFrame->AddLine(mBusyIndicator);
	mVisitedFileList = new VisitedFileList(mVisitedFileListFrame);
	mVisitedFileListFrame->AddLine(mVisitedFileList);

	mMainSubject->PopulateStatusBar(mBusyIndicatorFrame,mProgressBar->GetSubject(),mVisitedFileListFrame);

	mGlobalFrame->SetPadding(true);

	//
	//  Extension
	//
	mExtensionWindow = new iggExtensionWindow(this);
	mExtensionWindow->CompleteInitialization();

	//
	//  Base frame
	//
	mBaseFrame = new iggFrameFlip(mGlobalFrame);

	//
	//  Create the book frame but not layout it
	//
	iggFrame *tmpf = new iggFrame(mBaseFrame);
	mLog = new iggWidgetTextBrowser(false,false,mGlobalFrame);
	mLog->SetBaloonHelp("Log window","In this window IFrIT logs information about the data files, non-critical execution error, performance information, etc.");
	if(iRequiredCast<iggShell>(INFO,this->GetShell())->IsDesktopSmall(true))
	{
		mLog->Show(false);
	}
	if(iRequiredCast<iggShell>(INFO,this->GetShell())->IsDesktopSmall(false))
	{
		mAttachWindows = false;
	}

	//
	//  Create dialogs (needed for book pages)
	//
	mDialogAbout = new DialogAbout(this); IERROR_ASSERT(mDialogAbout);
	mDialogAnimatingProgress = new iggDialogAnimatingProgress(this); IERROR_ASSERT(mDialogAnimatingProgress);
	mDialogAxesLabels = new AxesLabelsDialog(this); IERROR_ASSERT(mDialogAxesLabels);
	mDialogCommandLine = new iggDialogCommandLine(this); IERROR_ASSERT(mDialogCommandLine);
	mDialogDataExplorer = new iggDialogDataExplorer(this); IERROR_ASSERT(mDialogDataExplorer);
	mDialogDocking = new DialogDocking(this); IERROR_ASSERT(mDialogDocking);
	mDialogEventRecorder = new iggDialogEventRecorder(this); IERROR_ASSERT(mDialogEventRecorder);
	mDialogFileSetExplorer = new iggDialogFileSetExplorer(this); IERROR_ASSERT(mDialogFileSetExplorer);
	mDialogHelp = new iggDialogHelp(this); IERROR_ASSERT(mDialogHelp);
	mDialogImageComposer = new iggDialogImageComposer(this); IERROR_ASSERT(mDialogImageComposer);
	mDialogLoadFile = new iggDialogLoadFile(this); IERROR_ASSERT(mDialogLoadFile);
	mDialogPaletteEditor = new iggDialogPaletteEditor(this); IERROR_ASSERT(mDialogPaletteEditor);
	mDialogParallelController = new iggDialogParallelController(this); IERROR_ASSERT(mDialogParallelController);
	mDialogPerformanceMeter = new iggDialogPerformanceMeter(this); IERROR_ASSERT(mDialogPerformanceMeter);
	mDialogPickerWindow = new iggDialogPickerWindow(this); IERROR_ASSERT(mDialogPickerWindow);
	mDialogRenderingProgress = 0; // we create this one later, so that it does not pop up during the initialization process
	mDialogScriptDebugger = new iggDialogScriptDebugger(this); IERROR_ASSERT(mDialogScriptDebugger);

	//
	//  Create the book
	//
	Book *book = new Book(tmpf);
	mBook = book;

//	iggPage::SetDelayedConstruction(false);
	mViewPage = new iggPageView(mBook);
	mDataPage = new iggPageData(mBook);
	
	mPages[0] = new iggPageSurface(mBook);
	mPages[1] = new iggPageCrossSection(mBook);
	mPages[2] = new iggPageVolume(mBook);
	mPages[3] = mParticlesPage = new iggPageParticles(mBook);
	mPages[4] = new iggPageVectorField(mBook);
	mPages[5] = new iggPageTensorField(mBook);
//	iggPage::SetDelayedConstruction(true);

	iParticlesViewSubject::UseFullState(false); // optimization, since we use iParticleGroup keys

	mBook->AddPage("View",iImageFactory::FindIcon("view.png"),mViewPage);
	mBook->AddPage("Surface",iImageFactory::FindIcon("surf.png"),mPages[0]);
	mBook->AddPage("Cross section",iImageFactory::FindIcon("xsec.png"),mPages[1]);
	mBook->AddPage("Volume",iImageFactory::FindIcon("volv.png"),mPages[2]);
	mBook->AddPage("Particles",iImageFactory::FindIcon("part.png"),mPages[3]);
	mBook->AddPage("Vector field",iImageFactory::FindIcon("vect.png"),mPages[4]);
	mBook->AddPage("Tensor field",iImageFactory::FindIcon("tens.png"),mPages[5]);
	mBook->AddPage("Data",iImageFactory::FindIcon("data.png"),mDataPage);

	//
	//  Layout the book frame
	//
	tmpf->AddLine(mBook);
	mBaseFrame->AddLayer(tmpf);

	//
	//  Interactor help frame
	//
	tmpf = new iggFrame("",mBaseFrame);
	tmpf->AddLine(new iggWidgetTextArea("%b%+Interactor Help",tmpf));
	tmpf->AddLine(new InteractorHelpTextView(tmpf));
	tmpf->SetRowStretch(0,1);
	tmpf->SetRowStretch(1,10);
	mBaseFrame->AddLayer(tmpf);
	mBaseFrame->ShowLayer(0);

	mGlobalFrame->AddLine(mBaseFrame,3);
	mGlobalFrame->AddLine(mLog,3);
	iggFrame *tmp1 = new iggFrame(mGlobalFrame,2);
	iggFrame *tmp2 = new iggFrame(mGlobalFrame);
	tmp1->AddSpace(10);
	mRenderButton = new RenderButton(tmp1);
	mRenderButton->Show(false);
	tmp1->SetPadding(true);
	tmp1->AddLine(new ImageButton(tmp1),mRenderButton);
	tmp1->AddSpace(10);
	mDataTypeFrame = new iggFrameFlip(tmp2,false);
	tmp2->AddSpace(10);
	tmp2->AddLine(mDataTypeFrame);
	tmp2->AddSpace(10);
	mGlobalFrame->AddLine(tmp1,new iggWidgetTextArea("",mGlobalFrame),tmp2);
	mGlobalFrame->SetRowStretch(0,0);
	mGlobalFrame->SetRowStretch(1,10);
	mGlobalFrame->SetRowStretch(2,0);

	book->PolishPages();

	iEdition::ApplySettings(this,this->Type());

	this->ShowToolTips(mToolTips);

#if defined(I_DEBUG)
	mDialogDebugHelper = new DebugHelperDialog(mGlobalFrame,this);
#endif

	//
	//  Connect pre-existing RenderWindows
	//
	int i;
	for(i=0; i<this->GetShell()->GetControlModule()->GetNumberOfViewModules(); i++)
	{
		iRequiredCast<iggRenderWindow>(INFO,this->GetShell()->GetControlModule()->GetViewModule(i)->GetRenderWindow())->AttachToMainWindow(this);
	}
}


void iggMainWindow::PreShowInitialization()
{
	this->BuildMenus();
	if(mInitialGeometry[2]>0 && mInitialGeometry[3]>0) this->GetSubject()->SetWindowGeometry(mInitialGeometry);
}


void iggMainWindow::PostShowInitialization()
{
	mDialogRenderingProgress = new iggDialogRenderingProgress(this); IERROR_ASSERT(mDialogRenderingProgress);

	int wg[4];
	this->GetSubject()->GetWindowGeometry(wg);
	mPrevPos[0] = wg[0];
	mPrevPos[1] = wg[1];

	mInitialized = true;
}


iggMainWindow::~iggMainWindow()
{
	if(mDocked)
	{
		mMainSubject->RestoreWindowsFromDockedPositions();
	}

	delete mExtensionWindow;
	mExtensionWindow = 0; // it does not exist any more

	if(mDialogAbout != 0) delete mDialogAbout;
	if(mDialogAnimatingProgress != 0) delete mDialogAnimatingProgress;
	if(mDialogAxesLabels != 0) delete mDialogAxesLabels;
	if(mDialogCommandLine != 0) delete mDialogCommandLine;
	if(mDialogDataExplorer != 0) delete mDialogDataExplorer;
	if(mDialogDocking != 0) delete mDialogDocking;
	if(mDialogEventRecorder != 0) delete mDialogEventRecorder;
	if(mDialogFileSetExplorer != 0) delete mDialogFileSetExplorer;
	if(mDialogHelp != 0) delete mDialogHelp;
	if(mDialogImageComposer != 0) delete mDialogImageComposer;
	if(mDialogLoadFile != 0) delete mDialogLoadFile;
	if(mDialogPaletteEditor != 0) delete mDialogPaletteEditor;
	if(mDialogParallelController != 0) delete mDialogParallelController;
	if(mDialogPerformanceMeter != 0) delete mDialogPerformanceMeter;
	if(mDialogPickerWindow != 0) delete mDialogPickerWindow;
	if(mDialogRenderingProgress != 0) delete mDialogRenderingProgress;
	if(mDialogScriptDebugger != 0) delete mDialogScriptDebugger;
#if defined(I_DEBUG)
	if(mDialogDebugHelper != 0) delete mDialogDebugHelper;
#endif

	delete mProgressBar;

	delete mGlobalFrame;
	delete mBusyIndicatorFrame;
	delete mVisitedFileListFrame;

	if(!mTwoSubjectsAreTheSame)
	{
		delete mMainSubject;
	}

	delete mReloadDataInfo;
	delete mEraseDataInfo;
}


void iggMainWindow::UpdateContents()
{
	mGlobalFrame->UpdateWidget();

	//
	//  Also update widgets that are not children of mGlobalFrame
	//
	mBusyIndicator->UpdateWidget();
	mVisitedFileList->UpdateWidget();

	//
	//  Update some of the dialogs
	//
	if(mDialogDataExplorer != 0) mDialogDataExplorer->UpdateDialog();
	if(mDialogFileSetExplorer != 0) mDialogFileSetExplorer->UpdateDialog();
	if(mDialogImageComposer != 0) mDialogImageComposer->UpdateDialog();
	if(mDialogPickerWindow != 0) mDialogPickerWindow->UpdateDialog();
	if(mViewPage != 0) mViewPage->GetWindowListDialog()->UpdateDialog();

	//
	//  Update script debugger
	//
	if(mDialogScriptDebugger != 0) mDialogScriptDebugger->UpdateAll();

	//
	//  Update extension too
	//
	mExtensionWindow->UpdateAll();
}


void iggMainWindow::Register(ibgWindowSubject *window)
{
	if(window != 0) mWindowList.AddUnique(window);
}


void iggMainWindow::UnRegister(ibgWindowSubject *window)
{
	if(window != 0) mWindowList.Remove(window);
}


void iggMainWindow::RegisterAutoDialog(iggDialogAuto *d)
{
	if(d != 0) mAutoDialogList.AddUnique(d);
}


void iggMainWindow::UnRegisterAutoDialog(iggDialogAuto *d)
{
	if(d != 0) mAutoDialogList.Remove(d);
}


//
//  **************************************************************************************
//
//      DATA MANIPULATION
//
//  **************************************************************************************
//
void iggMainWindow::LoadData(const iDataType &type, const iString &filename, int mod)
{
	iDataInfo info(type);
	
	mProgressBarMode = 1;
	mDialogLoadFile->LoadData(info,filename,mod);
	mProgressBarMode = 0;
}


void iggMainWindow::AddReloadDataType(const iDataType &type)
{
	*mReloadDataInfo += type;
}


void iggMainWindow::ReloadData()
{
	mProgressBarMode = 1;
	mDialogLoadFile->ReloadData(*mReloadDataInfo);
	mProgressBarMode = 0;
	mReloadDataInfo->Erase();
}


void iggMainWindow::RequestReloadData(const iDataInfo &info)
{
	int i;

	for(i=0; i<info.Count(); i++) this->AddReloadDataType(info.Type(i));

	if(!mDoNotOfferToReload && this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsThereData(info))
	{
		int ret = this->PopupWindow("This control only takes effect when the data set is reloaded.",_PopupWindowMessage,"Reload now","Later","Do not ask again");
		if(ret == 0) this->ReloadData();
		if(ret == 2) mDoNotOfferToReload = true;
	}
}


void iggMainWindow::AddEraseDataType(const iDataType &type)
{
	*mEraseDataInfo += type;
}


void iggMainWindow::RemoveEraseDataType(const iDataType &type)
{
	*mEraseDataInfo -= type;
}


void iggMainWindow::EraseData()
{
	int i;

	this->Block(true);
	
	for(i=0; i<mEraseDataInfo->Count(); i++)
	{
		this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->EraseData(mEraseDataInfo->Type(i));
		this->AfterEraseData(mEraseDataInfo->Type(i));
	}
	
	this->Block(false);
	mEraseDataInfo->Erase();
}


void iggMainWindow::AfterLoadData(const iDataType &type, const iString &filename)
{
	iDataReader *r = this->GetShell()->GetControlModule()->GetViewModule()->GetReader();

	if(r->GetErrorStatus()->NoError() || r->GetErrorStatus()->Level()<0)  // warning only
	{
		this->WriteToLog("");
		this->WriteToLog("Loaded file",filename);
		if(mDialogDataExplorer!=0 && mDialogDataExplorer->IsVisible()) mDialogDataExplorer->UpdateDialog();
		this->GetShell()->GetControlModule()->Render(_RenderOptionClones);
		this->LogDataInfo(type,mDetailLoadedDataInfo);

		//
		//  Provide the FileSet info
		//
		if(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsSet())
		{
			int m = 0;
			iString s;
			while(!this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetFileSetDataType(m).IsNull())
			{
				if(m > 0) s += " + ";
				s += this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetFileSetDataType(m).GetTextName();
				m++;
			}
			this->WriteToLog("File Set: ",s);
		}
		mDataPage->UpdateOnDataLoad();

		if(r->GetErrorStatus()->IsError()) this->PopupWindow(r->GetErrorStatus()->Message());
	}
	else
	{
		if(!r->GetErrorStatus()->IsAbort()) this->PopupWindow("Error in loading data:\n"+r->GetErrorStatus()->Message(),_PopupWindowError);
	}
	this->UpdateAll();
}


void iggMainWindow::AfterReloadData(const iDataType &)
{
	if(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetErrorStatus()->Message().IsEmpty())
	{
		if(mDialogDataExplorer!=0 && mDialogDataExplorer->IsVisible()) mDialogDataExplorer->UpdateDialog();
		this->GetShell()->GetControlModule()->Render(_RenderOptionClones);
	}
	else
	{
		this->PopupWindow("Reloading of the current file set failed:\n"+this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetErrorStatus()->Message(),_PopupWindowWarning);
	}
	this->UpdateAll();
}


void iggMainWindow::AfterEraseData(const iDataType &)
{
	if(mDialogDataExplorer!=0 && mDialogDataExplorer->IsVisible()) mDialogDataExplorer->UpdateDialog();
	this->GetShell()->GetControlModule()->Render(_RenderOptionClones);

	this->UpdateAll();
}


void iggMainWindow::LogDataInfo(const iFileLoader *loader, bool details)
{
	int i, j;

	IERROR_ASSERT(loader);

	bool work = false;
	iDataSubject *subject;
	for(j=0; j<loader->NumStreams(); j++) if((subject = loader->GetSubject(j))->IsThereData())
	{
		iDataLimits *limits = subject->GetLimits();
		work = true;

		//
		//  Extension did not produce its own info, use the default one
		//
		this->WriteToLog(subject->GetDataType().GetTextName(),subject->GetDataType().MatchesKeyword("basic")?"":iString("( VTK class name: ")+subject->GetData()->GetClassName()+")");

		//
		//  Are we ImageData?
		//
		vtkImageData *im = vtkImageData::SafeDownCast(subject->GetData());
		if(im != 0)
		{
			int dims[3];
			im->GetDimensions(dims);
			this->WriteToLog("Dimensions: "+iString::FromNumber(dims[0])+" x "+iString::FromNumber(dims[1])+" x "+iString::FromNumber(dims[2]));
		}

		vtkPolyData *pl = vtkPolyData::SafeDownCast(subject->GetData());
		if(pl != 0)
		{
			this->WriteToLog("Number of particles: "+iString::FromNumber(long(pl->GetNumberOfPoints())));
		}

		vtkPointData *pd = subject->GetData()->GetPointData();
		if(pd!=0 && pd->GetScalars() != 0)
		{
			this->WriteToLog("Number of scalar components: "+iString::FromNumber(pd->GetScalars()->GetNumberOfComponents()));
		}
		if(pd!=0 && pd->GetVectors() != 0)
		{
			this->WriteToLog("Number of vector components: "+iString::FromNumber(pd->GetVectors()->GetNumberOfComponents()));
		}
		if(pd!=0 && pd->GetTensors() != 0)
		{
			this->WriteToLog("Number of tensor components: "+iString::FromNumber(pd->GetTensors()->GetNumberOfComponents()));
		}

		if(details)
		{
			iDataExplorer *de = iDataExplorer::New(this->GetShell()->GetControlModule()->GetViewModule()); IERROR_ASSERT(de);
			de->SetActiveDataType(subject->GetDataType());
			for(i=0; i<de->GetNumberOfInputComponents(); i++)
			{
				de->SetInputComponent(i);
				iDataExplorer::Info di = de->GetInfo(true);
				this->WriteToLog(subject->GetLimits()->GetName(i)+": from "+iString::FromNumber(di.Minimum)+" ("+iString::FromNumber(log10(1.0e-30+fabs(di.Minimum)))+" dex) to "+iString::FromNumber(di.Maximum)+" ("+iString::FromNumber(log10(1.0e-30+fabs(di.Maximum)))+" dex)");
			}
			de->Delete();
		}
	}

	if(!work) this->WriteToLog("This file contains no data that IFrIT can use.");
}


void iggMainWindow::LogDataInfo(const iDataType &type, bool details)
{
	iDataReader *reader = this->GetShell()->GetControlModule()->GetViewModule()->GetReader();
	iDataSubject *subject = reader->GetSubject(type);

	if(!mExtensionWindow->LogDataInfo(type,details))
	{
		this->LogDataInfo(subject->GetLoader(),details);
	}

	if(subject!=0 && subject->GetLoader()->IsBoxPeriodic())
	{
		iString ws;
		if(subject->GetLoader()->IsDirectionPeriodic(0)) ws += "X, ";
		if(subject->GetLoader()->IsDirectionPeriodic(1)) ws += "Y, ";
		if(subject->GetLoader()->IsDirectionPeriodic(2)) ws += "Z";
		if(!ws.IsEmpty()) this->WriteToLog("Data periodic in "+ws);
	}
}


//
//  **************************************************************************************
//
//      MISC FUNCTIONS
//
//  **************************************************************************************
//
iggWidgetProgressBar* iggMainWindow::GetProgressBar() const
{
	switch(mProgressBarMode)
	{
	case 1:
		{
			return mDialogLoadFile->GetProgressBar();
		}
	case 0:
	default:
		{
			return mProgressBar;
		}
	}
}


int iggMainWindow::PopupWindow(const iString &text, int type, const char *b0, const char *b1, const char *b2)
{
	return mMainSubject->PopupWindow(mGlobalFrame,text,type,b0,b1,b2);
}


int iggMainWindow::PopupWindow(const iggFrame *parent, const iString &text, int type, const char *b0, const char *b1, const char *b2)
{
	return mMainSubject->PopupWindow(parent,text,type,b0,b1,b2);
}

	
int iggMainWindow::PopupWindow(const iggRenderWindow *parent, const iString &text, int type, const char *b0, const char *b1, const char *b2)
{
	return mMainSubject->PopupWindow(parent,text,type,b0,b1,b2);
}


void iggMainWindow::ShowToolTips(bool s)
{
	mToolTips = s;
	mMainSubject->ShowToolTips(s);
	this->ClearCache();
}


void iggMainWindow::ShowInteractorHelp(bool s, iViewModule* /*vm*/)
{
	mBaseFrame->ShowLayer(s?1:0);
}


void iggMainWindow::Block(bool s)
{
	if(s)
	{
		if(mBlockLevel == 0)
		{
			ibgWindowSubject::Block(true);
			iDynamicCast<BusyIndicator,iggWidget>(INFO,mBusyIndicator)->SetBusy(true);
		}
		mBlockLevel++;
	}
	else
	{
		mBlockLevel--;
		if(mBlockLevel == 0)
		{
			iDynamicCast<BusyIndicator,iggWidget>(INFO,mBusyIndicator)->SetBusy(false);
			ibgWindowSubject::Block(false);
		}
		if(mDialogRenderingProgress!=0 && mDialogRenderingProgress->IsCancelled() && mDialogRenderingProgress->IsVisible())
		{
			mDialogRenderingProgress->Show(false);
		}
		if(mBlockLevel < 0)
		{
#ifdef I_CHECK1
			IERROR_HIGH("Incorrect order if MainWindow->Block() calls. This is a bug, but IFrIT may continue to work properly.");
#endif
			mBlockLevel = 0;
		}
	}
}


bool iggMainWindow::IsExitAllowed()
{
	return this->AskForConfirmation("Are you sure you want to exit IFrIT?","Exit");
}


void iggMainWindow::WriteToLog(const iString &prefix, const iString &message, const iColor &color)
{
	mLog->AppendTextLine(prefix,message,color);

	if(mLog->GetNumberOfLines() > 1000)
	{
		while(mLog->GetNumberOfLines() > 700) mLog->RemoveLine(0);
	}
}


void iggMainWindow::WriteToLog(const iString &message)
{
	mLog->AppendTextLine("\t",message);

	if(mLog->GetNumberOfLines() > 1000)
	{
		while(mLog->GetNumberOfLines() > 700) mLog->RemoveLine(0);
	}
}


void iggMainWindow::ClearLog()
{
	mLog->Clear();
}


void iggMainWindow::ShowLog(bool s)
{
	mLog->Show(s);
	if(!s)
	{
		this->ProcessEvents();

		int wg[4];
		this->GetSubject()->GetWindowGeometry(wg,true);
		wg[3] = 10;
		this->GetSubject()->SetWindowGeometry(wg,true);
	}
}


bool iggMainWindow::IsLogVisible() const
{
	return mLog->IsVisible();
}


void iggMainWindow::OpenBookPage(int n)
{
	mBook->OpenPage(n);
}


void iggMainWindow::SetTabMode(int m)
{
	switch(m)
	{
	case 0:
		{
			mBook->SetTabMode(_BookTitleOnly);
			break;
		}
	case 1:
		{
			mBook->SetTabMode(_BookImageOnly);
			break;
		}
	case 2:
		{
			mBook->SetTabMode(_BookTitleAndImage);
			break;
		}
	}
	if(mExtensionWindow != 0) mExtensionWindow->SetTabMode(m);
	this->ClearCache();
}


void iggMainWindow::ShowAll(bool s)
{
	this->GetSubject()->Show(s);
	if(!mDocked)
	{
		mExtensionWindow->GetSubject()->Show(s);
		FOR_EVERY_RENDER_WINDOW(w)
		{
			w->GetSubject()->Show(s);
		}
	}
}


void iggMainWindow::DockWindows(bool s, bool show)
{
	if(s == mDocked) return;
	
	this->ClearCache();

	if(show) mDialogDocking->Show(true);

	this->ProcessEvents(true); // god knows why this is needed

	if(s)
	{
		mDocked = true;

		//
		//  Save all geometries
		//
		this->GetSubject()->GetHelper()->SaveWindowGeometry();
		mExtensionWindow->GetSubject()->GetHelper()->SaveWindowGeometry();

		FOR_EVERY_RENDER_WINDOW(w)
		{
			w->GetSubject()->GetHelper()->SaveWindowGeometry();
		}

		mGlobalFrame->SetPadding(false);
		mExtensionWindow->SetPadding(false);

		mMainSubject->PlaceWindowsInDockedPositions();
	} 
	else 
	{
		mMainSubject->RestoreWindowsFromDockedPositions();

		mGlobalFrame->SetPadding(true);
		mExtensionWindow->SetPadding(true);

		this->GetSubject()->GetHelper()->RestoreWindowGeometry();
		this->GetSubject()->Show(true);
		mExtensionWindow->GetSubject()->GetHelper()->RestoreWindowGeometry();
		mExtensionWindow->GetSubject()->GetHelper()->RestoreIcon();
		mExtensionWindow->GetSubject()->Show(true);

		FOR_EVERY_RENDER_WINDOW(w)
		{
			w->GetSubject()->GetHelper()->RestoreWindowGeometry();
			w->GetSubject()->GetHelper()->RestoreIcon();
			w->GetSubject()->Show(true);
		}

		mDocked = false;
	}

	this->ProcessEvents(true); // god knows why this is needed

	FOR_EVERY_RENDER_WINDOW(w)
	{
		w->Render();
	}

	this->ProcessEvents(true); // god knows why this is needed

	if(show) mDialogDocking->Show(false);
}


void iggMainWindow::PlaceAutoDialogs()
{
	this->ProcessEvents(true); // god knows why this is needed
	//
	//  Shift dialogs off the way
	//
	int mwg[4], fs[2], wg[4];
	this->GetSubject()->GetWindowGeometry(mwg);
	this->GetSubject()->GetFrameSize(fs[0],fs[1]);

	int i, j, tmp, n = 0, topOffset = 0;
	int *width = new int[mAutoDialogList.Size()];
	int *index = new int[mAutoDialogList.Size()];
	if(width!=0 && index!=0)
	{
		for(i=0; i<mAutoDialogList.Size(); i++) if(mAutoDialogList[i]->IsVisible())
		{
			mAutoDialogList[i]->GetSubject()->GetWindowGeometry(wg);
			index[n] = i;
			width[n] = wg[2];
			n++;
		}

		for(i=0; i<n-1; i++)
		{
			for(j=i+1; j<n; j++)
			{
				if(width[i] < width[j])
				{
					tmp = width[i];
					width[i] = width[j];
					width[j] = tmp;
					tmp = index[i];
					index[i] = index[j];
					index[j] = tmp;
				}
			}
		}

		for(i=0; i<n; i++)
		{
			this->ProcessEvents(true); // god knows why this is needed

			mAutoDialogList[index[i]]->GetSubject()->GetWindowGeometry(wg);
			wg[0] = mwg[0] + mwg[2] + fs[0];
			wg[1] = mwg[1] + topOffset;
			mAutoDialogList[index[i]]->GetSubject()->SetWindowGeometry(wg);

			this->ProcessEvents(true); // god knows why this is needed

			topOffset += (wg[3]+fs[1]);
		}

		delete [] width;
		delete [] index;
	}
}


bool iggMainWindow::AskForConfirmation(const iString &message, const char *action)
{
	return !mIdiosyncratic || this->PopupWindow(message,_PopupWindowMessage,action,"Cancel")==0;
}


void iggMainWindow::UpdateOnPick()
{
	mExtensionWindow->UpdateOnPick();
}


void iggMainWindow::UpdateMarkerWidgets()
{
	if(mViewPage != 0) mViewPage->UpdateMarkerWidgets();
}


void iggMainWindow::UpdateParticleWidgets(const iImage *icon)
{
	if(icon != 0)
	{
		mBook->ChangeIcon(4,icon);
	}
	if(mParticlesPage != 0)
	{
		if(icon != 0)
		{
			mParticlesPage->GetBook()->ChangeIcon(-1,icon);
		}
//		mParticlesPage->UpdateWidget();
	}
	if(icon != 0)
	{
		this->GetSubject()->SetToolBarIcon(_ToolBarShowParticles,*icon);
	}
	this->GetSubject()->UpdateMenus();

	mExtensionWindow->UpdateParticleWidgets(icon);
}


void iggMainWindow::AddTheme(const iString &name)
{
	mThemeList.Add(name);
}


void iggMainWindow::SetTheme(int n)
{
	if(mCurrentTheme!=n && n>=0 && n<mThemeList.Size())
	{
		mCurrentTheme = n;
		mMainSubject->SetTheme(mThemeList[n]);
		this->ClearCache();
	}
}


void iggMainWindow::ProcessEvents(bool sync) const
{
	iRequiredCast<iggShell>(INFO,this->GetShell())->ProcessEvents(sync);
}


//
//  **************************************************************************************
//
//  iObject functionality
//
//  **************************************************************************************
//
void iggMainWindow::PackStateBody(iString &s) const
{
	int wg[4];

	this->GetSubject()->GetWindowGeometry(wg);
	this->PackValue(s,KeyGeometry(),wg,4);

	mExtensionWindow->GetSubject()->GetWindowGeometry(wg);
	this->PackValue(s,KeyExtensionGeometry(),wg,4);

	this->PackValue(s,KeyDocked(),mDocked);
	this->PackValue(s,KeyInteractorHelp(),mInteractorHelp);
	this->PackValue(s,KeyToolTips(),mToolTips);
	this->PackValue(s,KeyOptionsAreGlobal(),mOptionsAreGlobal);
	this->PackValue(s,KeyWindowUnderFocusCurrent(),mWindowUnderFocusCurrent);
	this->PackValue(s,KeyAllowPrePopulateToolBar(),mAllowPrePopulateToolBar);
	this->PackValue(s,KeyIsIdiosyncratic(),mIdiosyncratic);
	this->PackValue(s,KeyAttachWindows(),mAttachWindows);
	this->PackValue(s,KeyTabMode(),mBook->GetTabMode());
	this->PackValue(s,KeyTheme(),(mCurrentTheme>=0 && mCurrentTheme<mThemeList.Size())?mThemeList[mCurrentTheme]:"");
}


void iggMainWindow::UnPackStateBody(const iString &s)
{
	int i, wg[4]; bool b; iString t;

	if(this->UnPackValue(s,KeyGeometry(),wg,4))
	{
		if(mInitialized)
		{
			this->GetSubject()->SetWindowGeometry(wg);
			this->ClearCache();
		}
		else
		{
			for(i=0; i<4; i++) mInitialGeometry[i] = wg[i];
		}
	}

	if(this->UnPackValue(s,KeyExtensionGeometry(),wg,4))
	{
		mExtensionWindow->GetSubject()->SetWindowGeometry(wg);
		this->ClearCache();
	}

	if(this->UnPackValue(s,KeyDocked(),b)) this->DockWindows(b,false);
	if(this->UnPackValue(s,KeyInteractorHelp(),b))
	{
		mInteractorHelp = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyToolTips(),b)) this->ShowToolTips(b);
	if(this->UnPackValue(s,KeyOptionsAreGlobal(),b))
	{
		mOptionsAreGlobal = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyWindowUnderFocusCurrent(),b))
	{
		mWindowUnderFocusCurrent = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyAllowPrePopulateToolBar(),b))
	{
		mAllowPrePopulateToolBar = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyIsIdiosyncratic(),b))
	{
		mIdiosyncratic = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyAttachWindows(),b))
	{
		mAttachWindows = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyTabMode(),i))	this->SetTabMode(i);
	if(this->UnPackValue(s,KeyTheme(),t))
	{
		for(i=0; i<mThemeList.Size(); i++)
		{
			if(t == mThemeList[i])
			{
				this->SetTheme(i);
				break;
			}
		}
	}

	//
	//  Special "action" keys
	//
	iObject::ReportMissingKeys(false); //action keys are not part of the states
	
	if(this->UnPackValue(s,KeySlidersAreEditable(),b))
	{
		iggWidgetSlider::SetAllEditable(b);
	}

	iObject::ReportMissingKeys(true);
}


//
//  **************************************************************************************
//
//    WINDOW MANIPULATION
//
//  **************************************************************************************
//
void iggMainWindow::OnRenderWindowMove(int /*wn*/)
{
}


void iggMainWindow::OnRenderWindowResize(int /*wn*/)
{
	if(mDialogImageComposer != 0) mDialogImageComposer->UpdateView();
}


void iggMainWindow::OnRenderWindowFocusIn(int wn)
{
	if(wn<-1 || wn>=this->GetShell()->GetControlModule()->GetNumberOfViewModules())
	{
		return; // can happen when a docked window is deleted
	}
	
	if(mWindowUnderFocusCurrent && wn>=0)
	{
		if(this->GetShell()->GetControlModule()->SetCurrentViewModuleIndex(wn)) this->UpdateAll();
	}

//	if(mInteractorHelp) this->ShowInteractorHelp(true,this->GetShell()->GetControlModule()->GetView(wn));
}


void iggMainWindow::OnRenderWindowFocusOut(int /*wn*/)
{
	if(mInteractorHelp) this->ShowInteractorHelp(false);
}


void iggMainWindow::OnRenderWindowEnter(int wn)
{
	if(wn<-1 || wn>=this->GetShell()->GetControlModule()->GetNumberOfViewModules())
	{
		return; // can happen when a docked window is deleted
	}
	
	if(mInteractorHelp) this->ShowInteractorHelp(true,this->GetShell()->GetControlModule()->GetViewModule(wn));
}


void iggMainWindow::OnRenderWindowLeave(int /*wn*/)
{
	if(mInteractorHelp) this->ShowInteractorHelp(false);
}


void iggMainWindow::DisplayWindowsAsIcons()
{
	if(!mDocked)
	{
		int i;
		for(i=0; i<mWindowList.Size(); i++)
		{
			//
			//  On multi-desktop systems this may cause weird behaviour unless all windows are on the same desktop.
			//
			mWindowList[i]->GetHelper()->ShowAsIcon();
		}
	}
}


void iggMainWindow::DisplayWindowsAsWindows()
{
	if(!mDocked)
	{
		int i;
		for(i=0; i<mWindowList.Size(); i++) 
		{
			//
			//  On multi-desktop systems this may cause weird behaviour unless all windows are on the same desktop.
			//
			mWindowList[i]->GetHelper()->ShowAsWindow();
		}
	}
}


void iggMainWindow::MoveWindows(bool all)
{
	if(mInMove) return; // ignore rapidly arriving events
	mInMove = true;

	int pos[2], wg[4];

	this->GetSubject()->GetWindowGeometry(wg);
	pos[0] = wg[0];
	pos[1] = wg[1];

	if(all && mInitialized && !mDocked && mAttachWindows) // do not move the first render window
	{
		int i, dp[2];
		dp[0] = pos[0] - mPrevPos[0];
		dp[1] = pos[1] - mPrevPos[1];
		if(dp[0]!=0 || dp[1]!=0)
		{
			ibgWindowSubject::Block(true); // needed to disable rendering while moving
			for(i=0; i<mWindowList.Size(); i++) if(mWindowList[i]->IsVisible())
			{
				mWindowList[i]->GetWindowGeometry(wg);
				wg[0] += dp[0];
				wg[1] += dp[1];
				mWindowList[i]->SetWindowGeometry(wg);
			}
			ibgWindowSubject::Block(false);
			this->ProcessEvents(true);
		}
	}

	mPrevPos[0] = pos[0];
	mPrevPos[1] = pos[1];

	mInMove = false;
}


void iggMainWindow::Exit()
{
	if(this->IsExitAllowed())
	{
		int i;
		for(i=0; i<mWindowList.Size(); i++) mWindowList[i]->Show(false);
		if(mDialogScriptDebugger != 0) mDialogScriptDebugger->Show(false);
		this->GetShell()->Exit();
	}
}



//
//  **************************************************************************************
//
//      MENU INTERACTION
//
//  **************************************************************************************
//
void iggMainWindow::BuildMenus()
{
	//
	//  1. File menu
	//
	this->GetSubject()->BeginMenu("&File",false);
	{
		this->GetSubject()->AddMenuItem(_MenuFileOpenUniformScalars,"Open Uniform&Scalars data file",iImageFactory::FindIcon("fileopenmesh.png"),"",false,false,&iViewModule::KeyNoClone());
		this->GetSubject()->AddMenuItem(_MenuFileOpenBasicParticles,"Open Basic&Particles data file",iImageFactory::FindIcon("fileopenpart.png"),"",false,false,&iViewModule::KeyNoClone());
		this->GetSubject()->AddMenuItem(_MenuFileOpenUniformVectors,"Open Uniform&Vectors data file",iImageFactory::FindIcon("fileopenvect.png"),"",false,false,&iViewModule::KeyNoClone());
		this->GetSubject()->AddMenuItem(_MenuFileOpenUniformTensors,"Open Uniform&Tensors data file",iImageFactory::FindIcon("fileopentens.png"),"",false,false,&iViewModule::KeyNoClone());

		mExtensionWindow->PopulateFileMenu();


		this->GetSubject()->AddMenuSeparator();

		this->GetSubject()->AddMenuItem(_MenuFileSaveState,"Save state",0,"",false,false);
		this->GetSubject()->AddMenuItem(_MenuFileSaveStateAs,"Save state as...",0,"",false,false);
		this->GetSubject()->AddMenuItem(_MenuFileLoadStateFrom,"Load state from...",0,"",false,false);

		this->GetSubject()->AddMenuSeparator();

		this->GetSubject()->AddMenuItem(_MenuFileExit,"&Exit",0,"",false,false);
	}
	this->GetSubject()->EndMenu();
	//
	//  2. Dialog menu
	//
	this->GetSubject()->BeginMenu("&Tools",false);
	{
		if(mDialogScriptDebugger != 0) this->GetSubject()->AddMenuItem(_MenuDialogScriptDebugger,"&Animation Script Debugger",iImageFactory::FindIcon("debug.png"),"Ctrl+A",false,false);
		if(mDialogPaletteEditor != 0) this->GetSubject()->AddMenuItem(_MenuDialogPaletteEditor,"Palette &Editor",iImageFactory::FindIcon("paled.png"),"Ctrl+E",false,false);
		if(mDialogPickerWindow != 0) this->GetSubject()->AddMenuItem(_MenuDialogPickerWindow,"&Picker Window",iImageFactory::FindIcon("picks.png"),"Ctrl+P",false,false);
		if(mDialogEventRecorder != 0) this->GetSubject()->AddMenuItem(_MenuDialogEventRecorder,"Event &Recorder",iImageFactory::FindIcon("recorder.png"),"Ctrl+R",false,false);
		if(mDialogFileSetExplorer != 0) this->GetSubject()->AddMenuItem(_MenuDialogFileSetExplorer,"&File Set Explorer",iImageFactory::FindIcon("setexp.png"),"Ctrl+F",false,false,&iDataReader::KeyIsSet());
		mExtensionWindow->PopulateLocalDialogMenu();
	
		this->GetSubject()->BeginMenu("Automatic Dialogs",false);
		{
			this->GetSubject()->AddMenuItem(_MenuDialogAutoBlock,"Do &not launch automatically",0,"",true,iggDialogAuto::mBlockShowing);
			int i;
			iString ws;
			for(i=0; i<mAutoDialogList.Size() && i<_MenuDialogAutoEnd-_MenuDialogAutoBegin; i++)
			{
				this->GetSubject()->AddMenuItem(_MenuDialogAutoBegin+i,"  "+mAutoDialogList[i]->GetTitle(),0,"",false,false);
			}
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->AddMenuSeparator();
		if(mDialogPerformanceMeter != 0) this->GetSubject()->AddMenuItem(_MenuDialogPerformanceMeter,"Performance &Meter",iImageFactory::FindIcon("perf.png"),"Ctrl+M",false,false);
		if(mDialogParallelController!=0 && this->GetSubject()->GetShell()->GetControlModule()->GetParallelManager()->GetMaxNumberOfProcessors()>1) this->GetSubject()->AddMenuItem(_MenuDialogParallelController,"Parallel &Controller",iImageFactory::FindIcon("parallel.png"),"Ctrl+C",false,false);
		if(mDialogDataExplorer != 0) this->GetSubject()->AddMenuItem(_MenuDialogDataExplorer,"&Data Explorer",iImageFactory::FindIcon("dataexp.png"),"Ctrl+D",false,false);
		if(mDialogImageComposer != 0) this->GetSubject()->AddMenuItem(_MenuDialogImageComposer,"&Image Composer",iImageFactory::FindIcon("imcomp.png"),"Ctrl+I",false,false);
		if(mDialogCommandLine != 0) this->GetSubject()->AddMenuItem(_MenuDialogCommandLine,"&Command Interpreter",iImageFactory::FindIcon("comline.png"),"Ctrl+N",false,false);
		mExtensionWindow->PopulateGlobalDialogMenu();
	}
	this->GetSubject()->EndMenu();
	//
	//  3. Style menu
	//
	this->GetSubject()->BeginMenu("&Style",false);
	{
		this->GetSubject()->AddMenuItem(_MenuStyleInteractorHelp,"Show &interactor help",0,"",true,mInteractorHelp);
		this->GetSubject()->AddMenuItem(_MenuStyleToolTips,"Show t&ooltips",0,"",true,mToolTips);

		this->GetSubject()->AddMenuSeparator();

		this->GetSubject()->BeginMenu("&Tab style",true);
		{
			this->GetSubject()->AddMenuItem(_MenuStyleTabText,"&Text only",0,"",true,mBook->GetTabMode()==_BookTitleOnly);
			this->GetSubject()->AddMenuItem(_MenuStyleTabIcon,"&Icon only",0,"",true,mBook->GetTabMode()==_BookImageOnly);
			this->GetSubject()->AddMenuItem(_MenuStyleTabBoth,"Icon &and text",0,"",true,mBook->GetTabMode()==_BookTitleAndImage);
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->BeginMenu("&Slider render style",true);
		{
			this->GetSubject()->AddMenuItem(_MenuStyleRenderImmediate,"&Immediate render",0,"",true,iggWidgetKeyHandlerBase::GetGlobalRenderMode()==_RenderModeImmediate);
			this->GetSubject()->AddMenuItem(_MenuStyleRenderDelayed,"&Delayed render",0,"",true,iggWidgetKeyHandlerBase::GetGlobalRenderMode()==_RenderModeDelayed);
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->AddMenuItem(_MenuStyleRenderResetAll,"&Reset all sliders to global mode",0,"",false,false);

		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->BeginMenu("&Advanced",false);
		{
			this->GetSubject()->AddMenuItem(_MenuStyleDockWindow,"&Dock windows",0,"",true,mDocked);
			this->GetSubject()->AddMenuItem(_MenuStyleSlidersEditable,"&Sliders are editable",0,"",true,iggWidgetSlider::IsEditableByDefault());
			this->GetSubject()->AddMenuItem(_MenuStyleIsIdiosyncratic,"&Ask for confirmations",0,"",true,mIdiosyncratic);
			this->GetSubject()->AddMenuItem(_MenuStyleAttachWindows,"&Move windows together",0,"",true,mAttachWindows);
			this->GetSubject()->AddMenuItem(_MenuStyleDetailLoadedDataInfo,"&Print detail info for loaded data",0,"",true,mDetailLoadedDataInfo);
			this->GetSubject()->AddMenuItem(_MenuStyleAutoRender,"&Render automatically",0,"",true,mAutoRender);
		}
		this->GetSubject()->EndMenu();
		this->GetSubject()->BeginMenu("The&me",true);
		{
			int i;
			for(i=0; i<mThemeList.Size(); i++) this->GetSubject()->AddMenuItem(_MenuStyleTheme+i,mThemeList[i],0,"",true,false);
		}
		this->GetSubject()->EndMenu();
	}
	this->GetSubject()->EndMenu();
	//
	//  4. Options menu
	//
	this->GetSubject()->BeginMenu("&Options",false);
	{
		this->GetSubject()->AddMenuItem(_MenuOptionAntialiasing,"Use antialiasing",0,"Shift+Ctrl+Z",true,false,0,false,&iViewModule::KeyAntialiasing());

		this->GetSubject()->BeginMenu("&Bounding box style",true);
		{
			this->GetSubject()->AddMenuItem(_MenuOptionIfritBox,"&Ifrit style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeDefault);
			this->GetSubject()->AddMenuItem(_MenuOptionClassicBox,"&Classic style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeClassic);
			this->GetSubject()->AddMenuItem(_MenuOptionHairBox,"&Hair-thin style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeHairThin);
			this->GetSubject()->AddMenuItem(_MenuOptionAxesBox,"&Axes style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeAxes);
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->BeginMenu("&Fonts",false);
		{
			this->GetSubject()->BeginMenu("Font &type",true);
			{
				this->GetSubject()->AddMenuItem(_MenuOptionFontTypeVector,"&Vector",0,"",true,false,0,false,&iViewModule::KeyFontType(),_TextTypeVector);
				this->GetSubject()->AddMenuItem(_MenuOptionFontTypeArial,"&Arial",0,"",true,false,0,false,&iViewModule::KeyFontType(),_TextTypeBitmapArial);
				this->GetSubject()->AddMenuItem(_MenuOptionFontTypeCourier,"&Courier",0,"",true,false,0,false,&iViewModule::KeyFontType(),_TextTypeBitmapCourier);
				this->GetSubject()->AddMenuItem(_MenuOptionFontTypeTimes,"&Times",0,"",true,false,0,false,&iViewModule::KeyFontType(),_TextTypeBitmapTimes);
			}
			this->GetSubject()->EndMenu();

			this->GetSubject()->BeginMenu("Font &size",true);
			{
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizem5," 20%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),-5);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizem4," 30%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),-4);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizem3," 40%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),-3);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizem2," 60%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),-2);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizem1," 75%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),-1);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizec0,"100%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),0);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizep1,"130%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),+1);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizep2,"180%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),+2);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizep3,"250%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),+3);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizep4,"330%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),+4);
				this->GetSubject()->AddMenuItem(_MenuOptionFontSizep5,"450%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),+5);
			}
			this->GetSubject()->EndMenu();
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->BeginMenu("&Images",false);
		{
			this->GetSubject()->BeginMenu("Image &format",true);
			{
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatPNG,"PNG (&Portable Network Graphics)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatPNG);
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatJPG,"JPG (&Joint Photographic Experts Group)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatJPG);
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatPPM,"PPM (Portable Pi&xmap)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatPNM);
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatBMP,"BMP (Window &Bitmap)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatBMP);
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatTIF,"TIF (&Tag Image File Format)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatTIF);
				this->GetSubject()->AddMenuItem(_MenuOptionImageFormatEPS,"EPS (&Encapsulated PostScript)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),_ImageFormatEPS);

				this->GetSubject()->BeginMenu("  Postscript paper format",true);
				{
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA0,"A0",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),0);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA1,"A1",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),1);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA2,"A2",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),2);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA3,"A3",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),3);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA4,"A4",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),4);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA5,"A5",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),5);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA6,"A6",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),6);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA7,"A7",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),7);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatA8,"A8",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),8);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatL1,"Letter",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),9);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptPaperFormatL2,"10x17",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),10);
				}
				this->GetSubject()->EndMenu();

				this->GetSubject()->BeginMenu("  Postscript orientation",true);
				{
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptOrientationPortrait,"Portrait",0,"",true,false,0,false,&iViewModule::KeyPostScriptOrientation(),0);
					this->GetSubject()->AddMenuItem(_MenuOptionPostScriptOrientationLandscape,"Landscape",0,"",true,false,0,false,&iViewModule::KeyPostScriptOrientation(),1);
				}
				this->GetSubject()->EndMenu();
			}
			this->GetSubject()->EndMenu();

			this->GetSubject()->BeginMenu("Image &zoom",true);
			{
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom001,"x 1",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),1);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom002,"x 2",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),2);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom003,"x 3",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),3);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom004,"x 4",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),4);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom005,"x 5",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),5);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom006,"x 6",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),6);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom008,"x 8",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),8);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom010,"x 10",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),10);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom015,"x 15",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),15);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom020,"x 20",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),20);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom030,"x 30",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),30);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom040,"x 40",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),40);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom050,"x 50",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),50);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom060,"x 60",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),60);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom080,"x 80",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),80);
				this->GetSubject()->AddMenuItem(_MenuOptionImageZoom100,"x 100",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),100);
			}
			this->GetSubject()->EndMenu();
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->AddMenuItem(_MenuOptionSettingsGlobal,"Options apply to all windows",0,"",true,mOptionsAreGlobal);
	}
	this->GetSubject()->EndMenu();
	//
	//  5. Help menu
	//
	this->GetSubject()->BeginMenu("&Help",false);
	{
		this->GetSubject()->AddMenuItem(_MenuHelpContents,"Contents",0,"F1",false,false);
#if defined(I_DEBUG)
		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->AddMenuItem(_MenuHelpMax-1,"Check Blocking",0,"",false,false);
		this->GetSubject()->AddMenuItem(_MenuHelpMax,"Debug Helper",0,"",false,false);
#endif
		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->AddMenuItem(_MenuHelpAbout,"About IFrIT",0,"",false,false);
	}
	this->GetSubject()->EndMenu();
	this->GetSubject()->CompleteMenu();

	//
	//  Build a tool bar
	//
	if(mAllowPrePopulateToolBar)
	{
		mExtensionWindow->PrePopulateFileToolBar();
		this->GetSubject()->AddToolBarSeparator();
	}
	this->GetSubject()->AddToolBarButton(_MenuFileOpenUniformScalars,"Open UniformScalars data file");
	this->GetSubject()->AddToolBarButton(_MenuFileOpenBasicParticles,"Open BasicParticles data file");
	this->GetSubject()->AddToolBarButton(_MenuFileOpenUniformVectors,"Open UniformVectors data file");
	this->GetSubject()->AddToolBarButton(_MenuFileOpenUniformTensors,"Open UniformTensors data file");
	mExtensionWindow->PopulateFileToolBar(mAllowPrePopulateToolBar);
	this->GetSubject()->AddToolBarSeparator();
	this->GetSubject()->AddToolBarButton(_ToolBarShowSurface,"Show surface",iImageFactory::FindIcon("surf.png"),true,&iSurfaceViewSubject::KeyReady(),false,&iSurfaceViewSubject::KeyVisible());
	this->GetSubject()->AddToolBarButton(_ToolBarShowCrossSection,"Show cross section",iImageFactory::FindIcon("xsec.png"),true,&iCrossSectionViewSubject::KeyReady(),false,&iCrossSectionViewSubject::KeyVisible());
	this->GetSubject()->AddToolBarButton(_ToolBarShowVolume,"Show volume",iImageFactory::FindIcon("volv.png"),true,&iVolumeViewSubject::KeyReady(),false,&iVolumeViewSubject::KeyVisible());
	this->GetSubject()->AddToolBarButton(_ToolBarShowParticles,"Show particles",iImageFactory::FindIcon("part.png"),true,&iParticlesViewSubject::KeyReady(),false,&iParticlesViewSubject::KeyVisible());
	this->GetSubject()->AddToolBarButton(_ToolBarShowVectorField,"Show vector field",iImageFactory::FindIcon("vect.png"),true,&iVectorFieldViewSubject::KeyReady(),false,&iVectorFieldViewSubject::KeyVisible());
	this->GetSubject()->AddToolBarButton(_ToolBarShowTensorField,"Show tensor field",iImageFactory::FindIcon("tens.png"),true,&iTensorFieldViewSubject::KeyReady(),false,&iTensorFieldViewSubject::KeyVisible());
	mExtensionWindow->PopulateShowToolBar();
	this->GetSubject()->AddToolBarSeparator();
	this->GetSubject()->AddToolBarButton(_ToolBarOpenWindowsPage,"Switch to page with multiple windows controls",iImageFactory::FindIcon("wins.png"),false);
	this->GetSubject()->AddToolBarButton(_ToolBarMinimizeWindowsPage,"Minimize all windows",iImageFactory::FindIcon("minimize.png"),false);
	this->GetSubject()->AddToolBarButton(_ToolBarMaximizeWindowsPage,"Restore all windows to normal size",iImageFactory::FindIcon("maximize.png"),false);

	this->GetSubject()->AddToolBarSeparator();
	if(mDialogPickerWindow != 0) this->GetSubject()->AddToolBarButton(_MenuDialogPickerWindow,"Open picker window");
	if(mDialogFileSetExplorer != 0) this->GetSubject()->AddToolBarButton(_MenuDialogFileSetExplorer,"Open file set explorer window");
	if(mDialogDataExplorer != 0) this->GetSubject()->AddToolBarButton(_MenuDialogDataExplorer,"Open data explorer window");
	if(mDialogImageComposer != 0) this->GetSubject()->AddToolBarButton(_MenuDialogImageComposer,"Open image composer window");
}


void iggMainWindow::OnMenuBody(int id, bool on)
{
	iString fname;

	if(id < 0)
	{
		IERROR_LOW("Invalid menu item id.");
	}

	if(id <= _MenuFileMax)
	{
		switch(id)
		{
		case _MenuFileOpenUniformScalars:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iUniformScalarsDataSubject::DataType());
				if(ws.IsEmpty()) ws = iUniformScalarsDataSubject::DataType().GetEnvironment(this->GetShell());
				fname = this->GetFileName("Choose a file",ws,"Scalar field files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iUniformScalarsDataSubject::DataType(),fname);
				break;
			}
		case _MenuFileOpenUniformVectors:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iUniformVectorsDataSubject::DataType());
				if(ws.IsEmpty()) ws = iUniformVectorsDataSubject::DataType().GetEnvironment(this->GetShell());
				fname = this->GetFileName("Choose a file",ws,"Vector field files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iUniformVectorsDataSubject::DataType(),fname);
				break;
			}
		case _MenuFileOpenUniformTensors:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iUniformTensorsDataSubject::DataType());
				if(ws.IsEmpty()) ws = iUniformTensorsDataSubject::DataType().GetEnvironment(this->GetShell());
				fname = this->GetFileName("Choose a file",ws,"Tensor field files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iUniformTensorsDataSubject::DataType(),fname);
				break;
			}
		case _MenuFileOpenBasicParticles:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iBasicParticlesDataSubject::DataType());
				if(ws.IsEmpty()) ws = iBasicParticlesDataSubject::DataType().GetEnvironment(this->GetShell());
				fname = this->GetFileName("Choose a file",ws,"Particle set files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iBasicParticlesDataSubject::DataType(),fname);
				break;
			}
		case _MenuFileSaveState:
			{
				if(mStateFileName.IsEmpty()) mStateFileName = this->GetFileName("Choose a file",this->GetShell()->GetEnvironment(_EnvironmentBase),"IFrIT state files (*.ini)",false);
				if(!mStateFileName.IsEmpty())
				{
					bool ok = true;
					if(iFile::IsReadable(mStateFileName))
					{
						if(this->PopupWindow("Do you want to overwrite the existing file?",iggParameter::_PopupWindowWarning,"Ok","Cancel") == 1) ok = false;
					}
					if(ok)
					{
						if(!this->GetShell()->GetControlModule()->SaveStateToFile(mStateFileName)) this->PopupWindow("Saving options failed for unknown reason.\n The current state will not be saved",_PopupWindowError);
					}
				}
				return;
			}
		case _MenuFileSaveStateAs:
			{
				fname = this->GetFileName("Choose a file",mStateFileName,"IFrIT state files (*.ini)",false);
				if(!fname.IsEmpty())
				{
					mStateFileName = fname;
					if(!this->GetShell()->GetControlModule()->SaveStateToFile(mStateFileName)) this->PopupWindow("Saving options failed for unknown reason.\n The current state will not be saved.",_PopupWindowError);
				}
				return;
			}
		case _MenuFileLoadStateFrom:
			{
				fname = this->GetFileName("Choose a file",mStateFileName,"IFrIT state files (*.ini)");
				if(!fname.IsEmpty())
				{
					if(this->GetShell()->GetControlModule()->LoadStateFromFile(fname))
					{
						this->UpdateAll();
						this->GetShell()->GetControlModule()->Render(_RenderOptionAll);
					}
					else
					{
						this->PopupWindow("Loading options failed for unknown reason.",_PopupWindowError);
					}
				}
				return;
			}
		case _MenuFileExit:
			{
				if(this->IsExitAllowed()) this->GetShell()->Exit();
				break;
			}
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _MenuDialogMax)
	{
		switch(id)
		{
		case _MenuDialogScriptDebugger:
			{
				if(mDialogScriptDebugger != 0) mDialogScriptDebugger->Show(true);
				break;
			}
		case _MenuDialogPaletteEditor:
			{
				if(mDialogPaletteEditor != 0) mDialogPaletteEditor->Show(true);
				break;
			}
		case _MenuDialogPickerWindow:
			{
				if(mDialogPickerWindow != 0) mDialogPickerWindow->Show(true);
				break;
			}
		case _MenuDialogEventRecorder:
			{
				if(mDialogEventRecorder != 0) mDialogEventRecorder->Show(true);
				break;
			}
		case _MenuDialogParallelController:
			{
				if(mDialogParallelController != 0) mDialogParallelController->Show(true);
				break;
			}
		case _MenuDialogPerformanceMeter:
			{
				if(mDialogPerformanceMeter != 0) mDialogPerformanceMeter->Show(true);
				break;
			}
		case _MenuDialogFileSetExplorer:
			{
				if(mDialogFileSetExplorer != 0) mDialogFileSetExplorer->Show(true);
				break;
			}
		case _MenuDialogDataExplorer:
			{
				if(mDialogDataExplorer != 0) mDialogDataExplorer->Show(true);
				break;
			}
		case _MenuDialogImageComposer:
			{
				if(mDialogImageComposer != 0) mDialogImageComposer->Show(true);
				break;
			}
		case _MenuDialogCommandLine:
			{
				if(mDialogCommandLine != 0) mDialogCommandLine->Show(true);
				break;
			}
		case _MenuDialogAutoBlock:
			{
				iggDialogAuto::mBlockShowing = on;
				break;
			}
		default:
			{
				if(id>=_MenuDialogAutoBegin && id<_MenuDialogAutoBegin+mAutoDialogList.Size())
				{
					mAutoDialogList[id-_MenuDialogAutoBegin]->ForceShow(true);
				}
				else IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _MenuStyleMax)
	{
		switch(id)
		{
		case _MenuStyleDockWindow:
			{
				this->DockWindows(on,true);
				break;
			}
		case _MenuStyleInteractorHelp:
			{
				mInteractorHelp = on;
				break;
			}
		case _MenuStyleToolTips:
			{
				this->ShowToolTips(on);
				break;
			}
		case _MenuStyleTabText:
			{
				if(on) mBook->SetTabMode(_BookTitleOnly);
				break;
			}
		case _MenuStyleTabIcon:
			{
				if(on) mBook->SetTabMode(_BookImageOnly);
				break;
			}
		case _MenuStyleTabBoth:
			{
				if(on) mBook->SetTabMode(_BookTitleAndImage);
				break;
			}
		case _MenuStyleRenderImmediate:
			{
				if(on)
				{
					iggWidgetKeyHandlerBase::SetGlobalRenderMode(_RenderModeImmediate);
					iggWidgetRenderModeButton::UpdateAll();
				}
				break;
			}
		case _MenuStyleRenderDelayed:
			{
				if(on)
				{
					iggWidgetKeyHandlerBase::SetGlobalRenderMode(_RenderModeDelayed);
					iggWidgetRenderModeButton::UpdateAll();
				}
				break;
			}
		case _MenuStyleRenderResetAll:
			{
				iggWidgetRenderModeButton::ResetAllToGlobal();
				break;
			}
		case _MenuStyleSlidersEditable:
			{
				iggWidgetSlider::SetAllEditable(on);
				break;
			}
		case _MenuStyleIsIdiosyncratic:
			{
				mIdiosyncratic = on;
				break;
			}
		case _MenuStyleAttachWindows:
			{
				mAttachWindows = on;
				break;
			}
		case _MenuStyleDetailLoadedDataInfo:
			{
				mDetailLoadedDataInfo = on;
				break;
			}
		case _MenuStyleAutoRender:
			{
				mAutoRender = on;
				mRenderButton->Show(!on);
				break;
			}
		default:
			{
				if(id>=_MenuStyleTheme && id<_MenuStyleTheme+mThemeList.Size())
				{
					if(on) this->SetTheme(id-_MenuStyleTheme);
				}
				else IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _MenuOptionMax)
	{
		iString ws;

		int option = _ObjectOptionOne | _RenderOptionAuto;
		if(mOptionsAreGlobal)
		{
			option |= _ModuleOptionAll;
		}
		else
		{
			option |= _ModuleOptionOne;
		}

		switch(id)
		{
		case _MenuOptionAntialiasing:
			{
				this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyAntialiasing(),on);
				break;
			}
		case _MenuOptionIfritBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeDefault);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(false);
				}
				break;
			}
		case _MenuOptionClassicBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeClassic);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(false);
				}
				break;
			}
		case _MenuOptionHairBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeHairThin);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(false);
				}
				break;
			}
		case _MenuOptionAxesBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),_BoundingBoxTypeAxes);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(true);
				}
				break;
			}
		case _MenuOptionFontTypeVector:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontType(),_TextTypeVector);
				break;
			}
		case _MenuOptionFontTypeArial:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontType(),_TextTypeBitmapArial);
				break;
			}
		case _MenuOptionFontTypeCourier:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontType(),_TextTypeBitmapCourier);
				break;
			}
		case _MenuOptionFontTypeTimes:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontType(),_TextTypeBitmapTimes);
				break;
			}
		case _MenuOptionFontSizem5:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),-5);
				break;
			}
		case _MenuOptionFontSizem4:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),-4);
				break;
			}
		case _MenuOptionFontSizem3:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),-3);
				break;
			}
		case _MenuOptionFontSizem2:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),-2);
				break;
			}
		case _MenuOptionFontSizem1:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),-1);
				break;
			}
		case _MenuOptionFontSizec0:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),0);
				break;
			}
		case _MenuOptionFontSizep1:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),1);
				break;
			}
		case _MenuOptionFontSizep2:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),2);
				break;
			}
		case _MenuOptionFontSizep3:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),3);
				break;
			}
		case _MenuOptionFontSizep4:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),4);
				break;
			}
		case _MenuOptionFontSizep5:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),5);
				break;
			}
		case _MenuOptionImageFormatPNG:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatPNG);
				break;
			}
		case _MenuOptionImageFormatJPG:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatJPG);
				break;
			}
		case _MenuOptionImageFormatPPM:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatPNM);
				break;
			}
		case _MenuOptionImageFormatBMP:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatBMP);
				break;
			}
		case _MenuOptionImageFormatTIF:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatTIF);
				break;
			}
		case _MenuOptionImageFormatEPS:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),_ImageFormatEPS);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA0:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),0);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA1:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),1);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA2:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),2);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA3:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),3);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA4:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),4);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA5:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),5);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA6:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),6);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA7:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),7);
				break;
			}
		case _MenuOptionPostScriptPaperFormatA8:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),8);
				break;
			}
		case _MenuOptionPostScriptPaperFormatL1:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),9);
				break;
			}
		case _MenuOptionPostScriptPaperFormatL2:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),10);
				break;
			}
		case _MenuOptionPostScriptOrientationPortrait:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptOrientation(),0);
				break;
			}
		case _MenuOptionPostScriptOrientationLandscape:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptOrientation(),1);
				break;
			}
		case _MenuOptionImageZoom001:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),1);
				break;
			}
		case _MenuOptionImageZoom002:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),2);
				break;
			}
		case _MenuOptionImageZoom003:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),3);
				break;
			}
		case _MenuOptionImageZoom004:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),4);
				break;
			}
		case _MenuOptionImageZoom005:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),5);
				break;
			}
		case _MenuOptionImageZoom006:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),6);
				break;
			}
		case _MenuOptionImageZoom008:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),8);
				break;
			}
		case _MenuOptionImageZoom010:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),10);
				break;
			}
		case _MenuOptionImageZoom015:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),15);
				break;
			}
		case _MenuOptionImageZoom020:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),20);
				break;
			}
		case _MenuOptionImageZoom030:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),30);
				break;
			}
		case _MenuOptionImageZoom040:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),40);
				break;
			}
		case _MenuOptionImageZoom050:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),50);
				break;
			}
		case _MenuOptionImageZoom060:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),60);
				break;
			}
		case _MenuOptionImageZoom080:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),80);
				break;
			}
		case _MenuOptionImageZoom100:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),100);
				break;
			}
		case _MenuOptionSettingsGlobal:
			{
				mOptionsAreGlobal = on;
				return;
			}
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		if(!ws.IsEmpty()) this->GetShell()->GetControlModule()->Execute(ws,mAutoRender,option);
		return;
	}

	if(id <= _MenuHelpMax)
	{
		switch(id)
		{
		case _MenuHelpContents:
			{
				if(mDialogHelp != 0) mDialogHelp->Show(true);
				break;
			}
		case _MenuHelpAbout:
			{
				if(mDialogAbout != 0) mDialogAbout->Show(true);
				break;
			}
#if defined(I_DEBUG)
		case _MenuHelpMax-1:
			{
				if(mBlockLevel != 1)
				{
					this->PopupWindow(mGlobalFrame,"Nonzero block level: "+iString::FromNumber(mBlockLevel-1),_PopupWindowError);
					mBlockLevel = 1;
				}
				break;
			}
		case _MenuHelpMax:
			{
				if(mDialogDebugHelper != 0) mDialogDebugHelper->Show(true);
				break;
			}
#endif
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= _ToolBarMax)
	{
		switch(id)
		{
		case _ToolBarOpenWindowsPage:
			{
				mBook->OpenPage(0);
				mViewPage->ShowPage(4);
				mViewPage->GetWindowListDialog()->Show(true);
				break;
			}
		case _ToolBarMinimizeWindowsPage:
			{
				this->DisplayWindowsAsIcons();
				this->GetSubject()->GetHelper()->ShowAsIcon();
				break;
			}
		case _ToolBarMaximizeWindowsPage:
			{
				this->DisplayWindowsAsWindows();
				break;
			}
		case _ToolBarShowSurface:
			{
				this->GetShell()->GetControlModule()->Show(iSurfaceViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(1);
					mExtensionWindow->OpenBookPageByIndex(1);
					mBook->GetPage(1)->UpdateWidget();
				}
				break;
			}
		case _ToolBarShowCrossSection:
			{
				this->GetShell()->GetControlModule()->Show(iCrossSectionViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(2);
					mExtensionWindow->OpenBookPageByIndex(2);
					mBook->GetPage(2)->UpdateWidget();
				}
				break;
			}
		case _ToolBarShowVolume:
			{
				this->GetShell()->GetControlModule()->Show(iVolumeViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(3);
					mExtensionWindow->OpenBookPageByIndex(3);
					mBook->GetPage(3)->UpdateWidget();
				}
				break;
			}
		case _ToolBarShowParticles:
			{
				this->GetShell()->GetControlModule()->Show(iParticlesViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(4);
					mExtensionWindow->OpenBookPageByIndex(4);
					mBook->GetPage(4)->UpdateWidget();
				}
				break;
			}
		case _ToolBarShowVectorField:
			{
				this->GetShell()->GetControlModule()->Show(iVectorFieldViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(5);
					mExtensionWindow->OpenBookPageByIndex(5);
					mBook->GetPage(5)->UpdateWidget();
				}
				break;
			}
		case _ToolBarShowTensorField:
			{
				this->GetShell()->GetControlModule()->Show(iTensorFieldViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(6);
					mExtensionWindow->OpenBookPageByIndex(6);
					mBook->GetPage(6)->UpdateWidget();
				}
				break;
			}
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(!mExtensionWindow->OnMenuBody(id,on))
	{
		IERROR_LOW("Invalid menu item id.");
	}
}

#endif
