/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggdialogdataexplorer.h"


#include "icontrolmodule.h"
#include "idata.h"
#include "idataexplorer.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "ihistogram.h"
#include "iimagefactory.h"
#include "ishell.h"
#include "iviewmodule.h"

#include "iggdatatypeprovider.h"
#include "iggframedatatypeselector.h"
#include "iggframedoublebutton.h"
#include "iggframehistogramareaextra.h"
#include "iggmainwindow.h"
#include "iggwidgetarea.h"
#include "iggwidgetmisc.h"
#include "iggwidgetotherbutton.h"
#include "iggwidgettext.h"

#include "ibgwidgetareasubject.h"
#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetcolorselectionsubject.h"
#include "ibgwidgetentrysubject.h"
#include "ibgwidgetselectionboxsubject.h"
#include "ibgwidgettexteditorsubject.h"

#include "iggsubjectfactory.h"
#include "iggparameter.h"
using namespace iggParameter;

#include <limits.h>

//
//  templates
//
#include "iarraytemplate.h"


namespace iggDialogDataExplorer_Private
{
	struct Variable
	{
		iString Name;
		iViewModule *View;
		iDataInfo Data;
		iDataExplorer *Explorer;
		iColor Color;
		float Opacity;
		Variable()
		{
			View = 0;
			Explorer = 0;
			Opacity = 0.0f;
		}
	};
	//
	//  A small dialog for setting variable's color and opacity, and its widgets
	//
	class VariablePropertyDialog;

	class VariablePropertyDialogOpacitySlider : public iggWidget
	{

	public:

		VariablePropertyDialogOpacitySlider(VariablePropertyDialog *dialog, iggFrame *parent) : iggWidget(parent)
		{
			mDialog = dialog;
			mSubject = iggSubjectFactory::CreateWidgetEntrySubject(this,true,0,"Opacity",0);
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void UpdateWidgetBody();
		virtual void OnInt1Body(int i);

		VariablePropertyDialog *mDialog;
		ibgWidgetEntrySubject *mSubject;
	};

	class VariablePropertyDialogColorSelection : public iggWidget
	{

	public:

		VariablePropertyDialogColorSelection(VariablePropertyDialog *dialog, iggFrame *parent) : iggWidget(parent)
		{
			mDialog = dialog;
			mSubject = iggSubjectFactory::CreateWidgetColorSelectionSubject(this,true);
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void UpdateWidgetBody();
		virtual void OnVoid1Body();

		VariablePropertyDialog *mDialog;
		ibgWidgetColorSelectionSubject *mSubject;
	};

	class VariablePropertyDialog : public iggDialog
	{

	public:

		VariablePropertyDialog(iggWidgetListView *list, iggWidgetHistogramArea *area, iggDialog *parent) : iggDialog(parent,_DialogModal,0,"Variable Properties",0,1)
		{
			mList = list;
			mArea = area;
			mVariable = 0;

			mFrame->AddLine(new VariablePropertyDialogOpacitySlider(this,mFrame));
			mFrame->AddLine(new VariablePropertyDialogColorSelection(this,mFrame));
		}

		void SetChanged(bool color)
		{
			mArea->UpdateWidget();
			if(color)
			{
				mList->Clear();
				mList->UpdateWidget();
			}
		}

		void AttachVariable(Variable *v)
		{
			mVariable = v;
		}

		Variable* GetVariable() const
		{
			return mVariable;
		}

	protected:

		Variable *mVariable;
		iggWidgetListView *mList;
		iggWidgetHistogramArea *mArea;
	};

	void VariablePropertyDialogOpacitySlider::UpdateWidgetBody()
	{
		if(mDialog->GetVariable() != 0) mSubject->SetValue(round(20.0*mDialog->GetVariable()->Opacity));
	}

	void VariablePropertyDialogOpacitySlider::OnInt1Body(int)
	{
		if(mDialog->GetVariable() != 0)
		{
			mDialog->GetVariable()->Opacity = 0.05*mSubject->GetValue();
			mDialog->SetChanged(false);
		}
	}

	void VariablePropertyDialogColorSelection::UpdateWidgetBody()
	{
		if(mDialog->GetVariable() != 0) mSubject->SetColor(mDialog->GetVariable()->Color);
	}

	void VariablePropertyDialogColorSelection::OnVoid1Body()
	{
		if(mDialog->GetVariable() != 0)
		{
			mDialog->GetVariable()->Color = mSubject->GetColor();
			mDialog->SetChanged(true);
		}
	}
	//
	//  List of explored variables - does most of work
	//
	class ExploredVariablesList : public iggWidgetListView
	{

	public:

		ExploredVariablesList(iggDialogDataExplorer *de, iggFrame *parent) : iggWidgetListView(true,parent)
		{
			mDataExplorer = de;
			mMaxNameLength = -1;

			//
			//  Set colors for first levels like in a pseudo-raibow palette
			//
			mDefaultColor[0] = iColor(255,  0,  0);
			mDefaultColor[1] = iColor(  0,  0,255);
			mDefaultColor[2] = iColor(  0,255,  0);
			mDefaultColor[3] = iColor(255,255,  0);
			mDefaultColor[4] = iColor(255,  0,255);
			mDefaultColor[5] = iColor(  0,255,255);

			mPropertiesDialog = 0;
		}

		~ExploredVariablesList()
		{
			delete mPropertiesDialog;
            while(mVariables.Size() > 0) this->RemoveVariable(0);
		}

		inline const iArray<Variable>& GetVariables() const { return mVariables; }

		void AddVariable(int n)
		{
			iDataReader *dr = this->GetShell()->GetControlModule()->GetViewModule()->GetReader();
			iDataLimits *l = dr->GetLimits(mDataExplorer->GetProvider()->GetActiveDataType());

			if(l!=0 && n>=0 && n<l->GetNumVars())
			{
				Variable v;

				v.Name = l->GetName(n);
				v.View = this->GetShell()->GetControlModule()->GetViewModule();
				v.Data = l->GetDataType();

				//
				//  Check for duplicates
				//
				int i;
				for(i=0; i<mVariables.Size(); i++)
				{
					if(mVariables[i].Name==v.Name && mVariables[i].View==v.View && mVariables[i].Data.Type(0)==v.Data.Type(0))
					{
						if(this->GetMainWindow()->PopupWindow(mDataExplorer->GetFrame(),"This variables is already on the list!",_PopupWindowMessage,"Cancel","Add anyway") == 0) return; else break;
					}
				}

				v.Explorer = iDataExplorer::New(v.View);
				if(v.Data.Count() > 0) v.Explorer->SetActiveDataType(v.Data.Type(0));
				v.Explorer->SetInputComponent(n);
				v.Explorer->SetStretch(mDataExplorer->GetStretch());

				if(mVariables.Size() < 6)
				{
					v.Color = mDefaultColor[mVariables.Size()];
				}
				else
				{
					v.Color = mDefaultColor[0];
				}
				v.Opacity = 0.5;

				mVariables.Add(v);
				if(v.Name.Length() > mMaxNameLength)
				{
					this->UpdateWidget();
				}
				else
				{
					this->InsertFormattedEntry(mVariables.MaxIndex());
				}
				this->UpdateDependents();
			}
		}

		void RemoveVariable(int n)
		{
			if(n>=0 && n<mVariables.Size())
			{
				mVariables[n].Explorer->Delete();
				mVariables.Remove(n);
			}
		}

		virtual void Select(int paraFrom, int paraTo)
		{
			iggWidgetListView::Select(paraFrom,paraTo);
			this->UpdateDependents();
		}

		void CreatePropertiesDialog(iggWidgetHistogramArea *area)
		{
			mPropertiesDialog = new VariablePropertyDialog(this,area,mDataExplorer);
		}

		void Modify(int type)
		{
			static Variable *buffer = 0;
			static int length = 0;
			int i;
			Variable v;

			if(!this->HasSelection()) return;

			switch(type)
			{
			case 0:
				{
					if(mSelectedRange[0] == 0) return; // already there

					int n = mSelectedRange[1] - mSelectedRange[0] + 1;
					if(n > length)
					{
						//
						//  Extend the buffer
						//
						Variable *tmp = new Variable[n];
						if(length > 0)
						{
							memcpy(tmp,buffer,length*sizeof(Variable));
							delete [] buffer;
						}
						buffer = tmp;
						length = n;
					}
					for(i=0; i<n; i++) buffer[i] = mVariables[mSelectedRange[0]+i];
					for(i=mSelectedRange[0]-1; i>=0; i--) mVariables[i+n] = mVariables[i];
					for(i=0; i<n; i++) mVariables[i] = buffer[i];
					break;
				}
			case 1:
				{
					if(mSelectedRange[0] == 0) return; // already there

					v = mVariables[mSelectedRange[0]-1];
					for(i=mSelectedRange[0]; i<=mSelectedRange[1]; i++)
					{
						mVariables[i-1] = mVariables[i];
					}
					mVariables[mSelectedRange[1]] = v;
					break;
				}
			case 2:
				{
					if(mSelectedRange[1] == mVariables.MaxIndex()) return; // already there

					v = mVariables[mSelectedRange[1]+1];
					for(i=mSelectedRange[1]; i>=mSelectedRange[0]; i--)
					{
						mVariables[i+1] = mVariables[i];
					}
					mVariables[mSelectedRange[0]] = v;
					break;
				}
			case 3:
				{
					int n = mSelectedRange[1] - mSelectedRange[0] + 1;
					for(i=0; i<n; i++) this->RemoveVariable(mSelectedRange[0]);
					break;
				}
			}
			this->UpdateWidget();
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			bool ok;
			int i, j, wn, n = mVariables.Size();
			iControlModule *cm = this->GetShell()->GetControlModule();

			mMaxNameLength = -1;
			bool redo = true;
			if(n == this->Count())
			{
				redo = false;
				iString ws;
				for(i=0; !redo && i<n; i++)
				{
					ws = this->GetItem(i);
					ws.ReduceWhiteSpace();
					if(mVariables[i].Name != ws.Section(" ",0,0))
					{
						redo = true;
					}

					j = ws.Find("window#");
					wn = -1;
					if(j > 0)
					{
						ws = ws.Part(j+7);
						if(mSubject->SupportsHTML())
						{
							//
							//  remove all tags
							//
							ws.RemoveFormatting("<",">");
						}
						ws.ReduceWhiteSpace();
						wn = ws.Section(" ",0,0).ToInt(ok) - 1;
#ifdef I_CHECK1
						if(!ok)
						{
							IERROR_REPORT_BUG;
						}
#endif
					}
#ifdef I_CHECK1
					else
					{
						IERROR_REPORT_BUG;
					}
#endif
					if(wn<0 || cm->GetViewModule(wn)!=mVariables[i].View)
					{
						redo = true;
					}
				}
			}

			if(redo)
			{
				//
				//  remove non-existent VMs
				//
				n = cm->GetNumberOfViewModules();
				for(i=0; i<mVariables.Size(); i++) // the upper limit is variable!!!
				{
					ok = false;
					for(j=0; !ok && j<n; j++) if(cm->GetViewModule(j) == mVariables[i].View) ok = true;
					if(!ok)
					{
						this->RemoveVariable(i--);
					}
				}

				this->Clear();
				n = mVariables.Size();
				for(i=0; i<n; i++)
				{
					this->InsertFormattedEntry(i);
				}
				this->UpdateDependents();
			}

			this->iggWidgetListView::UpdateWidgetBody();
		}

		void InsertFormattedEntry(int n)
		{
			if(!mSubject->SupportsHTML()) mSubject->SetTextColor(mVariables[n].Color);
			this->InsertItem(this->GetFormattedEntry(n));
		}

		iString GetFormattedEntry(int n)
		{
			static iString dummy;

			if(mMaxNameLength < 0)
			{
				int i, n = mVariables.Size();
				for(i=0; i<n; i++)
				{
					if(mVariables[i].Name.Length() > mMaxNameLength) mMaxNameLength = mVariables[i].Name.Length();
				}
				while(dummy.Length() < mMaxNameLength+3) dummy += ".........";
			}

			iString ws;
			if(mSubject->SupportsHTML())
			{
				ws += "<b><font color=#" + iString::FromNumber(mVariables[n].Color.Red(),"%02x") + iString::FromNumber(mVariables[n].Color.Green(),"%02x") + iString::FromNumber(mVariables[n].Color.Blue(),"%02x") + ">";
			}
			ws += mVariables[n].Name;
			if(mSubject->SupportsHTML()) ws += "</font></b>";
			ws += dummy.Part(0,3+mMaxNameLength-mVariables[n].Name.Length());
			ws += " window#";
			if(mSubject->SupportsHTML()) ws += "<b>";
			ws += iString::FromNumber(mVariables[n].View->GetWindowNumber()+1,"%2d");
			if(mSubject->SupportsHTML()) ws += "</b>";
			ws += " data: ";
			if(mSubject->SupportsHTML()) ws += "<b>";
			ws += mVariables[n].Data.Type(0).GetTextName();
			if(mSubject->SupportsHTML()) ws += "</b>";
			return ws;
		}

		virtual void OnReturnPressed()
		{
			if(this->HasSelection())
			{
				mPropertiesDialog->AttachVariable(&mVariables[mSelectedRange[0]]);
				mPropertiesDialog->Show(true);
				this->UpdateWidget();
			}
		}

		int mMaxNameLength;
		iColor mDefaultColor[9];
		iArray<Variable> mVariables;  // not pointers for cache coherence
		VariablePropertyDialog *mPropertiesDialog;
		iggDialogDataExplorer *mDataExplorer;
	};
	//
	//  Fraction info list
	//
	class FractionInfoList : public iggFrameFlip
	{

	public:

		FractionInfoList(ExploredVariablesList *list, iggFrame *parent) : iggFrameFlip(parent)
		{
			mList = list;
			mValue = 0.0;
			
			//
			//  Use double buffer
			//
			iggFrame *tmp;
			tmp = new iggFrame(this);
			mView0 = new iggWidgetTextBrowser(false,false,tmp);
			tmp->AddLine(mView0);
			this->AddLayer(tmp);
			tmp = new iggFrame(this);
			mView1 = new iggWidgetTextBrowser(false,false,tmp);
			tmp->AddLine(mView1);
			this->AddLayer(tmp);
			mView = mView1;
			this->ShowLayer(0);
		}

		void SetValue(float v)
		{
			mValue = v;
			this->UpdateWidget();
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			static iString dummy;
			//
			//  Cache a few things
			//
			const iArray<Variable> &v = mList->GetVariables();
			int i, j, stretch, n = v.Size();
			iDataExplorer::Info d;
			const iHistogram *h;
			float xv, sMin, sMax;
			double sum1, sum2;

			int maxlength = 0;
			for(i=0; i<n; i++)
			{
				if(v[i].Name.Length() > maxlength) maxlength = v[i].Name.Length();
			}
			while(dummy.Length() < maxlength+3) dummy += ".........";

			mView->Clear();
			mView->AppendTextLine("Value",iString::FromNumber(mValue));
			for(i=0; i<n; i++)
			{
				d = v[i].Explorer->GetInfo();
				h = d.Histogram;
				stretch = h->GetStretch();
				sMin = iDataStretch::ApplyStretch(d.Minimum,stretch,false);
				sMax = iDataStretch::ApplyStretch(d.Maximum,stretch,true);
				xv = (iDataStretch::ApplyStretch(mValue,stretch,false)-sMin)/(sMax-sMin+1.0e-36f);
				if(xv < 0.0)
				{
					sum1 = 0.0;
					sum2 = 1.0;
				}
				else if(xv > 1.0)
				{
					sum1 = 1.0;
					sum2 = 0.0;
				}
				else
				{
					sum1 = sum2 = 0.0;
					for(j=0; j<h->N(); j++)
					{
						if(xv < h->X(j)) sum2 += h->Y(j); else sum1 += h->Y(j); 
					}
					if(sum1+sum2 < 0.5) sum1 = sum2 = 0.5;
				}
				mView->AppendTextLine(v[i].Name+dummy.Part(0,3+maxlength-v[i].Name.Length()),"below "+iString::FromNumber(sum1/(sum1+sum2))+", above "+iString::FromNumber(sum2/(sum1+sum2)),v[i].Color);
			}
			if(mView == mView1)
			{
				mView = mView0;
				this->ShowLayer(1);
			}
			else
			{
				mView = mView1;
				this->ShowLayer(0);
			}

			this->iggFrameFlip::UpdateWidgetBody();
		}

		float mValue;
		ExploredVariablesList *mList;
		iggWidgetTextBrowser *mView, *mView0, *mView1;
	};


	//
	//  Draw area
	//
	class Area : public iggWidgetHistogramArea
	{

	public:

		Area(iggDialogDataExplorer *dialog, ExploredVariablesList *list, iggFrame *parent) : iggWidgetHistogramArea(false,true,parent)
		{
			mList = list;
			mDialog = dialog;
		
			mGlobalMin = 0.0;
			mGlobalMax = 1.0;

			list->AddDependent(this);

			mUnderInteraction = false;
			mInfoList = 0;

			mNeedsBaloonHelp = false;
		}

		void AttachInfoList(FractionInfoList *l)
		{
			mInfoList = l;
		}

		virtual void OnMousePress(int x, int y, int b)
		{
			if(b != _MouseLeftButton) return;

			if(!mUnderInteraction && mInfoList!=0)
			{
				mUnderInteraction = true;
				mPosX = x;
				mPosY = y;
				this->UpdateInfoList();
				mDialog->ShowFractionInfo(true);
				mSubject->RequestPainting(ibgWidgetDrawAreaSubject::_Foreground);
			}
		}

		virtual void OnMouseRelease(int /*x*/, int /*y*/, int b)
		{
			if(b != _MouseLeftButton) return;

			if(mUnderInteraction)
			{
				mUnderInteraction = false;
				mDialog->ShowFractionInfo(false);
				mSubject->RequestPainting(ibgWidgetDrawAreaSubject::_Foreground);
			}
		}

		virtual void OnMouseMove(int x, int y, int b)
		{
			if(b != _MouseLeftButton) return;

			if(mUnderInteraction)
			{
				mPosX = x;
				mPosY = y;
				this->UpdateInfoList();
				mSubject->RequestPainting(ibgWidgetDrawAreaSubject::_Foreground);
			}
		}

	protected:

		void UpdateInfoList()
		{
			float min = mDialog->ApplyStretch(mGlobalMin,false);
			float max = mDialog->ApplyStretch(mGlobalMax,true);
			float f = float(mPosX)/float(mSubject->Width()-1);
			if(f < 0.0f) f = 0.0f;
			if(f > 1.0f) f = 1.0f;
			mInfoList->SetValue(mDialog->ResetStretch(min+(max-min)*f));
		}

		virtual void DrawBackgroundBody()
		{
			//
			//  Cache a few things
			//
			const iArray<Variable> &v = mList->GetVariables();
			int i, n = v.Size();
			iDataExplorer::Info d;

			this->GetMainWindow()->Block(true);

			//
			//  Compute the global range first
			//
			float range[2];
			if(n > 0)
			{
				d = v[0].Explorer->GetInfo();
				mGlobalMin = d.Minimum;
				mGlobalMax = d.Maximum;
				for(i=1; i<n; i++)
				{
					d = v[i].Explorer->GetInfo();
					if(mGlobalMin > d.Minimum) mGlobalMin = d.Minimum;
					if(mGlobalMax < d.Maximum) mGlobalMax = d.Maximum;
				}
			}

			//
			//  We use the background to accumulate the image
			//  Paint in the reverse order, so that the variable on top in the
			//  list is on top in the image too
			//
			float min = mDialog->ApplyStretch(mGlobalMin,false);
			float max = mDialog->ApplyStretch(mGlobalMax,true);
			for(i=n-1; i>=0; i--)
			{
				if(i < n-1)
				{
					mSubject->Begin(ibgWidgetDrawAreaSubject::_Foreground,true);
				}
				d = v[i].Explorer->GetInfo();
				range[0] = (mDialog->ApplyStretch(d.Minimum,false)-min)/(max-min+1.0e-36f);
				range[1] = (mDialog->ApplyStretch(d.Maximum,true)-min)/(max-min+1.0e-36f);
				this->SetHistogram(d.Histogram,range);
				this->DrawHistogram(v[i].Color);
				if(i > 0) mSubject->End();
				//
				//  Blend images
				//
				if(i < n-1) mSubject->BlendForegroundOntoBackground(v[i].Opacity);
			}

			this->GetMainWindow()->Block(false);
		}

		virtual void DrawForegroundBody()
		{
			static const iColor markerColor(0,0,0);

			//
			// draw a marker line if under interaction
			//
			if(mUnderInteraction)
			{
				mSubject->DrawLine(mPosX,0,mPosX,mSubject->Height()-1,markerColor,5);
			}
		}

		virtual void GetHistogramRange(float &min, float &max) const
		{
			min = mGlobalMin;
			max = mGlobalMax;
		}

		int mPosX, mPosY;
		float mGlobalMin, mGlobalMax;
		bool mUnderInteraction;
		ExploredVariablesList *mList;
		FractionInfoList *mInfoList;
		iggDialogDataExplorer *mDialog;
	};
	//
	//  List of available variables
	//
	class AvailableVariablesList : public iggWidgetListView
	{

	public:

		AvailableVariablesList(iggDialogDataExplorer *de, ExploredVariablesList *list, iggFrameDataTypeSelector *dts, iggFrame *parent) : iggWidgetListView(false,parent)
		{
			mSelector = dts;
			mDataExplorer = de;
			mList = list;
			mAddButton = 0;
		}

		void SetButton(iggWidget *addbutton)
		{
			mAddButton = addbutton;
		}

		virtual void Select(int paraFrom, int paraTo)
		{
			iggWidgetListView::Select(paraFrom,paraTo);
			mAddButton->Enable(this->HasSelection());
		}

		void AddSelectedVariables()
		{
			int i;
			for(i=mSelectedRange[0]; i<=mSelectedRange[1] && i<this->GetNumberOfLines(); i++)
			{
				mList->AddVariable(i);
			}
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			mSelector->UpdateWidget();

			iDataReader *dr = this->GetShell()->GetControlModule()->GetViewModule()->GetReader();
			iDataLimits *l = dr->GetLimits(mDataExplorer->GetProvider()->GetActiveDataType());

			if(l!=0 && dr->IsThereData(l->GetDataType()))
			{
				int i, n = l->GetNumVars();

				bool redo = true;
				if(n == this->Count())
				{
					redo = false;
					iString ws;
					for(i=0; !redo && i<n; i++)
					{
						ws = this->GetItem(i);
						ws.ReduceWhiteSpace();
						if(l->GetName(i) != ws)
						{
							redo = true;
						}
					}
				}

				if(redo)
				{
					this->Clear();
					for(i=0; i<n; i++)
					{
						this->InsertItem(l->GetName(i));
					}
				}
			}
			else this->Clear();

			this->iggWidgetListView::UpdateWidgetBody();
		}

		virtual void OnReturnPressed()
		{
			this->AddSelectedVariables();
		}

		iggWidget *mAddButton;
		iggFrameDataTypeSelector *mSelector;
		ExploredVariablesList *mList;
		iggDialogDataExplorer *mDataExplorer;
	};


	class AddVariableButton : public iggWidgetSimpleButton
	{

	public:

		AddVariableButton(AvailableVariablesList *list, iggFrame *parent) : iggWidgetSimpleButton("",parent,true)
		{
			mList = list;
			mSubject->SetIcon(*iImageFactory::FindIcon("moveleft.png"));

			this->SetBaloonHelp("Show selected variables");
		}

		~AddVariableButton()
		{
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->Enable(mList->HasSelection());
		}

		virtual void Execute()
		{
			mList->AddSelectedVariables();
		}

		AvailableVariablesList *mList;
	};


	class ControlButton : public iggWidgetSimpleButton
	{

	public:

		ControlButton(int type, ExploredVariablesList *list, iggFrame *parent) : iggWidgetSimpleButton("",parent,true)
		{
			mType = type;
			mList = list;
			switch(type)
			{
			case 0:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("movetop.png"));
					this->SetBaloonHelp("Move the current variables to the top");
					break;
				}
			case 1:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("moveup.png"));
					this->SetBaloonHelp("Move the current variables up");
					break;
				}
			case 2:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("movedown.png"));
					this->SetBaloonHelp("Move the current variables down");
					break;
				}
			case 3:
				{
					mSubject->SetIcon(*iImageFactory::FindIcon("close.png"));
					this->SetBaloonHelp("Remove selected variables");
					break;
				}
			default:
				{
					IERROR_FATAL("Invalid type");
				}
			}
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->Enable(mList->HasSelection());
		}

		virtual void Execute()
		{
			mList->Modify(mType);
		}

		int mType;
		ExploredVariablesList *mList;
	};


	class DataComboBox : public iggWidget
	{

	public:

		DataComboBox(int type, ExploredVariablesList *list, iggFrame *parent) : iggWidget(parent)
		{
			mList = list;
			mType = type;

			switch(type)
			{
			case 0:
			case 1:
				{
					mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,"");
					break;
				}
			default:
				{
					IERROR_FATAL("Invalid type");
				}
			}
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			mSubject->Clear();
			
			if(mList->GetVariables().Size() > 0)
			{
				const iHistogram *h = mList->GetVariables()[0].Explorer->GetInfo().Histogram;
				if(h != 0)
				{
					switch(mType)
					{
					case 0:
						{
							mSubject->InsertItem(iString::FromNumber(h->GetMaxValue()));
							mSubject->InsertItem("Cell: "+iString::FromNumber(h->GetMaxCellIndex()));
							mSubject->InsertItem("at X: "+iString::FromNumber(h->GetMaxCellPosition()[0]));
							mSubject->InsertItem("at Y: "+iString::FromNumber(h->GetMaxCellPosition()[1]));
							mSubject->InsertItem("at Z: "+iString::FromNumber(h->GetMaxCellPosition()[2]));
							break;
						}
					case 1:
						{
							mSubject->InsertItem(iString::FromNumber(h->GetMinValue()));
							mSubject->InsertItem("Cell: "+iString::FromNumber(h->GetMinCellIndex()));
							mSubject->InsertItem("at X: "+iString::FromNumber(h->GetMinCellPosition()[0]));
							mSubject->InsertItem("at Y: "+iString::FromNumber(h->GetMinCellPosition()[1]));
							mSubject->InsertItem("at Z: "+iString::FromNumber(h->GetMinCellPosition()[2]));
							break;
						}
					}
				}
			}
		}

		int mType;
		ibgWidgetComboBoxSubject *mSubject;
		ExploredVariablesList *mList;
	};


	class DataDisplay : public iggWidgetNumberDisplay
	{

	public:

		DataDisplay(int type, ExploredVariablesList *list, iggFrame *parent) : iggWidgetNumberDisplay("",parent)
		{
			mList = list;
			mType = type;
			mNeedsBaloonHelp = false;
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			if(mList->GetVariables().Size() > 0)
			{
				const iDataExplorer::Info &d = mList->GetVariables()[0].Explorer->GetInfo();
				switch(mType)
				{
				case 0:
					{
						this->Display(d.Median);
						break;
					}
				case 1:
					{
						this->Display(d.Average);
						break;
					}
				case 2:
					{
						this->Display(d.Dispersion);
						break;
					}
				}
			}
			else this->Display(0.0);
		}

		int mType;
		ExploredVariablesList *mList;
	};


	class MoveFocalPointFrame : public iggFrameMoveFocalPointButton
	{
		
	public:
		
		MoveFocalPointFrame(ExploredVariablesList *list, iggFrame *parent) : iggFrameMoveFocalPointButton("Select a point",parent,true)
		{
			mBox = new iggWidgetSimpleRadioBox(2,"",this);
			mBox->InsertItem("Maximum");
			mBox->InsertItem("Minimum");
			mBox->SetBaloonHelp("Choose where to move the focal point to","This box is used to select whether the focal point in moved to the location where the top variable has the maximum or the minimum value.");
			this->AddLine(mBox,2);

			mList = list;

			this->SetBaloonHelp("Move the camera focal point or place a marker at the location of the maximum or minimum value");
		}

		virtual int GetWindowNumber() const 
		{ 
			if(mList->GetVariables().Size() > 0)
			{
				return mList->GetVariables()[0].View->GetWindowNumber();
			}
			else return -1;
		}

		virtual void GetPosition(double x[3]) const
		{
			const double *p = 0;

			if(mList->GetVariables().Size() > 0)
			{
				const Variable &v = mList->GetVariables()[0];
				switch(mBox->GetValue())
				{
				case 0:
					{
						p = v.Explorer->GetInfo().Histogram->GetMaxCellPosition();
						break;
					}
				case 1:
					{
						p = v.Explorer->GetInfo().Histogram->GetMinCellPosition();
						break;
					}
				default:
					{
						IERROR_LOW("Invalid choice.");
						p = 0;
					}
				}
			}
			if(p != 0)
			{
				x[0] = p[0];
				x[1] = p[1];
				x[2] = p[2];
			}
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			this->Enable(mList->GetVariables().Size() > 0);
			this->iggFrameMoveFocalPointButton::UpdateWidgetBody();
		}

		ExploredVariablesList *mList;
		iggWidgetSimpleRadioBox *mBox;
	};

	//
	//  Stretch combo box
	//
	class StretchComboBox : public iggWidget
	{

	public:

		StretchComboBox(iggDialogDataExplorer *de, ExploredVariablesList *list, iggFrameBase *parent) : iggWidget(parent)
		{
			mDataExplorer = de;
			mList = list;

			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,"Data stretch");
			
			int i, n;
			const iString &s = iDataStretch::GetStretchNames();
			n = s.Contains(',') + 1;
			for(i=0; i<n; i++)
			{
				mSubject->InsertItem(s.Section(",",i,i));
			}

			this->OnInt1Body(mDataExplorer->GetStretch());

			this->SetBaloonHelp("Set the stretch for X-axis","This box sets the stretch (scaling) for the horizontal axis. All variables must have the same stretch, since they are shown on one plot.");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			mSubject->SetValue(mDataExplorer->GetStretch());
		}

		void OnInt1Body(int s)
		{
			mDataExplorer->SetStretch(s);
			//
			//  Cache a few things
			//
			const iArray<Variable> &v = mList->GetVariables();
			int i, n = v.Size();

			for(i=0; i<n; i++)
			{
				v[i].Explorer->SetStretch(s);
			}

			mDataExplorer->UpdateDialog();
		}

		ibgWidgetComboBoxSubject *mSubject;
		ExploredVariablesList *mList;
		iggDialogDataExplorer *mDataExplorer;
	};
};


using namespace iggDialogDataExplorer_Private;


iggDialogDataExplorer::iggDialogDataExplorer(iggMainWindow *parent) : iggDialog(parent,0U,iImageFactory::FindIcon("dataexp.png"),"Data Explorer","sr.gg.dd",2)
{
	iggFrame *box1, *box2, *tmp1, *tmp2;

	this->SetStretch(iParameter::_StretchLog);

	mProvider = new iggKeywordDataTypeProvider("",this->GetFrame()); IERROR_ASSERT(mProvider);

	mFlashFrame = new iggFrameFlip(mFrame);
	tmp1 = new iggFrame(mFlashFrame,2);
	mFlashFrame->AddLayer(tmp1);

	box1 = new iggFrame(tmp1,2);
	box2 = new iggFrame(tmp1,2);

	iggFrameDataTypeSelector *dts = new iggFrameDataTypeSelector(mProvider,false,"Use data",box2,_ListDataTypesWithVariables);
	ExploredVariablesList *el = new ExploredVariablesList(this,box1);
	AvailableVariablesList *al = new AvailableVariablesList(this,el,dts,box2);
	tmp1->AddLine(new iggWidgetTextArea("   Shown variables",tmp1),new iggWidgetTextArea("   All variables",tmp1));
	tmp1->AddLine(box1,box2);

	tmp1 = new iggFrame(mFrame,1);
	Area *area = new Area(this,el,tmp1);
	el->CreatePropertiesDialog(area);
	tmp1->AddLine(area);
	tmp1->AddLine(new iggFrameHistogramAreaExtra(area,tmp1,false));
	tmp1->AddLine(new StretchComboBox(this,el,tmp1));
	tmp1->SetRowStretch(0,10);

	tmp2 = new iggFrame(mFrame);
	iggFrame *b = new iggFrame("Data for top variable",tmp2,2);
	iggWidget *w;
	w = new DataComboBox(0,el,b);
	el->AddDependent(w);
	b->AddLine(new iggWidgetTextArea("Maximum",b),w);
	w = new DataComboBox(1,el,b);
	el->AddDependent(w);
	b->AddLine(new iggWidgetTextArea("Minimum",b),w);
	w = new DataDisplay(0,el,b);
	el->AddDependent(w);
	b->AddLine(new iggWidgetTextArea("Median",b),w);
	w = new DataDisplay(1,el,b);
	el->AddDependent(w);
	b->AddLine(new iggWidgetTextArea("Average",b),w);
	w = new DataDisplay(2,el,b);
	el->AddDependent(w);
	b->AddLine(new iggWidgetTextArea("Dispersion",b),w);
	tmp2->AddLine(b);
	w = new MoveFocalPointFrame(el,tmp2);
	el->AddDependent(w);
	tmp2->AddLine(w);
	tmp2->AddSpace(10);

	mFrame->AddLine(tmp1,tmp2);

//	box1->AddLine(0,new iggWidgetTextArea("Shown variables",box1));
	tmp1 = new iggFrame(box1);
	ControlButton *b2;
	b2 = new ControlButton(0,el,tmp1);
	el->AddDependent(b2);
	tmp1->AddLine(b2);
	b2 = new ControlButton(1,el,tmp1);
	el->AddDependent(b2);
	tmp1->AddLine(b2);
	b2 = new ControlButton(2,el,tmp1);
	el->AddDependent(b2);
	tmp1->AddLine(b2);
	b2 = new ControlButton(3,el,tmp1);
	el->AddDependent(b2);
	tmp1->AddLine(b2);
	tmp1->AddSpace(10);
	box1->AddLine(tmp1,el);

//	box2->AddLine(0,new iggWidgetTextArea("All variables",box2));
	tmp1 = new iggFrame(box2);
	AddVariableButton *b1 = new AddVariableButton(al,tmp1);
	al->SetButton(b1);
	tmp1->AddSpace(10);
	tmp1->AddLine(b1);
	tmp1->AddSpace(10);
	box2->AddLine(tmp1,al);
	box2->AddLine(static_cast<iggWidget*>(0),dts);

//	mFrame->AddLine(box1,box2);
	mFrame->AddLine(mFlashFrame,2);

	tmp1 = new iggFrame("",mFlashFrame);
	tmp1->SetPadding(false);
	FractionInfoList *fi = new FractionInfoList(el,tmp1);
	area->AttachInfoList(fi);
	tmp1->AddLine(fi);
	mFlashFrame->AddLayer(tmp1);
	mFlashFrame->ShowLayer(0);

	mFrame->SetRowStretch(0,10);
	mFrame->SetRowStretch(1,5);

	this->ResizeContents(700,500);
}


iggDialogDataExplorer::~iggDialogDataExplorer()
{
	delete mProvider;
}


void iggDialogDataExplorer::ShowFractionInfo(bool s)
{
	mFlashFrame->ShowLayer(s?1:0);
}

#endif
