/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  DataFileLoader for reading uniform mesh data
//
#ifndef IUNIFORMGRIDFILELOADER_H
#define IUNIFORMGRIDFILELOADER_H


#include "ifieldfileloader.h"


#include "iarray.h"
#include "ibuffer.h"

#include <vtkObject.h>

class iUniformGridHelper;
class iUniformScalarsHelper;

class vtkStructuredPoints;


class iUniformGridFileLoader : public iFieldFileLoader
{

	friend class iFieldDataLimits;

public:

	vtkTypeMacro(iUniformGridFileLoader,iFieldFileLoader);

	void SetVoxelLocation(int s);
	inline int GetVoxelLocation() const { return mVoxelLocation; }

	void SetScaledDimension(int v);
	inline int GetScaledDimension() const { return mScaledDim; }

protected:

	iUniformGridFileLoader(iDataReader *r, int priority, bool usePeriodicOffsets);
	virtual ~iUniformGridFileLoader();

	inline float* GetDataPointer() const { return mBuffer; }

	virtual void ReadFileBody(const iString &suffix, const iString &fname);
	virtual void FinalizeBody();
	virtual void ShiftDataBody(vtkDataSet *data, double dx[3]);
	virtual void Polish(vtkDataSet *ds);

	virtual void ReadBinFile(const iString &fname);
    virtual void ReadTxtFile(const iString &fname);

	virtual void SetIndicies(int &nvar, int &ncomp) = 0;
	virtual void AssignBinData(int nvar, vtkIdType ntot, int com, float *d) = 0;
	virtual void AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f) = 0;
	virtual void AttachBuffer(vtkStructuredPoints *data) = 0;
	virtual void SetArraysAndComponents() = 0;
	virtual void OperateOnData(){}

	void EraseBuffer();
	void ComputeSpacing(double org[3], double spa[3]);
	void ReleaseData();

	bool IsCompatible(iUniformGridFileLoader *other) const;

	double mOriginOffset[3];
	int mScaledDim;
	int mNumVars, mNumComponents;
	const int mMaxDimension;
	const int mBufferSizeOffset;
	iBuffer<float> mVarMin;
	iBuffer<float> mVarMax;

	int mVoxelLocation;
	int mFileDims[3], mDataDims[3];

	vtkIdType mBufferSize;
	float *mBuffer;

private:

	const bool mUsePeriodicOffsets;
	iUniformGridHelper *mHelper;
};


//
//  Concrete classes
//
class iUniformVectorsFileLoader : public iUniformGridFileLoader
{

	friend class iDataReader;
	friend class iUniformScalarsHelper;

public:

	vtkTypeMacro(iUniformVectorsFileLoader,iUniformGridFileLoader);

protected:

	iUniformVectorsFileLoader(iDataReader *r);

	virtual void SetIndicies(int &nvar, int &ncomp);
	virtual void AssignBinData(int nvar, vtkIdType ntot, int com, float *d);
	virtual void AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f);
	virtual void AttachBuffer(vtkStructuredPoints *data);
	virtual void SetArraysAndComponents();
};


class iUniformTensorsFileLoader : public iUniformGridFileLoader
{

	friend class iDataReader;

public:

	vtkTypeMacro(iUniformTensorsFileLoader,iUniformGridFileLoader);

protected:

	iUniformTensorsFileLoader(iDataReader *r);

	virtual void SetIndicies(int &nvar, int &ncomp);
	virtual void AssignBinData(int nvar, vtkIdType ntot, int com, float *d);
	virtual void AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f);
	virtual void AttachBuffer(vtkStructuredPoints *data);
	virtual void SetArraysAndComponents();
};


class iUniformScalarsFileLoader : public iUniformGridFileLoader
{

	friend class iDataReader;
	friend class iUniformScalarsHelper;

public:

	vtkTypeMacro(iUniformScalarsFileLoader,iUniformGridFileLoader);

	void SetCalculatorExpression(const iString &s);
	inline const iString& GetCalculatorExpression() const { return mCalculatorExpression; }

	void SetCalculatorOutput(int n);
	inline int GetCalculatorOutput() const { return mCalculatorOutput; }

protected:

	iUniformScalarsFileLoader(iDataReader *r, iUniformVectorsFileLoader *vs);
	virtual ~iUniformScalarsFileLoader();

    virtual void OperateOnData();

	virtual void SetIndicies(int &nvar, int &ncomp);
	virtual void AssignBinData(int nvar, vtkIdType ntot, int com, float *d);
	virtual void AssignTxtData(int nvar, vtkIdType ntot, vtkIdType ind, float *f);
	virtual void AttachBuffer(vtkStructuredPoints *data);
	virtual void SetArraysAndComponents();

private:

	iString mCalculatorExpression;
	int mCalculatorOutput;

	iUniformVectorsFileLoader *mVectorFieldLoader;
	iUniformScalarsHelper *mHelper2;
};

#endif

