#pragma ident	"$Id: iiimp-data-collector.c,v 1.4 2002/02/26 00:16:54 kasha Exp kasha $"


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <pthread.h>
#include <errno.h>
#include <sys/uio.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "iiimp-opcode.h"

int		fd_log = -1;

#define FD_MAX	(1024)

static int	fd_type[FD_MAX];
static char	pad[16];

#define IO_READ		(0xfe)
#define IO_WRITE	(0xef)
#define MAGIC		"IIIMP DUMP DATA"
#define DATA_VERSION	(1)

#define FD_UNDEFINED	(0)
#define FD_CANDIDATE	(1)
#define FD_IIIMP	(2)
#define FD_NOT_IIIMP	(3)


extern ssize_t	_read(int, void *, size_t);
extern ssize_t	_write(int, const void *, size_t);
extern int	_close(int);
extern int	_socket(int, int, int);
extern int	_accept(int s, struct sockaddr * addr, socklen_t * addrlen);
extern ssize_t	_send(int fildes, const void * msg, size_t len, int flags);
extern ssize_t	_recv(int s, void * msg, size_t len, int flags);

int		pad_size(int nbyte);


static void
log_prepare()
{
	char		logfile[1024];
	char *		p;
	struct iovec	iov[4];
	uint32_t	byte_order;
	uint32_t	version;

	if (-1 != fd_log) return;

	memset(pad, 0xff, sizeof (pad));

	p = getenv("IIIMP_LOG_FILE");
	if (NULL == p) {
		if (0 == getuid()) {
			p = "/var/run/iiimp-data.%d";
		} else {
			p = "/var/tmp/iiimp-data.%d";
		}
	}
	sprintf(logfile, p, getpid());
	fd_log = open(logfile, O_CREAT | O_WRONLY | O_APPEND, 0666);

	if (-1 != fd_log) {
		iov[0].iov_base = (caddr_t)MAGIC;
		iov[0].iov_len = 16;

		byte_order = 1;
		iov[1].iov_base = (caddr_t)(&byte_order);
		iov[1].iov_len = (sizeof (byte_order));

		version = DATA_VERSION;
		iov[2].iov_base = (caddr_t)(&version);
		iov[2].iov_len = (sizeof (version));

		iov[3].iov_base = (caddr_t)(pad);
		iov[3].iov_len = (16 - iov[1].iov_len - iov[2].iov_len);

		writev(fd_log, iov, 4);
	}

	return;
}


void
dump(int iot, int fd, const void * buf, size_t nbyte) {
	struct iovec	iov[6];
	int		iov_count;
	int		iov_data_size;
	uint32_t	thread_id;
	uint32_t	io_type;
	uint32_t	fildes;
	uint32_t	data_size;
	int		n;

	if (-1 == fd_log) return;

	n = 0;
	iov_data_size = 0;

	thread_id = pthread_self();
	iov[n].iov_base = (caddr_t)(&thread_id);
	iov[n].iov_len = (sizeof (thread_id));
	iov_data_size += (sizeof (thread_id));
	n += 1;

	io_type = iot;
	iov[n].iov_base = (caddr_t)(&io_type);
	iov[n].iov_len = (sizeof (io_type));
	iov_data_size += (sizeof (io_type));
	n += 1;

	fildes = fd;
	iov[n].iov_base = (caddr_t)(&fildes);
	iov[n].iov_len = (sizeof (fildes));
	iov_data_size += (sizeof (fildes));
	n += 1;

	data_size = nbyte;
	iov[n].iov_base = (caddr_t)(&data_size);
	iov[n].iov_len = (sizeof (data_size));
	iov_data_size += (sizeof (data_size));
	n += 1;

	iov[n].iov_base = (caddr_t)(buf);
	iov[n].iov_len = data_size;
	iov_data_size += data_size;
	n += 1;

	iov[n].iov_base = pad;
	iov[n].iov_len = pad_size(iov_data_size);
	iov_count = ((0 == iov[n].iov_len) ? n : (n + 1));

	writev(fd_log, iov, iov_count);

	return;
}


int
pad_size(int nbyte) {
	int	i;

	i = (nbyte % 16);
	return ((0 == i) ? 0 : (16 - i));
}


ssize_t
read(int fildes, void * buf, size_t nbyte)
{
	ssize_t		r;
	int		errno_save;
	struct iovec	iov[5];
	int		iov_count;
	int		iov_data_size;
	uint32_t	thread_id;
	uint32_t	io_type;
	uint32_t	data_size;

	r = _read(fildes, buf, nbyte);
	errno_save = errno;

	if (r <= 0) return r;

	if ((0 <= fildes) && (fildes < FD_MAX) &&
	    (FD_CANDIDATE == fd_type[fildes])) {
		if ((4 <= r) && (4 <= nbyte) &&
		    (IM_CONNECT == (0x7f & *((unsigned char *)buf)))) {
			log_prepare();
			if (-1 == fd_log) {
				fd_type[fildes] = FD_NOT_IIIMP;	/* give up */
				return r;
			}
			fd_type[fildes] = FD_IIIMP;	/* probably IIIMP port */
		} else {
			fd_type[fildes] = FD_NOT_IIIMP;	/* not IIIMP port */
			return r;
		}
	}

	if ((0 <= fildes) && (fildes < FD_MAX) &&
	    (FD_IIIMP == fd_type[fildes])) {
		dump(IO_READ, fildes, buf, r);
	}

	errno = errno_save;
	return r;
}


ssize_t
write(int fildes, const void * buf, size_t nbyte)
{
	unsigned char *	p;
	struct iovec	iov[4];
	int		iov_count;
	int		iov_data_size;
	uint32_t	thread_id;
	uint32_t	io_type;
	uint32_t	data_size;

	p = (unsigned char *)buf;

	while ((0 <= fildes) && (fildes < FD_MAX) &&
	       (FD_CANDIDATE == fd_type[fildes])) {
		int		op;
		int		len;
		int		byte_order;

		fd_type[fildes] = FD_NOT_IIIMP; /* temporary */

		if (nbyte <= 8) break;	/* too short for IM_CONNECT */

		op = (0x7f & *(p + 0));
		if (IM_CONNECT != op) break;

		byte_order = *(p + 4);
		if ((0x42 != byte_order) && (0x6c != byte_order)) break;

		len = (((*(p + 1) << 16) +
			 (*(p + 2) << 8) +
			 (*(p + 3) << 0)) * 4);
		if ((len + 4) != nbyte) break;

		log_prepare();
		if (-1 == fd_log) break;

		fd_type[fildes] = FD_IIIMP;	/* probably IIIMP port */

		if (FD_UNDEFINED != fd_type[fildes]) break;
	}

	if ((0 <= fildes) && (fildes < FD_MAX) &&
	    (FD_IIIMP == fd_type[fildes])) {
		dump(IO_WRITE, fildes, buf, nbyte);
	}

	return _write(fildes, buf, nbyte);
}


int
close(int fildes)
{
	if ((0 <= fildes) && (fildes < FD_MAX)) {
		if (FD_IIIMP == fd_type[fildes]) {
			dump(IO_WRITE, fildes, NULL, 0);
		}
		fd_type[fildes] = FD_UNDEFINED;
	}
	return _close(fildes);
}


int
socket(int domain, int type, int protocol)
{
	int	fildes;

	fildes = _socket(domain, type, protocol);
	if ((0 <= fildes) && (fildes < FD_MAX)) {
		if (FD_IIIMP == fd_type[fildes]) {
			dump(IO_WRITE, fildes, NULL, 0);
		}
		fd_type[fildes] = FD_CANDIDATE;
	}

	return fildes;
}


int
accept(int s, struct sockaddr * addr, void * addrlen)
{
	int	fildes;

	fildes = _accept(s, addr, addrlen);
	if ((0 <= fildes) && (fildes < FD_MAX)) {
		if (FD_IIIMP == fd_type[fildes]) {
			dump(IO_WRITE, fildes, NULL, 0);
		}
		fd_type[fildes] = FD_CANDIDATE;
	}

	return fildes;
}


ssize_t
recv(int fildes, void * msg, size_t len, int flags)
{
	ssize_t		r;
	int		errno_save;
	struct iovec	iov[5];
	int		iov_count;
	int		iov_data_size;
	uint32_t	thread_id;
	uint32_t	io_type;
	uint32_t	data_size;

	r = _recv(fildes, msg, len, flags);
	errno_save = errno;

	if (r <= 0) return r;

	if ((0 <= fildes) && (fildes < FD_MAX) &&
	    (FD_CANDIDATE == fd_type[fildes])) {
		if ((4 <= r) && (4 <= len) &&
		    (IM_CONNECT == (0x7f & *((unsigned char *)msg)))) {
			log_prepare();
			if (-1 == fd_log) {
				fd_type[fildes] = FD_NOT_IIIMP;	/* give up */
				return r;
			}
			fd_type[fildes] = FD_IIIMP;	/* probably IIIMP port */
		} else {
			fd_type[fildes] = FD_NOT_IIIMP;	/* not IIIMP port */
			return r;
		}
	}

	if ((0 <= fildes) && (fildes < FD_MAX) &&
	    (FD_IIIMP == fd_type[fildes])) {
		dump(IO_READ, fildes, msg, r);
	}

	errno = errno_save;
	return r;
}


ssize_t
send(int s, const void * msg, size_t len, int flags)
{
	unsigned char *	p;
	struct iovec	iov[4];
	int		iov_count;
	int		iov_data_size;
	uint32_t	thread_id;
	uint32_t	io_type;
	uint32_t	data_size;

	p = (unsigned char *)msg;

	while ((0 <= s) && (s < FD_MAX) &&
	       (FD_CANDIDATE == fd_type[s])) {
		int		op;
		int		len;
		int		byte_order;

		fd_type[s] = FD_NOT_IIIMP; /* temporary */

		if (len <= 8) break;	/* too short for IM_CONNECT */

		op = (0x7f & *(p + 0));
		if (IM_CONNECT != op) break;

		byte_order = *(p + 4);
		if ((0x42 != byte_order) && (0x6c != byte_order)) break;

		len = (((*(p + 1) << 16) +
			 (*(p + 2) << 8) +
			 (*(p + 3) << 0)) * 4);
		if ((len + 4) != len) break;

		log_prepare();
		if (-1 == fd_log) break;

		fd_type[s] = FD_IIIMP;	/* probably IIIMP port */

		if (FD_UNDEFINED != fd_type[s]) break;
	}

	if ((0 <= s) && (s < FD_MAX) &&
	    (FD_IIIMP == fd_type[s])) {
		dump(IO_WRITE, s, msg, len);
	}

	return _send(s, msg, len, flags);
}
