#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import logging

from twisted.internet.defer import succeed

from imagestore.lib.service import ServiceError

from imagestore.installservice import (
    GetStatesTask, InstallImageTask, CancelChangesTask)
from imagestore.model import ImageStoreResponse
from imagestore.proxyservice import (
    GetDashboardTask, GetUpgradesTask, SearchTask)
from imagestore.storageservice import (
    SetUpgradeImagesTask, StoreImagesTask, GetStoredImagesTask,
    ClearErrorMessageTask, GetInstalledImageUrisTask)
from imagestore.eucaservice import GetEucaInfo


CACHED_IMAGES_THRESHOLD = 30


class Presenter(object):

    def __init__(self, serviceHub):
        self._serviceHub = serviceHub
        self._getStateArgs = object()

    def getDashboard(self):
        logging.debug("Retrieving dashboard.")
        def getUpgrades(uris):
            if uris is not None and uris:
                logging.debug("Retrieving upgrades for uris: %r" % (uris,))
                return self._serviceHub.addTask(GetUpgradesTask(uris))
            return {}

        def resetStoreAndGetDashboard(upgradeImagesResponse):
            upgradeImages = upgradeImagesResponse.get("images")
            if upgradeImages:
                logging.debug("Upstream says upgrades are available: %r" %
                              ([image["uri"] for image in upgradeImages],))
                self._serviceHub.addTask(SetUpgradeImagesTask(upgradeImages))
                return upgradeImages
            else:
                logging.debug("Upstream says no upgrades are available.")
            return None

        def getDashboardWithUpgrades(upgradeImages):
            upgradeUris = []
            if upgradeImages is not None and upgradeImages:
                for image in upgradeImages:
                    upgradeUris.append(image["uri"])
            logging.debug("Asking upstream for the dashboard.")
            return self._serviceHub.addTask(GetDashboardTask(upgradeUris))

        deferred = self._serviceHub.addTask(GetInstalledImageUrisTask())
        deferred.addCallback(getUpgrades)
        deferred.addCallback(resetStoreAndGetDashboard)
        deferred.addCallback(getDashboardWithUpgrades)
        deferred.addCallback(self._storeImages)
        deferred.addCallback(self._setResponseStates)
        return deferred

    def search(self, searchTerms):
        logging.debug("Searching for %r" % (searchTerms,))
        deferred = self._serviceHub.addTask(SearchTask(searchTerms))
        deferred.addCallback(self._storeImages)
        deferred.addCallback(self._setResponseStates)
        return deferred

    def _storeImages(self, imageStoreResponse):
        images = imageStoreResponse["images"]
        logging.debug("Storing %d images from result." % len(images))
        deferred = self._serviceHub.addTask(StoreImagesTask(images))
        deferred.addCallback(lambda result: imageStoreResponse)
        return deferred

    def _setResponseStates(self, imageStoreResponse):
        uris = []
        for image in imageStoreResponse.get("images", ()):
            uris.append(image["uri"])
        def setStates(states):
            imageStoreResponse["states"] = states
            return imageStoreResponse
        deferred = self._serviceHub.addTask(GetStatesTask(uris))
        deferred.addCallback(setStates)
        return deferred

    def getStates(self, imageUris):
        if imageUris != self._getStateArgs:
            self._getStateArgs = imageUris
            logging.debug("Retrieving states for URIs: %r "
                          "(won't repeat while URIs don't change)"
                          % (imageUris,))
        if not imageUris:
            return succeed([])
        return self._serviceHub.addTask(GetStatesTask(imageUris))

    def getSecretKey(self, clientId):
        # XXX Untested
        logging.debug("Presenter asking EucaService for credentials "
                      "from admin.")
        deferred = self._serviceHub.addTask(
            GetEucaInfo("admin"))
        deferred.addCallback(lambda result: result.secretKey)
        def logSuccess(result):
            logging.debug("Presenter got EucaInfo and returned it.")
            return result
        def logFailure(result):
            logging.debug("Presenter got an error back from the "
                          "GetEucaInfoTask.")
            return result
        deferred.addCallbacks(logSuccess, logFailure)
        return deferred

    def startInstall(self, imageUri):
        logging.info("Starting install of image %s" % (imageUri,))
        deferred = self._serviceHub.addTask(GetStoredImagesTask([imageUri]))
        def installImage(result):
            return self._serviceHub.addTask(InstallImageTask(result[0]))
        deferred.addCallback(installImage)
        return deferred

    def cancelChanges(self, imageUri):
        logging.info("Cancelling changes for image %s" % (imageUri,))
        deferred = self._serviceHub.addTask(CancelChangesTask(imageUri))
        def getStates(result):
            return self._serviceHub.addTask(GetStatesTask([imageUri]))
        deferred.addCallback(getStates)
        return deferred

    def clearError(self, imageUri):
        logging.info("Clearing errors for image %s" % (imageUri,))
        deferred = self._serviceHub.addTask(ClearErrorMessageTask(imageUri))
        def getStates(result):
            return self._serviceHub.addTask(GetStatesTask([imageUri]))
        deferred.addCallback(getStates)
        return deferred
