#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from imagestore.lib.tests import TestCase
from imagestore.lib.service import (
    Service, taskHandler, ServiceHub)

from imagestore.model import (
    ImageState, ImageSection, ImageStoreResponse, ImageRegistration)
from imagestore.presenter import Presenter, CACHED_IMAGES_THRESHOLD

from imagestore.installservice import (
    GetStatesTask, InstallImageTask,
    InstallServiceError, CancelChangesTask)
from imagestore.proxyservice import (
    GetDashboardTask, GetUpgradesTask, SearchTask, ProxyServiceError)
from imagestore.storageservice import (
    GetStoredImagesTask, SetUpgradeImagesTask, StoreImagesTask,
    StorageServiceError, ClearErrorMessageTask, GetInstalledImageUrisTask)
from imagestore.tests.helpers import (
    ServiceTestCase, createRegistration)


class FakeInstallService(Service):

    def __init__(self, states, imageRegistration, cancelChangesResult):
        self._states = states
        self._imageRegistration = imageRegistration
        self._cancelChangesResult = cancelChangesResult
        self.tasks = []
        Service.__init__(self)

    @taskHandler(GetStatesTask)
    def _getStates(self, task):
        self.tasks.append(task)
        if callable(self._states):
            return self._states()
        return self._states

    @taskHandler(InstallImageTask)
    def _installImage(self, task):
        self.tasks.append(task)
        if callable(self._imageRegistration):
            return self._imageRegistration()
        return self._imageRegistration

    @taskHandler(CancelChangesTask)
    def _cancelChanges(self, task):
        self.tasks.append(task)
        if callable(self._cancelChangesResult):
            return self._cancelChangesResult()
        return self._cancelChangesResult


class FakeProxyService(Service):

    def __init__(self, dashboard, upgrades=None):
        self._dashboard = dashboard
        self._upgrades = upgrades
        self.tasks = []
        Service.__init__(self)

    @taskHandler(GetDashboardTask)
    def _getDashboard(self, task):
        self.tasks.append(task)
        if callable(self._dashboard):
            return self._dashboard()
        return self._dashboard

    @taskHandler(GetUpgradesTask)
    def _getUpgrades(self, task):
        self.tasks.append(task)
        if callable(self._upgrades):
            upgrades = self._upgrades()
        else:
            upgrades = self._upgrades
        return ImageStoreResponse({"images": upgrades})


class FakeStorageService(Service):

    def __init__(self, upgradeImages, storedImages, installedImages,
                 clearErrorMessageResult):
        self._upgradeImages = upgradeImages
        self._storedImages = storedImages
        self._installedImages = installedImages
        self._clearErrorMessageResult = clearErrorMessageResult

        self.tasks = []
        Service.__init__(self)

    @taskHandler(SetUpgradeImagesTask)
    def _setUpgradeImages(self, task):
        self.tasks.append(task)
        if callable(task.images):
            self._upgradeImages = task.images()
        else:
            self._upgradeImages = task.images

    @taskHandler(StoreImagesTask)
    def _storeImages(self, task):
        self.tasks.append(task)
        if callable(task.images):
            self._storeImages = task.images()
        else:
            self._storeImages = task.images

    @taskHandler(GetStoredImagesTask)
    def _getStoredImages(self, task):
        self.tasks.append(task)
        if callable(self._storedImages):
            return self._storedImages()
        return self._storedImages

    @taskHandler(ClearErrorMessageTask)
    def _clearErrorMessage(self, task):
        self.tasks.append(task)
        if callable(self._clearErrorMessageResult):
            return self._clearErrorMessageResult()
        return self._clearErrorMessageResult

    @taskHandler(GetInstalledImageUrisTask)
    def _getInstalledImageUris(self, task):
        self.tasks.append(task)
        if callable(self._installedImages):
            return self._installedImages()
        return self._installedImages



class PresenterTest(ServiceTestCase):

    def createPresenterAndFakeServices(self, proxyDashboardResponse=None,
                                       installServiceStates=None,
                                       installedImages=None,
                                       installImageRegistration=None,
                                       proxyUpgradeImages=None,
                                       storageUpgradeImages=None,
                                       storageStoredImages=None,
                                       cancelChangesResult=None,
                                       clearErrorMessageResult=None):
        if installServiceStates is None:
            installServiceStates = []
        if installedImages is None:
            installedImages = []
        if proxyUpgradeImages is None:
            proxyUpgradeImages = []
        if storageUpgradeImages is None:
            storageUpgradeImages = []
        if storageStoredImages is None:
            storageStoredImages = []
        installService = FakeInstallService(
            states=installServiceStates,
            imageRegistration=installImageRegistration,
            cancelChangesResult=cancelChangesResult)
        proxyService = FakeProxyService(
            dashboard=proxyDashboardResponse,
            upgrades=proxyUpgradeImages)
        storageService = FakeStorageService(
            installedImages=installedImages,
            upgradeImages=storageUpgradeImages,
            storedImages=storageStoredImages,
            clearErrorMessageResult=clearErrorMessageResult)

        serviceHub = ServiceHub()
        serviceHub.addService(installService)
        serviceHub.addService(proxyService)
        serviceHub.addService(storageService)

        presenter = Presenter(serviceHub)
        return presenter, serviceHub

    def createResponse(self, newImages=None, upgradeImages=None):
        if newImages is None:
            newImages = []
        if upgradeImages is None:
            upgradeImages = []
        allImages = newImages + upgradeImages

        response = ImageStoreResponse(
            {"sections": [],
             "images": allImages,
             })

        if upgradeImages:
            response["sections"].append(ImageSection({
                "title": "Upgrades",
                "summary": ("The following recent versions of appliances you "
                            "have installed are available:"),
                "image-uris": [image["uri"] for image in upgradeImages],
                }))
        if newImages:
            response["sections"].append(ImageSection({
                "title": "New",
                "summary": "Appliances recently made available:",
                "image-uris": [image["uri"] for image in newImages],
                }))
        return response

    def testGetDashboardInjectsImageStates(self):
        image1 = self.createImage(1)
        image2 = self.createImage(2)
        image3 = self.createImage(3)

        response = self.createResponse(newImages=[image1, image2])

        installServiceStates = [
            ImageState({"image-uri": image1["uri"],
                        "status": "uninstalled",
                        "is-upgrade": True}),
            ImageState({"image-uri": image2["uri"],
                        "status": "installed",
                        "emi": "emi-123456789"}),
            ]
        expectedResponse = ImageStoreResponse(str(response),
                                              states=installServiceStates)

        def callback(result, serviceHub):
            installService = serviceHub.getService(FakeInstallService)
            self.assertEquals(result, expectedResponse)

            self.assertEquals(len(installService.tasks), 1)
            self.assertIdentical(type(installService.tasks[0]),
                                 GetStatesTask)

            proxyService = serviceHub.getService(FakeProxyService)
            self.assertEquals(len(proxyService.tasks), 2)
            self.assertIdentical(type(proxyService.tasks[0]),
                                 GetUpgradesTask)
            self.assertIdentical(type(proxyService.tasks[1]),
                                 GetDashboardTask)
            self.assertEquals(proxyService.tasks[1].upgrade_uris,
                              [image3["uri"]])

            storageService = serviceHub.getService(FakeStorageService)
            self.assertEquals(len(storageService.tasks), 3)
            self.assertIdentical(type(storageService.tasks[0]),
                                 GetInstalledImageUrisTask)
            self.assertIdentical(type(storageService.tasks[1]),
                                 SetUpgradeImagesTask)
            self.assertEquals(storageService.tasks[1].images,
                              [image3])
            self.assertIdentical(type(storageService.tasks[2]),
                                 StoreImagesTask)
            self.assertEquals(storageService.tasks[2].images,
                              [image1, image2])


        presenter, serviceHub = self.createPresenterAndFakeServices(
            installServiceStates=installServiceStates,
            installedImages=[image1["uri"], image2["uri"]],
            proxyDashboardResponse=response,
            proxyUpgradeImages=[image3],
            storageUpgradeImages=[])
        deferred = presenter.getDashboard()
        deferred.addCallback(callback, serviceHub)

        return self.runServicesAndWaitForDeferred(
            [serviceHub], deferred)

    def testGetDashboardNoneInstalledSkipsGetUpgrades(self):
        image1 = self.createImage(1)
        image2 = self.createImage(2)

        def failGetUpgrades():
            raise ProxyServiceError("I shouldn't be called since, "
                                    "there are no installed images.")

        response = self.createResponse(newImages=[image1, image2])

        installServiceStates = [
            ImageState({"image-uri": image1["uri"],
                        "status": "uninstalled"}),
            ImageState({"image-uri": image2["uri"],
                        "status": "uninstalled"})
            ]

        presenter, serviceHub = self.createPresenterAndFakeServices(
            installServiceStates=installServiceStates,
            installedImages=[],
            proxyDashboardResponse=response,
            proxyUpgradeImages=failGetUpgrades,
            storageUpgradeImages=[])
        deferred = presenter.getDashboard()

        expectedResponse = ImageStoreResponse(str(response),
                                              states=installServiceStates)

        def callback(result):
            self.assertEquals(result, expectedResponse)

        deferred.addCallback(callback)
        return self.runServicesAndWaitForDeferred(
            [serviceHub], deferred)

    def testStartInstallRunsToCompletion(self):
        image1 = self.createImage(1)

        installImageRegistration = createRegistration(1, image=image1)

        def callback(result, serviceHub):
            self.assertEquals(result, installImageRegistration)

            storageService = serviceHub.getService(FakeStorageService)
            self.assertEquals(len(storageService.tasks), 1)
            self.assertIdentical(type(storageService.tasks[0]),
                                 GetStoredImagesTask)
            self.assertEquals(storageService.tasks[0].uris,
                              [image1["uri"]])

            installService = serviceHub.getService(FakeInstallService)
            self.assertEquals(len(installService.tasks), 1)
            self.assertIdentical(type(installService.tasks[0]),
                                 InstallImageTask)
            self.assertEquals(installService.tasks[0].image,
                              image1)

        presenter, serviceHub = self.createPresenterAndFakeServices(
            storageStoredImages=[image1],
            installImageRegistration=installImageRegistration)
        deferred = presenter.startInstall(image1["uri"])
        deferred.addCallback(callback, serviceHub)

        return self.runServicesAndWaitForDeferred([serviceHub], deferred)

    def testGetStates(self):
        image = self.createImage(1)

        def callback(result, serviceHub):
            installService = serviceHub.getService(FakeInstallService)
            self.assertEquals(len(installService.tasks), 1)
            task = installService.tasks[0]
            self.assertIdentical(type(task), GetStatesTask)
            self.assertEquals(task.uris, [image["uri"]])

            self.assertEquals(result, ["woot!"])

        presenter, serviceHub = self.createPresenterAndFakeServices(
            installServiceStates=["woot!"])
        deferred = presenter.getStates([image["uri"]])
        deferred.addCallback(callback, serviceHub)

        return self.runServicesAndWaitForDeferred([serviceHub], deferred)

    def testGetStatesWithEmptyList(self):
        image = self.createImage(1)

        def callback(result, serviceHub):
            installService = serviceHub.getService(FakeInstallService)
            self.assertEquals(installService.tasks, [])
            self.assertEquals(result, [])

        presenter, serviceHub = self.createPresenterAndFakeServices(
            installServiceStates=["woot!"])
        deferred = presenter.getStates([])
        deferred.addCallback(callback, serviceHub)

        return self.runServicesAndWaitForDeferred([serviceHub], deferred)

    def testCancelChanges(self):
        image = self.createImage(1)

        def callback(result, serviceHub):
            installService = serviceHub.getService(FakeInstallService)
            self.assertEquals(len(installService.tasks), 2)

            cancelTask = installService.tasks[0]
            self.assertIdentical(type(cancelTask), CancelChangesTask)
            self.assertEquals(cancelTask.imageUri, image["uri"])

            statesTask = installService.tasks[1]
            self.assertIdentical(type(statesTask), GetStatesTask)
            self.assertEquals(statesTask.uris, [image["uri"]])

            self.assertEquals(result, ["woot!"])

        presenter, serviceHub = self.createPresenterAndFakeServices(
            installServiceStates=["woot!"])
        deferred = presenter.cancelChanges(image["uri"])
        deferred.addCallback(callback, serviceHub)

        return self.runServicesAndWaitForDeferred([serviceHub], deferred)

    def testClearError(self):
        image = self.createImage(1)

        def callback(result, serviceHub):
            installService = serviceHub.getService(FakeInstallService)
            storageService = serviceHub.getService(FakeStorageService)
            self.assertEquals(len(installService.tasks), 1)
            self.assertEquals(len(storageService.tasks), 1)

            clearErrorTask = storageService.tasks[0]
            self.assertIdentical(type(clearErrorTask), ClearErrorMessageTask)
            self.assertEquals(clearErrorTask.imageUri, image["uri"])

            statesTask = installService.tasks[0]
            self.assertIdentical(type(statesTask), GetStatesTask)
            self.assertEquals(statesTask.uris, [image["uri"]])

            self.assertEquals(result, ["woot!"])

        presenter, serviceHub = self.createPresenterAndFakeServices(
            installServiceStates=["woot!"])
        deferred = presenter.clearError(image["uri"])
        deferred.addCallback(callback, serviceHub)

        return self.runServicesAndWaitForDeferred([serviceHub], deferred)

    def testSearch(self):

        image1 = self.createImage(1)
        image2 = self.createImage(2)

        storedImages = []

        class FakeHandlers(Service):

            @taskHandler(SearchTask)
            def _search(s, task):
                self.assertEquals(task.searchTerms, "foo/bar")
                return ImageStoreResponse({"images": [image1, image2]})

            @taskHandler(StoreImagesTask)
            def _search(s, task):
                storedImages.extend(task.images)

            @taskHandler(GetStatesTask)
            def _getStates(s, task):
                self.assertEquals(task.uris, [image1["uri"], image2["uri"]])
                return ["state1", "state2"]

        fakeHandlers = FakeHandlers()
        serviceHub = ServiceHub()
        serviceHub.addService(fakeHandlers)

        presenter = Presenter(serviceHub)

        def checkResult(result):
            self.assertEquals(result["images"], [image1, image2])
            self.assertEquals(result["states"], ["state1", "state2"])
            self.assertEquals(storedImages, [image1, image2])

        deferred = presenter.search("foo/bar")
        deferred.addCallback(checkResult)
        return self.runServicesAndWaitForDeferred([serviceHub], deferred)
