#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <stdarg.h>

#include "imapfilter.h"
#include "session.h"
#include "buffer.h"


extern options_t opts;

buffer_t obuf;			/* Output buffer. */

static int tag = 0x1000;	/* Every IMAP command is prefixed with a
				 * unique [:alnum:] string. */


int send_command(session_t * ssn, char *cmd, char *alt);
void prepare_command(const char *fmt,...);


/*
 * Send to server data; a command.
 */
int
send_command(session_t * ssn, char *cmd, char *alt)
{
	ssize_t w;
	char *b;

	if (ssn->socket == -1)
		return -1;

	debug("sending command (%d):\n\n%s\n", ssn->socket,
	    (opts.debug == 1 && alt ? alt : cmd));

	verbose("C (%d): %s", ssn->socket, (alt ? alt : cmd));

	for (w = 0, b = cmd; w != (ssize_t) strlen(b); b += w, w = 0)
		if ((w = socket_write(ssn, b)) == -1)
			return -1;

	if (tag == 0xFFFF)	/* Tag always between 0x1000 and 0xFFFF. */
		tag = 0x0FFF;

	return tag++;
}


/*
 * Prepare data for sending and check that the output buffer size is
 * sufficient.
 */
void
prepare_command(const char *fmt,...)
{
	int n;
	va_list args;

	va_start(args, fmt);

	buffer_reset(&obuf);
	n = vsnprintf(obuf.data, obuf.size + 1, fmt, args);
	if (n > (int)obuf.size) {
		buffer_check(&obuf, n);
		vsnprintf(obuf.data, obuf.size + 1, fmt, args);
	}
	va_end(args);
}


/*
 * Send a response to a command continuation request.
 */
int
imap_continuation(session_t * ssn, const char *cont)
{
	ssize_t w;
	const char *b;

	if (ssn->socket == -1)
		return -1;

	for (w = 0, b = cont; w != (ssize_t) strlen(b); b += w, w = 0)
		if ((w = socket_write(ssn, b)) == -1)
			return -1;

	socket_write(ssn, "\r\n");

	debug("sending continuation data (%d):\n\n%s\r\n\n", ssn->socket, cont);

	return 0;
}


/*
 * IMAP NOOP: does nothing always succeeds.
 */
int
imap_noop(session_t * ssn)
{

	prepare_command("%04X NOOP\r\n", tag);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP CAPABILITY: requests listing of capabilities that the server supports.
 */
int
imap_capability(session_t * ssn)
{

	prepare_command("%04X CAPABILITY\r\n", tag);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP NAMESPACE: discovers the prefix and delimeter of namespaces used by the
 * server for mailboxes (RFC 2342).
 */
int
imap_namespace(session_t * ssn)
{

	prepare_command("%04X NAMESPACE\r\n", tag);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP LOGOUT: informs server that client is done.
 */
int
imap_logout(session_t * ssn)
{

	prepare_command("%04X LOGOUT\r\n", tag);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP STARTTLS: begin TLS negotiation.
 */
int
imap_starttls(session_t * ssn)
{

	prepare_command("%04X STARTTLS\r\n", tag);

	return send_command(ssn, obuf.data, NULL);
}


#ifndef NO_CRAMMD5
/*
 * IMAP AUTHENTICATE: indicates authentication mechanism and performs an
 * authentication protocol exchange.
 */
int
imap_authenticate(session_t * ssn, const char *auth)
{

	prepare_command("%04X AUTHENTICATE %s\r\n", tag, auth);

	return send_command(ssn, obuf.data, NULL);
}
#endif


/*
 * IMAP LOGIN: identifies client to server.
 */
int
imap_login(session_t * ssn, const char *user, const char *pass)
{
	int n, r;
	char c;
	char *s;

	/* Command to send to server. */
	prepare_command("%04X LOGIN \"%s\" \"%s\"\r\n", tag, user, pass);

	/* Alternate command with password shrouded for safe printing. */
	n = snprintf(&c, 1, "%04X LOGIN \"%s\" *\r\n", tag, user);
	s = (char *)xmalloc(n + 1);
	snprintf(s, n + 1, "%04X LOGIN \"%s\" *\r\n", tag, user);

	r = send_command(ssn, obuf.data, s);

	xfree(s);

	return r;
}


/*
 * IMAP SUBSCRIBE: adds the specified mailbox name to the server's set of
 * "active" or "subscribed" mailboxes.
 */
int
imap_subscribe(session_t * ssn, const char *mbox)
{

	prepare_command("%04X SUBSCRIBE \"%s\"\r\n", tag, mbox);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP EXAMINE: access a mailbox in READ-ONLY mode.
 */
int
imap_examine(session_t * ssn, const char *mbox)
{

	prepare_command("%04X EXAMINE \"%s\"\r\n", tag, mbox);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP SELECT: access a mailbox in READ-WRITE mode.
 */
int
imap_select(session_t * ssn, const char *mbox)
{

	prepare_command("%04X SELECT \"%s\"\r\n", tag, mbox);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP STATUS: requests status of the indicated mailbox.
 */
int
imap_status(session_t * ssn, const char *mbox, const char *items)
{

	prepare_command("%04X STATUS \"%s\" (%s)\r\n", tag, mbox, items);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP CREATE: create mailbox.
 */
int
imap_create(session_t * ssn, const char *mbox)
{

	prepare_command("%04X CREATE \"%s\"\r\n", tag, mbox);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP SEARCH: searches the mailbox for messages that match certain criteria.
 */
int
imap_search(session_t * ssn, int uid, const char *charset, const char *criteria)
{

	if (charset != NULL && *charset != '\0')
		prepare_command("%04X %sSEARCH CHARSET \"%s\" %s\r\n", tag,
		    (uid ? "UID " : ""), charset, criteria);
	else
		prepare_command("%04X %sSEARCH %s\r\n", tag,
		    (uid ? "UID " : ""), criteria);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP FETCH: retrieves data associated with a message.
 */
int
imap_fetch(session_t * ssn, int uid, const char *mesg, const char *items)
{

	prepare_command("%04X %sFETCH %s %s\r\n", tag, (uid ? "UID " : ""),
	    mesg, items);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP STORE: alters data associated with a message.
 */
int
imap_store(session_t * ssn, int uid, const char *mesg, const char *mode, const char *flags)
{

	prepare_command("%04X %sSTORE %s %sFLAGS.SILENT (%s)\r\n", tag,
	    (uid ? "UID " : ""), mesg, (!strncasecmp(mode, "add", 3) ? "+" :
	    !strncasecmp(mode, "remove", 6) ? "-" : ""), flags);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP COPY: copy messages to mailbox.
 */
int
imap_copy(session_t * ssn, int uid, const char *mesg, const char *mbox)
{

	prepare_command("%04X %sCOPY %s \"%s\"\r\n", tag, (uid ? "UID " : ""),
	    mesg, mbox);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP APPEND: append message to the end of a mailbox.
 */
int
imap_append(session_t * ssn, const char *mbox, const char *flags, const char *date,
    unsigned int size)
{

	prepare_command("%04X APPEND \"%s\"%s%s%s%s%s%s {%d}\r\n", tag, mbox,
	    (flags ? " (" : ""), (flags ? flags : ""), (flags ? ")" : ""),
	    (date ? " \"" : ""), (date ? date : ""), (date ? "\"" : ""), size);

	return send_command(ssn, obuf.data, NULL);
}



/*
 * IMAP CLOSE: delete messages and return to authenticated state.
 */
int
imap_close(session_t * ssn)
{

	prepare_command("%04X CLOSE\r\n", tag);

	return send_command(ssn, obuf.data, NULL);
}


/*
 * IMAP EXPUNGE: permanently removes any messages with the \Deleted flag set.
 */
int
imap_expunge(session_t * ssn)
{

	prepare_command("%04X EXPUNGE\r\n", tag);

	return send_command(ssn, obuf.data, NULL);
}
