#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <regex.h>

#include "imapfilter.h"
#include "session.h"
#include "buffer.h"
#include "regexp.h"


buffer ibuf;			/* Input buffer. */
enum {				/* Server data responses to be parsed;
				 * regular expressions index. */
	DATA_RESPONSE_TAGGED,
	DATA_RESPONSE_CAPABILITY,
	DATA_RESPONSE_AUTHENTICATE,
	DATA_RESPONSE_NAMESPACE,
	DATA_RESPONSE_STATUS,
	DATA_RESPONSE_STATUS_MESSAGES,
	DATA_RESPONSE_STATUS_RECENT,
	DATA_RESPONSE_STATUS_UNSEEN,
	DATA_RESPONSE_EXAMINE_EXISTS,
	DATA_RESPONSE_EXAMINE_RECENT,
	DATA_RESPONSE_LIST,
	DATA_RESPONSE_LSUB,
	DATA_RESPONSE_SEARCH,
	DATA_RESPONSE_FETCH,
	DATA_RESPONSE_FETCH_FLAGS,
	DATA_RESPONSE_FETCH_DATE,
	DATA_RESPONSE_FETCH_SIZE,
	DATA_RESPONSE_FETCH_HEADER,
	DATA_RESPONSE_FETCH_BODY,
	DATA_RESPONSE_FETCH_FIELDS,
};
regexp responses[] = {		/* Server data responses to be parsed;
				 * regular expressions patterns. */
	{ "([[:xdigit:]]{4,4}) (OK|NO|BAD) [[:print:]]*\r\n", NULL, 0, NULL },
	{ "\\* CAPABILITY ([[:print:]]*)\r\n", NULL, 0, NULL },
	{ "\\+ ([[:graph:]]*)\r\n", NULL, 0, NULL },
	{ "\\* NAMESPACE (NIL|\\(\\(\"([[:graph:]]*)\" \"([[:print:]])\"\\)"
	  "[[:print:]]*\\)) (NIL|\\([[:print:]]*\\)) (NIL|\\([[:print:]]*\\))"
	  "\r\n", NULL, 0, NULL },
	{ "\\* STATUS [[:print:]]* \\(([[:alnum:] ]*)\\)\r\n", NULL, 0, NULL },
	{ "MESSAGES ([[:digit:]]+)", NULL, 0, NULL },
	{ "RECENT ([[:digit:]]+)", NULL, 0, NULL },
	{ "UNSEEN ([[:digit:]]+)", NULL, 0, NULL },
	{ "\\* ([[:digit:]]+) EXISTS\r\n", NULL, 0, NULL },
	{ "\\* ([[:digit:]]+) RECENT\r\n", NULL, 0, NULL },
	{ "\\* LIST \\(([[:print:]]*)\\) (\"[[:print:]]\"|NIL) "
	  "(\"([[:print:]]+)\"|([[:print:]]+)|\\{([[:digit:]]+)\\}\r\n"
	  "([[:print:]]*))\r\n", NULL, 0, NULL },
	{ "\\* LSUB \\(([[:print:]]*)\\) (\"[[:print:]]\"|NIL) "
	  "(\"([[:print:]]+)\"|([[:print:]]+)|\\{([[:digit:]]+)\\}\r\n"
	  "([[:print:]]*))\r\n", NULL, 0, NULL },
	{ "\\* SEARCH ?([[:digit:] ]*)\r\n", NULL, 0, NULL },
	{ "\\* [[:digit:]]+ FETCH \\(([[:print:]]*)\\)\r\n", NULL, 0, NULL },
	{ "FLAGS \\(([[:print:]]*)\\)", NULL, 0, NULL },
	{ "INTERNALDATE \"([[:print:]]*)\"", NULL, 0, NULL },
	{ "RFC822.SIZE ([[:digit:]]+)", NULL, 0, NULL },
	{ "BODY\\[HEADER\\] \\{([[:digit:]]+)\\}\r\n([[:print:]\r\n]*)", NULL, 0,
	  NULL },
	{ "BODY\\[TEXT\\] (\\{([[:digit:]]+)\\}\r\n([[:print:]\r\n]*)|"
	  "\"([[:print:]]*)\")", NULL, 0, NULL },
	{ "BODY\\[HEADER\\.FIELDS \\([[:print:]]*\\)\\] \\{([[:digit:]]+)\\}"
	  "\r\n([[:print:]\r\n]*)", NULL, 0, NULL },
	{ NULL, NULL, 0, NULL }
};


int receive_response(session *ssn, char *buf);

int check_tag(session *ssn, int tag);
int check_bye(void);
int check_continuation(void);
int check_trycreate(void);


/*
 * Read data the server sent.
 */
int
receive_response(session *ssn, char *buf)
{

	if (socket_read(ssn, buf, INPUT_BUF) == -1)
		return -1;

	debug("getting response (%d):\n\n%s\n", ssn->socket, buf);

	return 0;
}


/*
 * Search for tagged response in the data that the server sent.
 */
int
check_tag(session *ssn, int tag)
{
	int r;
	char t[4 + 1];
	regexp *re;

	r = STATUS_RESPONSE_NONE;

	snprintf(t, sizeof(t), "%04X", tag);

	re = &responses[DATA_RESPONSE_TAGGED];

	if (!regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0)) {
		if (!strncasecmp(ibuf.data + re->pmatch[1].rm_so, t,
		    strlen(t))) {
			if (!strncasecmp(ibuf.data + re->pmatch[2].rm_so,
			    "OK", strlen("OK")))
				r = STATUS_RESPONSE_OK;
			else if (!strncasecmp(ibuf.data + re->pmatch[2].rm_so,
			    "NO", strlen("NO")))
				r = STATUS_RESPONSE_NO;
			else if (!strncasecmp(ibuf.data + re->pmatch[2].rm_so,
			    "BAD", strlen("BAD")))
				r = STATUS_RESPONSE_BAD;
		}
	}

	if (r != STATUS_RESPONSE_NONE)
		verbose("S (%d): %s", ssn->socket, ibuf.data +
		    re->pmatch[0].rm_so);

	return r;
}


/*
 * Check if server sent a BYE response (connection is closed immediately).
 */
int
check_bye(void)
{

	if (xstrcasestr(ibuf.data, "* BYE") &&
	    !xstrcasestr(ibuf.data, " LOGOUT "))
		return 1;
	else
		return 0;
}


/*
 * Check if the server sent a continuation request.
 */
int
check_continuation(void)
{

	if (ibuf.data[0] == '+' && ibuf.data[1] == ' ')
		return 1;
	else
		return 0;
}


/*
 * Check if the server sent a TRYCREATE response.
 */
int
check_trycreate(void)
{

	if (xstrcasestr(ibuf.data, "[TRYCREATE]"))
		return 1;
	else
		return 0;
}


/*
 * Get server data and make sure there is a tagged response inside them.
 */
int
response_generic(session *ssn, int tag)
{
	int r;

	if (tag == -1)
		return -1;

	buffer_reset(&ibuf);

	do {
		buffer_check(&ibuf, strlen(ibuf.data) + INPUT_BUF);
		if (receive_response(ssn, ibuf.data + strlen(ibuf.data)) == -1)
			return -1;
		if (check_bye())
			return -1;
		if (check_continuation())
			return STATUS_RESPONSE_CONTINUE;
	} while ((r = check_tag(ssn, tag)) == STATUS_RESPONSE_NONE);

	if (r == STATUS_RESPONSE_NO &&
	    (check_trycreate() || get_option_boolean("create")))
		return STATUS_RESPONSE_TRYCREATE;

	return r;
}


/*
 * Process the greeting that server sends during connection.
 */
int
response_greeting(session *ssn)
{

	buffer_reset(&ibuf);

	if (receive_response(ssn, ibuf.data) == -1)
		return -1;

	if (check_bye())
		return -1;

	verbose("S (%d): %s", ssn->socket, ibuf);

	if (xstrcasestr(ibuf.data, "* PREAUTH"))
		return STATUS_RESPONSE_PREAUTH;

	return STATUS_RESPONSE_NONE;
}


/*
 * Process the data that server sent due to IMAP CAPABILITY client request.
 */
int
response_capability(session *ssn, int tag)
{
	int r;
	char *s;
	regexp *re;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	ssn->protocol = PROTOCOL_NONE;

	re = &responses[DATA_RESPONSE_CAPABILITY];

	if (!regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0)) {
		s = xstrndup(ibuf.data + re->pmatch[1].rm_so,
		    re->pmatch[1].rm_eo - re->pmatch[1].rm_so);

		if (xstrcasestr(ibuf.data, "IMAP4rev1"))
			ssn->protocol = PROTOCOL_IMAP4REV1;
		else if (xstrcasestr(ibuf.data, "IMAP4"))
			ssn->protocol = PROTOCOL_IMAP4;
		else {
			error("server supports neither the IMAP4rev1 nor the "
			    "IMAP4 protocol\n");
			return -1;
		}

		ssn->capabilities = CAPABILITY_NONE;

		if (xstrcasestr(ibuf.data, "NAMESPACE"))
			ssn->capabilities |= CAPABILITY_NAMESPACE;
#ifndef NO_CRAMMD5
		if (xstrcasestr(ibuf.data, "AUTH=CRAM-MD5"))
			ssn->capabilities |= CAPABILITY_CRAMMD5;
#endif
#ifndef NO_SSLTLS
		if (xstrcasestr(ibuf.data, "STARTTLS"))
			ssn->capabilities |= CAPABILITY_STARTTLS;
#endif
		if (xstrcasestr(ibuf.data, "CHILDREN"))
			ssn->capabilities |= CAPABILITY_CHILDREN;

		xfree(s);
	}

	return r;
}


#ifndef NO_CRAMMD5
/*
 * Process the data that server sent due to IMAP AUTHENTICATE client request.
 */
int
response_authenticate(session *ssn, int tag, unsigned char **cont)
{
	int r;
	regexp *re;

	re = &responses[DATA_RESPONSE_AUTHENTICATE];

	if ((r = response_generic(ssn, tag)) == STATUS_RESPONSE_CONTINUE &&
	    !regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0))
		*cont = (unsigned char *)xstrndup(ibuf.data + re->pmatch[1].rm_so,
		    re->pmatch[1].rm_eo - re->pmatch[1].rm_so);

	return r;
}
#endif


/*
 * Process the data that server sent due to IMAP NAMESPACE client request.
 */
int
response_namespace(session *ssn, int tag)
{
	int r, n;
	regexp *re;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	ssn->ns.prefix = NULL;
	ssn->ns.delim = '\0';

	re = &responses[DATA_RESPONSE_NAMESPACE];

	if (!regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0)) {
		n = re->pmatch[2].rm_eo - re->pmatch[2].rm_so;
		if (n > 0)
			ssn->ns.prefix = xstrndup(ibuf.data +
			    re->pmatch[2].rm_so, n);
		ssn->ns.delim = *(ibuf.data + re->pmatch[3].rm_so);
	}
	debug("namespace (%d): '%s' '%c'\n", ssn->socket,
	    (ssn->ns.prefix ? ssn->ns.prefix : ""), ssn->ns.delim);

	return r;
}


/*
 * Process the data that server sent due to IMAP STATUS client request.
 */
int
response_status(session *ssn, int tag, unsigned int *exist,
    unsigned int *recent, unsigned int *unseen)
{
	int r;
	char *s;
	regexp *re;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	re = &responses[DATA_RESPONSE_STATUS];

	if (!regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0)) {
		s = xstrndup(ibuf.data + re->pmatch[1].rm_so,
		    re->pmatch[1].rm_eo - re->pmatch[1].rm_so);

		re = &responses[DATA_RESPONSE_STATUS_MESSAGES];
		if (!regexec(re->preg, s, re->nmatch, re->pmatch, 0))
			*exist = strtol(s + re->pmatch[1].rm_so, NULL, 10);

		re = &responses[DATA_RESPONSE_STATUS_RECENT];
		if (!regexec(re->preg, s, re->nmatch, re->pmatch, 0))
			*recent = strtol(s + re->pmatch[1].rm_so, NULL, 10);

		re = &responses[DATA_RESPONSE_STATUS_UNSEEN];
		if (!regexec(re->preg, s, re->nmatch, re->pmatch, 0))
			*unseen = strtol(s + re->pmatch[1].rm_so, NULL, 10);

		xfree(s);
	}

	return r;
}


/*
 * Process the data that server sent due to IMAP EXAMINE client request.
 */
int
response_examine(session *ssn, int tag, unsigned int *exist,
    unsigned int *recent)
{
	int r;
	regexp *re;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	re = &responses[DATA_RESPONSE_EXAMINE_EXISTS];
	if (!regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0))
		*exist = strtol(ibuf.data + re->pmatch[1].rm_so, NULL, 10);

	re = &responses[DATA_RESPONSE_EXAMINE_RECENT];
	if (!regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0))
		*recent = strtol(ibuf.data + re->pmatch[1].rm_so, NULL, 10);

	return r;
}


/*
 * Process the data that server sent due to IMAP SELECT client request.
 */
int
response_select(session *ssn, int tag)
{
	int r;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	if (xstrcasestr(ibuf.data, "[READ-ONLY]"))
		return STATUS_RESPONSE_READONLY;

	return r;
}


/*
 * Process the data that server sent due to IMAP LIST client request.
 */
int
response_list(session *ssn, int tag, char **mboxs, char **folders)
{
	int r, n;
	char *b, *a, *s, *m, *f;
	const char *v;
	regexp *re;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	m = *mboxs = (char *)xmalloc((strlen(ibuf.data) + 1) * sizeof(char));
	f = *folders = (char *)xmalloc((strlen(ibuf.data) + 1) * sizeof(char));
	*m = *f = '\0';

	re = &responses[DATA_RESPONSE_LIST];

	b = ibuf.data;
	while (!regexec(re->preg, b, re->nmatch, re->pmatch, 0)) {
		a = xstrndup(b + re->pmatch[1].rm_so,
		    re->pmatch[1].rm_eo - re->pmatch[1].rm_so);

		if (re->pmatch[4].rm_so != -1 && re->pmatch[4].rm_so != -1)
			s = xstrndup(b + re->pmatch[4].rm_so,
			    re->pmatch[4].rm_eo - re->pmatch[4].rm_so);
		else if (re->pmatch[5].rm_so != -1 &&
		    re->pmatch[5].rm_so != -1)
			s = xstrndup(b + re->pmatch[5].rm_so,
			    re->pmatch[5].rm_eo - re->pmatch[5].rm_so);
		else
			s = xstrndup(b + re->pmatch[7].rm_so, strtoul(b +
			    re->pmatch[6].rm_so, NULL, 10));

		v = reverse_namespace(s, ssn->ns.prefix, ssn->ns.delim);
		n = strlen(v);

		if (!xstrcasestr(a, "\\NoSelect")) {
			xstrncpy(m, v, n);
			m += n;
			xstrncpy(m++, "\n", 1);
		}

		if (!xstrcasestr(a, "\\NoInferiors") &&
		    (!(ssn->capabilities & CAPABILITY_CHILDREN) ||
		    ((ssn->capabilities & CAPABILITY_CHILDREN) &&
		    (xstrcasestr(a, "\\HasChildren")) &&
		    !xstrcasestr(a, "\\HasNoChildren")))) {
			xstrncpy(f, v, n);
			f += n;
			xstrncpy(f++, "\n", 1);
		}

		b += re->pmatch[0].rm_eo;

		xfree(a);
		xfree(s);
	}

	return r;
}


/*
 * Process the data that server sent due to IMAP LSUB client request.
 */
int
response_lsub(session *ssn, int tag, char **mboxs, char **folders)
{
	int r, n;
	char *b, *a, *s, *m, *f;
	const char *v;
	regexp *re;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	m = *mboxs = (char *)xmalloc((strlen(ibuf.data) + 1) * sizeof(char));
	f = *folders = (char *)xmalloc((strlen(ibuf.data) + 1) * sizeof(char));
	*m = *f = '\0';

	re = &responses[DATA_RESPONSE_LSUB];

	b = ibuf.data;
	while (!regexec(re->preg, b, re->nmatch, re->pmatch, 0)) {
		a = xstrndup(b + re->pmatch[1].rm_so,
		    re->pmatch[1].rm_eo - re->pmatch[1].rm_so);

		if (re->pmatch[4].rm_so != -1 && re->pmatch[4].rm_so != -1)
			s = xstrndup(b + re->pmatch[4].rm_so,
			    re->pmatch[4].rm_eo - re->pmatch[4].rm_so);
		else if (re->pmatch[5].rm_so != -1 &&
		    re->pmatch[5].rm_so != -1)
			s = xstrndup(b + re->pmatch[5].rm_so,
			    re->pmatch[5].rm_eo - re->pmatch[5].rm_so);
		else
			s = xstrndup(b + re->pmatch[7].rm_so, strtoul(b +
			    re->pmatch[6].rm_so, NULL, 10));

		v = reverse_namespace(s, ssn->ns.prefix, ssn->ns.delim);
		n = strlen(v);

		if (!xstrcasestr(a, "\\NoSelect")) {
			xstrncpy(m, v, n);
			m += n;
			xstrncpy(m++, "\n", 1);
		}

		if (!xstrcasestr(a, "\\NoInferiors") &&
		    (!(ssn->capabilities & CAPABILITY_CHILDREN) ||
		    ((ssn->capabilities & CAPABILITY_CHILDREN) &&
		    (xstrcasestr(a, "\\HasChildren")) &&
		    !xstrcasestr(a, "\\HasNoChildren")))) {
			xstrncpy(f, v, n);
			f += n;
			xstrncpy(f++, "\n", 1);
		}

		b += re->pmatch[0].rm_eo;

		xfree(a);
		xfree(s);
	}

	return r;
}


/*
 * Process the data that server sent due to IMAP SEARCH client request.
 */
int
response_search(session *ssn, int tag, char **mesgs)
{
	int r;
	regexp *re;
	char *b, *m;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	re = &responses[DATA_RESPONSE_SEARCH];

	b = ibuf.data;
	m = NULL;
	while (!regexec(re->preg, b, re->nmatch, re->pmatch, 0)) {
		if (!*mesgs) {
			m = *mesgs = (char *)xmalloc((strlen(ibuf.data) + 1) *
			    sizeof(char));
			*m = '\0';
		}

		xstrncpy(m, b + re->pmatch[1].rm_so, re->pmatch[1].rm_eo -
		    re->pmatch[1].rm_so);
		m += re->pmatch[1].rm_eo - re->pmatch[1].rm_so;
		xstrncpy(m++, " ", 1);

		b += re->pmatch[0].rm_eo;
	}

	return r;
}


/*
 * Process the data that server sent due to IMAP FETCH FAST client request.
 */
int
response_fetchfast(session *ssn, int tag, char **flags, char **date,
    char **size)
{
	int r;
	char *s;
	regexp *re;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	re = &responses[DATA_RESPONSE_FETCH];
	if (!regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0)) { 
		s = xstrndup(ibuf.data + re->pmatch[1].rm_so,
		    re->pmatch[1].rm_eo - re->pmatch[1].rm_so);

		re = &responses[DATA_RESPONSE_FETCH_FLAGS];
		if (!regexec(re->preg, s, re->nmatch, re->pmatch, 0))
			*flags = xstrndup(s + re->pmatch[1].rm_so,
			    re->pmatch[1].rm_eo - re->pmatch[1].rm_so);

		re = &responses[DATA_RESPONSE_FETCH_DATE];
		if (!regexec(re->preg, s, re->nmatch, re->pmatch, 0))
			*date = xstrndup(s + re->pmatch[1].rm_so,
			    re->pmatch[1].rm_eo - re->pmatch[1].rm_so);

		re = &responses[DATA_RESPONSE_FETCH_SIZE];
		if (!regexec(re->preg, s, re->nmatch, re->pmatch, 0))
			*size = xstrndup(s + re->pmatch[1].rm_so,
			    re->pmatch[1].rm_eo - re->pmatch[1].rm_so);
	}

	return r;
}


/*
 * Process the data that server sent due to IMAP FETCH BODY[HEADER] client
 * request.
 */
int
response_fetchheader(session *ssn, int tag, char **header,
    unsigned long int *len)
{
	int r;
	regexp *re;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	re = &responses[DATA_RESPONSE_FETCH_HEADER];
	if (!regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0)) { 
		*header = ibuf.data + re->pmatch[2].rm_so;
		*len = strtoul(ibuf.data + re->pmatch[1].rm_so, NULL, 10);
	}

	return r;
}


/*
 * Process the data that server sent due to IMAP FETCH BODY[TEXT] client
 * request.
 */
int
response_fetchbody(session *ssn, int tag, char **body,
    unsigned long int *len)
{
	int r;
	regexp *re;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	re = &responses[DATA_RESPONSE_FETCH_BODY];
	if (!regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0)) { 
		if (re->pmatch[2].rm_so != -1 && re->pmatch[2].rm_eo != -1) {
			*body = ibuf.data + re->pmatch[3].rm_so;
			*len = strtoul(ibuf.data + re->pmatch[2].rm_so, NULL,
			    10);
		} else {

			*body = ibuf.data + re->pmatch[4].rm_so;
			*len = re->pmatch[4].rm_eo - re->pmatch[4].rm_so;
		}
	}

	return r;
}


/*
 * Process the data that server sent due to IMAP FETCH BODY[HEADER.FIELDS ()]
 * client request.
 */
int
response_fetchfields(session *ssn, int tag, char **fields,
    unsigned long int *len)
{
	int r;
	regexp *re;

	if ((r = response_generic(ssn, tag)) == -1)
		return -1;

	re = &responses[DATA_RESPONSE_FETCH_FIELDS];
	if (!regexec(re->preg, ibuf.data, re->nmatch, re->pmatch, 0)) { 
		*fields = ibuf.data + re->pmatch[2].rm_so;
		*len = strtoul(ibuf.data + re->pmatch[1].rm_so, NULL, 10);
	}

	return r;
}
