#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/select.h>
#include <fcntl.h>

#include "imapfilter.h"
#include "session.h"

#ifndef NO_SSLTLS
#include <openssl/ssl.h>
#include <openssl/err.h>
#endif


/*
 * Connect to mail server.
 */
int
open_connection(session *ssn, const char *server, unsigned int port,
    const char *sprotocol)
{
	struct sockaddr_in sa;
	struct hostent *he;

#ifdef NO_SSLTLS
	if (sprotocol) {
		error("SSL not supported by this build\n");
		return -1;
	}
#endif

	memset((char *)&sa, 0, sizeof(struct sockaddr_in));

	ssn->socket = socket(AF_INET, SOCK_STREAM, 0);

	if (ssn->socket < 0) {
		error("create socket; %s\n", strerror(errno));
		return -1;
	}
	if (!(he = gethostbyname(server))) {
		error("get network host entry of %s; %s\n", server,
		    strerror(errno));
		close_connection(ssn);
		return -1;
	}
	sa.sin_family = AF_INET;
	sa.sin_port = htons(port);
	sa.sin_addr = *(struct in_addr *)he->h_addr;

	if (connect(ssn->socket, (struct sockaddr *)&sa,
	    sizeof(struct sockaddr))) {
		error("initiating connection to %s; %s\n", server,
		    strerror(errno));
		close_connection(ssn);
		return -1;
	}
#ifndef NO_SSLTLS
	if (sprotocol)
		if (!open_secure_connection(ssn, sprotocol))
			return ssn->socket;
		else
			return -1;
	else
		ssn->ssl = NULL;
#endif

	return ssn->socket;
}


#ifndef NO_SSLTLS
/*
 * Initialize SSL/TLS connection.
 */
int
open_secure_connection(session *ssn, const char *protocol)
{
	int e;
	SSL_CTX *ctx;
	SSL_METHOD *method;

	method = NULL;

	SSL_library_init();
	SSL_load_error_strings();

	if (!strncasecmp(protocol, "tls1", 4))
		method = TLSv1_client_method();
	else if (!strncasecmp(protocol, "ssl3", 4) ||
	    !strncasecmp(protocol, "ssl2", 4))
		method = SSLv23_client_method();

	if (!(ctx = SSL_CTX_new(method)))
		goto fail;

	if (!(ssn->ssl = SSL_new(ctx)))
		goto fail;

	SSL_set_fd(ssn->ssl, ssn->socket);

	if ((e = SSL_connect(ssn->ssl)) <= 0) {
		SSL_get_error(ssn->ssl, e);
		error("initiating SSL connection; %s\n",
		    ERR_error_string(ERR_get_error(), NULL));
		goto fail;
	}
	if (get_cert(ssn) == -1)
		goto fail;

	SSL_CTX_free(ctx);

	return 0;

fail:
	ssn->ssl = NULL;
	SSL_CTX_free(ctx);

	return -1;
}
#endif				/* NO_SSLTLS */


/*
 * Disconnect from mail server.
 */
int
close_connection(session *ssn)
{
	int r;

	r = 0;

#ifndef NO_SSLTLS
	close_secure_connection(ssn);
#endif

	if (ssn->socket != -1) {
		r = close(ssn->socket);
		ssn->socket = -1;

		if (r == -1)
			error("closing socket; %s\n", strerror(errno));
	}
	return r;
}


#ifndef NO_SSLTLS
/*
 * Shutdown SSL/TLS connection.
 */
int
close_secure_connection(session *ssn)
{

	if (ssn->ssl) {
		SSL_shutdown(ssn->ssl);
		SSL_free(ssn->ssl);
		ssn->ssl = NULL;
	}

	return 0;
}
#endif


/*
 * Read data from socket.
 */
ssize_t
socket_read(session *ssn, char *buf, size_t len)
{
	int f, s, t;
	ssize_t r;
	fd_set fds;

	struct timeval tv;

	struct timeval *tvp;

	r = 0;
	s = 1;
	tvp = NULL;

	memset(buf, 0, len + 1);

	t = (int)(get_option_number("timeout"));
	if (t > 0) {
		tv.tv_sec = t;
		tv.tv_usec = 0;
		tvp = &tv;
	}
	f = fcntl(ssn->socket, F_GETFL, 0);
	fcntl(ssn->socket, F_SETFL, f | O_NONBLOCK);

	FD_ZERO(&fds);
	FD_SET(ssn->socket, &fds);

#ifndef NO_SSLTLS
	if (ssn->ssl) {
		while (SSL_pending(ssn->ssl) > 0 || ((s =
		    select(ssn->socket + 1, &fds, NULL, NULL, tvp)) > 0 &&
		    FD_ISSET(ssn->socket, &fds))) {
			r = (ssize_t) SSL_read(ssn->ssl, buf, len);

			if (r > 0)
				break;

			switch (SSL_get_error(ssn->ssl, r)) {
			case SSL_ERROR_WANT_READ:
			case SSL_ERROR_WANT_WRITE:
				continue;
			case SSL_ERROR_ZERO_RETURN:
				goto fail;
			case SSL_ERROR_SYSCALL:
			case SSL_ERROR_SSL:
				error("reading data; %s\n",
				    ERR_error_string(ERR_get_error(), NULL));
				goto fail;
			default:
				goto fail;
			}
		}
	} else
#endif
	{
		if ((s = select(ssn->socket + 1, &fds, NULL, NULL, tvp)) > 0 &&
		    FD_ISSET(ssn->socket, &fds))
			r = read(ssn->socket, buf, len);

		if (r == -1) {
			error("reading data; %s", strerror(errno));
			goto fail;
		} else if (r == 0) {
			goto fail;
		}
	}

	fcntl(ssn->socket, F_SETFL, f);

	if (s == -1) {
		error("waiting to read from socket; %s\n", strerror(errno));
		goto fail;
	} else if (s == 0) {
		error("timeout period expired while waiting to read data\n");
		goto fail;
	}

	return r;
fail:
	close_connection(ssn);

	return -1;

}


/*
 * Write data to socket.
 */
ssize_t
socket_write(session *ssn, const char *buf, size_t len)
{
	int f, s, t;
	ssize_t w, wt;
	fd_set fds;

	struct timeval tv;

	struct timeval *tvp = NULL;

	w = wt = 0;
	s = 1;

	t = (int)(get_option_number("timeout"));
	if (t > 0) {
		tv.tv_sec = t;
		tv.tv_usec = 0;
		tvp = &tv;
	}
	f = fcntl(ssn->socket, F_GETFL, 0);
	fcntl(ssn->socket, F_SETFL, f | O_NONBLOCK);

	FD_ZERO(&fds);
	FD_SET(ssn->socket, &fds);

	while (len) {
#ifndef NO_SSLTLS
		if (ssn->ssl) {
			while ((s = select(ssn->socket + 1, NULL, &fds, NULL,
			    tvp) > 0 && FD_ISSET(ssn->socket, &fds))) {
				w = (ssize_t) SSL_write(ssn->ssl, buf, len);

				if (w > 0)
					break;

				switch (SSL_get_error(ssn->ssl, w)) {
				case SSL_ERROR_WANT_READ:
				case SSL_ERROR_WANT_WRITE:
					continue;
				case SSL_ERROR_ZERO_RETURN:
					goto fail;
				case SSL_ERROR_SYSCALL:
				case SSL_ERROR_SSL:
					error("writing data; %s\n",
					    ERR_error_string(ERR_get_error(),
					    NULL));
					goto fail;
				default:
					error("undefined ssl error "
					    "while writing data\n");
					goto fail;
				}
			}
		} else
#endif
		{
			if ((s = select(ssn->socket + 1, NULL, &fds, NULL,
			    tvp)) > 0 && FD_ISSET(ssn->socket, &fds))
				w = write(ssn->socket, buf, len);

			if (w == -1) {
				error("writing data; %s", strerror(errno));
				goto fail;
			} else if (w == 0) {
				goto fail;
			}
		}

		if (w > 0) {
			len -= w;
			buf += w;
			wt += w;
		}
	}

	fcntl(ssn->socket, F_SETFL, f);

	if (s == -1) {
		error("waiting to write to socket; %s\n", strerror(errno));
		goto fail;
	} else if (s == 0) {
		error("timeout period expired while waiting to write data\n");
		goto fail;
	}

	return wt;
fail:
	close_connection(ssn);

	return -1;
}
