""" Experimental alternative database module, using SQL.
    ***********************************************************************
    begin                : 2003/Aug/23
    copyright            : (C) 2003 by Tuomas Airaksinen
    email                : tuomas.airaksinen|at|tuma.stc.cx
    $Id: sqlitedb.py,v 1.1.2.58 2003/09/20 05:02:17 niederberger Exp $    
    ***********************************************************************    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 US
    ***********************************************************************
    SQL code follows the Python Database API Specification v2.0:
    http://www.python.org/peps/pep-0249.html
"""
__version__ = "$Revision: 1.1.2.58 $"

try:
    import sys
    import os
    import traceback
    import md5                          # hashing filenames to generate thumbnails
    import random                       # for generating unique id's
    import imghdr                       # to determine image type
    import time                         # to test performance increase during development, and generating batch names
    import marshal                      # for loading and saving img databases
    from string import *
    import Error
except:
    Error.PrintTB("Error importing the necessary python modules. Unable to continue.")
    sys.exit()
try:
    import EXIF
except:
    Error.PrintTB("Error importing EXIF module.")
try:
    from qt import *
except:
    Error.PrintTB("You system doesn't seem to have PyQT installed. Please install it before running this application. See http://www.riverbankcomputing.co.uk/pyqt/download.php and http://imgseek.sourceforge.net/requirements.html")
try:
    import imgdb
except:
    Error.PrintTB("Warning: Unable to load the C++ extension \"imgdb.so\" module. Make sure you installed imgSeek correctly, and email any install bug to \"imgseek-devel@lists.sourceforge.net\".")
    sys.exit()

####### Optional modules
## IPTC from PIL
try:
    import IptcExtract
except:
    pass
try:
    import Image
except:
    pass

####### SQL function/modules mapping
# all entries should have a 'module' entry, telling the external db module to be imported as SqlModule
SqlPluginMap = {
    'mysql':
        {'module': 'MySQLdb',
         },
    'sqlite':
        {'module': 'sqlite',
         },
    'postgres':
        {'module': 'pyPgSQL',
         'submodule': 'PgSQL',
         },
    }

class AddFilter:
    """This class should hold all restrictions that a user want to apply on a database Add """
    
    def __init__(self,env):
        # igntext and minsize and mindim are file criteria to ignore
        self.igntext = ""
        self.minsize = None
        self.mindim = 0
        self.ignext = env.doc_ext
        self.exexif = 0
        self.exiptc = 0
        self.mounted = 0
        self.probeext = 0
        self.addfname = ""
        self.volid = 1
        self.groupid = 1
        self.removeEmptyGroup = 0       # if restr.removeEmptyGroup is set and this dir has no images, then remove the group this dir should belong to
        
    def getFromDialog(self,dlg):
        """get restrictions from the add tab"""
        ## not sure if this code should be on imgSeek::addImageBtn_clicked or here.
        pass

class SqlDB:
    """imgSeek SQL backend. This is class should act just like ImgDB
    """
    print "Imported SqlDB: %s" % __version__
    
    def __init__(self, env, dbType, dbArgs=None, dbFile=None):
        """ SqlDB constructor

        @param dbfilename: database filename to read. Only makes sense with sqlite. When not present, a new database file will be created on ~/.imgseek/
        """        
        # check db type sanity
        assert SqlPluginMap.has_key(dbType), "Invalid database type"
        self.dbtype = dbType
        # import proper db
        #  __import__ docstring:
        #    When importing a module from a package, note that __import__('A.B', ...)
        #    returns package A when fromlist is empty, but its submodule B when
        #    fromlist is not empty.
        # bind desired sql module to local object
        modMap = SqlPluginMap[dbType]
        print modMap
        if modMap.has_key('submodule'): 
            self.SqlModule = __import__(modMap['module']+'.'+modMap['submodule'], globals(), locals(), [modMap['submodule']])
        else:
            self.SqlModule = __import__(modMap['module'])
        self.do_commit = 1               #whether to save the changes to DB after each execute(). Set it off if you change db alot.
        self.env = env
        self.imgdb = imgdb              # bind c++ img database module to local object
        self.dbargs = dbArgs            # sql database connection arguments (a dictionary, will be used on a dbModule.connect call)
        self.dbfile = dbFile            # c++ database module filename (the .img file)
        #### image metadata:
        self.imgexts = [ 'jpeg', 'jpg', 'gif', 'png', 'rgb', 'pbm', 'pgm', 'ppm', 'tiff', 'tif', 'rast', 'xbm', 'bmp' ] # to help determining img format from extension
        #### Thumbnail stuff
        self.thdir = os.path.expanduser(os.path.join("~",".thumbnails","")) # path for storing thumbnails. Calculate it only once here.
        self.thdir2 = os.path.expanduser(os.path.join("~",".thumbnails","normal","")) # path for storing thumbnails. Calculate it only once here.
        # just to make sure the thumbnail dir exists, so I dont need to be checking it over and over on self.getThumb()
        if not os.path.exists(self.thdir):
            os.mkdir(self.thdir)
        if not os.path.exists(self.thdir2):
            os.mkdir(self.thdir2)        
        #### Callbacks (lists of functions that should be called on changes            
        self.wnd = None
        self.app = None
        # cbs maps what could change to a list of functions that will get callled when that thing changes
        # these functions should accept a map describing what changed on db
        self.cbs = {}
        self.cbs["Batch"] = []
        self.cbs["Dir"] = []
        self.cbs["Group"] = []
        self.cbs["SimGroup"] = []
        self.cbs["Img"] = []
        self.cbs["Volume"] = []
        self.cbs["Meta"] = []        
        self.cbs["Database"] = []         # generic changes like, dbname, location. etc
        self.sysdirbmCb = []              # list of functions that should get called when a bookmark is changed.
        ### Init C++ db engine
        imgdb.initDbase()
        ### Misc inits
        self.textqueryhistory = []      # key is text str, value is list of [field,value] pairs
        self.imageParms = ["Description","Dimensions","Filesize","Filename","Format","Modify date","Database date","Modify_date_epoch"] # default meta keys all image should have
        self.metafields = self.imageParms[:] # history of entered metafields        
        self.readonlyfields = ["Filesize","Filename","Format","Dimensions","Modify date","Database date","Volume","Mounted"] # default readonly metadata
        self.invisiblefields = ["ViewRotate","Modify_date_epoch"] # fields which shouldn't even appear on the metadata dialog
        self.openGroups = []
        self.openDirs = []
        if dbArgs and dbFile:
            self.opendb()

    def __del__(self):
        self.closedb()
        
    ########## DB MANAGEMENT ****************
    def opendb(self,dbfilename=None,dbArgs=None):
        """open metadata db, losing current db. Will also tell imgdb C++ module to load corresponding image data

        @param dbfilename: full path filename for the c++ image module database coefficients file
        @param dbAargs: database connection arguments (dictionary)
        @return: success
        """
        if dbfilename:
            self.dbfile = dbfilename
        if dbArgs:
            self.dbargs = dbArgs
            
        self.dbfile = os.path.expanduser(self.dbfile)
        assert self.dbfile and self.dbtype and self.dbargs , "NULL dbfile, dbtype or dbargs"
        
        print "Opening coefficients database file "+self.dbfile+" ..."
        if not imgdb.loaddb(self.dbfile):
            print "Error loading image database"

        nonKwParms = ()                 # use it for db connectors that require non-keyword parms
        if "db" in self.dbargs.keys():
            self.dbargs['db'] = os.path.expanduser(self.dbargs['db'])
        print "sql Module:", self.SqlModule
        try:
            self.conn = apply(self.SqlModule.connect, nonKwParms , self.dbargs )
            self.cursor = self.conn.cursor()
        except:
            Error.PrintTB("Trying to connect to database and retrieve a cursor.")
            return 0
        try:                            # get db file size for statistical purposes
            self.kbsize = os.stat(self.dbfile).st_size
        except:
            self.kbsize = 0
        print "Done."
        print "opendb callbacks" 
        self.changedDB({"scope":"Database","reason":"opened","subject":self.dbfile})
        for cb in self.sysdirbmCb: cb()
        return 1

    def closedb(self):
        print "Closing relational database"
        imgdb.closeDbase()
        self.conn.commit()
        self.cursor.close()
        del self.cursor
        self.conn.close()
        del self.conn

    def execute(self, sql):
#        print "exec: " + sql
        try:
            self.cursor.execute(sql)
        except:
            print "Error: SQL command " + sql + " failed!"
            pass
        if self.do_commit:
            self.conn.commit()

    def reset(self):
        """drops and then creates all necessary tables"""
        self.cursor.close()
        del self.cursor
        self.conn.autocommit = 1
        self.cursor = self.conn.cursor()
        SQL = [ 
            'DROP TABLE Pics;',
            'DROP TABLE Groups;',
            'DROP TABLE PicsInGroups;',
            'DROP TABLE GroupsInGroups;',
            'DROP TABLE Metadata;',
            'DROP TABLE Volumes;',
            'DROP TABLE Batches;',
            'DROP TABLE ContentsInBatches;',
            'DROP TABLE SimGroups;',
            'DROP TABLE PicsInSimGroups;',
            'DROP TABLE ContSimGroups;',
            'DROP TABLE PicsInContSimGroups;',
            'DROP TABLE ColorSimGroups;',
            'DROP TABLE PicsInColorSimGroups;',
            'DROP TABLE DateSimGroups;',
            'DROP TABLE PicsInDateSimGroups;',
            'DROP TABLE FileSimGroups;',
            'DROP TABLE PicsInFileSimGroups;',
            'DROP TABLE Dirs;',
            'DROP TABLE DirsInDirs;',
            # it's better to have a lot of constraints and see errors occur while doing an insert, than seeing the database corrupt silently. 
            'CREATE TABLE Pics (id INT PRIMARY KEY, filename TEXT NOT NULL, dirid INT NOT NULL, filesize TEXT, modifydate TEXT);',
            'CREATE TABLE Groups (gid INT PRIMARY KEY, name TEXT NOT NULL, descr TEXT);',
            'CREATE TABLE PicsInGroups (id INT, gid INT NOT NULL DEFAULT 1, UNIQUE(id,gid) );', # when gid is ommited, Orphan group is assumed (gid=1) 
            'CREATE TABLE GroupsInGroups (gid1 INT, gid2 INT NOT NULL DEFAULT 1, UNIQUE(gid1,gid2));', # when gid is ommited, Orphan group is assumed (gid=1)
            'CREATE TABLE Metadata (id INT, akey TEXT NOT NULL, value TEXT);',
            'ALTER TABLE Metadata ADD INDEX(id);', #works for mysql. postgresql has CREATE INDEX. dunno about sqlite. #TODO: test
            'CREATE TABLE Volumes (volid INT PRIMARY KEY, path TEXT, descr TEXT, name TEXT NOT NULL);',
            'CREATE TABLE Batches ( bid INT PRIMARY KEY, name TEXT);',
            'CREATE TABLE ContentsInBatches ( bid INT, id INT, type TEXT);',
            'CREATE TABLE SimGroups ( sid INT PRIMARY KEY, mainimg INT, name TEXT NOT NULL);',
            'CREATE TABLE PicsInSimGroups ( id INT, sid INT NOT NULL, UNIQUE(id,sid));',
            'CREATE TABLE ContSimGroups ( sid INT PRIMARY KEY, mainimg INT, name TEXT );',
            'CREATE TABLE PicsInContSimGroups ( id INT, sid INT NOT NULL, UNIQUE(id,sid));',
            'CREATE TABLE ColorSimGroups ( sid INT PRIMARY KEY, mainimg INT, name TEXT );',
            'CREATE TABLE PicsInColorSimGroups ( id INT, sid INT NOT NULL, UNIQUE(id,sid));',
            'CREATE TABLE DateSimGroups ( sid INT PRIMARY KEY, mainimg INT, name TEXT );',
            'CREATE TABLE PicsInDateSimGroups ( id INT, sid INT NOT NULL, UNIQUE(id,sid));',
            'CREATE TABLE FileSimGroups ( sid INT PRIMARY KEY, mainimg INT, name TEXT );',
            'CREATE TABLE PicsInFileSimGroups ( id INT, sid INT NOT NULL, UNIQUE(id,sid));',
            'CREATE TABLE Dirs ( dirid INT PRIMARY KEY, path TEXT NOT NULL, descr TEXT, volid INT NOT NULL);',
            'CREATE TABLE DirsInDirs ( dirid1 INT PRIMARY KEY, dirid2 INT NOT NULL, UNIQUE(dirid1, dirid2));' ] #  /* dirid1=parent   dirid2=child */' 
        
        for i in SQL:
            try:
                self.execute(i)
            except:
                if i.find("DROP") == -1: # ommit errors related to table DROPs
                    Error.PrintTB("Error while converting, couldn't do SQL statement \"%s\"" % i)
        self.cursor.close()
        del self.cursor
        self.conn.autocommit = 0
        self.cursor = self.conn.cursor()
                
        self.openGroups = []            # group ids with listvierw item expanded (so ui will set the listview state when it changed)
        self.openDirs = []              # dir ids with listvierw item expanded (so ui will set the listview state when it changed)
        self.metafields = self.imageParms[:] # history of entered metafields        
        self.textqueryhistory = []      # key is text str, value is list of [field,value] pairs
        self.sysdirbm = []              # browse by sysdir bookmark list. List of paths
        self.dbAddGroup(1, "Orphan", "Images which haven't been assigned to any group.")
        self.dbAddVolume(1, os.sep, "Default volume","Local filesystem") 
        self.kbsize = 0
        self.curBatch = 1               # current batch being added.
        imgdb.resetdb()
        self.changedDB({"scope":"Database","reason":"reset"})
 
    def refreshDB(self,cb = None,restr = None):
        """call to scan db dirs for new files

        @param cb: Defaults to None. If present, should be a QProgressBar or QProgressDialog with a QApp parent().
        @param restr: restrictions to consider when adding new files
        @return: count of new files found and sucessfully added to db
        """
        
        if not restr:
            restr = AddFilter(self.env)
        restr.removeEmptyGroup = 1
        restr.groupid = 1
        restr.removeEmptyGroup = 0
        
        for vid in self.getVolumeList():
            lst = []
            self.crawlVolumeForDir(vid,lst)
            for did in lst:                
                path = self.getDirsPath(did)
                if not os.path.exists(path): continue
                newid = self.addDir(vid,path,1,[],0,-1,restr = restr)
        self.remove_dead()
        if cb:
            cb.cancel()
            cb.hide()
            cb.close()
            del cb
        return count
        #TODO: add other initializations from reset()

    def convert_old(self, old_db):
        """call to init current db object from an old ImgDB data file

        @param old_db: full path for a classic marshalled img-db.iqd file.
        """
        #TODO3: see how the conversions of old classic db files would work.
        print "Converting, please wait..."
        self.do_commit = 0
        self.reset()
        #pictures
        for id, value in old_db.img.items():
            self.dbAddPics(id, value[0], value[1], old_db.meta[id]["Filesize"], old_db.meta[id]["Modify date"])
        self.conn.commit()
        #metadata
        for id, data in old_db.meta.items():
            for key, val in data.items():
                self.setMetadata(id, key, val)
        self.conn.commit()
        #dirs
        for dirid, value in old_db.dirs.items():
            self.dbAddDir(dirid, value[0], value[5], value[6])
            self.dbAddDirsInDirs(dirid, value[1]);
        self.conn.commit()
        #volumes
        for volid, value in old_db.volumes.items():
            self.dbAddVolume(volid, value[1], value[2], value[3])
        self.conn.commit()
        #groups
        for gid, value in old_db.groups.items():
            self.dbAddGroup(gid, value[0], value[1])
            for id in value[3]:
                self.dbAddImageInGroup(id, gid)
            for gid2 in value[4]:
                self.dbAddGroupsInGroups(gid2, gid)
        self.conn.commit()
        for gid in self.getGroupList():
            if self.getGroupsParent(gid) == None:
                self.dbAddGroupsInGroups(gid, -1)
        #TODO: the following need to be converted ?? investigate
        #simgroups * blah, who needs those :) (no dbAdd-method available)
        #contsimgroups *same here
        #datesimgroups ...
        #filesimgroups ...
        self.conn.commit()
        self.do_commit = 1
        print "... conversion finished." 

    def changefname(self,nname):
        """ call to change db filename name
        """
        self.fname = nname
        print "NOT IMPLEMENTED" #TODO
        return
        self.changedDB({"scope":"Database","reason":"changedname","subject":nname})        

    def savedb(self, param):
        raise Error.deprecationError, "savedb: deprecated, do not use"    

    def changedDB(self, param):
        raise Error.deprecationError, "changeDB: deprecated, do not use"

    ########### GET METHODS #################
    #General get methods
    def getTheseInThat(self, table, tag1, tag2 = None, val2 = None): #list of values
        if tag2 == None:
            self.execute('SELECT %s FROM %s;'%(tag1, table))
        else:
            self.execute('SELECT %s FROM %s WHERE %s=\'%s\';'%(tag1, table, tag2, val2))
        return  [ i for i, in self.cursor.fetchall() ]

    def getSomething(self, table, tag1, tag2, val2): #one value
        self.execute('SELECT %s FROM %s WHERE %s = \'%s\';'%(tag1, table, tag2, val2))
        retval = self.cursor.fetchone()
        try:
            return retval[0]
        except:
            return None

    # LISTS:
    def getImageList(self):
        return self.getTheseInThat("Pics", "id")

    def getGroupList(self):
        return self.getTheseInThat("Groups", "gid")

    def getDirList(self):
        return self.getTheseInThat("Dirs", "dirid")

    def getVolumeList(self):
        return self.getTheseInThat("Volumes", "volid")

    def getBatchList(self):
        return self.getTheseInThat("Batches", "bid")

    def getSimGroupList(self):
        return self.getTheseInThat("SimGroups", "sid")

    # NAMES & etc, returning one value only
    def getImageByName(self, name):
        return self.getSomething("Pics", "id", "filename", name)

    def getVolumeByName(self, name):
        return self.getSomething("Volumes", "volid", "name", name)

    def getVolumeName(self, volid):
        return self.getSomething("Volumes", "name", "volid", volid)

    def getPicsFileName(self, id):
        return self.getSomething("Pics", "filename", "id", id)

    def getDirsParent(self, dirid):
        return self.getSomething("DirsInDirs", "dirid2", "dirid1", dirid)

    def getGroupsParent(self, gid):
        # what if it belongs to many groups? not possible atm, but..
        return self.getSomething("GroupsInGroups", "gid2", "gid1", gid)

    def getBatchName(self, bid):
        return self.getSomething("Batches", "name", "bid", bid)

    def getPicsFilesize(self, id):
        return self.getSomething("Pics", "modifydate", "id", id)
        
    def getPicsFilesize(self, id):
        return self.getSomething("Pics", "filesize", "id", id)
        
    def getPicsShortFilename(self, id): #TODO
        return os.path.split(self.getPicsFilename(id))[-1]
        #return self.getSomething("Pics", "filename", "id", id)
        
    def getPicsFilename(self, id):
        return self.getSomething("Pics", "filename", "id", id)

    def getGroupsName(self, gid):
        return self.getSomething("Groups", "name", "gid", gid)

    def getSimGroupsMainImageId(self,sid):
        return self.getSomething("SimGroups", "mainimg", "sid", sid)

    def getDirsPath(self, dirid):
        return self.getSomething("Dirs", "path", "dirid", dirid)

    def getMetadata(self, id, key):
        #return "not implemented!"
        self.execute('SELECT value FROM Metadata WHERE id = %d AND akey = \'%s\';'%(id,key))
        return self.cursor.fetchone()[0]

    def isGroupChildToParent(self, childgid, parentgid):
    #TODO4: develop some SQL algorithm (if possible) to find out this. Will be faster.
        for son in self.getGroupsInGroups(parentgid):
            if son == childgid:
                return 1
            else:
                return self.isGroupChildToParent(childgid, son)

    #CONTENT LISTS, returning many values
    def getDirsInVolume(self, volid):
        return self.getTheseInThat("Dirs", "dirid", "volid", volid);

    def getPicsInDirs(self, dirid):  # should use another GET method
        return self.getTheseInThat( "Pics", "id", "dirid",dirid);

    def getDirsInDirs(self, dirid):
        return self.getTheseInThat("DirsInDirs", "dirid1", "dirid2", dirid);

    def getGroupsInGroups(self,gid):
        return self.getTheseInThat("GroupsInGroups", "gid1", "gid2", gid);

    def getPicsInGroups(self,gid):
        return self.getTheseInThat("PicsInGroups", "id", "gid", gid);

    def getSimGroupsSimilarImages(self, sid):
        return self.getTheseInThat("PicsInSimGroups", "id", "sid", sid)

    def getContentsInBatch(self, bid):
        self.execute('SELECT type, id FROM ContentsInBatches WHERE bid = %d;'%bid)
        return self.cursor.fetchall()

    ########### SET METHODS #################
    def setSomething(self, table, settag, setval, tag, val):
        try:
            self.execute('UPDATE %s SET %s = \'%s\' WHERE %s = \'%s\';'%(table, settag, setval, tag, val))
        except:
            self.execute('INSERT INTO %s VALUES (\'%s\', \'%s\');'%(table, settag, setval))

    def setVolumesName(self, volid, name):
        self.setSomething("Volumes", "name", name, "volid", volid)

    def setVolumesDescription(self, volid, descr):
        self.setSomething("Volumes", "descr", descr, "volid", volid)

    def setPicsFilename(self, id, filename):
        self.setSomething("Pics", "filename", filename, "id", id)

    def setDirsVolId(self, dirid, volid):
        self.setSomething("Dirs", "dirid", dirid, "volid", volid)

    def setGroupsDescription(self, gid, descr):
        self.setSomething("Groups", "descr", descr, "gid", gid)

    def setGroupName(self, gid, name):
        self.setSomething("Groups", "name", name, "gid", gid)

    def setMetadata(self, id, key, value):
        self.execute('INSERT INTO Metadata VALUES (%d,\'%s\', \'%s\');'%(id, key, value))

   ### dbAdd ###
    def dbAddSInS(self, table, tag1, val1, tag2, val2):
        """ add something in something"""
        #No need to test if a pair is already in, since most "pair" tables have a UNIQUE constraint
        self.execute('INSERT INTO %s VALUES (\'%s\', \'%s\');'%(table,val1,val2))

    def dbAddGroup(self, gid, name, descr):
        self.execute('INSERT INTO Groups VALUES (%d, \'%s\', \'%s\');'%(gid, name, descr))

    def dbAddBatch(self, bid, name):
        self.dbAddSInS("Batches", "bid", bid, "name", name)

    def dbAddVolume(self, volid, path, descr,  name):
        self.execute('INSERT INTO Volumes VALUES (%d, \'%s\', \'%s\', \'%s\');'%(volid, path, descr, name))

    def dbAddPics(self, id, filename, dirid, filesize=None, modifydate=None):
        if not filesize:
            try:
                self.prettysize(os.stat(fname).st_size)
            except:
                Error.PrintTB("Getting file size")                
                filesize = "0 b"
        if not modifydate:
            try:
                modifydate = time.strftime("%Y/%m/%d %H:%M:%S", time.gmtime(os.stat(filename).st_mtime))
            except:
                Error.PrintTB("Getting file modify time")
                modifydate = "Never"
                
        self.execute('INSERT INTO Pics VALUES (%d, \'%s\', %d, \'%s\',\'%s\');'%(id, filename, dirid, filesize, modifydate))

    def dbAddImageInGroup(self, id, gid):#should check if already exists
        self.dbAddSInS("PicsInGroups", "id", id, "gid", gid)

    def dbAddDirInVolume(self, dirid, volid):
        self.dbAddSInS("DirsInVolumes", "dirid", dirid, "volid", volid)

    def dbAddDir(self, dirid, path, descr, volid):
        self.execute('INSERT INTO Dirs VALUES (%d,\'%s\',\'%s\', %d);'%(dirid, path, descr, volid))

    def dbAddGroupsInGroups(self, gid1, gid2):
        self.dbAddSInS("GroupsInGroups", "gid1", gid1, "gid2", gid2)

    def dbAddDirsInDirs(self, dirid1, dirid2):
        self.dbAddSInS("DirsInDirs", "dirid1", dirid1, "dirid2", dirid2)

    #### dbDELETE ###
    def dbDeleteSomething(self, tables, tag, val):
        for i in tables:
            self.execute('DELETE FROM %s WHERE %s = \'%s\';'%(i, tag, val))

    def dbDeleteSInS(self, table, tag1, val1, tag2, val2):
        self.execute("DELETE FROM %s WHERE %s = \'%s\' AND %s = \'%s\';"%(table, tag1, val1, tag2, val2))

    def dbDeleteContentsInBatch(self, id, bid):
        self.dbDeleteSInS("ContentsInBatch", "bid", bid, "id", id)

    def dbDeleteDirsInDirs(self, dirid1, dirid2):
        self.dbDeleteSInS("DirsInDirs", "dirid1", dirid1, "dirid2", dirid2)

    def dbDeletePicsInGroups(self, id, gid):
        self.dbDeleteSInS("PicsInGroups", "id", id, "gid", gid)

    def dbDeleteGroupsInGroups(self, gid1, gid2):
        self.dbDeleteSInS("GroupsInGroups", "gid1", gid1, "gid2", gid2)

    def dbDeleteGroup(self, gid):
        self.dbDeleteSomething(["Groups", "PicsInGroups"], "gid", gid)
        self.dbDeleteSomething(["GroupsInGroups"], "gid2", gid)

    def dbClearBatch(self, bid):
        self.dbDeleteSomething(["ContentsInBatches"], "bid", bid)

    def dbDeleteAllBatches(self):
        self.execute("DELETE FROM Batches;")
        self.execute("DELETE FROM ContentsInBatches;")

    def dbDeleteAllMetadata(self,id):
        self.execute("DELETE FROM Metadata WHERE id=%d;"%id)

    def dbDeleteBatch(self, bid):
        self.dbDeleteSomething(["Batches", "ContentsInBatches"], "bid", bid)

    def dbDeleteDir(self, dirid):
        self.dbDeleteSomething(["Dirs", "Pics"], "dirid", dirid)
        self.dbDeleteSomething(["DirsInDirs"], "dirid2", dirid)

    def dbDeleteVolume(self, volid):
        self.dbDeleteSomething(["Volumes", "Dirs"], "volid", volid)

    def dbDeleteImage(self, id):
        where = ["Pics", "PicsInGroups", "PicsInSimGroups", "PicsInContSimGroups", "PicsInColorSimGroups", "PicsInDateSimGroups", "PicsInFileSimGroups", "Metadata"] 
        self.dbDeleteSomething(where, "id", id)

    ########### GROUP MANAGEMENT ************
    def copyGroup(self,sid,gid):
        """call to copy a group into another

        @param sid: source group id
        @param gid: destination group id. New group will be a child of group gid
        @return: new group id (ie, the id of the new group which is a copy of sid)
        """
        if self.isGroupAncestor(gid,sid):
            print "Attempt to move/copy parent to a child, ignoring request."
            return -1
        nid = self.newGroup(self.getGroupsName(sid)[:],pargid = gid)
        self.setGroupsDescription(nid, self.getGroupsName(sid)[:])
        for i in self.getPicsInGroups(sid):
            self.dbAddImageInGroup(i, nid)
        for sg in self.getGroupsInGroups(sid):
            self.copyGroup(sg,nid)
        return nid

    def copyObjectsToGroup(self, ids,gid):
        """ call to copy a group of objects to a destination group

        @param gid: destination group
        @param ids: is a list of maps. {'type':{Group,Img},'id':obj_id}
        @return: True if something got changed
        """        
        exitc = 0
        try:
            for id in ids:
                if id["type"]=="Group":
                    self.copyGroup(id["id"],gid)
                if id["type"]=="Img":
                    self.moveFileGroup(id["id"],gid)
            exitc = 1
        except:
            traceback.print_exc()
            exitc = 1
        self.changedDB({"scope":"Group","reason":"copiedimages"})
        return exitc

    def isGroupAncestor(self,chid,pid):
        """ check a group pid is the ancestor (directly or inderectly, no matter how many generations apart) of another group chid.

        To avoid infinite loops when moving/copying groups
        
        @param pid: id of the parent group to check.
        @param chid: id of the child group to check
        @return: result
        @rtype: boolean
        """
        return self.isGroupChildToParent(chid,pid) #TODO: remove either of these functions

    def moveGroup(self,fromid,toid):
        """ call to move groups

        @param fromid: id of the group that will be moved
        @param toid: id of the group that will become the parent of fromid
        """
        if fromid==1:
            print "Attempt to move Orphan group. Ignored request"
            return
        if fromid==toid:
            return 0
        if self.isGroupChildToParent(toid, fromid):
            print "Invalid move. Parent -> Child"
            return 0
        self.dbDeleteGroupsInGroups( fromid, self.getGroupsParent(fromid) ) # assert: can belong to only one group
        self.dbAddGroupsInGroups(fromid, toid)

    def addImageToGroup(self,id,gid):
        """append image to a group

        @param id: image id
        @param gid: group id. This image will now also belong to this group
        """
        self.dbAddImageInGroup(id,gid)
        self.changedDB({"scope":"Group","reason":"addedimage","subject":gid,"target":id})            

    def moveFileGroup(self,fromid,toid,fromgroup = -1):
        """ call to make file fromid to be a child of group toid
        if fromgroup is defined, erase this image from the group fromgroup (this would be a MOVE operation)
        """
        if fromid==toid:
            return
        if toid==-1:
            print "Invalid move: Only groups may be copied/moved to root. Ignoring request."
            return
        self.dbDeletePicsInGroups(fromid, fromgroup)
        self.addImageToGroup(fromid, toid)

    def moveObjectsToGroup(self, ids,gid): 
        """ call to move a group of objects to gid
        
        @param ids: a list of maps.
        @param gid: destination group
        @return: success or not
        """
        exitc = 0
        self.doNotSave = 1        
        try:
            for id in ids:
                if id["type"]=="Group":
                    self.moveGroup(id["id"],gid)
                if id["type"]=="Img":
                    if id.has_key("FromId"):
                        self.moveFileGroup(id["id"],gid,id["FromId"])
                    else:
                        self.moveFileGroup(id["id"],gid)
            exitc = 1
        except:
            traceback.print_exc()
            exitc = 1
        self.changedDB({"scope":"Group","reason":"movedimages"})
        return exitc

    def newGroup(self,text,pargid = -1):
        """create new group

        @param text: new name
        @param pargid: parent group id
        @return: new group id
        """
        nid = self.generateImageID()
        while nid in self.getGroupList():
            nid = self.generateImageID()
        self.dbAddGroup(nid, text, "")
        self.dbAddGroupsInGroups(nid, pargid)
        self.changedDB({"scope":"Group","reason":"new","subject":nid})        
        return nid

    def removeGroup(self,gid):
        """remove group from database
        child imgs will be moved to Orphan group
        
        @param gid: group id to be removed        
        """        
        if gid==1:
            print "Attempt to remove default Orphan group, aborting remove process."
            return
        # move child imgs to default orphan group (id = 1)
        try:
            for id in self.getPicsInGroups(gid):
                self.moveFileGroup(id,1)
        except:
            print "Strange error removing group and adding children to Orphan group"
            traceback.print_exc()
        # remove children groups
        for gid2 in self.getGroupsInGroups(gid):
            self.removeGroup(gid2)
        self.dbDeleteGroup(gid)
        self.changedDB({"scope":"Group","reason":"removed","subject":gid})

    def removeVolume(self,volid):
        """remove volume and all it's subdirs

        @param volid: volume id
        """
        for dirid in self.getDirsInVolume(volid):
            self.removeDir(dirid)
        self.dbDeleteVolume(volid)
        self.changedDB({"scope":"Volume","reason":"removed","subject":gid})

    def removeDir(self,dirid):
        """remove dir and all it's images

        @param dirid: dir id
        """        
        #remove child dirs
        for dirid2 in self.getDirsInDirs(dirid):
            self.removeDir(dirid2)
        #remove child imgs
        try:
            for id in self.getPicsInDirs(dirid):
                self.removeFile(id)
        except:
            traceback.print_exc()
        #remove the dir itself
        self.dbDeleteDir(dirid)
        self.changedDB({"scope":"Dir","reason":"removed","subject":gid})

    def newVolume(self,name):
        """ create a new volume with the provided name and return its id. -1 is returned on a failure

        @param name: new volume name
        """
        if self.getVolumeByName(name) != None:
            return -1
        nid = self.generateImageID()
        while nid in self.getVolumeList():
            nid = self.generateImageID()
        self.dbAddVolume(nid, "", "No description given.", name)
        self.changedDB({"scope":"Volume","reason":"new","subject":nid})                
        return nid

    def renameVolume(self,vid,text):
        """ rename volume

        @param vid: volume id to be changed
        @param text: new volume name
        """
        self.setVolumesName(vid, text)
        self.changedDB({"scope":"Volume","reason":"renamed","subject":vid})

    def renameGroup(self,gid,text):
        self.setGroupName(gid, text)
        self.changedDB({"scope":"Group","reason":"renamed","subject":vid})        

    def describeVolume(self,vid,text):
        self.setVolumesDescription(vid,text)
        self.changedDB({"scope":"Volume","reason":"described","subject":vid})        

    def describeGroup(self,gid,text):
        self.setGroupsDescription(gid,text)
        self.changedDB({"scope":"Group","reason":"described","subject":vid})        

    ########## SIM GROUPS *******************
    def updateGroups(self, app, simthresd = 50,by = "Color (Fast)"):
        print "updateGroups: NOT IMPLEMENTED" #TODO
        return
        self.changedDB({"scope":"SimGroup","reason":"updated"})

    def createGroupsFromSymGroups(self):
        """create a new logical group for every similarity group  """
        for sid in self.getSimGroupList():
            ngid = self.newGroup(self.getPicsFilename( "["+self.getSimGroupsMainImageId(sid) + "]" ))
            self.setGroupDescription(ngid, "Images similar to " + self.getPicsFilename( self.getSimGroupsMainImageId(sid) ))
            for imid in self.getSimGroupsSimilarImages(sid):
                self.addImageToGroup(imid, ngid)

    ########## DIRECTORIES & FILES***********
    def renameFiles(self,renpairs):
        """call when files (fullpath) were physically renamed
        renpairs is dict. key is old fname value is [new fname,QListView item] -- QListView item can be None
        """
        listItemsToRemove = []          # will collect the list o QListView items to be removed from the widget when the rename process is finished
        for old in renpairs:
            if not self.renameFile(old, renpairs[old][0], warn = 0):
                print "Error renaming:%s" % old
            else:
                listItemsToRemove.append(renpairs[old][1])
        self.changedDB({"scope":"Batch","reason":"changeditem"})
        self.changedDB({"scope":"Img","reason":"multiplerenamed"})
        return filter(None,listItemsToRemove) # filter here is to remove all None objects from list

    def renameFile(self,old,new,warn = 1):
        """call when file (fullpath) old is renamed to new
        if warn = 1,issue the proper changedDB calls
        """
        print "renameFile: NOT IMPLEMENTED YET" #TODO
        return 0

        
#        if self.fullfilenamedict.has_key(old): # if renamed file is on db
#            oldbasepath = os.path.split(old)[0]+os.sep
#            newbasepath = os.path.split(new)[0]+os.sep
#            fid = self.fullfilenamedict[old] # holds file id being moved
#            
#            # check all db dirs to see if the new file now belongs to any of them
#            for did in self.getDirList():
#                if newbasepath == self.getDirsPath(did):
#                    self.dbAddPics(fid, new, did)
#                try:
#                    # the following line will raise an exception if old is not
#                    # found, but that's ok. I guess it's cheaper to just let it
#                    # happen, than to keep doing "if old in bla_list" checks and
#                    # then removing the item in question.
#                    self.dirs[did][4].remove(old) # remove from nominal (full path) list
#                    self.dirs[did][3].remove(fid) # remove from id list
#                except:                 # not on this dir.... move along
#                    pass
#            try:
#                self.fullfilenamedict[new] = fid
#                del self.fullfilenamedict[old]
#                id = self.fullfilenamedict[new]
#                self.meta[id]["Filename"] = os.path.split(new)[-1]
#                self.filenamedict[self.meta[id]["Filename"]] = id
#                del self.filenamedict[os.path.split(old)[-1]]
#                self.img[id][0] = new
#            except:
#                Error.PrintTB("Renaming files")
#                return 0
#            if warn:
#                self.changedDB({"scope":"Batch","reason":"changeditem"})
#                self.changedDB({"scope":"Img","reason":"renamed","subject":id})
#        return 1

    def removeFile(self,fid):
        """ call to remove a given image from db.
        This function won't fire a chagedDb event, because the function calling this
        one is responsible for that (e.g. removeDir)
        
        @param fid: id of the file that will be removed
        """
        imgdb.removeID(fid)
        self.dbDeleteImage(fid)

    def getTotalFiles(self):
        """ number of images in db

        @return: number of images in db
        @rtype: int
        """
        return len(self.getImageList())

    def addStubDir(self,path,volid):
        """ make sure this path exists on db

        if it already exists, return its id, otherwise, create new dir and return new id

        @param path: full dir path
        @param volid: volume id this dir belongs to
        """
        ndid = -1
        for dirid in self.getDirsInVolume(volid):        
            if self.getDirsPath(dirid)==path:
                ndid = dirid                 # this dir is already on dbase
                break
        if ndid == -1:                  # its a new dir
            ndid = self.generateImageID()
            while ndid in self.getGroupList():
                ndid = self.generateImageID()

            # guess parent
            parent = -1
            for dirid2 in self.getDirsInVolume(volid):
                if path[:rfind(path[:-1],os.sep)+1]==self.getDirsPath(dirid2):
                    parent = dirid2
                    break
            self.dbAddDir(ndid, path, "", volid)
            self.dbAddDirsInDirs(ndid, parent)
            #self.dbAddDirInVolume(ndid,volid)
        return ndid

    def addDir(self,volid,path,groupid,aborted = [],recursive = 0,calledRecurs=-1,pb=None,restr=None):
        """Add to db all the files on this dir.
        
        aborted is an empty list that should be nonempty when you want this adddir call to be aborted
        calledRecurs is the id of the dir that should be the parent to the one at path (ie, the one being processed)
        pb is a QProgressDialog
        TODO: all addDir options should be a dictionary and a dict instance should be passed as the only parm
        TODO1: This function & friends are MESSY, rewrite. Also: most parameters are deprecated, as they are now passed through restr
        """
        if aborted: return -1
        #startt = time.time()
        path = os.path.abspath(path)      # make sure we have absolute path, so the stubdir code below won't choke on "wallpaper/"-like paths
        dfiles = os.listdir(path)
        if self.env.verbose: print "Scanning dir: " + path
        hasImage = 0        
        if not restr:
            restr = AddFilter(self.env)
        groupid = restr.groupid
        if not len(dfiles):
            if calledRecurs==-1 and restr.removeEmptyGroup and not hasImage and self.groups.has_key(groupid):
                self.removeGroup(groupid)
            return -1
        if path[-1] != os.sep[0]: path = path+os.sep
        myLabel = path
        mySteps = len(dfiles)
        hasImage = 0
        if self.wnd:                    # init progress bars
            if calledRecurs==-1:
                addDirPB = self.wnd.progressBar
                self.wnd.progressList.clear()
                self.wnd.progressList.insertItem("Started adding, please wait...")
            else:
                self.wnd.progressLabel.setText(myLabel)
                addDirPB = self.wnd.subprogressBar
            addDirPB.setTotalSteps(mySteps)
            addDirPB.setProgress(0)
        ndid = -1
        childdirs = []
        parts = split(path,os.sep)[1:-1]
        cparts = []
        acc = ""
        for part in parts:
            acc = acc+part+os.sep
            cparts.append(os.sep+acc[:])
        for part in cparts[:-1]:
            self.addStubDir(part,volid)
        ndid = self.addStubDir(cparts[-1],volid)
        if ndid == -1:                  # its a new dir
            print "[ERROR] New dir should have an id"
            ndid = self.generateImageID()
            while ndid in self.getDirList():
                ndid = self.generateImageID()
        childfiles = []
        childfilenames = []
        addcnt = 0
        for file in dfiles:             # note: file is not full path. Use path+file when needed
            addcnt = addcnt+1
            if not (addcnt % 20) and self.wnd:
                addDirPB.setProgress(addcnt)
                if self.app: self.app.processEvents()
                if self.wnd.abortedAdd:
                    aborted.append(1)
                    self.wnd.abortedAdd = 0
                    break            
            if os.path.isdir(path+file) and recursive:
                if file[0]=='.': continue # do not add dirs starting with . (also excludes .thumbnail dirs)
                # update paths tree                
                thasimg = self.addDir(volid,path+file,groupid,aborted,recursive,ndid,restr = restr)
                if self.wnd:
                    self.wnd.progressLabel.setText(myLabel)
                    addDirPB.setTotalSteps(mySteps)
                    addDirPB.setProgress(addcnt)
                if thasimg == -1: continue
                childdirs.append(thasimg)
                hasImage = 1
                continue
            # is file
            ext = self.extIsImg(path+file,probe = restr.probeext)
            if ext: #its a file, now just check if its an image
                if file[0]=='.': continue # do not add imgs starting with . (also excludes .thumbnail dirs)
                if restr.igntext and find(path+file,restr.igntext) != -1: continue
                if restr.ignext and ext in restr.ignext: continue
                try:
                    if restr.minsize and os.stat(path+file).st_size < restr.minsize*1024: continue
                except:
                    # what the hell am I supposed to do with a file that cant even be stat'd ?
                    continue
                nfid = self.generateImageID()
                while nfid in self.getFileList():
                    nfid = self.generateImageID()

                self.setPicsFilename(nfid, path+file)
                self.dbAddImageInGroup(nfid, groupid)
                #TODO: save parent directory's id var ndid
                try:
                    ret = self.addFile(path+file,nfid,ndid,ext, restr)
                except:
                    traceback.print_exc()
                    print "Unhandled error while adding file. Please report this bug to \"imgseek-devel@lists.sourceforge.net\""
                if ret==1:
                    hasImage = 1
                    childfiles.append(nfid)
                    childfilenames.append(path+file)
                    self.dbAddImageInGroup(nfid,groupid)
                    ### extract optional metadata
                    if (ext in self.env.exif_ext) and restr.exexif:
                        try:
                            self.extractEXIF(nfid)
                        except:
                            traceback.print_exc()                            
                            print "Error extracting EXIF metadata for:",path+file
                    if (ext in self.env.iptc_ext) and self.env.hasIPTC and restr.exiptc:
                        try:
                            self.extractIPTC(nfid)
                        except:
                            traceback.print_exc()
                            print "Error extracting IPTC metadata for:",path+file
                elif not ret: #error adding image or image already on dbase
                    if self.wnd: self.wnd.progressList.insertItem("Unable to add: "+path+file)
                    self.dbDeleteImage(nfid)
                elif ret==2:
                    self.dbDeleteImage(nfid)
        if hasImage: # only add to databse if has image isinde            
            if ndid not in self.getDirsInVolume(volid): self.dbAddDirInVolume(ndid,volid)
            if ndid in self.getDirList():
                for ddd in childdirs:
                    while ddd in self.getDirsInDirs(ndid):
                        self.dbDeleteDirsInDirs(ndid,ddd)
                for i in childdirs:
                    self.dbAddDirsInDirs(i, ndid)
                for i in childfiles:
                    self.dbAddPics(i, childfilenames[i], ndid)
                self.setDirsVolId(ndid, volid)
            else:
                print "Dir (id %d) should already be in dbase, doing NOTHING"%ndid
                #self.dirs[ndid] = [path,calledRecurs,childdirs,childfiles,childfilenames,"",volid]
        if self.env.verbose: print "Added dir: " + path
        if self.wnd:
            if calledRecurs==-1:
                self.wnd.progressList.insertItem("Finished successfully.")
        # if restr.removeEmptyGroup is set and this dir has no images, then remove the group this dir should belong to
        if calledRecurs==-1 and restr.removeEmptyGroup and not hasImage and groupid in self.getGroupList():
            self.removeGroup(groupid)
        if hasImage: return ndid
        else: return -1

    def addFile(self,fname,newid,dirid,ext, restr = None):
        """adds this filename to database.

        First, loading it, then calculating its haar transform
        and then adding the image index to all respective buckets.
        Each self.files list element is [filename,avg lum]

        @param fname: full filename
        @type fname: string
        @param newid: id this file should be assigned to
        @param dirid: id of the dir this file belongs to
        @param ext: file extension or image type, should be the result of a call to extIsImg()
        @param restr: restrictions to consider
        @return True if image succesfully added
        """
        
        if dirid in self.getDirList():
            if self.getImageByName(fname) in self.getPicsInDirs(dirid): #self.dirs[dirid][4]:
                return 2
        mountDiff = ""
        newdbdate = time.asctime(time.localtime())        
        if restr and restr.mounted:
            mountDiff = newdbdate
        thname = self.thdir2+ md5.new("file://"+fname+mountDiff).hexdigest()+".png"
        if restr:
            mindim = restr.mindim
        else:
            mindim = 0
        if self.env.verbose: print "Adding file: " + fname
        if self.env.qtv==2:
            if ext in self.env.qt_ext:
                ret = imgdb.addImage(newid,fname,thname,not os.path.exists(thname),mindim)                
            else:
                im = Image.open(fname)
                im.save(self.thdir2+".cachev.bmp")
                del im
                ret = imgdb.addImage(newid,self.thdir2+".cachev.bmp",thname,not os.path.exists(thname),mindim)                
        else:
            ret = imgdb.addImage(newid,fname,thname,not os.path.exists(thname),mindim)            
        if not ret:
            print "Error adding image:",fname
            return 0
        if ret==2:                      # dimension too small or too white
            print "Ignored (small dimensions): "+fname
            return 0
        imgDims = [imgdb.getImageWidth(newid),imgdb.getImageHeight(newid)]
        ## init metadata
        try:                            # *** NOTE *** when changing this code, also change ImgDB.py:initImgMetadata **********************
            #TODO self.meta[newid] = self.blankMetaDict.copy()
            if mountDiff:
                self.setMetadata(newid,"Mounted","yes")
            else:
                self.setMetadata(newid,"Mounted", "no")
            self.setMetadata(newid, "Filename", os.path.split(fname)[-1])
            self.setMetadata(newid,"Filesize", self.prettysize(os.stat(fname).st_size))
            self.setMetadata(newid,"Format", ext)
            if restr:
                self.setMetadata(newid,"Volume", self.getVolumeName(restr.volid))
            self.setMetadata(newid,"Database date", newdbdate)
            self.setMetadata(newid,"Modify date", time.strftime("%Y/%m/%d %H:%M:%S", time.gmtime(os.stat(fname).st_mtime)))
            self.setMetadata(newid,"Modify_date_epoch", os.stat(fname).st_mtime) # for time clustering
            if imgDims:
                self.setMetadata(newid,"Dimensions", str(imgDims[0])+" x "+str(imgDims[1]))
        except:
            traceback.print_exc()
            print "Error setting basic metadata for "+fname
            self.setMetadata(newid,"Filename", fname)
        return 1

    def remove_dead(self):
        """removes any non existant file on database """        
        #TODO: fix: volume attrib telling it's a removable vol ?
        remd = []
        for fidx in self.getImageList():
            if (self.getMetaData(fixd, "Mounted") == "no") and (not os.path.exists(self.getPicsFilename(fidx))): # only remove unmounted imgs
                self.removeFile(fidx)
                remd.append(fidx)
        self.changedDB({"scope":"Img","reason":"removed","subjects":remd})
        return len(remd)

    def extIsImg(self,file,probe = 0): #NOT MODIFIED!
        """ tests if this file has a supported image format extension or is an image.

        @param file: full filename
        @type file: string
        @param probe: if true, makes it check even if file has no extension
        @type probe: bool
        """
        ext = rfind(file,".")        
        if ext==-1:
            if not probe:
                return 0
            else:
                try:
                    return imghdr.what(file)
                except:
                    return None            
        ext = lower(file[ext+1:])        
        if ext in self.env.supported_ext:  # if it is a supported image filetype
            return ext
        if probe:
            try:
                return imghdr.what(file)
            except:
                return 0
        return 0

    ########## IMAGE RELATED ****************
    def regenerateThumbs(self):
        """call it to iterate all db images and force de re-generation of their cached thumbnail.
        """
        for id in self.getImageList():
            self.getThumbDB(id,force = 1)

    def queryImage(self,filen,numres,scanned,removeFirst = 1): #not modified
        """query for similar images

        @param filen: full filename
        @param numres: maximum number of results
        @param scanned: true if image is a photo. False if it's a drawing
        @param removeFirst: if the first result (most similar) should be removed from results. Defaults to 1
        @return: list of similar images. List of pairs [result id,result score]  Score is 0-100
        @rtype: list
        """        
        imgdb.queryImgFile(filen,numres,not scanned)
        nres = imgdb.getNumResults()
        res = []
        for i in range(nres):
            rid = imgdb.getResultID()
            rsc = imgdb.getResultScore()
            rsc = -100.0*rsc/38.70
            #sanity checks
            if rsc<0:rsc = 0
            if rsc>100:rsc = 100
            res.append([rid,rsc])
        res.reverse()
        if not res: return res
        if removeFirst:
            return res[1:]
        else:
            return res

    def queryData(self,id,numres): #not modified
        """query for similar images. Source is a db image

        @param id: img id
        @param numres: maximum number of results
        @return: list of similar images. List of pairs [result id,result score]  Score is 0-100
        @rtype: list
        """
        ### NOTE: some code here should be on a separate function (so it can also get called by queryImage(). It's not in order
        # to avoid an extra function call when querying
        
        imgdb.queryImgID(id,numres)
        nres = imgdb.getNumResults()
        res = []
        for i in range(nres):
            rid = imgdb.getResultID()
            rsc = imgdb.getResultScore()
            rsc = -100.0*rsc/38.70
            #sanity checks
            if rsc<0:rsc = 0
            if rsc>100:rsc = 100
            res.append([rid,rsc])
        res.reverse()
        if not res: return res        
        return res[1:]

    def extractEXIF(self,fid):
        """hook for exif library.

        @param fid: set metadata for the image with this id.
        """
        try:
            file = open(self.getPicsFilename(fid), 'rb')
        except:
            print 'Unable to open file in order to extract EXIF data.'
            return
        data = EXIF.process_file(file)
        if data:
            x = data.keys()
            for i in x:
                if i in ('JPEGThumbnail', 'TIFFThumbnail'):
                    continue
                try:
                    for key,val in data[i].printable.items():
                        self.setMetadata(fid, key, val)
                    if i not in self.metafields: 
                        self.metafields.append(i)
                except:
                    try:
                        for key,val in data[i].items():
                            self.setMetadata(fid, key, val)
                    except:
                        print "Error extracting EXIF field:",i
            if data.has_key('JPEGThumbnail'):
                self.setMetadata(fid, 'JPEGThumbnail', "True")
            else:
                self.setMetadata(fid,'JPEGThumbnail', "False")

    def extractIPTC(self,fid):
        """hook for iptc PIL library.

        @param fid: set metadata for the image with this id.
        """
        try:
            data = IptcExtract.getiptcinfo(self.getPicsFilename(fid))
        except:
            traceback.print_exc()
            print 'Error extracting IPTC data.'
            return
        if not data:
            return
        else:
            metacache={}
            for k, v in data.items():
                try:
                    if not k[1]: continue # skip key 0, which is \x00\x02 for example. I don't know what it means
                    i = IptcExtract.infomap[k[1]]
                    if metacache.has_key(i):
                        metacache[i] = metacache[i] + "," + str(v)
                    else:
                        metacache[i] = str(v)                    
                except:
                    print "----------\nError extracting IPTC field: %s %s\nIf you know what this field means according to IPTC standards, please tell us at \"imgseek-devel@lists.sourceforge.net\"\n----------"% (k, repr(v))
            for k in metacache.keys():
                self.setMetadata(fid,k,metacache[k])

    def initImgMetadata(self,fname,newid,imgDims = None):
        """add basic metadata to img

        @param newid: image id
        @param fname: image fullpath fname, used to extract short filename
        @param imgDims: image dimension
        @type imgDims: tuple (width,height)
        """
        try:
            self.setMetadata(newid,"Filename", os.path.split(fname)[-1])
            self.setMetadata(newid,"Filesize", self.prettysize(os.stat(fname).st_size))
            self.setMetadata(newid,"Format", self.extIsImg(fname))
            self.setMetadata(newid,"Database date", time.asctime(time.localtime()))
            self.setMetadata(newid,"Modify date", time.strftime("%Y/%m/%d %H:%M:%S", time.gmtime(os.stat(fname).st_mtime)))
            self.setMetadata(newid,"Modify_date_epoch", os.stat(fname).st_mtime) # for time clustering            
            if imgDims:
                self.setMetadata(newid,"Dimensions", str(imgDims[0])+" x "+str(imgDims[1]))
        except:
            traceback.print_exc()
            print "Error setting basic metadata for "+fname
            self.setMetadata(newid,"Filename", fname)

    def getThumbDB(self,id,force = 0):
        """returns a QPixmap with the thumbnail for an img in db

        @param id: db image id.
        """
        if self.getMetadata(id, "Mounted") == "yes":
            tfname = self.getPicsFilename(id)
            return self.getThumb(self.getPicsFilename(id),force=force, mountDiff=self.getMetaData(id, "Database date"))
        else:
            return self.getThumb(self.getPicsFilename(id),force=force)

    def getThumb(self,fname,imgDims = None,force = 0, mountDiff=""): #NOT MODIFIED AT ALL
        """use to get a QPixmap with the thumbnail for a file

        Using the standard from http://triq.net/~pearl/thumbnail-spec/
        
        @param fname: full path for image
        @type fname: string
        @param force: if true, thumbnail will be regenerated (ie, cached copy will be ignored and rewritten)
        @type force: int
        @return: a QPixmap() instance with the proportional thumbnail.If thummbn found on cache, open it, otherwise create one.
        """
        aImage = None
        thname = self.thdir2+ md5.new("file://"+fname+mountDiff).hexdigest()+".png"
        (base,ext) = os.path.splitext(fname)
        fext = ext[1:].lower()
        if not os.path.exists(thname) or force:
            if fext in self.env.doc_ext:
                print "Not generating thumbnail for document file. Email \"imgseek-devel@lists.sourceforge.net\" if you think imgSeek should thumbnail documents."
                return None
            if fext in self.env.qt_ext:            
                aImage = QImage()
                if not aImage.load(fname):
                    print "Error loading image file while creating a thumbnail for it:" + fname
                    return None
                if self.env.qtv>=3:
                    aImage = aImage.smoothScale(128,128,QImage.ScaleMin)
                else:
                    nw = 128
                    nh = 128
                    if aImage.width()>aImage.height():
                        nw = 128
                        nh = int(128*aImage.height()/aImage.width())
                    else:
                        nh = 128
                        nw = int(128*aImage.width()/aImage.height())
                    aImage = aImage.smoothScale(nw,nh)
                aImage.save(thname,"PNG")
            elif self.env.hasImMagick and (fext in self.env.magick_ext):
                imgdb.magickThumb(fname,thname)
            elif self.env.hasPIL and (fext in self.env.pil_ext):
                im = Image.open(fname)
                nw = 128
                nh = 128
                sz = im.size
                if sz[0] > sz[1]:
                    nw = 128
                    nh = int(128*sz[1]/sz[0])
                else:
                    nh = 128
                    nw = int(128*sz[0]/sz[1])
                im = im.resize((nw,nh))
                im.save(thname)
                del im
        if not aImage: # load it from thumbnail file
            aImage = QPixmap()
            if not aImage.load(thname):
                print "Error loading thumbnail file " + thname
        else:
            aPix = QPixmap()
            aPix.convertFromImage(aImage)
            return aPix
        return aImage

    def importMeta(self,header,data,key,fullpath = 1):
        """import data from an array of arrays into the internal metadata dict.

        This function is called directly by the Import CSV wizard.
        Will iterate through every element in data, trying to find the corresponding image according to the key field indicated by the key
        parameter, and then import the data for the file.
        
        @param header: list of meta fields (strings), to which data will be associated to on every imported file
        @param data: list of lists of strings.
        @param fullpath: if True, match key field to full path instead of filename
        @param key: index of column to match when looking which file in db to set this data
        @type key: int
        @return: Human readable text informing how many items were imported.
        @rtype: string
        """
        print "importMeta: Not implemented yet" #TODO
        
        resp = ""
        count = 0
        self.syncFilenames()        
        for it in data:
            try:
                id = -1
                if fullpath:
                    for itd in self.img.keys():
                        if self.img[itd][0]==it[key]:
                            id = itd
                            break
                else:
                    if self.filenamedict.has_key(it[key]):
                        id = self.filenamedict[it[key]]
                if id==-1:
                    continue
                for hid in range(len(header)):
                    if header[hid] in self.readonlyfields: continue
                    if hid==key: continue
                    self.meta[id][header[hid]] = it[hid]
                count = count+1
                self.changedDB({"scope":"Meta","reason":"changed","subject":id})        
            except:
                traceback.print_exc()
                continue
        resp = resp+"%d item(s) imported."%count
        return resp

    def openImage(self,fname): #not modified
        """abstract method for opening an image. This function should call whatever is available to open the given image

        @param fname: full path for an image
        @return QImage instance
        """
        (base,ext) = os.path.splitext(fname)        
        fext = ext[1:].lower()        
        aImage = QImage()
        if fext in self.env.qt_ext:
            if not aImage.load(fname):
                print "Error loading image file " + fname
        elif self.env.hasImMagick and (fext in self.env.magick_ext):
            imgdb.convert(fname,self.thdir2+".cachev.bmp")
            if not aImage.load(self.thdir2+".cachev.bmp"):
                print "Error loading cached image file " + fname
        elif self.env.hasPIL and (fext in self.env.pil_ext):
            im = Image.open(fname)
            im.save(self.thdir2+".cachev.bmp")
            del im
            if not aImage.load(self.thdir2+".cachev.bmp"):
                print "Error loading cached image file " + fname                
        return aImage

    def openPixmap(self,fname): #not modified
        """abstract method for opening an image. This function should call whatever is available to open the given image

        @param fname: full path for an image
        @return QPixmap instance
        """
        (base,ext) = os.path.splitext(fname)        
        fext = ext[1:].lower()        
        aImage = QPixmap()
        if fext in self.env.qt_ext:
            if not aImage.load(fname):
                print "Error loading image file " + fname
        elif self.env.hasImMagick and (fext in self.env.magick_ext):
            imgdb.convert(fname,self.thdir2+".cachev.bmp")
            if not aImage.load(self.thdir2+".cachev.bmp"):
                print "Error loading cached image file " + fname
        elif self.env.hasPIL and (fext in self.env.pil_ext):
            im = Image.open(fname)
            im.save(self.thdir2+".cachev.bmp")
            del im
            if not aImage.load(self.thdir2+".cachev.bmp"):
                print "Error loading cached image file " + fname                
        return aImage

    def scanForMetaTags(self,bid):
        """rescan all images in this batch for metadata on the files

        @param bid: batch id
        """
        ids = []
        self.crawlBatchForImg(bid,ids)
        for nfid in ids:
            fname = self.getPicsFilename(nfid)
            try:
                self.dbDeleteAllMetadata(nfid)
                try:
                    self.initImgMetadata(fname,nfid)
                except:
                    traceback.print_exc()
                    print "Error setting basic metadata for "+fname
                    self.setMetadata(nfid, "Filename", fname)
            except:
                print "Error gathering metadata for ",fname
                traceback.print_exc()
            ### extract optional metadata
            ext = self.extIsImg(fname,probe = 1)
            if (ext in self.env.exif_ext):
                try:
                    self.extractEXIF(nfid)
                except:
                    traceback.print_exc()                            
                    print "Error extracting EXIF metadata for:",fname
            if (ext in self.env.iptc_ext) and self.env.hasIPTC:
                try:
                    self.extractIPTC(nfid)
                except:
                    traceback.print_exc()
                    print "Error extracting IPTC metadata for:",fname

    def createGroupsFromSymGroups(self):
        """create a new logical group for every similarity group
        """
        for sid in self.getSimGroupList():
            ngid = self.newGroup("["+ self.getMetadata( self.getSimGroupsMainImageId(sid), "Filename") +"]")
            self.setGroupsDescription(ngid, "Images similar to "+ self.getPicsFilename( getSimGroupsMainImageId( sid ))) 
            for imid in self.getSimGroupsSimilarImages(sid):
                self.addPicsInGroups(imid, ngid)
        self.changedDB({"scope":"Group","reason":"added"})

    ########## FILENAME RELATED *************
    def syncFilenames(self,force = 0):
        """call it to update dictionary of filenames:id """
        print "syncFileNames: NOT IMPLEMENTED" #TODO
        return -1
        if not force and (len(self.filenamedict.keys())==len(self.img.keys())):return # do not sync if almost nothing changed
        self.filenamedict = {}
        for id in self.img.keys():
            try:
                self.filenamedict[self.meta[id]["Filename"]] = id
            except:
                print "Image without metadata found. Trying to rebuild it."
                self.initImgMetadata(self.img[id][0],id)

    def syncFullFilenames(self,force = 0):
        """call it to update dictionary of filenames:id """
        print "syncFullfilenames: NOT IMPLEMENTED"
        return 0
        if not force and (len(self.fullfilenamedict.keys())==len(self.img.keys())):return # do not sync if almost nothing changed
        self.fullfilenamedict = {}
        for id in self.img.keys():
            try:
                self.fullfilenamedict[self.img[id][0]] = id
            except:
                print "Image without metadata found. Trying to rebuild it."
                self.initImgMetadata(self.img[id][0],id)                

    def queryFilename(self,fname,numres = 10,target = None,thresd=0.5): # not modified
        """query for similar filenames

        @param fname: full
        @param numres: maximum number of results
        @param target: list of file names to search on
        @return: list of similar filenames
        @rtype: list
        """
        if not target:
            self.syncFilenames()
            target = self.filenamedict
        import difflib                      # to match similar filenames        
        try:
            fname = os.path.split(fname)[-1]
        except:
            pass
        
        mats = difflib.get_close_matches(fname, target.keys(), numres,thresd)
        mats = mats[1:]
        fres = []
        for mat in mats:
            fres.append([target[mat],0])
        return fres

    def getThumbName(self,fname):
        """return the thumbnail full path for the image
        given by fname (fname is full path) """

    def extension(self, fname):
        """ returns the .3 extension of a given filename, including the dot"""
        return fname[-(len(fname)-rfind(fname,'.')):]
        return self.thdir2+ md5.new("file://"+fname).hexdigest()+".png"

    def uniquename(self, fname,ext,path=None,attempt=0):
        """ returns unique 8.3 filename for a file with filename fname
        placed on specified path.

        @param fname: source filename
        @param path: if non-Null, this function will make sure that
        there is no file on path with the returned unique filename
        @param ext: extension for the resulting filename (".png" for example)
        @rtype: string
        """
        if not path:   # return unique name without checking
                return md5.new(fname).hexdigest()[-8:]+ext
        else:
                if attempt > 64: raise "Possible error generating unique file name."
                uname = md5.new(fname + str(attempt)).hexdigest()[-8:] + ext
                if os.path.exists(path+uname) or os.path.exists(path+os.sep+uname):
                        return self.uniquename(fname,ext,path, attempt+1)
                return uname

    ########## CRAWL ************************
    def crawlVolumeForDir(self,vid,lst):
        """crawl this volume adding to lst all dir id's found"""
        for dirid in self.getDirsInVolume(vid):
            self.crawlDirForDir(dirid,lst)
    
    def crawlDirForDir(self,dirid,lst):
        """crawl this volume adding to lst all dir id's found"""
        for chid in self.getDirsInDirs(dirid):
            self.crawlDirForDir(chid,lst)
        lst.append(dirid)

    def crawlVolumeForImg(self,vid,lst):
        """crawl this volume adding to lst all img id's found"""
        for dirid in self.getDirsInVolume(vid):
            self.crawlDirForImg(dirid,lst,0)

    def crawlDirForImg(self,dirid,lst,recurse = 1):
        """crawl this volume adding to lst all img id's found"""
        if recurse:
            for chid in self.getDirsInDirs(dirid):
                self.crawlDirForImg(chid,lst)
        for chid in self.getPicsInDirs(dirid):
            lst.append(chid)

    def crawlGroupForImg(self,gid,lst):
        """crawl this volume adding to lst all img id's found"""
        for chid in self.getGroupsInGroups(gid):
            self.crawlGroupForImg(chid,lst)
        for chid in self.getPicsInGroups(gid):
            lst.append(chid)

    def crawlVolumeForFile(self,vid,lst):
        """crawl this volume adding to lst all img id's found"""
        for dirid in self.getDirsInVolume(vid):
            self.crawlDirForFile(dirid,lst,0)

    def crawlDirForFile(self,dirid,lst,recurse = 1):
        """crawl this volume adding to lst all img id's found"""
        if recurse:
            for chid in self.getDirsInDirs(dirid):
                self.crawlDirForFile(chid,lst)
        for chid in self.getPicsInDirs(dirid):
            lst.append(self.getPicsFilename(chid))

    def crawlSysDirForFile(self,dirid,lst,recurse = 1): #unmodified
        """crawl this volume adding to lst all img id's found"""
        if dirid[-1]!='/':dirid = dirid+'/'
        dfiles = os.listdir(dirid)
        for file in dfiles:
            if os.path.isdir(dirid+file) and recurse:
                self.crawlSysDirForFile(dirid+file,lst,recurse)
                continue            
            ext = self.extIsImg(file)
            if ext: #its a file, now just check if its an image
                if file[0]=='.': continue # do not add imgs starting with . (also excludes .thumbnail dirs)
                lst.append(dirid+file)

    def crawlGroupForFile(self,gid,lst):
        """crawl this volume adding to lst all img id's found"""
        for chid in self.getGroupsInGroups(gid):
            self.crawlGroupForFile(chid,lst)
        for chid in self.getPicsInGroups(gid):
            lst.append(self.getPicsFilename(chid))

    def crawlBatchForImg(self,bid,idlist,dumpids = 1):
        """ here bid is the a batch id of a batch in db
        if dumpids==1 it will dump ids, otherwise it will dump filenames (fullpaths)
        """
        for type, id in self.getContentsInBatch(bid):
            if type=="Volume":
                self.crawlVolumeForImg(id,idlist)
            if type=="Dir":
                self.crawlDirForImg(id,idlist)
            if type=="Group":
                self.crawlGroupForImg(id,idlist)
            if type=="Img":
                if dumpids:
                    idlist.append(id)
                else:
                    idlist.append(self.getPicsFilename(id))

    def crawlBatchForFile(self,bid,idlist):
        """ here bid is the a batch id of a batch in db
        will dump on idlist all image files
        """
        for type, id in self.getContentsInBatch(bid):
            if type=="Volume":
                self.crawlVolumeForFile(id,idlist)
            if type=="Dir":
                self.crawlDirForFile(id,idlist)
            if type=="SysDir":
                self.crawlSysDirForFile(id,idlist)
            if type=="Group":
                self.crawlGroupForFile(id,idlist)
            if type=="Img":
                idlist.append(self.getPicsFilename(id))
            if type=="File":
                idlist.append(id)

    def crawlBatchContentsForImg(self,bid,idlist):
        """here bid is a list of items, with the same format a batch content would have """
        for type,id in bid:
            if type=="Volume":
                self.crawlVolumeForImg(id,idlist)
            if type=="Dir":
                self.crawlDirForImg(id,idlist)
            if type=="Group":
                self.crawlGroupForImg(id,idlist)
            if type=="Img":
                idlist.append(id)

    ########## BATCH STUFF ******************
    def saveBatchMeta(self,dct,bid):
        """apply data on dict dct to the batch """
        idlist = []
        self.crawlBatchForImg(bid,idlist)
        for id in idlist:
            for fi in dct.keys():
                self.setMetadata(id, fi, dct[fi])

    def BatchToText(self,bid):
        """ returns a pair of strings, which represents this batch in text form"""
        try:
            cnt = self.getContentsInBatch(bid)
        except:
            traceback.print_exc()
            print "Attempt to show invalid batch"
            return
        ret = []
        for it in cnt:
            tp = "Unknown"
            tx = ""
            if it[0] == "SysDir":
                tp = "System directory"
                tx = it[1]
            if it[0] == "Dir":
                tp = "Database directory"
                tx= "%s - %d image(s)" % (self.dirs[it[1]][0],len(self.dirs[it[1]][3]))
            if it[0] == "Group":
                tp = "Group"
                tx= "%s - %d image(s)" % (self.groups[it[1]][0],len(self.groups[it[1]][3]))
            if it[0] == "SimGroup":
                tp = "Similarity Group"
                tx= "%s - %d image(s)" % (self.img[self.simgroups[it[1]][1]][0],len(self.simgroups[it[1]][2]))
            if it[0] == "Img":
                tp = "Database image"
                tx= "%s - %s" % (self.img[it[1]][0],self.meta[it[1]]["Filesize"])
            if it[0] == "Volume":
                tp = "Volume"
                tx= "%s - %d dirs(s)" % (self.volumes[it[1]][3],len(self.volumes[it[1]][0]))                
            if it[0] == "File":
                tp = "System file"
                tx = it[1]
            ret.append([tp,tx])
        return ret

    def removeItemBatch(self,idx,bid):
        """call it to remove item idx from batch bid """
        self.dbDeleteContentsInBatch(idx, bid)
        self.changedDB({"scope":"Batch","reason":"removeditem","subject":bid,"target":idx})        

    def addItemBatch(self,it,bid = -1):
        """call to populate a batch with an item

        @param it: map of type {'type','id'}
        @param bid: destination batch id
        @return: 0 if item already in this batch, 1 is successful and -1 on unkown error.
        """
        it = [it["type"],it["id"]]
        if bid==-1: bid = self.curBatch
        try:
            if self.idInBatch(it,bid):
                return 0
            self.addContentsToBatch(it, bid)
        except:
            traceback.print_exc()
            print "Error adding  item to batch"
            return -1        
        self.changedDB({"scope":"Batch","reason":"addeditem","subject":bid,"target":it})
        return 1

    def removeBatch(self,bid):
        """call it to remove batch bid """
        self.dbDeleteBatch(bid)
        self.changedDB({"scope":"Batch","reason":"removed","subject":bid})

    def resetBatch(self,bid):
        """call it to remove batch bid """
        self.dbClearBatch()
        self.changedDB({"scope":"Batch","reason":"reseted","subject":bid})

    def resetBatchHistory(self):
        """ remove all batches from db """
        self.dbDeleteAllBatches()
        self.changedDB({"scope":"Batch","reason":"resetedbatchhistory"})        
        #format: self.batches = {1:[time.asctime(time.localtime()),[]]}

    def addBatch(self,name = ""):
        """call create new batch in db

        @param name: new batch name. If undefined, the current time will be used
        @type name: string
        @return: new batch id
        """
        bid = 0
        try:
            if not name: name = time.asctime(time.localtime())
            bid = self.generateImageID()
            while bid in self.getBatchList():
                bid = self.generateImageID()
            self.dbAddBatch(bid, name)
        except:
            traceback.print_exc()
            print "Error adding  batch"
            return 0
        self.changedDB({"scope":"Batch","reason":"new","subject":bid})
        return bid

    def idInBatch(self,it,bid = -1):
        """checks if the given id is in one of the groups/dirs already included on a batch """
        if bid==-1: bid = self.curBatch
        if (it[0], it[1]) in self.getContentsInBatch(bid):
            return 1
        return 0

    ########## MISC METHODS *****************
    def changedDB(self,what = None):
        """helper function, call it whenever something on db is changed and the GUI should know of

        @param what: describe the change
        @type what: dict
        """
        if not what:
            print "Changes should have a reason."
            return
        try:
            for cb in self.cbs[what["scope"]]:
                cb(what) 
        except:
            Error.PrintTB("db callback system error - no scope")

    def TextReport(self):
        """get database statistics

        @return: human readable statistics about db
        @rtype: string
        """
        ntext = "Database status:\n"
        ntext = ntext+"Images: %d\n"%len(self.getPicsList())
        ntext = ntext+"Volumes: %d\n"%len(self.getVolumeList())
        ntext = ntext+"Directories: %d\n"%len(self.getDirList())
        ntext = ntext+"Groups: %d\n"%len(self.getGroupList())
        ntext = ntext+"Work Batches: %d\n"%len(self.getBatchList())
        ntext = ntext+"Similarity groups: %d\n"%len(self.getSimGroupList())
        ntext = ntext+"Metadata fields cached: %d\n"%len(self.metafields)
        ntext = ntext+"Text query history entries: %d\n"%len(self.textqueryhistory)
        ntext = ntext+"Database size: %d bytes\n"%self.kbsize
        return ntext

    def refreshModifyDate(self):
        """ call to refresh the Modify_date_epoch and Modify date metafields"""
        for id in self.getImageList():
            try:
                fname = self.getPicsFilename(id)
                self.setMetadata(id,"Modify date", time.strftime("%Y/%m/%d %H:%M:%S", time.gmtime(os.stat(fname).st_mtime)))
                self.setMetaData(id,"Modify_date_epoch", os.stat(fname).st_mtime) # for time clustering 
            except:
                Error.PrintTB("Error setting Modify Date... Ignore this error if file is non-existant, and run \"Remove dead files\" later.")

    def generateImageID(self):
        """ need to return something unique"""
        # Idea/Tuomas: what if we would use timestamps as id's? We could take timestamp
        # from images and if there is already another with the same id, then, for example,
        # we could take the timestamp of current moment. again, check, and then add some random.
        #
        try:
            a = random.randrange(999999999)
        except:
            try:
                a = random.randrange(99999999)
            except:                
                try:
                    a = random.randrange(9999999)
                except:
                    a = random.randrange(999999)                
        # 1 is a reserved id for default items
        while a==1: a = self.generateImageID()
        return a

    def hashStr(self,str):
        """returns MD5 hex representation """
        return md5.new(str).hexdigest()[:8]

    def prettysize(self,uint):          # Not modified
        """returns a string after converting the integer uint and adding periods to enhance readability

        @param uint: int to be transformed
        @type uint: int
        @return: pretty string (human readable)
        @rtype: string
        """
        a = str(uint)
        b = ""
        for i in range(len(a)):
            if not i%3:
                b = "."+b
            b = a[len(a)-1-i]+b
        return b[:-1] + " bytes"

    def addSysDirBookmark(self,txt):
        print "addSysDirBookmark: NOT IMPLEMENTED" #TODO
        return -1
        self.sysdirbm.append(txt) 
        for cb in self.sysdirbmCb: cb()

    def delSysDirBookmark(self,id):
        """remove a bookmark
        
        @param id: 0-based index of the bmark to be removed
        """
        print "delSysDirBookmark: NOT IMPLEMENTED" #TODO
        return -1
        try:
            del self.sysdirbm[id]
            self.dirty = 1
            for cb in self.sysdirbmCb: cb()
        except:
            traceback.print_exc()
            print "Error removing the supplied bookmark id"

    def removeAllMeta(self,id):
        self.dbDeleteAllMetadata(id) 
        self.changedDB({"scope":"Meta","reason":"removedall","subject":id})

    def textquery(self,txt,numres = 10):
        """start text query with logical operators.

        @param txt: txt[1] = {'AND','OR'} is the logic to be used. txt[0] is a list of pairs [field_name,query_keyword]
        @param numres: maximum number of results
        @type numres: int
        @return: list of pairs [img_id,0]. (0 is fixed and should be ignored.) What matters is the img_id integer, which means this image is a query result
        """
        
        metaq = txt[0]
        logic = txt[1]
        
        SQL = 'SELECT id FROM Metadata WHERE '
        for key,val in metaq:
            SQL = SQL + 'key LIKE \"%%%s%%\"" AND value LIKE \"%%%s%%\" %s '%(key,value, logic)
        
        SQL = SQL[:-4] #removing AND or OR in the end (4 chars)
        SQL = SQL + ';'
        print "textquery: " + SQL
        self.execute(SQL)
        res = self.cursor.fetchall() 
        return [ [i,j] for i,j in res ] 


if __name__ == '__main__':    

    def testConvert():
        print "#################################### Test importing an old db"

        import ImgDB
        from Settings import Env
        myNewDB = SqlDB(Env(), 'mysql',dbArgs={'user': 'tuma', 'passwd':'mp2gra5', 'db':'imgseek'}, dbFile='/home/tuma/.imgseek/img-db.iqd.img')
        #myNewDB = SqlDB(Env(), 'postgres',dbArgs={'user': 'tuma', 'database':'imgseek'}, dbFile='/home/tuma/.imgseek/img-db.iqd.img')
        #'/home/tuma/.imgseek/img-db.iqd.newdb'}, dbFile='/home/tuma/.imgseek/img-db.iqd.img')       myOldDB = ImgDB.ImgDB(Env())
        myOldDB = ImgDB.ImgDB(Env())
        myOldDB.opendb('/home/tuma/.imgseek/img-db.iqd.olddb')

        myNewDB.convert_old(myOldDB)

    testConvert()
