/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "agent.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <locale.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include "pin-dialog.h"
#include "passphrase-dialog.h"
#include "connect-error-dialog.h"
#include "wireless-connect-dialog.h"

#include "indicator-network-agent-server.h"

#include "dbus-shared-names.h"
#include "log.h"

typedef struct _NetworkAgentPrivate NetworkAgentPrivate;

struct _NetworkAgentPrivate
{
  DBusGConnection *session_bus;
  PinDialog *pin_dialog;
  PassphraseDialog *passphrase_dialog;
  ConnectErrorDialog *error_dialog;
  WirelessConnectDialog *wireless_connect_dialog;
};

#define NETWORK_AGENT_GET_PRIVATE(o)			 \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), NETWORK_AGENT_TYPE, \
				NetworkAgentPrivate))

static void network_agent_class_init(NetworkAgentClass *klass);
static void network_agent_init(NetworkAgent *self);
static void network_agent_dispose(GObject *object);
static void network_agent_finalize(GObject *object);

G_DEFINE_TYPE (NetworkAgent, network_agent, G_TYPE_OBJECT);

static void free_dialogs(NetworkAgent *self)
{
  NetworkAgentPrivate *priv = NETWORK_AGENT_GET_PRIVATE(self);

  if (priv->pin_dialog != NULL) {
    gtk_widget_destroy(GTK_WIDGET(priv->pin_dialog));
    priv->pin_dialog = NULL;
  }

  if (priv->passphrase_dialog != NULL) {
    gtk_widget_destroy(GTK_WIDGET(priv->passphrase_dialog));
    priv->passphrase_dialog = NULL;
  }

  if (priv->error_dialog != NULL) {
    /* g_object_unref(priv->error_dialog); */
    priv->error_dialog = NULL;
  }

  if (priv->wireless_connect_dialog != NULL) {
    gtk_widget_destroy(GTK_WIDGET(priv->wireless_connect_dialog));
    priv->wireless_connect_dialog = NULL;
  }
}

static void network_agent_class_init(NetworkAgentClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS(klass);

  g_debug("%s()", __func__);

  g_type_class_add_private(object_class, sizeof(NetworkAgentPrivate));

  object_class->dispose = network_agent_dispose;
  object_class->finalize = network_agent_finalize;

  g_assert(klass != NULL);

  dbus_g_object_type_install_info(NETWORK_AGENT_TYPE,
				  &dbus_glib__network_agent_server_object_info);
}

static void network_agent_init (NetworkAgent *self)
{
  NetworkAgentPrivate *priv = NETWORK_AGENT_GET_PRIVATE(self);
  DBusGProxy *driver_proxy;
  GError *error = NULL;
  guint ret;

  g_debug("%s()", __func__);

  priv->pin_dialog = NULL;
  priv->passphrase_dialog = NULL;
  priv->error_dialog = NULL;

  priv->session_bus = dbus_g_bus_get(DBUS_BUS_SESSION, &error);
  if (error != NULL) {
    g_critical("Unable to connect to the dbus session bus: %s",
	       error->message);
    g_error_free(error);
    return;
  }

  dbus_g_connection_register_g_object(priv->session_bus,
				      INDICATOR_NETWORK_AGENT_OBJECT,
				      G_OBJECT(self));

  driver_proxy = dbus_g_proxy_new_for_name(priv->session_bus,
					   DBUS_SERVICE_DBUS,
					   DBUS_PATH_DBUS,
					   DBUS_INTERFACE_DBUS);

  org_freedesktop_DBus_request_name(driver_proxy,
				    INDICATOR_NETWORK_AGENT_NAME,
				    0, &ret, &error);

  if (error != NULL) {
    g_warning("Unable to register service %s: %s",
	      INDICATOR_NETWORK_AGENT_OBJECT, error->message);
    g_error_free (error);
    return;
  }

  g_object_unref(driver_proxy);
}

static void network_agent_dispose(GObject *object)
{
  NetworkAgent *self = NETWORK_AGENT(object);

  free_dialogs(self);

  G_OBJECT_CLASS(network_agent_parent_class)->dispose(object);
}

static void network_agent_finalize(GObject *object)
{
  G_OBJECT_CLASS(network_agent_parent_class)->finalize(object);
}

void network_agent_start(NetworkAgent *self)
{
  NetworkAgentPrivate *priv = NETWORK_AGENT_GET_PRIVATE(self);

  g_debug("%s()", __func__);

  g_return_if_fail(IS_NETWORK_AGENT(self));

  priv->pin_dialog = pin_dialog_new();
  if (priv->pin_dialog == NULL)
	  g_warning("Failed to create pin dialog");

  priv->passphrase_dialog = passphrase_dialog_new();
  if (priv->passphrase_dialog == NULL)
	  g_warning("Failed to create passphrase dialog");

  priv->error_dialog = connect_error_dialog_new();
  if (priv->error_dialog == NULL)
	  g_warning("Failed to create connect error dialog");

  priv->wireless_connect_dialog = wireless_connect_dialog_new();
  if (priv->wireless_connect_dialog == NULL)
	  g_warning("Failed to create wireless connect dialog");
}

void network_agent_stop(NetworkAgent *self)
{
  g_debug("%s()", __func__);

  g_return_if_fail(IS_NETWORK_AGENT(self));

  free_dialogs(self);
  /* FIXME: uncomment this */
  /* gtk_main_quit(); */
}

gboolean network_agent_show_connect_error(NetworkAgent *self,
					  const char *error_type,
					  GError **error) {
  NetworkAgentPrivate *priv = NETWORK_AGENT_GET_PRIVATE(self);

  g_return_val_if_fail(IS_NETWORK_AGENT(self), TRUE);
  g_return_val_if_fail(priv->error_dialog != NULL, TRUE);

  g_debug("%s()", __func__);

  connect_error_dialog_show_error(priv->error_dialog, error_type);

  return TRUE;
}

void network_agent_show_wireless_connect(NetworkAgent *self)
{
  NetworkAgentPrivate *priv = NETWORK_AGENT_GET_PRIVATE(self);

  g_debug("%s()", __func__);

  wireless_connect_dialog_show(priv->wireless_connect_dialog);
}

gboolean network_agent_set_debug(NetworkAgent* self, gint level,
				 GError **error)
{
  log_set_debug(level > 0);

  return TRUE;
}

int main(int argc, char **argv)
{
  NetworkAgent *agent;

  g_debug("%s()", __func__);

  g_type_init();

  gtk_init(&argc, &argv);

  log_init("indicator-network-agent");

  agent = g_object_new(NETWORK_AGENT_TYPE, NULL);

  setlocale(LC_ALL, "");
  bindtextdomain(GETTEXT_PACKAGE, GNOMELOCALEDIR);
  textdomain(GETTEXT_PACKAGE);

  /* run the loop */
  gtk_main();

  g_object_unref(agent);
  agent = NULL;

  return 0;
}
