/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "passphrase-dialog.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <dbus/dbus-glib.h>
#include <gtk/gtk.h>
#include <locale.h>

#include "connman-manager.h"
#include "connman-service.h"
#include "connman.h"
#include "marshal.h"

G_DEFINE_TYPE(PassphraseDialog, passphrase_dialog, GTK_TYPE_DIALOG);

typedef struct _PassphraseDialogPrivate PassphraseDialogPrivate;

struct _PassphraseDialogPrivate
{
  DBusGConnection *system_bus;
  DBusGProxy *connman_proxy, *service_proxy;
  GtkWidget *entry;
  gchar *path, *security_type;
};

#define PASSPHRASE_DIALOG_GET_PRIVATE(o)				\
  (G_TYPE_INSTANCE_GET_PRIVATE((o), PASSPHRASE_DIALOG_TYPE,		\
			       PassphraseDialogPrivate))

#define NETWORK_AGENT_OBJECT "/indicator/agent"

#define WEP_40BIT_HEXKEY_LEN 10
#define WEP_104BIT_HEXKEY_LEN 26
#define WEP_40BIT_ASCIIKEY_LEN 5
#define WEP_104BIT_ASCIIKEY_LEN 13
#define WPA_MIN_LEN 8
#define WPA_MAX_LEN 64

gboolean agent_request_passphrase(PassphraseDialog *self, char *service_proxy,
				  GError **error_out);
gboolean agent_release();

#include "connman-agent-server.h"

static gboolean is_security_wpa(PassphraseDialog *self)
{
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  if (g_strcmp0(priv->security_type, "wpa") == 0)
    return TRUE;
  else if (g_strcmp0(priv->security_type, "psk") == 0)
    return TRUE;
  else if (g_strcmp0(priv->security_type, "rsn") == 0)
    return TRUE;

  return FALSE;
}

static gboolean is_security_wep(PassphraseDialog *self)
{
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  return g_strcmp0(priv->security_type, "wep") == 0;
}

static int register_agent(PassphraseDialog *self)
{
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);
  GError *error = NULL;

  /* FIXME: maybe dbus_g_proxy_new_for_name_owner()? */
  priv->connman_proxy = dbus_g_proxy_new_for_name(priv->system_bus,
						  CONNMAN_SERVICE,
						  CONNMAN_MANAGER_PATH,
						  CONNMAN_MANAGER_INTERFACE);

  if (error != NULL) {
    g_critical("Unable to get connman manager proxy: %s", error->message);
    g_error_free(error);
    return 1;
  }

  org_moblin_connman_Manager_register_agent(priv->connman_proxy,
					    NETWORK_AGENT_OBJECT,
					    &error);

  if (error != NULL) {
    g_critical("Unable to register as connman agent: %s", error->message);
    g_error_free(error);
    return 1;
  }

  return 0;
}

static void passphrase_dialog_init(PassphraseDialog *self)
{
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);
  GError *error = NULL;

  memset(priv, 0, sizeof(*priv));

  priv->system_bus = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
  if (error) {
    g_critical("Unable to connect to the dbus system bus: %s",
	       error->message);
    g_error_free(error);
    return;
  }

  dbus_g_connection_register_g_object(priv->system_bus, NETWORK_AGENT_OBJECT,
				      G_OBJECT(self));

  register_agent(self);

  return;
}

static void passphrase_dialog_dispose(GObject *object)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(object);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  if (priv->connman_proxy != NULL) {
    /*
     * unregister our connman agent, but we don't care about the error as
     * our object is being destroyed anyway and there's not much we can do
     * here so no need to have callback
     */
    org_moblin_connman_Manager_unregister_agent_async(priv->connman_proxy,
						      NETWORK_AGENT_OBJECT,
						      NULL, NULL);
    g_object_unref(priv->connman_proxy);
    priv->connman_proxy = NULL;
  }

  if (priv->service_proxy != NULL) {
    g_object_unref(priv->service_proxy);
    priv->service_proxy = NULL;
  }

  if (priv->entry != NULL) {
    gtk_widget_destroy(priv->entry);
    priv->entry = NULL;
  }

  if (priv->path != NULL) {
    g_free(priv->path);
    priv->path = NULL;
  }

  if (priv->security_type != NULL) {
    g_free(priv->security_type);
    priv->security_type = NULL;
  }

  G_OBJECT_CLASS(passphrase_dialog_parent_class)->dispose(object);
}

static void passphrase_dialog_finalize(GObject *object)
{
  G_OBJECT_CLASS(passphrase_dialog_parent_class)->finalize(object);
}

static void passphrase_dialog_class_init(PassphraseDialogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private(object_class, sizeof(PassphraseDialogPrivate));

  object_class->dispose = passphrase_dialog_dispose;
  object_class->finalize = passphrase_dialog_finalize;

  g_assert(klass != NULL);

  dbus_g_object_type_install_info(PASSPHRASE_DIALOG_TYPE,
				  &dbus_glib__agent_server_object_info);
}

void connect_cb(DBusGProxy *proxy, GError *error, gpointer user_data)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(user_data);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  if (error) {
    g_warning("failed to connect service: %s", error->message);
    g_error_free(error);
  }

  g_object_unref(priv->service_proxy);
  priv->service_proxy = NULL;
}

static void responded(GtkDialog *dialog, gint response_id, gpointer user_data)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(user_data);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);
  const gchar *passphrase;
  GValue *value;
  GError *error = NULL;

  g_return_if_fail(priv->path != NULL);
  g_return_if_fail(priv->service_proxy != NULL);

  g_debug("%s(): response_id %d", __func__, response_id);

  gtk_widget_hide_all(GTK_WIDGET(self));

  if (response_id != GTK_RESPONSE_ACCEPT)
    return;

  passphrase = gtk_entry_get_text(GTK_ENTRY(priv->entry));

  value = g_malloc0(sizeof(*value));
  g_value_init(value, G_TYPE_STRING);
  g_value_set_string(value, passphrase);

  /* FIXME: should be async */
  org_moblin_connman_Service_set_property(priv->service_proxy,
					  CONNMAN_PROPERTY_PASSPHRASE,
					  value, &error);

  if (error) {
    g_warning("%s(): failed to set passphrase for service %s: %s",
	      __func__, priv->path, error->message);
    g_error_free(error);
    return;
  }

  org_moblin_connman_Service_connect_async(priv->service_proxy, connect_cb,
					   self);
}

static void ask_passphrase(PassphraseDialog *self)
{
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  g_debug("%s()", __func__);

  gtk_entry_set_text(GTK_ENTRY(priv->entry), "");

  gtk_widget_show_all(GTK_WIDGET(self));
}

static void validate(GtkWidget *widget, gpointer user_data)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(user_data);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);
  const gchar *text, *security;
  guint len;

  text = gtk_entry_get_text(GTK_ENTRY(priv->entry));
  len = strlen(text);

  security = priv->security_type;
  if (security == NULL) {
    g_warning("%s(): security type not defined", __func__);
    return;
  }

  if (is_security_wep(self)) {
    switch (len) {
    case WEP_40BIT_HEXKEY_LEN:
    case WEP_104BIT_HEXKEY_LEN:
    case WEP_40BIT_ASCIIKEY_LEN:
    case WEP_104BIT_ASCIIKEY_LEN:
      gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
					TRUE);
      break;
    default:
      gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
					FALSE);
      break;
    }
  } else if (is_security_wpa(self)) {
    if ((len >= WPA_MIN_LEN) && (len <= WPA_MAX_LEN)) {
      gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
					TRUE);
    } else {
      gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
					FALSE);
    }
  }
}

void show_password_toggled(GtkToggleButton *button, gpointer user_data)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(user_data);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  if (gtk_toggle_button_get_active(button))
    gtk_entry_set_visibility(GTK_ENTRY(priv->entry), TRUE);
  else
    gtk_entry_set_visibility(GTK_ENTRY(priv->entry), FALSE);
}

void get_properties_cb(DBusGProxy *proxy, GHashTable *properties,
		       GError *error, gpointer user_data)
{
  PassphraseDialog *self = PASSPHRASE_DIALOG(user_data);
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);
  GValue *value;
  guint len;

  if (error) {
    g_warning("failed to get properties for service %s:", error->message);
    g_error_free(error);
    return;
  }

  value = g_hash_table_lookup(properties, CONNMAN_PROPERTY_SECURITY);
  if (value) {
    g_free(priv->security_type);
    priv->security_type = g_value_dup_string(value);
    g_debug("security_type %s", priv->security_type);
  }

  if (priv->security_type != NULL) {
    if (is_security_wpa(self))
      len = WPA_MAX_LEN;
    else if (is_security_wep(self))
      len = WEP_104BIT_HEXKEY_LEN;
    else
      len = WPA_MAX_LEN;

    gtk_entry_set_max_length(GTK_ENTRY(priv->entry), len);
  }
  
  g_hash_table_destroy(properties);
}

gboolean agent_request_passphrase(PassphraseDialog *self, char *path,
				  GError **error_out)
{
  PassphraseDialogPrivate *priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  g_debug("%s() %s", __func__, path);

  priv->path = g_strdup(path);
  if (priv->path == NULL)
    return TRUE;

  priv->service_proxy = dbus_g_proxy_new_for_name(priv->system_bus,
						  CONNMAN_SERVICE,
						  priv->path,
						  CONNMAN_SERVICE_INTERFACE);
  if (priv->service_proxy == NULL) {
    g_debug("Failed to create dbus proxy for service %s", priv->path);
    return TRUE;
  }

  org_moblin_connman_Service_get_properties_async(priv->service_proxy,
						  get_properties_cb, self);

  ask_passphrase(self);

  return TRUE;
}

gboolean agent_release(PassphraseDialog *self, GError **error)
{
  g_debug("%s()", __func__);

  /* connman has released us, most probably due because it's shutting down */

  /* FIXME: wait for connman to reappear and then re-register */

  return FALSE;
}

PassphraseDialog *passphrase_dialog_new(void)
{
  GtkWidget *vbox, *widget;
  PassphraseDialogPrivate *priv;
  PassphraseDialog *self;

  self = g_object_new(PASSPHRASE_DIALOG_TYPE, NULL);
  priv = PASSPHRASE_DIALOG_GET_PRIVATE(self);

  gtk_window_set_title(GTK_WINDOW(self), _("Network passphrase"));

  gtk_dialog_add_buttons(GTK_DIALOG(self),
			 GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
			 GTK_STOCK_CONNECT, GTK_RESPONSE_ACCEPT,
			 NULL);
  gtk_dialog_set_response_sensitive(GTK_DIALOG(self), GTK_RESPONSE_ACCEPT,
				    FALSE);

  priv->entry = gtk_entry_new();
  gtk_entry_set_visibility(GTK_ENTRY(priv->entry), FALSE);
  g_signal_connect(G_OBJECT(priv->entry), "changed", (GCallback) validate,
		   self);
  vbox = gtk_dialog_get_content_area(GTK_DIALOG(self));
  gtk_box_pack_start(GTK_BOX(vbox), priv->entry, TRUE, TRUE, 5);

  widget = gtk_check_button_new_with_label(_("Show password"));
  g_signal_connect(G_OBJECT(widget), "toggled",
		   G_CALLBACK(show_password_toggled), self);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), FALSE);
  gtk_box_pack_start(GTK_BOX(vbox), widget, TRUE, TRUE, 5);

  g_signal_connect(G_OBJECT(self), "response", G_CALLBACK(responded),
		   self);

  return self;
}
