/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "pin-dialog.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <dbus/dbus-glib.h>
#include <gtk/gtk.h>
#include <locale.h>

#include "ofono.h"
#include "ofono-manager.h"
#include "ofono-simmanager.h"
#include "ofono-modem.h"
#include "marshal.h"

G_DEFINE_TYPE(PinDialog, pin_dialog, GTK_TYPE_DIALOG);

typedef struct _PinDialogPrivate PinDialogPrivate;

struct _PinDialogPrivate
{
  DBusGConnection *system_bus;
  DBusGProxy *ofono_manager;
  DBusGProxy *modem;
  DBusGProxy *sim_manager;
  GtkWidget *entry;
  gchar *modem_path;
};

#define PIN_DIALOG_GET_PRIVATE(o)					\
  (G_TYPE_INSTANCE_GET_PRIVATE((o), PIN_DIALOG_TYPE, PinDialogPrivate))


static void pin_dialog_init(PinDialog *self)
{
  PinDialogPrivate *priv = PIN_DIALOG_GET_PRIVATE(self);
  GError *error = NULL;

  priv->system_bus = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
  if (error) {
    g_critical("Unable to connect to the dbus system bus: %s",
	       error->message);
    g_error_free(error);
    return;
  }

  priv->ofono_manager = NULL;
  priv->modem = NULL;
  priv->sim_manager = NULL;
  priv->entry = NULL;
  priv->modem_path = NULL;

  return;
}

static void pin_dialog_dispose(GObject *object)
{
  G_OBJECT_CLASS(pin_dialog_parent_class)->dispose(object);
}

static void pin_dialog_finalize(GObject *object)
{
  G_OBJECT_CLASS(pin_dialog_parent_class)->finalize(object);
}

static void pin_dialog_class_init(PinDialogClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private(object_class, sizeof(PinDialogPrivate));

  object_class->dispose = pin_dialog_dispose;
  object_class->finalize = pin_dialog_finalize;

  g_assert(klass != NULL);
}

static void enter_pin_cb(DBusGProxy *proxy, GError *error, gpointer user_data) 
{
  if (error) {
    g_warning("%s(): enter pin failed: %s", __func__, error->message);
    g_error_free(error);
    return;
  }
}

static void enter_pin(PinDialog *self, const gchar *pin)
{
  PinDialogPrivate *priv = PIN_DIALOG_GET_PRIVATE(self);
  const gchar *type = "pin";

  g_debug("%s(): sending enter_pin %s type %s", __func__, pin, type);

  org_ofono_SimManager_enter_pin_async(priv->sim_manager, type, pin,
				       enter_pin_cb, self);
}

static void responded(GtkDialog *dialog, gint response_id, gpointer user_data)
{
  PinDialog *self = user_data;
  PinDialogPrivate *priv = PIN_DIALOG_GET_PRIVATE(self);
  const gchar *pin;

  g_debug("%s(): response_id %d", __func__, response_id);

  gtk_widget_hide_all(GTK_WIDGET(self));

  if (response_id != GTK_RESPONSE_ACCEPT)
    return;

  pin = gtk_entry_get_text(GTK_ENTRY(priv->entry));

  enter_pin(self, pin);
}

static void ask_pin(PinDialog *self)
{
  PinDialogPrivate *priv = PIN_DIALOG_GET_PRIVATE(self);

  g_debug("%s()", __func__);

  gtk_entry_set_text(GTK_ENTRY(priv->entry), "");

  gtk_widget_show_all(GTK_WIDGET(self));
}

static void update_pin_required(PinDialog *self, const GValue *value)
{
  const gchar *state;

  state = g_value_get_string(value);

  g_debug("%s(): state %s", __func__, state);

  if (g_strcmp0(state, "pin") == 0) {
    /* pin required */
    ask_pin(self);
  }
}

static void sim_manager_property_changed(DBusGProxy *proxy,
					 const gchar *property, GValue *value,
					 gpointer user_data)
{
  PinDialog *self = user_data;

  if (g_strcmp0(property, OFONO_PROPERTY_PIN_REQUIRED) == 0)
    update_pin_required(self, value);

  g_value_unset(value);
}

static void sim_manager_get_properties_cb(DBusGProxy *proxy,
					  GHashTable *properties,
					  GError *error,
					  gpointer user_data)
{
  PinDialog *self = user_data;
  GValue *value;

  if (error) {
    g_warning("%s(): failed to get properties from ofono sim manager: %s",
	      __func__, error->message);
    g_error_free(error);
    return;
  }

  value = g_hash_table_lookup(properties, OFONO_PROPERTY_PIN_REQUIRED);
  if (value != NULL)
    update_pin_required(self, value);

  g_hash_table_destroy(properties);
}

static void get_sim_manager(PinDialog *self)
{
  PinDialogPrivate *priv = PIN_DIALOG_GET_PRIVATE(self);

  g_debug("%s(): modem_path %s", __func__, priv->modem_path);

  priv->sim_manager = dbus_g_proxy_new_for_name(priv->system_bus,
					       OFONO_SERVICE,
					       priv->modem_path,
					       OFONO_SIM_INTERFACE);
  if (priv->sim_manager == NULL)
    return;

  dbus_g_proxy_add_signal(priv->sim_manager, OFONO_SIGNAL_PROPERTY_CHANGED,
			  G_TYPE_STRING, G_TYPE_VALUE, G_TYPE_INVALID);

  dbus_g_proxy_connect_signal(priv->sim_manager,
			      OFONO_SIGNAL_PROPERTY_CHANGED,
			      G_CALLBACK(sim_manager_property_changed),
			      self, NULL);

  org_ofono_SimManager_get_properties_async(priv->sim_manager,
					    sim_manager_get_properties_cb,
					    self);
  return;
}

static void update_interfaces(PinDialog *self, const GValue *value)
{
  gchar **interfaces;
  gboolean found = FALSE;
  int i;

  interfaces = g_value_get_boxed(value);

  for (i = 0; interfaces[i] != NULL; i++) {
    if (g_strcmp0(interfaces[i], OFONO_SIM_INTERFACE) == 0) {
      found = TRUE;
    }
  }

  if (found)
    get_sim_manager(self);
}

static void modem_property_changed(DBusGProxy *proxy, const gchar *property,
				   GValue *value, gpointer user_data)
{
  PinDialog *self = user_data;

  if (g_strcmp0(property, OFONO_PROPERTY_INTERFACES) == 0)
    update_interfaces(self, value);

  g_value_unset(value);
}

static void modem_get_properties_cb(DBusGProxy *proxy, GHashTable *properties,
				    GError *error, gpointer user_data)
{
  PinDialog *self = user_data;
  GValue *value;

  if (error != NULL) {
    g_warning("%s(): failed to get properties from ofono modem: %s",
	      __func__, error->message);
    g_error_free(error);
    return;
  }

  value = g_hash_table_lookup(properties, OFONO_PROPERTY_INTERFACES);
  if (value != NULL)
    update_interfaces(self, value);

  g_hash_table_destroy(properties);


}


static void get_modem(PinDialog *self)
{
  PinDialogPrivate *priv = PIN_DIALOG_GET_PRIVATE(self);

  g_debug("%s(): modem_path %s", __func__, priv->modem_path);

  priv->modem = dbus_g_proxy_new_for_name(priv->system_bus, OFONO_SERVICE,
					  priv->modem_path,
					  OFONO_MODEM_INTERFACE);
  if (priv->modem == NULL)
    return;

  dbus_g_proxy_add_signal(priv->modem, OFONO_SIGNAL_PROPERTY_CHANGED,
			  G_TYPE_STRING, G_TYPE_VALUE, G_TYPE_INVALID);

  dbus_g_proxy_connect_signal(priv->modem, OFONO_SIGNAL_PROPERTY_CHANGED,
			      G_CALLBACK(modem_property_changed),
			      self, NULL);

  org_ofono_Modem_get_properties_async(priv->modem, modem_get_properties_cb,
				       self);

  return;
}

static void iterate_modems(const GValue *value, gpointer user_data)
{
  GList **modems = user_data;
  gchar *path;

  path = g_value_dup_boxed(value);
  if (!path)
    return;

  *modems = g_list_append(*modems, path);
}

static void update_modems(PinDialog *self, const GValue *value)
{
  PinDialogPrivate *priv = PIN_DIALOG_GET_PRIVATE(self);
  GList *modems = NULL, *iter;

  dbus_g_type_collection_value_iterate(value, iterate_modems, &modems);

  /* use only first modem for now */
  iter = g_list_first(modems);
  if (iter == NULL)
    return;

  g_free(priv->modem_path);
  priv->modem_path = g_strdup(iter->data);

  get_modem(self);

  for (iter = modems; iter != NULL; iter = iter->next) {
    g_free(iter->data);
  }
  g_list_free(modems);
}

static void ofono_manager_property_changed(DBusGProxy *proxy,
					   const gchar *property, GValue *value,
					   gpointer user_data)
{
  PinDialog *self = user_data;

  if (g_strcmp0(property, OFONO_PROPERTY_MODEMS) == 0)
    update_modems(self, value);

  g_value_unset(value);
}

static void manager_get_properties_cb(DBusGProxy *proxy,
				      GHashTable *properties,
				      GError *error,
				      gpointer user_data) 
{
  PinDialog *self = user_data;
  GValue *value;

  if (error) {
    g_warning("%s(): failed to get properties from ofono manager: %s",
	      __func__, error->message);
    g_error_free(error);
    return;
  }

  value = g_hash_table_lookup(properties, OFONO_PROPERTY_MODEMS);
  if (value != NULL)
    update_modems(self, value);

  g_hash_table_destroy(properties);
}

static void get_ofono_manager(PinDialog *self)
{
  PinDialogPrivate *priv = PIN_DIALOG_GET_PRIVATE(self);

  /* FIXME: handle ofono crashes */
  priv->ofono_manager = dbus_g_proxy_new_for_name(priv->system_bus,
						  OFONO_SERVICE,
						  OFONO_MANAGER_PATH,
						  OFONO_MANAGER_INTERFACE);
  if (priv->ofono_manager == NULL)
    return;

  dbus_g_proxy_add_signal(priv->ofono_manager, OFONO_SIGNAL_PROPERTY_CHANGED,
			  G_TYPE_STRING, G_TYPE_VALUE, G_TYPE_INVALID);

  dbus_g_proxy_connect_signal(priv->ofono_manager,
			      OFONO_SIGNAL_PROPERTY_CHANGED,
			      G_CALLBACK(ofono_manager_property_changed),
			      self, NULL);

  org_ofono_Manager_get_properties_async(priv->ofono_manager,
					 manager_get_properties_cb,
					 self);
}


PinDialog *pin_dialog_new(void)
{
  GtkWidget *vbox;
  PinDialogPrivate *priv;
  PinDialog *self;

  self = g_object_new(PIN_DIALOG_TYPE, NULL);
  priv = PIN_DIALOG_GET_PRIVATE(self);

  gtk_window_set_title(GTK_WINDOW(self), _("Modem PIN query"));

  gtk_dialog_add_buttons(GTK_DIALOG(self),
			 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
			 GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
			 NULL);

  priv->entry = gtk_entry_new();
  vbox = gtk_dialog_get_content_area(GTK_DIALOG(self));
  gtk_box_pack_start(GTK_BOX(vbox), priv->entry, TRUE, TRUE, 5);

  g_signal_connect(G_OBJECT(self), "response", G_CALLBACK(responded),
		   self);

  dbus_g_object_register_marshaller(_marshal_VOID__STRING_BOXED,
				    G_TYPE_NONE,
				    G_TYPE_STRING,
				    G_TYPE_VALUE,
				    G_TYPE_INVALID);

  get_ofono_manager(self);

  return self;
}
