/*
 * test-service.c: unit test for the service module
 *
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * David Barth <david.barth@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib.h>
#include <glib-object.h>
#include <sys/types.h>
#include <unistd.h>

#include "indicator-network-service.h"
#include "services.h"
#include "network-menu.h"

/* For reference: process pending mainloop events (dbusmenu items) */
/* while (	g_main_context_iteration (g_main_context_default (), FALSE)); */

static gboolean
nothing_fatal (const gchar *log_domain, GLogLevelFlags log_level,
			   const gchar *message, gpointer user_data)
{ return FALSE; }

static void
test_dummy (void)
{
	g_assert (TRUE);
}

static void
test_pin_dialog (void)
{
	/* reminder: test that the pin dialog handles an async call failure */
	g_debug ("test not implemented");

	/* TODO: implement a test for the pin dialog */
	g_assert (FALSE);
}

static struct network_service *
create_ns_mock_object (void)
{
	/* Prevent some g_object errors to stop us:
	   we're creating half initialized instances to get
	   just what it takes to have a sufficient mock object.
	*/
	g_test_log_set_fatal_handler (nothing_fatal, NULL);

	/* let's create a barebone i-service with a fake dbus proxy */
	struct network_service *ns;
	ns = g_malloc0 (sizeof(*ns));
	ns->connman_proxy = NULL;
	ns->services = services_new(ns);
	g_debug ("the next critical warning is normal");
	ns->network_menu = network_menu_new (ns);
	ns->ns_dbus = g_object_new (NETWORK_SERVICE_DBUS_TYPE, NULL);

	/* Create fake dbus constructs */
   	DBusGConnection *fake_bus = g_object_new (DBUS_TYPE_G_CONNECTION, NULL);
	network_service_set_system_bus (ns, fake_bus);

	return ns;
}

static void
test_connman_down (void)
{
	/* let's create a barebone i-service with a fake dbus proxy */
	struct network_service *ns = create_ns_mock_object ();

	/* now pretend connman is getting down */
	connman_proxy_destroy (ns->connman_proxy, ns);

	/* and check that the indicator reflects the right state */
	gchar *icon;
	network_service_dbus_get_icon (ns->ns_dbus, &icon);
	g_assert (g_strcmp0 (icon, ICON_DISCONNECTED) == 0);

	/* and that the menu explains it */
    while (	g_main_context_iteration (g_main_context_default (), FALSE));
	g_assert (network_menu_is_disabled (ns->network_menu));
}

void
test_service_create_suite (void)
{
#define DOMAIN "/Network/Service"

  g_test_add_func (DOMAIN"/Dummy", test_dummy);
  g_test_add_func (DOMAIN"/State/ConnmanDown", test_connman_down);

  /* FIXME: implement and enable this test */
  /* g_test_add_func (DOMAIN"/PinDialogAsyncFail", test_pin_dialog); */
}

gint
main (gint argc, gchar *argv[])
{
  g_type_init ();
  g_test_init (&argc, &argv, NULL);

  test_service_create_suite ();
  
  return g_test_run ();
}
