/* ink.c
 *
 * (c) 2006 Markus Heinz
 *
 * This software is licensed under the terms of the GPL.
 * For details see file COPYING.
 */

#include <inklevel.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

void usage(void) {
  printf("ink -p \"usb\"|\"parport\" [-n <portnumber>]\n");
}

int main(int argc, char *argv[]) {
  struct ink_level *level = NULL;
  int result = 0;
  int port = 0;
  int portnumber = 0;
  int c;
  int i;

  char *strCartridges[MAX_CARTRIDGE_TYPES] = {
    "Not present:",
    "Black:",
    "Color:",
    "Photo:",
    "Cyan:",
    "Magenta:",
    "Yellow:",
    "Photoblack:",
    "Photocyan:",
    "Photomagenta:",
    "Photoyellow:",
    "Red:",
    "Green:",
    "Blue:",
    "Light Black:",
    "Light Cyan:",
    "Light Magenta:",
    "Light Light Black:"
    "Matte Black:",
    "Gloss Optimizer:",
    "Unknown:"
  };

  printf("ink v0.3.2 (c) 2006 Markus Heinz\n\n");

  if (argc == 1) {
    usage();
    return 1;
  }

  while ((c = getopt(argc, argv, "p:n:")) != -1) {
    switch (c) {
    case 'p':
      if (strcmp(optarg, "parport") == 0) {
	port = PARPORT;
      } else if (strcmp(optarg, "usb") == 0) {
	port = USB;
      } else {
	usage();
	return 1;
      }
      break;
    case 'n':
      if (optarg) {
	portnumber = atoi(optarg);
      } else {
	usage();
	return 1;
      }
      break;
    default:
      usage();
      return 1;
    }
  }

  level = (struct ink_level *) malloc(sizeof(struct ink_level));

  if (level == NULL) {
    printf("Not enough memory available.\n");
    return 1;
  }
  
  result = get_ink_level(port, portnumber, level);

  if (result != OK) {
    switch (result) {
    case ERROR:
      printf("An unknown error occured.\n");
      break;
    case DEV_PARPORT_INACCESSIBLE:
      printf("Could not access '/dev/parport%d'.\n", portnumber);
      break;
    case DEV_LP_INACCESSIBLE:
      printf("Could not access '/dev/lp%d'.\n", portnumber);
      break;
    case COULD_NOT_GET_DEVICE_ID:
      printf("Could not get device id.\n");
      break;
    case DEV_USB_LP_INACCESSIBLE:
      printf("Could not access '/dev/usb/lp%d' or '/dev/usblp%d'.\n", 
             portnumber, portnumber);
      break;
    case UNKNOWN_PORT_SPECIFIED:
      printf("Unknown port specified.\n");
      break;
    case NO_PRINTER_FOUND:
      printf("No printer found.\n");
      break;
    case NO_DEVICE_CLASS_FOUND:
      printf("No device class found.\n");
      break;
    case NO_CMD_TAG_FOUND:
      printf("No cmd tag found.\n");
      break;
    case PRINTER_NOT_SUPPORTED:
      printf("Printer not supported.\n");
      break;
    case NO_INK_LEVEL_FOUND:
      printf("No ink level found.\n");
      break;
    case COULD_NOT_WRITE_TO_PRINTER:
      printf("Could not write to printer.\n");
      break;
    case COULD_NOT_READ_FROM_PRINTER:
      printf("Could not read from printer.\n");
      break;
    case COULD_NOT_PARSE_RESPONSE_FROM_PRINTER:
      printf("Could not parse response from printer.\n");
      break;
    case COULD_NOT_GET_CREDIT:
      printf("Could not get credit.\n");
      break;
    }
    printf("Could not get ink level.\n");
    free(level);
    return 1;
  }

  printf("%s\n\n", level->model);

  switch (level->status) {
  case RESPONSE_INVALID:
    printf("No ink level found\n");
    break;

  case RESPONSE_VALID:
    for(i = 0; i < MAX_CARTRIDGE_TYPES; i++) {
      if(level->levels[i][INDEX_TYPE] != CARTRIDGE_NOT_PRESENT) {
        printf("%-18s %3d%%\n", strCartridges[level->levels[i][INDEX_TYPE]],
               level->levels[i][INDEX_LEVEL]);
      } else {
        break;
      }
    }
    break;

  default:
    printf("Printer returned unknown status\n");
    break;
  }
  
  free(level);

  return 0;
}
