// --------------------------------------------------------------------
// Style dialog
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "styles.h"
#include "ipeq.h"
#include "ipeiml.h"
#include "ipepdfdoc.h"
#include "ipestyle.h"
#include "debugwin.h"

#include <qwhatsthis.h>
#include <qlistbox.h>
#include <qfiledialog.h>
#include <qcombobox.h>
#include <qmessagebox.h>
#include <qtoolbutton.h>

// --------------------------------------------------------------------

/*! \class DialogStyles
  \brief Dialog to modify style sheets.
*/

/*! Create dialog, given layer names (read-only), and an
  IpePagePresentation object.  Note that \c data is updated during the
  execution of the dialog, and must be discarded if the return value
  of exec() is not QDialog::Accepted. */
DialogStyles::DialogStyles(QWidget* parent, IpePdfDocument *doc)
  : DlgStyles(parent, "dialogstyles", true), iDoc(doc)
{
  iDelButton->setAccel(Key_Delete);
  UpdateList();
  iSheets->setCurrentItem(0);
  UpdateButtons();
}

void DialogStyles::UpdateList()
{
  iSheets->clear();
  const IpeStyleSheet *sheet = iDoc->StyleSheet();
  while (sheet) {
    if (sheet->Name().empty())
      iSheets->insertItem("<unnamed>");
    else
      iSheets->insertItem(QIpe(sheet->Name()));
    sheet = sheet->Cascade();
  }
}

void DialogStyles::UpdateButtons()
{
  uint cur = iSheets->currentItem();
  iUpButton->setEnabled(!(cur == 0 || cur == iSheets->count() - 1));
  iDownButton->setEnabled(cur + 2 < iSheets->count());
  iDelButton->setEnabled(cur + 1 < iSheets->count());
}

void DialogStyles::ErrMsg(QString str)
{
  QMessageBox::warning(this, "Ipe", str, tr("Dismiss"));
}

void DialogStyles::ShowHelp()
{
  QWhatsThis::enterWhatsThisMode();
}

void DialogStyles::AddNew()
{
  QString fn = QFileDialog::getOpenFileName
    (QString::null, tr("XML files (*.xml)"), this);
  if (fn.isEmpty())
    return;
  if (!iDoc->AddStyleSheet(fn)) {
    ErrMsg(tr("Cannot open or parse file '%1', style sheet not added")
	   .arg(fn));
    return;
  }
  IpeAttributeSeq seq;
  iDoc->CheckStyle(seq);
  UpdateList();
  iSheets->setCurrentItem(0);
  UpdateButtons();
}

void DialogStyles::Save()
{
  int cur = iSheets->currentItem();
  const IpeStyleSheet *sheet = iDoc->StyleSheet();
  while (cur--)
    sheet = sheet->Cascade();

  QString fn = QFileDialog::getSaveFileName
    (QString::null, tr("XML files (*.xml)"), this);
  if (fn.isEmpty())
    return;
  QFile file(fn);
  if (!file.open(IO_WriteOnly)) {
    ErrMsg(tr("Cannot open file '%1', style sheet not saved").arg(fn));
    return;
  }
  IpeQStream stream(&file);
  sheet->SaveAsXml(stream);
  file.close();
}

void DialogStyles::Remove()
{
  uint cur = iSheets->currentItem();
  if (cur + 1 == iSheets->count())
    return;
  IpeStyleSheet *delSheet = 0;
  IpeStyleSheet *prev = 0;
  if (cur == 0) {
    delSheet = iDoc->GetStyleSheet();
    iDoc->SetStyleSheet(delSheet->Cascade());
  } else {
    prev = iDoc->GetStyleSheet();
    for (int cur1 = cur - 1; cur1 > 0; cur1--)
      prev = prev->Cascade();
    delSheet = prev->Cascade();
    prev->SetCascade(delSheet->Cascade());
  }
  // check document
  IpeAttributeSeq seq;
  if (!iDoc->CheckStyle(seq)) {
    for (IpeAttributeSeq::iterator it = seq.begin(); it != seq.end(); ++it)
      IPEDBG << "Undefined symbolic attribute '" <<
	iDoc->Repository()->String(*it) << "'\n";
    ErrMsg(tr("Removing this style sheet would cause some symbolic "
	      "attributes in your document to become undefined.<br>"
	      "See the message window for a list of these attributes.<br>"
	      "The sheet has not been removed."));
    // Undo operation
    if (cur == 0)
      iDoc->SetStyleSheet(delSheet);
    else
      prev->SetCascade(delSheet);
    seq.clear();
    iDoc->CheckStyle(seq);
  } else {
    // Removal successfull
    delSheet->SetCascade(0);
    delete delSheet;
    UpdateList();
    iSheets->setCurrentItem(0);
    UpdateButtons();
  }
}

void DialogStyles::MoveUp()
{
  uint cur = iSheets->currentItem();
  if (!cur || cur + 1 == iSheets->count())
    return;
  if (cur == 1) {
    IpeStyleSheet *sheet0 = iDoc->GetStyleSheet();
    IpeStyleSheet *sheet = sheet0->Cascade();
    sheet0->SetCascade(sheet->Cascade());
    sheet->SetCascade(sheet0);
    iDoc->SetStyleSheet(sheet);
  } else {
    IpeStyleSheet *sheet = iDoc->GetStyleSheet();
    for (int cur2 = cur - 2; cur2 > 0; cur2--)
      sheet = sheet->Cascade();
    // sheet1 and sheet2 need to be flipped
    IpeStyleSheet *sheet1  = sheet->Cascade();
    IpeStyleSheet *sheet2  = sheet1->Cascade();
    sheet1->SetCascade(sheet2->Cascade());
    sheet2->SetCascade(sheet1);
    sheet->SetCascade(sheet2);
  }
  iDoc->SetEdited(true);
  UpdateList();
  iSheets->setCurrentItem(cur - 1);
  UpdateButtons();
}

void DialogStyles::MoveDown()
{
  uint cur = iSheets->currentItem();
  if (cur + 2 >= iSheets->count())
    return;
  if (cur == 0) {
    // flip sheet1 and sheet2
    IpeStyleSheet *sheet1 = iDoc->GetStyleSheet();
    IpeStyleSheet *sheet2 = sheet1->Cascade();
    sheet1->SetCascade(sheet2->Cascade());
    sheet2->SetCascade(sheet1);
    iDoc->SetStyleSheet(sheet2);
  } else {
    IpeStyleSheet *sheet = iDoc->GetStyleSheet();
    for (int cur2 = cur - 1; cur2 > 0; cur2--)
      sheet = sheet->Cascade();
    // sheet1 and sheet2 need to be flipped
    IpeStyleSheet *sheet1  = sheet->Cascade();
    IpeStyleSheet *sheet2  = sheet1->Cascade();
    sheet1->SetCascade(sheet2->Cascade());
    sheet2->SetCascade(sheet1);
    sheet->SetCascade(sheet2);
  }
  iDoc->SetEdited(true);
  UpdateList();
  iSheets->setCurrentItem(cur + 1);
  UpdateButtons();
}

void DialogStyles::CurrentChanged(QListBoxItem *)
{
  UpdateButtons();
}

// --------------------------------------------------------------------
