// --------------------------------------------------------------------
// Canvas Painter
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipetext.h"

#include "ipecanvas.h"

#include <qpainter.h>
#include <qpointarray.h>
#include <qimage.h>
#include <qpaintdevice.h>
#include <qapplication.h>

inline QPoint QPt(const IpeVector &v)
{
  return QPoint(int(v.iX), int(v.iY));
}

inline IpeString IpeQ(const QString &str)
{
  return IpeString(str.utf8());
}

inline QString QIpe(const IpeString &str)
{
  return QString::fromUtf8(str.CString());
}

// --------------------------------------------------------------------

IpeCanvasPainter::IpeCanvasPainter(const IpeStyleSheet *sheet,
				   IpeCanvasPaintServices *services,
				   QPixmap *pixmap, QPainter *painter,
				   double zoom, bool pretty)
  : IpePainter(sheet), iServices(services),
    iPixmap(pixmap), iPainter(painter), iZoom(zoom), iPretty(pretty)
{
  assert(iServices);
  iDimmed = false;
  iPainter->setFont(QApplication::font());
  // these symbolic dash styles are on fixed indices
  for (int i = 0; i < 4; ++i)
    iDash[i] =
      StyleSheet()->Find(IpeAttribute(IpeAttribute::EDashStyle, true, i+1));
}

void IpeCanvasPainter::BeginPath(const IpeVector &v)
{
  iV.push_back(QPt(Matrix() * v));
  iType.push_back(EBeginPath);
}

void IpeCanvasPainter::BeginClosedPath(const IpeVector &v)
{
  iV.push_back(QPt(Matrix() * v));
  iType.push_back(EBeginClosedPath);
}

void IpeCanvasPainter::LineTo(const IpeVector &v)
{
  iV.push_back(QPt(Matrix() * v));
  iType.push_back(ELineTo);
}

void IpeCanvasPainter::CurveTo(const IpeVector &v1, const IpeVector &v2,
			    const IpeVector &v3)
{
  iV.push_back(QPt(Matrix() * v1));
  iV.push_back(QPt(Matrix() * v2));
  iV.push_back(QPt(Matrix() * v3));
  iType.push_back(ECurveTo);
}

void IpeCanvasPainter::EndPath()
{
  iType.push_back(EEndPath);
}

void IpeCanvasPainter::EndClosedPath()
{
  iType.push_back(EEndClosedPath);
}

void IpeCanvasPainter::DrawPath()
{
  const IpeRepository *rep = StyleSheet()->Repository();
  // draw interior
  IpeAttribute fill = Fill();
  if (!fill.IsNullOrVoid()) {
    // fill color must be absolute
    assert(fill.IsAbsolute());
    IpeColor fillColor = rep->ToColor(fill);
    QColor qfill(int(fillColor.iRed * 255),
		 int(fillColor.iGreen * 255),
		 int(fillColor.iBlue * 255));
    QBrush brush;
    brush.setColor(iDimmed ? qfill.light() : qfill);
    brush.setStyle(QBrush::SolidPattern);
    /*
    if (qfill.red() > qfill.green())
      brush.setStyle(QBrush::Dense4Pattern);
    else
      brush.setStyle(QBrush::CrossPattern);
    */
    iPainter->setBrush(brush);
    iPainter->setPen(QPainter::NoPen);

    QPoint org;
    std::vector<QPoint> pv;
    pv.reserve(iV.size());

    uint k = 0;
    QPoint start = iV[0];

    for (uint i = 0; i < iType.size(); ++i) {
      switch (iType[i]) {
      case EBeginPath:
      case EBeginClosedPath:
	org = iV[k];
      case ELineTo:
	pv.push_back(iV[k++]);
	break;
      case ECurveTo:
	{
	  QPointArray ctrlPts(4);
	  ctrlPts.setPoint(0, iV[k-1]);
	  ctrlPts.setPoint(1, iV[k++]);
	  ctrlPts.setPoint(2, iV[k++]);
	  ctrlPts.setPoint(3, iV[k++]);
#if QT_VERSION >= 300
	  QPointArray arr = ctrlPts.cubicBezier();
#else
	  QPointArray arr = ctrlPts.quadBezier();
#endif
	  std::copy(arr.begin(), arr.end(), std::back_inserter(pv));
	}
	break;
      case EEndPath:
      case EEndClosedPath:
	pv.push_back(org);
	if (org != start)
	  pv.push_back(start);
	break; // nothing
      }
    }
    QPointArray arr;
    int n = pv.size();
    arr.setRawData(&pv[0], n);
    // Exploit that Index() of null attribute is zero.
    bool wind = (WindRule().Index() != 0);
    iPainter->drawPolygon(arr, wind);
    arr.resetRawData(&pv[0], n);
  }
  IpeAttribute stroke = Stroke();
  if (!stroke.IsNullOrVoid()) {
    // Now 'stroke' must be an absolute color
    assert(stroke.IsAbsolute());
    IpeColor strokeColor = rep->ToColor(stroke);
    iPainter->setBrush(QPainter::NoBrush);
    QPen pen;
    QColor qstroke(int(strokeColor.iRed * 255),
		   int(strokeColor.iGreen * 255),
		   int(strokeColor.iBlue * 255));
    pen.setColor(iDimmed ? qstroke.light() : qstroke);
    // width
    if (LineWidth()) {
      assert(LineWidth().IsAbsolute());
      double wid = iZoom * rep->ToScalar(LineWidth());
      pen.setWidth(uint(wid + 0.5));
    }
    // Exploit that Index() of null attribute is zero.
    switch (LineJoin().Index()) {
    case IpeStrokeStyle::EMiterJoin:
      pen.setJoinStyle(Qt::MiterJoin);
      break;
    case IpeStrokeStyle::ERoundJoin:
      pen.setJoinStyle(Qt::RoundJoin);
      break;
    case IpeStrokeStyle::EBevelJoin:
      pen.setJoinStyle(Qt::BevelJoin);
      break;
    }
    switch (LineCap().Index()) {
    case IpeStrokeStyle::EButtCap:
      pen.setCapStyle(Qt::FlatCap);
      break;
    case IpeStrokeStyle::ERoundCap:
      pen.setCapStyle(Qt::RoundCap);
      break;
    case IpeStrokeStyle::ESquareCap:
      pen.setCapStyle(Qt::SquareCap);
      break;
    }
    IpeAttribute dash = DashStyle();
    if (!dash.IsNullOrSolid()) {
      // now comes the tricky part
      if (dash == iDash[0])
	pen.setStyle(Qt::DashLine);
      else if (dash == iDash[1])
	pen.setStyle(Qt::DotLine);
      else if (dash == iDash[2])
	pen.setStyle(Qt::DashDotLine);
      else if (dash == iDash[3])
	pen.setStyle(Qt::DashDotDotLine);
      else
	pen.setStyle(Qt::DotLine);
    }
    iPainter->setPen(pen);

    QPoint org;
    std::vector<QPoint> pv;
    pv.reserve(iV.size());

    uint k = 0;
    for (uint i = 0; i < iType.size(); ++i) {
      switch (iType[i]) {
      case EBeginPath:
      case EBeginClosedPath:
	org = iV[k];
      case ELineTo:
	pv.push_back(iV[k++]);
	break;
      case ECurveTo:
	{
	  QPointArray ctrlPts(4);
	  ctrlPts.setPoint(0, iV[k-1]);
	  ctrlPts.setPoint(1, iV[k++]);
	  ctrlPts.setPoint(2, iV[k++]);
	  ctrlPts.setPoint(3, iV[k++]);
#if QT_VERSION >= 300
	  QPointArray arr = ctrlPts.cubicBezier();
#else
	  QPointArray arr = ctrlPts.quadBezier();
#endif
	  std::copy(arr.begin(), arr.end(), std::back_inserter(pv));
	}
	break;
      case EEndClosedPath:
#ifdef WIN32
	pv.push_back(org);
#endif
      case EEndPath:
	{
	  QPointArray arr;
	  int n = pv.size();
	  arr.setRawData(&pv[0], n);
#ifndef WIN32
	  // drawPolygon appears to be buggy, at least on Qt 2.3.0 on Win2000
	  if (iType[i] == EEndClosedPath)
	    iPainter->drawPolygon(arr);
	  else
#endif
	    iPainter->drawPolyline(arr);
	  arr.resetRawData(&pv[0], n);
	  pv.clear();
	  break;
	}
      }
    }
  }
  iType.clear();
  iV.clear();
}

void IpeCanvasPainter::DrawBitmap(IpeBitmap bitmap)
{
  if (!bitmap.RenderData())
    iServices->CvSvcSetRenderData(bitmap);

  if (bitmap.RenderData()) {
    QImage *qimg = static_cast<QImage *>(bitmap.RenderData());
    IpeMatrix tf = Matrix() * IpeMatrix(1.0 / qimg->width(), 0.0,
					0.0, -1.0 / qimg->height(),
					0.0, 1.0);
    QWMatrix m;
    m.setMatrix(tf.iA[0], tf.iA[1], tf.iA[2], tf.iA[3], tf.iA[4], tf.iA[5]);
    iPainter->setWorldMatrix(m, true);
    iPainter->drawImage(0, 0, *qimg);
    iPainter->resetXForm();
  }
}

void IpeCanvasPainter::DrawText(const IpeText *text)
{
  if (Stroke().IsNullOrVoid())
    return;
  // 'stroke' must be an absolute color
  assert(Stroke().IsAbsolute());

  IpeColor stroke = StyleSheet()->Repository()->ToColor(Stroke());
  QColor col(int(stroke.iRed * 255), int(stroke.iGreen * 255),
	     int(stroke.iBlue * 255));
  if (iDimmed)
    col = col.light();
  QRgb rgb = col.rgb();

  // Draw bounding box rectangle
  if (!iPretty) {
    QPen pen;
    pen.setColor(Qt::green);
    pen.setStyle(Qt::DotLine);
    iPainter->setPen(pen);
    iPainter->moveTo(QPt(Matrix() * IpeVector(0,0)));
    iPainter->lineTo(QPt(Matrix() * IpeVector(0, text->TotalHeight())));
    iPainter->lineTo(QPt(Matrix() * IpeVector(text->Width(),
					      text->TotalHeight())));
    iPainter->lineTo(QPt(Matrix() * IpeVector(text->Width(), 0)));
    iPainter->lineTo(QPt(Matrix() * IpeVector(0,0)));

    iPainter->setPen(QPainter::NoPen);
    iPainter->setBrush(Qt::green);
    QPoint ref = QPt(Matrix() * text->Align());
    iPainter->drawRect(ref.x() - 3, ref.y() - 3, 6, 6);
  }

  const IpeText::XForm *xf = text->GetXForm();

  if (!xf) {
    QString s = QIpe(text->Text());
    uint olen = s.length();
    int i = s.find('\n');
    if (i >= 0)
      s.truncate(i);
    s.truncate(30);
    if (s.length() < olen)
      s += "...";
    iPainter->setPen(col);
    // QRect r = iPainter->fontMetrics().boundingRect(s);
    QPoint pt = QPt(Matrix().Translation());
    pt.setY(pt.y() - iPainter->fontMetrics().descent());
    //if (text->IsMiniPage())
    //pt.setY(pt.y() + 15.0);
    iPainter->drawText(pt, s);
    return;
  }

  IpeMatrix m1 =
    Matrix() * IpeMatrix(xf->iStretch.iX, 0, 0, xf->iStretch.iY, 0, 0);
  for (IpeText::PdfCharSeq::const_iterator it = xf->iRender.begin();
       it != xf->iRender.end(); ++it) {
    IpeVector pos = m1 * it->iPos;
    QRgb cRgb = (it->iRgb == 0) ? rgb : it->iRgb;
    // compute world transformation
    IpeMatrix m = m1 * IpeLinear(it->iMatrix[0], it->iMatrix[1],
				 it->iMatrix[2], it->iMatrix[3]);
    iServices->CvSvcRenderCharacter(iPixmap, it->iFontObject, m,
			       pos, cRgb, it->iCharCode, it->iUnicode);
  }

  for (IpeText::PdfSegSeq::const_iterator it = xf->iLines.begin();
       it != xf->iLines.end(); ++it) {
    QPen pen;
    QColor qstroke(QRgb(it->iRgb));
    pen.setColor(qstroke);
    double wid = iZoom * it->iWidth;
    pen.setWidth(uint(wid + 0.5));
    iPainter->setPen(pen);
    iPainter->moveTo(QPt(m1 * it->iSeg.iP));
    iPainter->lineTo(QPt(m1 * it->iSeg.iQ));
  }
}

// --------------------------------------------------------------------
