// --------------------------------------------------------------------
// Overlay for creating IpeText objects
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipecreatetext.h"
#include "ipecanvas.h"

#include <qevent.h>
#include <qpainter.h>
#include <qstring.h>
#include <qmultilineedit.h>
#include <qlabel.h>

inline IpeString IpeQ(const QString &str)
{
  return IpeString(str.utf8());
}

inline QString QIpe(const IpeString &str)
{
  return QString::fromUtf8(str.CString());
}

// --------------------------------------------------------------------

/*! \class IpeDialogCreateText
  \brief Dialog to edit a text paragraph
*/

//! The argument \c data is only updated if the dialog is accepted.
IpeDialogCreateText::IpeDialogCreateText(QWidget* parent, QString caption,
				   QString label, QString &data)
  : IpeDlgCreateText(parent, "dlgcreatetext", true), iData(data)
{
  setCaption(caption);
  iLabel->setText(label);
  iEditor->setText(iData);
  iEditor->setFocus();
#if 0
  iEditor->setFont(SPreferences::Static()->iTextFont);
#endif
}

void IpeDialogCreateText::UpdateData()
{
  iData = iEditor->text();
}

void IpeDialogCreateText::FlushLeft()
{
  QString s = iEditor->text().stripWhiteSpace();
  iEditor->setText("\\begin{flushleft}\n" + s + "\n\\end{flushleft}\n");
}

// --------------------------------------------------------------------

/*! \class CreateText
  \brief Overlay for creating IpeText objects.
*/

//! Constructor starts selection.
IpeCreateText::IpeCreateText(QMouseEvent *, IpeCanvas *canvas,
			     IpeOverlayServices *services)
  : IpeOverlay(canvas), iServices(services)
{
  iV[0] = iCanvas->Pos();
  iV[1] = iV[0];
}

void IpeCreateText::New(QMouseEvent *, IpeCanvas *canvas,
			IpeOverlayServices *services, TMode mode,
			IpeRect *rect, const char *init)
{
  IpeVector pos = canvas->Pos();
  QString text;
  if (init)
    text = init;
  IpeDialogCreateText *dialog =
    new IpeDialogCreateText(canvas, QObject::tr("Create text object"),
			 (mode == EMath ?
			  QObject::tr("Enter formula (LaTeX source)") :
			  QObject::tr("Enter text (LaTeX source)")), text);

  if (dialog->exec() == QDialog::Accepted) {
    text = text.stripWhiteSpace();
    if (!text.isEmpty()) {
      IpeString data = IpeQ(text);
      if (mode == EMath)
	data = IpeString("$") + data + "$";
      IpeText *obj;
      if (rect && mode == EMinipage)
	obj = new IpeText(services->OvSvcAttributes(),
			  data, rect->TopLeft(),
			  IpeText::ETextbox, rect->Width());
      else
	obj = new IpeText(services->OvSvcAttributes(),
			  data, pos, IpeText::ELabel);

      services->OvSvcAddObject(obj);
      canvas->Update();
    }
  }
}

bool IpeCreateText::Edit(IpeText *obj, const IpeStyleSheet *)
{
  QString text;
  text = QIpe(obj->Text());
  IpeDialogCreateText *dialog =
    new IpeDialogCreateText(0, QObject::tr("Edit text object"),
			 QObject::tr("Edit LaTeX source"), text);
  if (dialog->exec() == QDialog::Accepted) {
    text = text.stripWhiteSpace();
    if (!text.isEmpty()) {
      obj->SetText(IpeQ(text));
      return true;
    }
  }
  return false;
}

void IpeCreateText::Draw(QPaintEvent *, QPainter *qPainter) const
{
  qPainter->setPen(CreateColor());
  IpeOverlayPainter painter(iCanvas->StyleSheet(), qPainter);
  painter.Transform(iCanvas->CanvasTfm());
  IpeVector d(0, -20.0);
  painter.BeginClosedPath(iV[0]);
  painter.LineTo(iV[1]);
  painter.LineTo(iV[1] + d);
  painter.LineTo(iV[0] + d);
  painter.EndClosedPath();
  painter.DrawPath();
}

void IpeCreateText::MouseMove(QMouseEvent *)
{
  iV[1].iX = iCanvas->Pos().iX;
  iCanvas->UpdateOverlay();
}

void IpeCreateText::MousePress(QMouseEvent *)
{
  IpeScalar wid = iCanvas->Pos().iX - iV[0].iX;
  if (wid < 0) {
    iV[0].iX += wid;
    wid = -wid;
  }
  QString text;
  IpeDialogCreateText *dialog =
    new IpeDialogCreateText(iCanvas, QObject::tr("Create text object"),
			 QObject::tr("Enter LaTeX source"), text);
  if (dialog->exec() == QDialog::Accepted) {
    text = text.stripWhiteSpace();
    if (!text.isEmpty()) {
      IpeText *obj = new IpeText(iServices->OvSvcAttributes(),
				 IpeQ(text), iV[0], IpeText::EMinipage, wid);
      iServices->OvSvcAddObject(obj);
    }
  }
  iCanvas->FinishOverlay();
}

// --------------------------------------------------------------------

IpeTextWidthChanger::IpeTextWidthChanger(IpeCanvas *canvas, IpeText *text)
  : IpeOverlay(canvas), iObj(text)
{
  iV[0] = iObj->Matrix() * iObj->Position();
  iV[1] = iV[0];
  iV[1].iX = iV[0].iX + iObj->Width();
  iDragging = false;
}

void IpeTextWidthChanger::Draw(QPaintEvent *, QPainter *qPainter) const
{
  qPainter->setPen(Qt::magenta);
  IpeOverlayPainter painter(iCanvas->StyleSheet(), qPainter);
  painter.Transform(iCanvas->CanvasTfm());
  IpeVector d(0, -20.0);
  painter.BeginClosedPath(iV[0]);
  painter.LineTo(iV[1]);
  painter.LineTo(iV[1] + d);
  painter.LineTo(iV[0] + d);
  painter.EndClosedPath();
  painter.DrawPath();
}

void IpeTextWidthChanger::MouseRelease(QMouseEvent *)
{
  IpeScalar wid = IpeAbs(iCanvas->Pos().iX - iV[0].iX);
  iObj->SetWidth(wid);
  iCanvas->FinishOverlay();
}

void IpeTextWidthChanger::MousePress(QMouseEvent *)
{
  iMouseDown = iCanvas->Pos();
  iDragging = true;
  iCanvas->UpdateOverlay();
}

void IpeTextWidthChanger::MouseMove(QMouseEvent *)
{
  if (iDragging) {
    IpeScalar d = iCanvas->Pos().iX - iMouseDown.iX;
    iV[1].iX = iV[0].iX + iObj->Width() + d;
    iCanvas->UpdateOverlay();
  }
}

// --------------------------------------------------------------------
