// -*- C++ -*-
// --------------------------------------------------------------------
// Overlay for the canvas --- shows moving objects etc.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPEOVERLAY_H
#define IPEOVERLAY_H

#include "ipegeo.h"
#include "ipepainter.h"
#include "ipepage.h"

#include <qpoint.h>
#include <qcolor.h>

// --------------------------------------------------------------------

class IpePage;
class IpeStyleSheet;
class IpeCanvas;
class IpeAllAttributes;

class QPainter;
class QMouseEvent;
class QPaintEvent;
class QKeyEvent;

class IpeOverlayPainter : public IpePainter {
public:
  IpeOverlayPainter(const IpeStyleSheet *sheet, QPainter *painter);
  virtual ~IpeOverlayPainter() { }

  virtual void BeginPath(const IpeVector &v);
  virtual void BeginClosedPath(const IpeVector &v);
  virtual void LineTo(const IpeVector &v);
  virtual void CurveTo(const IpeVector &v1, const IpeVector &v2,
		       const IpeVector &v3);
  virtual void EndPath();
  virtual void EndClosedPath();

  virtual void DrawBitmap(IpeBitmap bitmap);
  virtual void DrawText(const IpeText *text);

private:
  QPainter *iPainter;
  QPoint iPathBegin;
  QPoint iPos;
};

// --------------------------------------------------------------------

/*! \class IpeOverlayServices
  \brief Interface for callbacks from the overlays.

  Provides services to (object-creating) overlays.
*/

class IpeOverlayServices {
public:
  virtual void OvSvcAddObject(IpeObject *obj) = 0;
  virtual const IpeAllAttributes &OvSvcAttributes() = 0;
  virtual const IpeStyleSheet *OvSvcStyleSheet() = 0;
  virtual void OvSvcAddUndoItem(IpePage *page, QString s) = 0;
  virtual void OvSvcAddUndoItem(IpePage::iterator it, IpePage *page,
				const IpePgObject &original, QString s) = 0;
};

// --------------------------------------------------------------------

class IpeOverlay {
public:
  virtual ~IpeOverlay();

  //! Redraw the current overlay.
  virtual void Draw(QPaintEvent *ev, QPainter *painter) const = 0;

  virtual void MousePress(QMouseEvent *ev);
  virtual void MouseRelease(QMouseEvent *ev);
  virtual void MouseMove(QMouseEvent *ev);
  virtual void KeyPress(QKeyEvent *ev);

  inline QColor CreateColor() const { return QColor("navy"); }

 protected:
  IpeOverlay(IpeCanvas *canvas);

 protected:
  IpeCanvas *iCanvas;
};

// --------------------------------------------------------------------

class IpePanning : public IpeOverlay {
 public:
  IpePanning(QMouseEvent *ev, const IpePage *page, IpeCanvas *canvas);

  virtual void Draw(QPaintEvent *ev, QPainter *painter) const;
  virtual void MouseRelease(QMouseEvent *ev);
  virtual void MouseMove(QMouseEvent *ev);

 private:
  const IpePage *iPage;
  IpeVector iMouseDown;
  IpeVector iPan;
};

// --------------------------------------------------------------------

class IpeSelecting : public IpeOverlay {
 public:
  IpeSelecting(QMouseEvent *ev, IpePage *page, IpeCanvas *canvas,
	       bool nonDestructive, int selectDistance, int view,
	       IpeOverlayServices *services);

  virtual void Draw(QPaintEvent *ev, QPainter *painter) const;
  virtual void MouseRelease(QMouseEvent *ev);
  virtual void MouseMove(QMouseEvent *ev);
  virtual void KeyPress(QKeyEvent *ev);

private:
  void EnsurePrimary();

private:
  friend class SelectCompare;

  IpePage *iPage;
  IpeOverlayServices *iServices;
  bool iNonDestructive;
  int iSelectDistance;
  IpeVector iMouseDown;
  struct SObj {
    IpePage::iterator iIt;
    double iDistance;
  };
  std::vector<SObj> iObjs;
  int iCur;

  bool iDragging;
  IpeVector iCorner;
};

// --------------------------------------------------------------------

class IpeTransforming : public IpeOverlay {
 public:
  enum TType { EMove, EScale, EStretch, ERotate };

  IpeTransforming(QMouseEvent *ev, IpePage *page,
		  IpeCanvas *canvas, TType type,
		  IpeOverlayServices *services);
  bool FindObjects(int selectDistance);

  virtual void Draw(QPaintEvent *ev, QPainter *painter) const;
  virtual void MouseMove(QMouseEvent *ev);
  virtual void MouseRelease(QMouseEvent *ev);

 private:
  void Compute(const IpeVector &v);
  bool UpdateCloseSelection(int selectDistance);

 private:
  IpePage *iPage;
  IpeOverlayServices *iServices;
  TType iType;
  bool iWithShift;
  IpeVector iMouseDown;
  IpeMatrix iTransform;
  IpeVector iOrigin;
};

// --------------------------------------------------------------------

class IpeOverlayFactory {
public:
  enum { ESelecting, EMoving, ERotating, EStretching, EPanning,
	 ELabel, EMath, EParagraph, EMark,
	 ERectangle, EPolyline, EPolygon,
	 EArc1, EArc2, EArc3,
	 ECircle1, ECircle2, ECircle3,
	 ESpline, ESplinegon, Num };

  IpeOverlayFactory(QMouseEvent *ev, IpeCanvas *canvas,
		    IpePage *page, int view,
		    IpeOverlayServices *services);
  void CreateOverlay(int index);
  IpeOverlay *MakeOverlay(int index);
  static const char *PixmapName(int index);
  static const char *Tooltip(int index);
  static const char *WhatsThis(int index);
  static const char *Shortcut(int index);
  static const char *Context(int index);
private:
  IpeOverlay *StartTransforming(IpeTransforming::TType type);
private:
  QMouseEvent *iEvent;
  IpeCanvas *iCanvas;
  IpePage *iPage;
  int iView;
  IpeOverlayServices *iServices;
};

// --------------------------------------------------------------------
#endif
