// --------------------------------------------------------------------
// Bitmaps
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipebitmap.h"

// --------------------------------------------------------------------

/*! \class IpeBitmap
  \ingroup base
  \brief A bitmap.

  Bitmaps are explicitely shared using reference-counting.  Copying is
  cheap, so IpeBitmap objects are meant to be passed by value.

  The bitmap can cache data to speed up rendering. This data can be
  set only once (as the bitmap is conceptually immutable).

  The bitmap also provides a slot for short-term storage of an "object
  number".  The PDF embedder, for instance, sets it to the PDF object
  number when embedding the bitmap, and can reuse it when "drawing"
  the bitmap.
*/

//! Default constructor constructs null bitmap.
IpeBitmap::IpeBitmap()
{
  iImp = 0;
}

//! Create from XML stream.
IpeBitmap::IpeBitmap(const IpeXmlAttributes &attr, IpeString data,
		     bool rawData)
{
  iImp = new Imp;
  iImp->iRefCount = 1;
  iImp->iObjNum = IpeLex(attr["id"]).GetInt();
  iImp->iRender = 0;
  iImp->iWidth = IpeLex(attr["width"]).GetInt();
  iImp->iHeight = IpeLex(attr["height"]).GetInt();
  int length = IpeLex(attr["length"]).GetInt();
  assert(iImp->iWidth > 0 && iImp->iHeight > 0);
  IpeString cs = attr["ColorSpace"];
  if (cs == "DeviceGray") {
    iImp->iComponents = 1;
    iImp->iColorSpace = EDeviceGray;
  } else if (cs == "DeviceCMYK") {
    iImp->iComponents = 4;
    iImp->iColorSpace = EDeviceCMYK;
  } else {
    iImp->iComponents = 3;
    iImp->iColorSpace = EDeviceRGB;
  }
  IpeString fi = attr["Filter"];
  if (fi == "DCTDecode")
    iImp->iFilter = EDCTDecode;
  else if (fi == "FlateDecode")
    iImp->iFilter = EFlateDecode;
  else
    iImp->iFilter = EDirect;
  iImp->iBitsPerComponent = IpeLex(attr["BitsPerComponent"]).GetInt();
  if (length == 0) {
    assert(iImp->iFilter == EDirect);
    int bitsPerRow = Width() * Components() * BitsPerComponent();
    int bytesPerRow = (bitsPerRow + 7) / 8;
    length = Height() * bytesPerRow;
  }
  if (rawData) {
    assert(length == data.size());
    iImp->iData = IpeBuffer(data.data(), length);
  } else {
    // decode data
    iImp->iData = IpeBuffer(length);
    char *p = iImp->iData.data();
    IpeLex datalex(data);
    while (length-- > 0)
      *p++ = char(datalex.GetHexByte());
  }
  ComputeChecksum();
}

//! Create a new image
IpeBitmap::IpeBitmap(int width, int height,
		     TColorSpace colorSpace, int bitsPerComponent,
		     IpeBuffer data, TFilter filter)
{
  iImp = new Imp;
  iImp->iRefCount = 1;
  iImp->iObjNum = -1;
  iImp->iRender = 0;
  iImp->iWidth = width;
  iImp->iHeight = height;
  assert(iImp->iWidth > 0 && iImp->iHeight > 0);
  iImp->iColorSpace = colorSpace;
  switch (colorSpace) {
  case EDeviceGray:
    iImp->iComponents = 1;
    break;
  case EDeviceRGB:
    iImp->iComponents = 3;
    break;
  case EDeviceCMYK:
    iImp->iComponents = 4;
    break;
  }
  iImp->iFilter = filter;
  iImp->iBitsPerComponent = bitsPerComponent;
  iImp->iData = data;
  ComputeChecksum();
}

//! Copy constructor.
IpeBitmap::IpeBitmap(const IpeBitmap &rhs)
{
  iImp = rhs.iImp;
  if (iImp)
    iImp->iRefCount++;
}

//! Destructor.
IpeBitmap::~IpeBitmap()
{
  if (iImp && --iImp->iRefCount == 0)
    delete iImp;
}

//! Assignment operator (takes care of reference counting).
IpeBitmap &IpeBitmap::operator=(const IpeBitmap &rhs)
{
  if (this != &rhs) {
    if (iImp && --iImp->iRefCount == 0)
      delete iImp;
    iImp = rhs.iImp;
    if (iImp)
      iImp->iRefCount++;
  }
  return *this;
}

//! Save bitmap in XML stream.
void IpeBitmap::SaveAsXml(IpeStream &stream, int id, int pdfObjNum) const
{
  assert(iImp);
  stream << "<bitmap";
  stream << " id=\"" << id << "\"";
  stream << " width=\"" << Width() << "\"";
  stream << " height=\"" << Height() << "\"";
  stream << " length=\"" << Size() << "\"";
  switch (ColorSpace()) {
  case EDeviceGray:
    stream << " ColorSpace=\"DeviceGray\"";
    break;
  case EDeviceRGB:
    stream << " ColorSpace=\"DeviceRGB\"";
    break;
  case EDeviceCMYK:
    stream << " ColorSpace=\"DeviceCMYK\"";
    break;
  }
  switch (Filter()) {
  case EFlateDecode:
    stream << " Filter=\"FlateDecode\"";
    break;
  case EDCTDecode:
    stream << " Filter=\"DCTDecode\"";
    break;
  default:
    // no filter
    break;
  }
  stream << " BitsPerComponent=\"" << BitsPerComponent() << "\"";

  if (pdfObjNum >= 0) {
    stream << " pdfObject=\"" << pdfObjNum << "\"/>\n";
  } else {
    // save data
    stream << ">\n";
    const char *data = Data();
    const char *fin = data + Size();
    int col = 0;
    while (data != fin) {
      stream.PutHexByte(*data++);
      if (++col == 36) {
	stream << "\n";
	col = 0;
      }
    }
    if (col > 0)
      stream << "\n";
    stream << "</bitmap>\n";
  }
}

//! Set a cached bitmap for fast rendering.
void IpeBitmap::SetRenderData(void *data) const
{
  assert(iImp && iImp->iRender == 0);
  iImp->iRender = data;
}

bool IpeBitmap::Equal(IpeBitmap rhs) const
{
  if (iImp == rhs.iImp)
    return true;
  if (!iImp || !rhs.iImp)
    return false;

  if (iImp->iColorSpace != rhs.iImp->iColorSpace ||
      iImp->iBitsPerComponent != rhs.iImp->iBitsPerComponent ||
      iImp->iWidth != rhs.iImp->iWidth ||
      iImp->iHeight != rhs.iImp->iHeight ||
      iImp->iComponents != rhs.iImp->iComponents ||
      iImp->iFilter != rhs.iImp->iFilter ||
      iImp->iChecksum != rhs.iImp->iChecksum ||
      iImp->iData.size() != rhs.iImp->iData.size())
    return false;
  // check actual data
  int len = iImp->iData.size();
  char *p = iImp->iData.data();
  char *q = rhs.iImp->iData.data();
  while (len--) {
    if (*p++ != *q++)
      return false;
  }
  return true;
}

void IpeBitmap::ComputeChecksum()
{
  int s = 0;
  int len = iImp->iData.size();
  char *p = iImp->iData.data();
  while (len--) {
    s = (s & 0x0fffffff) << 3;
    s += *p++;
  }
  iImp->iChecksum = s;
}

// --------------------------------------------------------------------
