// -*- C++ -*-
// --------------------------------------------------------------------
// Ipelets
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPELET_H
#define IPELET_H

#include "ipebase.h"

#ifdef WIN32
#define IPELET_DECLARE extern "C" __declspec(dllexport)
#else
#define IPELET_DECLARE extern "C"
#endif

// --------------------------------------------------------------------

class IpeAllAttributes;
class IpePage;
class IpeSnapData;
class IpeDocument;
class IpeStyleSheet;

/*! \class IpeletHelper
  \ingroup ipelet
  \brief Service provider for Ipelets.

  Ipelets can ask Ipe to perform various services and request information
  using this class.
*/

class IPE_EXPORT IpeletHelper {
 public:
  //! Show a message in the status bar.
  virtual void Message(const char *msg) = 0;
  /*! Pop up a modal message box with the \a text, and up to three
    buttons (Use 0 pointer to remove a button).  Returns index (0
    to 2) of button pressed.
  */
  virtual int MessageBox(const char *text, const char *button1,
			 const char *button2, const char *button3) = 0;
  /*! Pop up a modal dialog asking the user to enter a string.
    Returns true if the user didn't cancel the dialog. */
  virtual bool GetString(const char *prompt, IpeString &str) = 0;
  /*! Return the style sheet. */
  virtual const IpeStyleSheet *StyleSheet() = 0;
  /*! Return read-only document. */
  virtual const IpeDocument *Document() = 0;
  /*! Get read-write access to the entire document.
    Marks the document as edited, and resets undo. */
  virtual IpeDocument *EditDocument() = 0;
  //! Return index of current page (starting with 0).
  virtual int CurrentPage() const = 0;
  //! Return index of current page view (starting with 0).
  virtual int CurrentView() const = 0;
  //! Return index of current layer in the page.
  virtual int CurrentLayer() const = 0;
  //! Return the attributes currently set in the UI.
  virtual const IpeAllAttributes &Attributes() const = 0;
  /*! Return the current snapping situation in Ipe. */
  virtual const IpeSnapData &SnapData() const = 0;
};

// --------------------------------------------------------------------

/*! \class Ipelet
  \ingroup ipelet
  \brief Abstract base class for Ipelets.
*/

class IPE_EXPORT Ipelet {
 public:
  virtual ~Ipelet() = 0;
  //! Return the version of Ipelib the Ipelet was linked against.
  virtual int IpelibVersion() const = 0;
  virtual int NumFunctions() const;
  //! Return the label of the Ipelet (to be shown in the menu).
  virtual const char *Label() const = 0;
  virtual const char *SubLabel(int function) const;
  virtual const char *KeySequence(int function) const;
  //! Run a function from the Ipelet.
  virtual void Run(int function, IpePage *page, IpeletHelper *helper) = 0;
};

// --------------------------------------------------------------------
#endif
