// --------------------------------------------------------------------
// Popup menu with object attributes
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "props.h"
#include "ipeoverlay.h"
#include "ipepath.h"
#include "ipemark.h"
#include "ipetext.h"
#include "ipeimage.h"
#include "iperef.h"
#include "ipegroup.h"
#include "ipestyle.h"
#include "ipeq.h"
#include "ipecanvas.h"
#include "ipecreatetext.h"
#include "ipeeditpath.h"
#include "ipescissors.h"

#include <qpopupmenu.h>
#include <qcursor.h>
#include <qlabel.h>

// --------------------------------------------------------------------

QString AttributePopup::String(IpeAttribute attr) const
{
  return QIpe(iStyleSheet->Repository()->String(attr));
}

void AttributePopup::SetTitle(QString str)
{
  iMenu->insertItem(str);
}

void AttributePopup::AddComment(QString str)
{
  iMenu->insertItem(str);
  /*
  QLabel *label = new QLabel(iMenu);
  label->setText(str);
  iMenu->insertItem(label);
  */
}

void AttributePopup::InsertColor(QString str, IpeAttribute color, bool fill)
{
  QPopupMenu *sub = new QPopupMenu(iMenu);
  for (uint i = 0; i < iSyms[IpeAttribute::EColor].size(); ++i) {
    IpeAttribute attr = iSyms[IpeAttribute::EColor][i];
    if (fill)
      sub->insertItem(ColorPixmap(attr, iStyleSheet), String(attr),
		      this, SLOT(SetFill(int)), 0, i);
    else
      sub->insertItem(ColorPixmap(attr, iStyleSheet), String(attr),
		      this, SLOT(SetStroke(int)), 0, i);
  }
  QString item = str + ": ";
  if (color.IsNull())
    item += "void";
  else
    item += String(color);
  iMenu->insertItem(item, sub);
}

void AttributePopup::InsertAttribute(QString str, IpeKind kind,
				     IpeAttribute attr)
{
  QPopupMenu *sub = new QPopupMenu(iMenu);
  for (uint i = 0; i < iSyms[kind].size(); ++i) {
    IpeAttribute attr = iSyms[kind][i];
    switch (kind) {
    case IpeAttribute::ELineWidth:
      sub->insertItem(String(attr), this, SLOT(SetLineWidth(int)), 0, i);
      break;
    case IpeAttribute::EDashStyle:
      sub->insertItem(String(attr), this, SLOT(SetDashStyle(int)), 0, i);
      break;
    case IpeAttribute::ETextSize:
      sub->insertItem(String(attr), this, SLOT(SetTextSize(int)), 0, i);
      break;
    case IpeAttribute::EMarkSize:
      sub->insertItem(String(attr), this, SLOT(SetMarkSize(int)), 0, i);
      break;
    default: // make compiler happy
      break;
    }
  }
  QString item = str + ": ";
  if (attr.IsNull() && kind == IpeAttribute::EDashStyle)
    item += "solid";
  else if (attr.IsNull())
    item += "(null)";
  else
    item += String(attr);
  iMenu->insertItem(item, sub);
}

void AttributePopup::InsertArrow(QString str, IpeAttribute attr, bool backw)
{
  QPopupMenu *sub = new QPopupMenu(iMenu);
  if (backw)
    sub->insertItem(tr("none"), this, SLOT(SetBackwardArrow(int)), 0, 0);
  else
    sub->insertItem(tr("none"), this, SLOT(SetForwardArrow(int)), 0, 0);

  for (uint i = 0; i < iSyms[IpeAttribute::EArrowSize].size(); ++i) {
    IpeAttribute attr1 = iSyms[IpeAttribute::EArrowSize][i];
    if (backw)
      sub->insertItem(String(attr1), this, SLOT(SetBackwardArrow(int)),
		      0, i+1);
    else
      sub->insertItem(String(attr1), this, SLOT(SetForwardArrow(int)),
		      0, i+1);
  }
  QString item = str + ": ";
  if (attr.IsNull())
    item += "none";
  else
    item += String(attr);
  iMenu->insertItem(item, sub);
}

const char *capText[] = { "Butt cap", "Round cap", "Square cap" };
const char *joinText[] = { "Miter join", "Round join", "Bevel join" };

void AttributePopup::InsertStrokeStyle(IpeStrokeStyle attr)
{
  // line cap
  QPopupMenu *sub = new QPopupMenu(iMenu);
  for (int i = 0; i < 3; ++i)
    sub->insertItem(capText[i], this, SLOT(SetCap(int)), 0, i);
  QString item = "Line cap: ";
  if (!attr.Cap())
    item += "default";
  else
    item += capText[attr.Cap().Index()];
  iMenu->insertItem(item, sub);
  // line join
  sub = new QPopupMenu(iMenu);
  for (int i = 0; i < 3; ++i)
    sub->insertItem(joinText[i], this, SLOT(SetJoin(int)), 0, i);
  item = "Line join: ";
  if (!attr.Cap())
    item += "default";
  else
    item += joinText[attr.Join().Index()];
  iMenu->insertItem(item, sub);
  // fill rule
  sub = new QPopupMenu(iMenu);
  sub->insertItem(tr("Even-odd fill"), this, SLOT(SetWindRule(int)), 0, 0);
  sub->insertItem(tr("Wind fill"), this, SLOT(SetWindRule(int)), 0, 1);
  item = "Default fill rule";
  if (attr.WindRule()) {
    if (attr.WindRule().Index() > 0)
      item = "Wind fill rule";
    else
      item = "Even-odd fill rule";
  }
  iMenu->insertItem(item, sub);
}

void AttributePopup::InsertEdit()
{
  iMenu->insertItem(tr("Edit"), this, SLOT(Edit()));
}

void AttributePopup::InsertLayer(int layer)
{
  QPopupMenu *sub = new QPopupMenu;
  for (int i = 0; i < iPage->CountLayers(); ++i) {
    sub->insertItem(QIpe(iPage->Layer(i).Name()),
		    this, SLOT(SetLayer(int)), 0, i);
  }
  iMenu->insertItem(tr("Layer: ") + QIpe(iPage->Layer(layer).Name()), sub);
}

// --------------------------------------------------------------------

void AttributePopup::VisitGroup(const IpeGroup *obj)
{
  SetTitle(tr("Group object"));
  AddComment(tr("%1 elements").arg(obj->size()));
  InsertLayer(iIt->Layer());
  InsertStrokeStyle(obj->StrokeStyle());
  if (!iLayerLocked)
    iMenu->insertItem(tr("Ungroup"), this, SLOT(Ungroup()));
}

void AttributePopup::VisitPath(const IpePath *obj)
{
  SetTitle(tr("Path object"));
  AddComment(tr("%1 subpaths").arg(obj->NumSubPaths()));
  InsertLayer(iIt->Layer());
  InsertColor(tr("Stroke"), obj->Stroke());
  InsertColor(tr("Fill"), obj->Fill(), true);
  InsertAttribute(tr("Line width"), IpeAttribute::ELineWidth,
		  obj->LineWidth());
  InsertAttribute(tr("Dash style"), IpeAttribute::EDashStyle,
		  obj->DashStyle());
  InsertArrow(tr("Arrow"), obj->ForwardArrow(), false);
  InsertArrow(tr("Backward arrow"), obj->BackwardArrow(), true);
  InsertStrokeStyle(obj->StrokeStyle());
  if (!iLayerLocked) {
    if (obj->NumSubPaths() > 1)
      iMenu->insertItem(tr("Decompose path"), this, SLOT(DecomposePath()));
    else
      iMenu->insertItem(tr("Scissors"), this, SLOT(UseScissors()));
  }
  // don't bother to provide editing of a single ellipse
  if (obj->NumSubPaths() > 1 ||
      obj->SubPath(0)->Type() != IpeSubPath::EEllipse)
    InsertEdit();
}

void AttributePopup::VisitMark(const IpeMark *obj)
{
  SetTitle(tr("Mark object"));
  InsertLayer(iIt->Layer());
  InsertColor(tr("Stroke"), obj->Stroke());
  InsertAttribute(tr("Size"), IpeAttribute::EMarkSize, obj->Size());
}

void AttributePopup::VisitText(const IpeText *obj)
{
  SetTitle(tr("Text object"));
  InsertLayer(iIt->Layer());
  InsertColor(tr("Stroke"), obj->Stroke());
  InsertAttribute(tr("Size"), IpeAttribute::ETextSize, obj->Size());

  QPopupMenu *sub = new QPopupMenu(iMenu);
  sub->insertItem(tr("transformable"),
		  this, SLOT(SetTransformable(int)), 0, 1);
  sub->insertItem(tr("fixed"),
		  this, SLOT(SetTransformable(int)), 0, 0);
  if (obj->IsTransformable())
    iMenu->insertItem(tr("Transformable"), sub);
  else
    iMenu->insertItem(tr("Fixed"), sub);
  if (obj->IsMiniPage())
    iMenu->insertItem(tr("Change width"), this, SLOT(ChangeWidth()));

  sub = new QPopupMenu(iMenu);
  sub->insertItem(tr("label"),
		  this, SLOT(SetParagraph(int)), 0, IpeText::ELabel);
  sub->insertItem(tr("minipage"),
		  this, SLOT(SetParagraph(int)), 0, IpeText::EMinipage);
  sub->insertItem(tr("textbox"),
		  this, SLOT(SetParagraph(int)), 0, IpeText::ETextbox);
  sub->insertItem(tr("title"),
		  this, SLOT(SetParagraph(int)), 0, IpeText::ETitle);
  switch (obj->Type()) {
  case IpeText::ELabel:
    iMenu->insertItem(tr("Label"), sub);
    break;
  case IpeText::EMinipage:
    iMenu->insertItem(tr("Minipage"), sub);
    break;
  case IpeText::ETextbox:
    iMenu->insertItem(tr("Textbox"), sub);
    break;
  case IpeText::ETitle:
    iMenu->insertItem(tr("Title"), sub);
    break;
  }

  sub = new QPopupMenu(iMenu);
  sub->insertItem(tr("Left"), this, SLOT(SetHAlign(int)), 0, IpeText::ELeft);
  sub->insertItem(tr("Right"), this, SLOT(SetHAlign(int)), 0, IpeText::ERight);
  sub->insertItem(tr("Center"), this, SLOT(SetHAlign(int)), 0,
		  IpeText::EHCenter);
  QString halign;
  switch (obj->HorizontalAlignment()) {
  case IpeText::ELeft:
    halign = "Left aligned";
    break;
  case IpeText::ERight:
    halign = "Right aligned";
    break;
  case IpeText::EHCenter:
    halign = "Horizontally centered";
    break;
  }
  iMenu->insertItem(halign, sub);

  sub = new QPopupMenu(iMenu);
  sub->insertItem(tr("Top"), this, SLOT(SetVAlign(int)), 0, IpeText::ETop);
  sub->insertItem(tr("Bottom"), this, SLOT(SetVAlign(int)), 0,
		  IpeText::EBottom);
  sub->insertItem(tr("Baseline"), this, SLOT(SetVAlign(int)), 0,
		  IpeText::EBaseline);
  sub->insertItem(tr("Center"), this, SLOT(SetVAlign(int)), 0,
		  IpeText::EVCenter);
  QString valign;
  switch (obj->VerticalAlignment()) {
  case IpeText::ETop:
    valign = "Top aligned";
    break;
  case IpeText::EBottom:
    valign = "Bottom aligned";
    break;
  case IpeText::EVCenter:
    valign = "Vertically centered";
    break;
  case IpeText::EBaseline:
    valign = "Aligned on baseline";
    break;
  }
  iMenu->insertItem(valign, sub);

  if (obj->IsMiniPage())
    iMenu->insertItem(tr("Change width"), this, SLOT(ChangeWidth()));

  InsertEdit();
}

void AttributePopup::VisitImage(const IpeImage *obj)
{
  SetTitle(tr("Image object"));
  IpeBitmap bm = obj->Bitmap();
  SetTitle(tr("%1 x %2 pixels").arg(bm.Width()).arg(bm.Height()));
  SetTitle(tr("%1 components with %2 bits").arg(bm.Components())
	   .arg(bm.BitsPerComponent()));
  SetTitle(tr("Filter %1").arg(int(bm.Filter())));
  InsertLayer(iIt->Layer());
}

void AttributePopup::VisitReference(const IpeReference *obj)
{
  SetTitle(tr("Reference object"));
  iMenu->insertItem(tr("Name: ") + String(obj->Name()));
  InsertLayer(iIt->Layer());
}

// --------------------------------------------------------------------

void AttributePopup::SetStroke(int id)
{
  iIt->Object()->SetStroke(iSyms[IpeAttribute::EColor][id]);
  iChanged = true;
}

void AttributePopup::SetFill(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    obj->SetFill(iSyms[IpeAttribute::EColor][id]);
    iChanged = true;
  }
}

void AttributePopup::SetLineWidth(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    obj->SetLineWidth(iSyms[IpeAttribute::ELineWidth][id]);
    iChanged = true;
  }
}

void AttributePopup::SetDashStyle(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    obj->SetDashStyle(iSyms[IpeAttribute::EDashStyle][id]);
    iChanged = true;
  }
}

void AttributePopup::SetTextSize(int id)
{
  IpeText *obj = iIt->Object()->AsText();
  if (obj) {
    obj->SetSize(iSyms[IpeAttribute::ETextSize][id]);
    iChanged = true;
  }
}

void AttributePopup::SetParagraph(int id)
{
  IpeText *obj = iIt->Object()->AsText();
  if (obj && obj->Type() != IpeText::TType(id)) {
    obj->SetType(IpeText::TType(id));
    iChanged = true;
  }
}

void AttributePopup::SetHAlign(int id)
{
  IpeText *obj = iIt->Object()->AsText();
  if (obj && obj->HorizontalAlignment() != id) {
    obj->SetHorizontalAlignment(IpeText::THorizontalAlignment(id));
    iChanged = true;
  }
}

void AttributePopup::SetVAlign(int id)
{
  IpeText *obj = iIt->Object()->AsText();
  if (obj && obj->VerticalAlignment() != id) {
    obj->SetVerticalAlignment(IpeText::TVerticalAlignment(id));
    iChanged = true;
  }
}

void AttributePopup::SetTransformable(int id)
{
  IpeText *obj = iIt->Object()->AsText();
  if (obj) {
    obj->SetTransformable(bool(id));
    iChanged = true;
  }
}

void AttributePopup::SetMarkShape(int id)
{
  IpeMark *obj = iIt->Object()->AsMark();
  if (obj) {
    obj->SetShape(id + 1);
    iChanged = true;
  }
}

void AttributePopup::SetMarkSize(int id)
{
  IpeMark *obj = iIt->Object()->AsMark();
  if (obj) {
    obj->SetSize(iSyms[IpeAttribute::EMarkSize][id]);
    iChanged = true;
  }
}

void AttributePopup::SetForwardArrow(int id)
{
  IpePath *obj = iIt->Object()->AsPath();
  if (obj) {
    if (id == 0)
      obj->SetForwardArrow(IpeAttribute());
    else
      obj->SetForwardArrow(iSyms[IpeAttribute::EArrowSize][id - 1]);
    iChanged = true;
  }
}

void AttributePopup::SetBackwardArrow(int id)
{
  IpePath *obj = iIt->Object()->AsPath();
  if (obj) {
    if (id == 0)
      obj->SetBackwardArrow(IpeAttribute());
    else
      obj->SetBackwardArrow(iSyms[IpeAttribute::EArrowSize][id - 1]);
    iChanged = true;
  }
}

void AttributePopup::SetCap(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    IpeStrokeStyle ss = obj->StrokeStyle();
    ss.SetCap(id);
    obj->SetStrokeStyle(ss);
    iChanged = true;
  }
}

void AttributePopup::SetJoin(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    IpeStrokeStyle ss = obj->StrokeStyle();
    ss.SetJoin(id);
    obj->SetStrokeStyle(ss);
    iChanged = true;
  }
}

void AttributePopup::SetWindRule(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    IpeStrokeStyle ss = obj->StrokeStyle();
    ss.SetWindRule(id != 0);
    obj->SetStrokeStyle(ss);
    iChanged = true;
  }
}

void AttributePopup::SetLayer(int id)
{
  iIt->SetLayer(id);
  iChanged = true;
}

// --------------------------------------------------------------------

void AttributePopup::Edit()
{
  if (iIt->Object()->AsText() &&
      IpeCreateText::Edit(iIt->Object()->AsText(), iStyleSheet)) {
    iServices->OvSvcAddUndoItem(iIt, iPage, iOriginal,
				     tr("text object edit"));
  } else if (iIt->Object()->AsPath())
    iCanvas->SetOverlay(new IpeEditPath(iCanvas, iIt->Object()->AsPath(),
					iPage, iServices, iIt));
}

void AttributePopup::Ungroup()
{
  if (iIt->Object()->AsGroup()) {
    iServices->OvSvcAddUndoItem(new IpePage(*iPage), tr("ungrouping"));
    iPage->Ungroup(iLayer);
  }
}

void AttributePopup::DecomposePath()
{
  if (iIt->Object()->AsPath()) {
    iServices->OvSvcAddUndoItem(new IpePage(*iPage),
				     tr("path decomposition"));
    iPage->DecomposePath(iLayer);
  }
}

void AttributePopup::UseScissors()
{
  if (iIt->Object()->AsPath())
    iCanvas->SetOverlay(new IpeScissors(iCanvas, iIt->Object()->AsPath(),
					iPage, iIt));
}

void AttributePopup::ChangeWidth()
{
  if (iIt->Object()->AsText())
    iCanvas->SetOverlay(new IpeTextWidthChanger(iCanvas,
						iIt->Object()->AsText()));
}

// --------------------------------------------------------------------

AttributePopup::AttributePopup(IpePage::iterator it, IpePage *page,
			       IpeCanvas *canvas, int layer,
			       const IpeStyleSheet *sheet,
			       const IpeAttributeSeq *syms,
			       IpeOverlayServices *services)
  : iCanvas(canvas), iServices(services), iStyleSheet(sheet),
    iSyms(syms), iPage(page), iIt(it), iOriginal(*it),
    iLayer(layer)
{
  iMenu = new QPopupMenu();
  iChanged = false;
  iLayerLocked = page->Layer(layer).IsLocked();
  it->Object()->Accept(*this);
}

void AttributePopup::Exec()
{
  iMenu->exec(QCursor::pos());
  if (iChanged) {
    iServices->OvSvcAddUndoItem(iIt, iPage, iOriginal,
				     tr("attribute change"));
    iPage->SetEdited(true);
    iCanvas->Update();
  }
}

// --------------------------------------------------------------------
