// --------------------------------------------------------------------
// Popup menu with object attributes
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2005  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "props.h"
#include "ipeoverlay.h"
#include "ipepath.h"
#include "ipemark.h"
#include "ipetext.h"
#include "ipeimage.h"
#include "iperef.h"
#include "ipegroup.h"
#include "ipestyle.h"
#include "ipeq.h"
#include "ipecanvas.h"
#include "ipecreatetext.h"
#include "ipeeditpath.h"
#include "ipescissors.h"
#include "widgets.h"

#include <QCursor>
#include <QLabel>

// In appui.cpp
extern QPixmap ColorPixmap(IpeAttribute sym, const IpeStyleSheet *sheet);

// --------------------------------------------------------------------

QString AttributePopup::String(IpeAttribute attr) const
{
  return QIpe(iStyleSheet->Repository()->String(attr));
}

void AttributePopup::connect(IpeAction *a, const char *receiver)
{
  QObject::connect(a, SIGNAL(triggered(int)), this, receiver);
}

void AttributePopup::SetTitle(QString str)
{
  iMenu->addAction(str);
}

void AttributePopup::AddComment(QString str)
{
  iMenu->addAction(str);
  /*
  QLabel *label = new QLabel(iMenu);
  label->setText(str);
  iMenu->insertItem(label);
  */
}

void AttributePopup::InsertColor(QString str, IpeAttribute color, bool fill)
{
  QMenu *sub = new QMenu(iMenu);
  for (uint i = 0; i < iSyms[IpeAttribute::EColor].size(); ++i) {
    IpeAttribute attr = iSyms[IpeAttribute::EColor][i];
    if (!fill && attr == IpeAttribute::Void())
      continue;
    IpeAction *a =
      new IpeAction(i, ColorPixmap(attr, iStyleSheet), String(attr), this);
    sub->addAction(a);
    if (fill)
      connect(a, SLOT(SetFill(int)));
    else
      connect(a, SLOT(SetStroke(int)));
  }
  QString item = str + QLatin1String(": ");
  if (color.IsNull())
    item += QLatin1String("void");
  else
    item += String(color);
  sub->setTitle(item);
  iMenu->addMenu(sub);
}

void AttributePopup::InsertAttribute(QString str, IpeKind kind,
				     IpeAttribute attr)
{
  QMenu *sub = new QMenu(iMenu);
  for (uint i = 0; i < iSyms[kind].size(); ++i) {
    IpeAttribute attr = iSyms[kind][i];
    IpeAction *a = new IpeAction(i, String(attr), this);
    sub->addAction(a);
    switch (kind) {
    case IpeAttribute::ELineWidth:
      connect(a, SLOT(SetLineWidth(int)));
      break;
    case IpeAttribute::EDashStyle:
      connect(a, SLOT(SetDashStyle(int)));
      break;
    case IpeAttribute::ETextSize:
      connect(a, SLOT(SetTextSize(int)));
      break;
    case IpeAttribute::EMarkSize:
      connect(a, SLOT(SetMarkSize(int)));
      break;
    default: // make compiler happy
      break;
    }
  }
  QString item = str + QLatin1String(": ");
  if (attr.IsNull() && kind == IpeAttribute::EDashStyle)
    item += QLatin1String("solid");
  else if (attr.IsNull())
    item += QLatin1String("(null)");  // why should this happen?
  else
    item += String(attr);
  sub->setTitle(item);
  iMenu->addMenu(sub);
}

void AttributePopup::InsertArrow(QString str, IpeAttribute attr, bool backw)
{
  QMenu *sub = new QMenu(iMenu);
  IpeAction *a = new IpeAction(0, tr("none"), this);
  sub->addAction(a);
  if (backw)
    connect(a, SLOT(SetBackwardArrow(int)));
  else
    connect(a, SLOT(SetForwardArrow(int)));

  for (uint i = 0; i < iSyms[IpeAttribute::EArrowSize].size(); ++i) {
    IpeAttribute attr1 = iSyms[IpeAttribute::EArrowSize][i];
    a = new IpeAction(i+1, String(attr1), this);
    sub->addAction(a);
    if (backw)
      connect(a, SLOT(SetBackwardArrow(int)));
    else
      connect(a, SLOT(SetForwardArrow(int)));
  }
  QString item = str + QLatin1String(": ");
  if (attr.IsNull())
    item += QLatin1String("none");
  else
    item += String(attr);
  sub->setTitle(item);
  iMenu->addMenu(sub);
}

const char *capText[] = { "Butt cap", "Round cap", "Square cap" };
const char *joinText[] = { "Miter join", "Round join", "Bevel join" };

void AttributePopup::InsertStrokeStyle(IpeStrokeStyle attr)
{
  // line cap
  QMenu *sub = new QMenu(iMenu);
  IpeAction *a = new IpeAction(99, QLatin1String("Default"), this);
  sub->addAction(a);
  connect(a, SLOT(SetCap(int)));
  for (int i = 0; i < 3; ++i) {
    a = new IpeAction(i, QLatin1String(capText[i]), this);
    sub->addAction(a);
    connect(a, SLOT(SetCap(int)));
  }
  QString item = QLatin1String("Line cap: ");
  if (!attr.Cap())
    item += QLatin1String("default");
  else
    item += QLatin1String(capText[attr.Cap().Index()]);
  sub->setTitle(item);
  iMenu->addMenu(sub);
  // line join
  sub = new QMenu(iMenu);
  a = new IpeAction(99, QLatin1String("Default"), this);
  sub->addAction(a);
  connect(a, SLOT(SetJoin(int)));
  for (int i = 0; i < 3; ++i) {
    a = new IpeAction(i, QLatin1String(joinText[i]), this);
    sub->addAction(a);
    connect(a, SLOT(SetJoin(int)));
  }
  item = QLatin1String("Line join: ");
  if (!attr.Join())
    item += QLatin1String("default");
  else
    item += QLatin1String(joinText[attr.Join().Index()]);
  sub->setTitle(item);
  iMenu->addMenu(sub);
  // fill rule
  sub = new QMenu(iMenu);
  a = new IpeAction(99, tr("Default fill rule"), this);
  sub->addAction(a);
  connect(a, SLOT(SetWindRule(int)));
  a = new IpeAction(0, tr("Even-odd fill"), this);
  sub->addAction(a);
  connect(a, SLOT(SetWindRule(int)));
  a = new IpeAction(1, tr("Wind fill"), this);
  sub->addAction(a);
  connect(a, SLOT(SetWindRule(int)));
  if (attr.WindRule()) {
    if (attr.WindRule().Index() > 0)
      item = QLatin1String("Wind fill rule");
    else
      item = QLatin1String("Even-odd fill rule");
  } else
    item = QLatin1String("Default fill rule");
  sub->setTitle(item);
  iMenu->addMenu(sub);
}

void AttributePopup::InsertEdit()
{
  iMenu->addAction(tr("Edit"), this, SLOT(Edit()));
}

void AttributePopup::InsertLayer(int layer)
{
  QMenu *sub = new QMenu;
  for (int i = 0; i < iPage->CountLayers(); ++i) {
    IpeAction *a = new IpeAction(i, QIpe(iPage->Layer(i).Name()), this);
    sub->addAction(a);
    connect(a, SLOT(SetLayer(int)));
  }
  sub->setTitle(tr("Layer: ") + QIpe(iPage->Layer(layer).Name()));
  iMenu->addMenu(sub);
}

// --------------------------------------------------------------------

void AttributePopup::VisitGroup(const IpeGroup *obj)
{
  SetTitle(tr("Group object"));
  AddComment(tr("%1 elements").arg(obj->size()));
  InsertLayer(iIt->Layer());
  InsertStrokeStyle(obj->StrokeStyle());
  if (!iLayerLocked)
    iMenu->addAction(tr("Ungroup"), this, SLOT(Ungroup()));
}

void AttributePopup::VisitPath(const IpePath *obj)
{
  SetTitle(tr("Path object"));
  AddComment(tr("%1 subpaths").arg(obj->NumSubPaths()));
  InsertLayer(iIt->Layer());
  InsertColor(tr("Stroke"), obj->Stroke());
  InsertColor(tr("Fill"), obj->Fill(), true);
  InsertAttribute(tr("Line width"), IpeAttribute::ELineWidth,
		  obj->LineWidth());
  InsertAttribute(tr("Dash style"), IpeAttribute::EDashStyle,
		  obj->DashStyle());
  InsertArrow(tr("Arrow"), obj->ForwardArrow(), false);
  InsertArrow(tr("Backward arrow"), obj->BackwardArrow(), true);
  InsertStrokeStyle(obj->StrokeStyle());
  if (!iLayerLocked) {
    if (obj->NumSubPaths() > 1)
      iMenu->addAction(tr("Decompose path"), this, SLOT(DecomposePath()));
    else
      iMenu->addAction(tr("Scissors"), this, SLOT(UseScissors()));
  }
  // don't bother to provide editing of a single ellipse
  if (obj->NumSubPaths() > 1 ||
      obj->SubPath(0)->Type() != IpeSubPath::EEllipse)
    InsertEdit();
}

void AttributePopup::VisitMark(const IpeMark *obj)
{
  SetTitle(tr("Mark object"));
  InsertLayer(iIt->Layer());
  InsertColor(tr("Stroke"), obj->Stroke());
  InsertAttribute(tr("Size"), IpeAttribute::EMarkSize, obj->Size());
}

void AttributePopup::VisitText(const IpeText *obj)
{
  SetTitle(tr("Text object"));
  InsertLayer(iIt->Layer());
  InsertColor(tr("Stroke"), obj->Stroke());
  InsertAttribute(tr("Size"), IpeAttribute::ETextSize, obj->Size());

  QMenu *sub = new QMenu(iMenu);
  IpeAction *a = new IpeAction(1, tr("transformable"), this);
  sub->addAction(a);
  connect(a, SLOT(SetTransformable(int)));
  a = new IpeAction(1, tr("fixed"), this);
  sub->addAction(a);
  connect(a, SLOT(SetTransformable(int)));
  if (obj->IsTransformable())
    sub->setTitle(tr("Transformable"));
  else
    sub->setTitle(tr("Fixed"));
  iMenu->addMenu(sub);

  sub = new QMenu(iMenu);
  a = new IpeAction(IpeText::ELabel, tr("label"), this);
  sub->addAction(a);
  connect(a, SLOT(SetParagraph(int)));
  a = new IpeAction(IpeText::EMinipage, tr("minipage"), this);
  sub->addAction(a);
  connect(a, SLOT(SetParagraph(int)));
  a = new IpeAction(IpeText::ETextbox, tr("textbox"), this);
  sub->addAction(a);
  connect(a, SLOT(SetParagraph(int)));
  a = new IpeAction(IpeText::ETitle, tr("title"), this);
  sub->addAction(a);
  connect(a, SLOT(SetParagraph(int)));
  switch (obj->Type()) {
  case IpeText::ELabel:
    sub->setTitle(tr("Label"));
    break;
  case IpeText::EMinipage:
    sub->setTitle(tr("Minipage"));
    break;
  case IpeText::ETextbox:
    sub->setTitle(tr("Textbox"));
    break;
  case IpeText::ETitle:
    sub->setTitle(tr("Title"));
    break;
  }
  iMenu->addMenu(sub);

  sub = new QMenu(iMenu);
  a = new IpeAction(IpeText::ELeft, tr("Left"), this);
  sub->addAction(a);
  connect(a, SLOT(SetHAlign(int)));
  a = new IpeAction(IpeText::ERight, tr("Right"), this);
  sub->addAction(a);
  connect(a, SLOT(SetHAlign(int)));
  a = new IpeAction(IpeText::EHCenter, tr("Center"), this);
  sub->addAction(a);
  connect(a, SLOT(SetHAlign(int)));
  QString halign;
  switch (obj->HorizontalAlignment()) {
  case IpeText::ELeft:
    halign = QLatin1String("Left aligned");
    break;
  case IpeText::ERight:
    halign = QLatin1String("Right aligned");
    break;
  case IpeText::EHCenter:
    halign = QLatin1String("Horizontally centered");
    break;
  }
  sub->setTitle(halign);
  iMenu->addMenu(sub);

  sub = new QMenu(iMenu);
  a = new IpeAction(IpeText::ETop, tr("Top"), this);
  sub->addAction(a);
  connect(a, SLOT(SetVAlign(int)));
  a = new IpeAction(IpeText::EBottom, tr("Bottom"), this);
  sub->addAction(a);
  connect(a, SLOT(SetVAlign(int)));
  a = new IpeAction(IpeText::EVCenter, tr("Center"), this);
  sub->addAction(a);
  connect(a, SLOT(SetVAlign(int)));
  a = new IpeAction(IpeText::EBaseline, tr("Baseline"), this);
  sub->addAction(a);
  connect(a, SLOT(SetVAlign(int)));
  QString valign;
  switch (obj->VerticalAlignment()) {
  case IpeText::ETop:
    valign = QLatin1String("Top aligned");
    break;
  case IpeText::EBottom:
    valign = QLatin1String("Bottom aligned");
    break;
  case IpeText::EVCenter:
    valign = QLatin1String("Vertically centered");
    break;
  case IpeText::EBaseline:
    valign = QLatin1String("Aligned on baseline");
    break;
  }
  sub->setTitle(valign);
  iMenu->addMenu(sub);

  if (obj->IsMiniPage())
    iMenu->addAction(tr("Change width"), this, SLOT(ChangeWidth()));

  InsertEdit();
}

void AttributePopup::VisitImage(const IpeImage *obj)
{
  SetTitle(tr("Image object"));
  IpeBitmap bm = obj->Bitmap();
  SetTitle(tr("%1 x %2 pixels").arg(bm.Width()).arg(bm.Height()));
  SetTitle(tr("%1 components with %2 bits").arg(bm.Components())
	   .arg(bm.BitsPerComponent()));
  SetTitle(tr("Filter %1").arg(int(bm.Filter())));
  InsertLayer(iIt->Layer());
}

void AttributePopup::VisitReference(const IpeReference *obj)
{
  SetTitle(tr("Reference object"));
  iMenu->addAction(tr("Name: ") + String(obj->Name()));
  InsertLayer(iIt->Layer());
}

// --------------------------------------------------------------------

void AttributePopup::SetStroke(int id)
{
  iIt->Object()->SetStroke(iSyms[IpeAttribute::EColor][id]);
  iChanged = true;
}

void AttributePopup::SetFill(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    obj->SetFill(iSyms[IpeAttribute::EColor][id]);
    iChanged = true;
  }
}

void AttributePopup::SetLineWidth(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    obj->SetLineWidth(iSyms[IpeAttribute::ELineWidth][id]);
    iChanged = true;
  }
}

void AttributePopup::SetDashStyle(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    obj->SetDashStyle(iSyms[IpeAttribute::EDashStyle][id]);
    iChanged = true;
  }
}

void AttributePopup::SetTextSize(int id)
{
  IpeText *obj = iIt->Object()->AsText();
  if (obj) {
    obj->SetSize(iSyms[IpeAttribute::ETextSize][id]);
    iChanged = true;
  }
}

void AttributePopup::SetParagraph(int id)
{
  IpeText *obj = iIt->Object()->AsText();
  if (obj && obj->Type() != IpeText::TType(id)) {
    obj->SetType(IpeText::TType(id));
    iChanged = true;
  }
}

void AttributePopup::SetHAlign(int id)
{
  IpeText *obj = iIt->Object()->AsText();
  if (obj && obj->HorizontalAlignment() != id) {
    obj->SetHorizontalAlignment(IpeText::THorizontalAlignment(id));
    iChanged = true;
  }
}

void AttributePopup::SetVAlign(int id)
{
  IpeText *obj = iIt->Object()->AsText();
  if (obj && obj->VerticalAlignment() != id) {
    obj->SetVerticalAlignment(IpeText::TVerticalAlignment(id));
    iChanged = true;
  }
}

void AttributePopup::SetTransformable(int id)
{
  IpeText *obj = iIt->Object()->AsText();
  if (obj) {
    obj->SetTransformable(bool(id));
    iChanged = true;
  }
}

void AttributePopup::SetMarkShape(int id)
{
  IpeMark *obj = iIt->Object()->AsMark();
  if (obj) {
    obj->SetShape(id + 1);
    iChanged = true;
  }
}

void AttributePopup::SetMarkSize(int id)
{
  IpeMark *obj = iIt->Object()->AsMark();
  if (obj) {
    obj->SetSize(iSyms[IpeAttribute::EMarkSize][id]);
    iChanged = true;
  }
}

void AttributePopup::SetForwardArrow(int id)
{
  IpePath *obj = iIt->Object()->AsPath();
  if (obj) {
    if (id == 0)
      obj->SetForwardArrow(IpeAttribute());
    else
      obj->SetForwardArrow(iSyms[IpeAttribute::EArrowSize][id - 1]);
    iChanged = true;
  }
}

void AttributePopup::SetBackwardArrow(int id)
{
  IpePath *obj = iIt->Object()->AsPath();
  if (obj) {
    if (id == 0)
      obj->SetBackwardArrow(IpeAttribute());
    else
      obj->SetBackwardArrow(iSyms[IpeAttribute::EArrowSize][id - 1]);
    iChanged = true;
  }
}

void AttributePopup::SetCap(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    IpeStrokeStyle ss = obj->StrokeStyle();
    ss.SetCap(id);
    obj->SetStrokeStyle(ss);
    iChanged = true;
  }
}

void AttributePopup::SetJoin(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    IpeStrokeStyle ss = obj->StrokeStyle();
    ss.SetJoin(id);
    obj->SetStrokeStyle(ss);
    iChanged = true;
  }
}

void AttributePopup::SetWindRule(int id)
{
  IpeFillable *obj = iIt->Object()->AsFillable();
  if (obj) {
    IpeStrokeStyle ss = obj->StrokeStyle();
    ss.SetWindRule(id);
    obj->SetStrokeStyle(ss);
    iChanged = true;
  }
}

void AttributePopup::SetLayer(int id)
{
  iIt->SetLayer(id);
  iChanged = true;
}

// --------------------------------------------------------------------

void AttributePopup::Edit()
{
  if (iIt->Object()->AsText() &&
      IpeCreateText::Edit(iIt->Object()->AsText(), iStyleSheet)) {
    iServices->OvSvcAddUndoItem(iIt, iPage, iOriginal,
				tr("text object edit"));
  } else if (iIt->Object()->AsPath())
    iCanvas->SetOverlay(new IpeEditPath(iCanvas, iIt->Object()->AsPath(),
					iPage, iServices, iIt));
}

void AttributePopup::Ungroup()
{
  if (iIt->Object()->AsGroup()) {
    iServices->OvSvcAddUndoItem(new IpePage(*iPage), tr("ungrouping"));
    iPage->Ungroup(iLayer);
  }
}

void AttributePopup::DecomposePath()
{
  if (iIt->Object()->AsPath()) {
    iServices->OvSvcAddUndoItem(new IpePage(*iPage),
				     tr("path decomposition"));
    iPage->DecomposePath(iLayer);
  }
}

void AttributePopup::UseScissors()
{
  if (iIt->Object()->AsPath())
    iCanvas->SetOverlay(new IpeScissors(iCanvas, iIt->Object()->AsPath(),
					iPage, iIt));
}

void AttributePopup::ChangeWidth()
{
  if (iIt->Object()->AsText())
    iCanvas->SetOverlay(new IpeTextWidthChanger(iCanvas,
						iIt->Object()->AsText()));
}

// --------------------------------------------------------------------

AttributePopup::AttributePopup(IpePage::iterator it, IpePage *page,
			       IpeCanvas *canvas, int layer,
			       const IpeStyleSheet *sheet,
			       const IpeAttributeSeq *syms,
			       IpeOverlayServices *services)
  : iCanvas(canvas), iServices(services), iStyleSheet(sheet),
    iSyms(syms), iPage(page), iIt(it), iOriginal(*it),
    iLayer(layer)
{
  iMenu = new QMenu();
  iChanged = false;
  iLayerLocked = page->Layer(layer).IsLocked();
  it->Object()->Accept(*this);
}

void AttributePopup::Exec()
{
  iMenu->exec(QCursor::pos());
  if (iChanged) {
    iServices->OvSvcAddUndoItem(iIt, iPage, iOriginal,
				     tr("attribute change"));
    iPage->SetEdited(true);
    iCanvas->Update();
  }
}

// --------------------------------------------------------------------
