// --------------------------------------------------------------------
// MainWindow
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (c) 1993-2019 Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "mainwindow_qt.h"
#include "canvasview_qt.h"
#include "ipetool.h"
#include <clocale>

using namespace ipe;

// --------------------------------------------------------------------

MainWindow::
MainWindow(CanvasView* cv, Qt::WindowFlags f):
  QMainWindow(0, f), canvas_view(cv)
{
  setupUi(this);

  iDoc = nullptr;

  iViewMenu = menuBar()->addMenu(tr("&View"));
  iMoveMenu = menuBar()->addMenu(tr("&Move"));

  iViewMenu->addAction(new IpeAction(EFullScreen, "Full screen", "F11", this));
  iViewMenu->addAction(new IpeAction(EFitPage, "Fit page", ".", this));
  iViewMenu->addAction(new IpeAction(EZoomIn, "Zoom in", "=", this));
  iViewMenu->addAction(new IpeAction(EZoomOut, "Zoom out", "-", this));
  iViewMenu->addAction(new IpeAction(ESetTime, "Set time", "", this));
  iTimeCountdown = new IpeAction(ETimeCountdown, "Count down", "/", this);
  iTimeCountdown->setCheckable(true);
  iViewMenu->addAction(iTimeCountdown);
  IpeAction* iToggleTime = new IpeAction(EToggleTimeCounting, "Count time", "T", this);
  iToggleTime->setCheckable(true);
  iViewMenu->addAction(iToggleTime);
  iViewMenu->addAction(new IpeAction(EShowPresentation, "Show presentation", "", this));

  std::vector<std::string> next,prev;
  next.push_back("Right"); next.push_back("PgDown");
  prev.push_back("Left");  prev.push_back("PgUp");
  iMoveMenu->addAction(new IpeAction(ENextView, "Next view", next, this));
  iMoveMenu->addAction(new IpeAction(EPreviousView, "Previous view", prev, this));
  iMoveMenu->addAction(new IpeAction(ENextPage, "Next page", "N", this));
  iMoveMenu->addAction(new IpeAction(EPreviousPage, "Previous ", "P", this));
  iMoveMenu->addAction(new IpeAction(EFirstView, "First view", "Home", this));
  iMoveMenu->addAction(new IpeAction(ELastView, "Last view", "End", this));
  iMoveMenu->addAction(new IpeAction(EJumpTo, "Jump to...", "J", this));

  Canvas::Style style = currentViewCanvas->canvasStyle();
  style.pretty = true;
  currentViewCanvas->setCanvasStyle(style);
  style = nextViewCanvas->canvasStyle();
  style.pretty = true;
  nextViewCanvas->setCanvasStyle(style);

  iSnap.iSnap = 0; // Snap::ESnapGrid;
  iSnap.iGridVisible = false;
  iSnap.iGridSize = 8;
  iSnap.iAngleSize = M_PI / 6.0;
  iSnap.iSnapDistance = 10;
  iSnap.iWithAxes = false;
  iSnap.iOrigin = Vector::ZERO;
  iSnap.iDir = 0;
  currentViewCanvas->setSnap(iSnap);
  nextViewCanvas->setSnap(iSnap);

  connect(splitter, SIGNAL(splitterMoved(int,int)), this, SLOT(splitterMoved(int,int)));
  connect(splitter_2, SIGNAL(splitterMoved(int,int)), this, SLOT(splitterMoved(int,int)));

  currentViewCanvas->setFocus();
}

void MainWindow::cmd(int c)
{
  ipeDebug("Command %d", c);
  switch (c) {
  case EOpen:
    break;
  case EQuit:
    QApplication::exit();
    break;
  case EToggleTimeCounting:
    timeLabel->toggleCounting();
    break;
  case EFullScreen:
    canvas_view->setWindowState(canvas_view->windowState() ^ Qt::WindowFullScreen);
    break;
  case EFitPage:
    fitBox(iDoc->cascade()->findLayout()->paper(), currentViewCanvas);
    fitBox(iDoc->cascade()->findLayout()->paper(), nextViewCanvas);
    canvas_view->fitBox(iDoc->cascade()->findLayout()->paper());
    break;
  case ENextView:
    nextView(+1);
    break;
  case EPreviousView:
    nextView(-1);
    break;
  case ENextPage:
    nextPage(+1);
    break;
  case EPreviousPage:
    nextPage(-1);
    break;
  case EFirstView:
    firstView();
    break;
  case ELastView:
    lastView();
    break;
  case ETimeCountdown:
    timeLabel->toggleCountdown();
    break;
  case ESetTime:
    timeLabel->setTime();
    break;
  case EJumpTo: {
    int page = CanvasBase::selectPageOrView(iDoc, -1, iPageNo, 240, this->width(), this->height());
    if (page >= 0) {
      int view = CanvasBase::selectPageOrView(iDoc, page, (page == iPageNo) ? iViewNo : 0, 240, this->width(),this->height());
      if (view >= 0) {
	iPageNo = page;
	iViewNo = view;
	setView();
      }
    }
    break; }
  case EShowPresentation:
    canvas_view->show();
    break;
  default:
    // unknown action
    return;
  }
}

bool MainWindow::load(const char *fname)
{
  Document *doc = Document::loadWithErrorReport(fname);
  if (!doc)
    return false;

  iFileName = String(fname);
  doc->runLatex();

  delete iDoc;
  iDoc = doc;

  iPageNo = 0;
  iViewNo = 0;

  currentViewCanvas->setResources(iDoc->resources());
  nextViewCanvas->setResources(iDoc->resources());

  currentViewCanvas->setPage(iDoc->page(iPageNo), iPageNo, iViewNo, iDoc->cascade());
  currentViewCanvas->setPan(Vector(300, 400));
  static_cast<CanvasBase*>(currentViewCanvas)->update();
  setNextViewCanvas();

  canvas_view->load(doc, iPageNo, iViewNo);

  setView();
  return true;
}

void MainWindow::updateLabel()
{
  String s = iFileName;
  if (iFileName.rfind('/') >= 0)
    s = iFileName.substr(iFileName.rfind('/') + 1);
  if (iDoc->countTotalViews() > 1) {
    ipe::StringStream ss(s);
    ss << " (" << iPageNo + 1 << "-" << iViewNo + 1 << ")";
  }
  setWindowTitle(QIpe(s));
}

void MainWindow::fitBox(const Rect &box, Canvas* canvas)
{
  if (box.isEmpty())
    return;
  ipeDebug("canvas: %d x %d", canvas->width(), canvas->height());
  double xfactor = box.width() > 0.0  ? (canvas->width() / box.width()) : 20.0;
  double yfactor = box.height() > 0.0 ? (canvas->height() / box.height()) : 20.0;
  double zoom = (xfactor > yfactor) ? yfactor : xfactor;
  canvas->setPan(0.5 * (box.bottomLeft() + box.topRight()));
  canvas->setZoom(zoom);
  static_cast<CanvasBase*>(canvas)->update();
}

std::pair<int,int> MainWindow::nextPageView(int delta)
{
  const Page *page = iDoc->page(iPageNo);
  int iPageNo = this->iPageNo;
  int iViewNo = this->iViewNo;
  if (0 <= iViewNo + delta && iViewNo + delta < page->countViews()) {
    iViewNo += delta;
  } else if (0 <= iPageNo + delta && iPageNo + delta < iDoc->countPages()) {
    iPageNo += delta;
    if (delta > 0)
      iViewNo = 0;
    else
      iViewNo = iDoc->page(iPageNo)->countViews() - 1;
  }

  return std::make_pair(iPageNo, iViewNo);
}

void MainWindow::setNextViewCanvas()
{
  std::pair<int,int> page_view = nextPageView(1);
  int iPageNo = page_view.first;
  int iViewNo = page_view.second;
  nextViewCanvas->setPage(iDoc->page(iPageNo), iPageNo, iViewNo, iDoc->cascade());
  static_cast<CanvasBase*>(nextViewCanvas)->update();
}

void MainWindow::nextView(int delta)
{
  std::pair<int,int> page_view = nextPageView(delta);
  iPageNo = page_view.first;
  iViewNo = page_view.second;
  setView();
}

void MainWindow::nextPage(int delta)
{
  int newPageNo = iPageNo + delta;
  newPageNo = (newPageNo < 0 ? 0 : newPageNo);
  newPageNo = (newPageNo > iDoc->countPages() - 1 ? iDoc->countPages() - 1 : newPageNo);
  if (newPageNo != iPageNo) {
    iPageNo = newPageNo;
    iViewNo = 0;
    setView();
  }
}

void MainWindow::firstView()
{
  iPageNo = 0;
  iViewNo = 0;
  setView();
}

void MainWindow::lastView()
{
  iPageNo = iDoc->countPages() - 1;
  iViewNo = iDoc->page(iPageNo)->countViews() - 1;
  setView();
}

void MainWindow::resizeEvent(QResizeEvent *event)
{
  cmd(EFitPage);
}

void MainWindow::splitterMoved(int pos, int index)
{
  cmd(EFitPage);
}

void MainWindow::closeEvent(QCloseEvent *event)
{
  canvas_view->close();
  QMainWindow::closeEvent(event);
}

void MainWindow::setView()
{
  currentViewCanvas->setPage(iDoc->page(iPageNo), iPageNo, iViewNo, iDoc->cascade());
  static_cast<CanvasBase*>(currentViewCanvas)->update();
  updateLabel();

  canvas_view->setView(iPageNo, iViewNo);
  notesText->setPlainText(QIpe(iDoc->page(iPageNo)->notes()));

  setNextViewCanvas();
}

// --------------------------------------------------------------------

static void usage()
{
  fprintf(stderr, "Usage: ipepresenter <filename>\n");
  exit(1);
}

int main(int argc, char *argv[])
{
  Platform::initLib(IPELIB_VERSION);
  QApplication a(argc, argv);
  setlocale(LC_NUMERIC, "C");

  if (argc != 2)
    usage();

  const char *load = argv[1];

  CanvasView *cv = new CanvasView();
  MainWindow *mw = new MainWindow(cv);

  if (!mw->load(load))
    exit(2);

  cv->show();
  mw->show();

  mw->cmd(MainWindow::EFitPage);

  QObject::connect(&a, SIGNAL(lastWindowClosed()), &a, SLOT(quit()));
  return a.exec();
}

// --------------------------------------------------------------------
