/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdarg.h>

#ifdef _WIN32
 #include <sys/stat.h>
 #include <windows.h>

 #ifndef S_ISREG
 #define S_ISREG(mode)  (((mode) & S_IFMT) == S_IFREG)
 #endif

 #ifndef FTW_F
 #define FTW_F 1
 #endif
#else
 #include <ftw.h>
#endif

/* From id3lib */
#include <id3.h>
/* From libogg/libvorbis */
#include <vorbis/codec.h>
#include <vorbis/vorbisfile.h>
/* From EasyTag */
#include "vcedit.h"

/* Consts and variables for Files and Dirs */
char start_dir[255];
int  be_verbose ;
int  add_empty ;

/* Consts and variables for writing the DB */
const char* DB_file = "iRivNavi.iDB";
FILE* DB_handle;
const int	FAT_start = 0x0080,
			head_start = 0x0040,
			tags_start = 0xA0C0;
int number_of_tags;
int last_tag_offset;

int deslash ( char * string )
{
	int count;
	
	for ( count=0 ; count < strlen ( string ) ; count ++)
	{
		if (string[count] == '/')
		{
				string[count] = '\\';
		}
	}
	return 0;
}


void str_ntoup ( char* str , int limit )
{
	char* this_char;
	
	this_char = str ;
	
	while ( (*this_char != 0x00) && ( this_char - str < limit )  )
	{
		*this_char = toupper ( *this_char );
		this_char++;
	}	
}


void get_lso16 ( char bytes[2], int number )
{
	bytes[0] = (unsigned char) number;
	number >>= 8 ;
	bytes[1] = (unsigned char) number;
}


void get_lso32 ( char bytes[4], int number)
{
	bytes[0] = (unsigned char) number;
	number >>= 8 ;
	bytes[1] = (unsigned char) number;
	number >>= 8 ;
	bytes[2] = (unsigned char) number;
	number >>= 8 ;
	bytes[3] = (unsigned char) number;
}


int startup_DB ( void )
{
	int byte_count = 0 ;
	char full_name[255];
	
	number_of_tags = 0;

	strcpy ( full_name , start_dir );
	strcat ( full_name , "/" );
	strcat ( full_name , DB_file );
	DB_handle = fopen ( full_name , "wb" );

	/* Check file was created */
	if (DB_handle == 0x00)
	{
		printf("Could not create DB file - check that the supplied path exists and is writeable\n");
		exit(1);
	}
	
	/* Ok, first we write a clean Header and FAT*/
	for ( byte_count = 0 ; byte_count < tags_start ; byte_count++ )
	{
		fputc ( 0 , DB_handle );
	}
	
	/* now we'll write some marks */
	fseek ( DB_handle , 0x0000 , SEEK_SET );
	fputs ( "iRivDB Ver 0.12" , DB_handle );	
	fseek ( DB_handle , 0x0020 , SEEK_SET );
	fputs ( "iRiver iHP-100 DB File" , DB_handle );	
	fseek ( DB_handle , 0xA080 , SEEK_SET );
	fputs ( "Designed by iRiver" , DB_handle );

	return 0;	
}


int update_DB ( char *filename,
				char *songtitle,
				char *artist,
				char *album,
				char *genre )
{
	int fn_lng,st_lng,at_lng,al_lng,gn_lng;
	int previous_tag_offset;
	char 	new_filename[255] = "";
	char lso16[2];
	char lso32[4];
	
	previous_tag_offset = last_tag_offset;
	
	strcat ( new_filename , filename + strlen ( start_dir ) );
	deslash ( new_filename );
	
	fn_lng = strlen ( new_filename ) +1 ;
	st_lng = strlen ( songtitle ) +1 ;
	at_lng = strlen ( artist ) + 1;
	al_lng = strlen ( album ) + 1 ;
	gn_lng = strlen ( genre ) + 1 ;
	
	
	fseek ( DB_handle , tags_start + last_tag_offset , SEEK_SET );
	get_lso16 ( lso16 , fn_lng ) ;
	fputc ( lso16[0] , DB_handle ) ;
	fputc ( lso16[1] , DB_handle ) ;
	get_lso16 ( lso16 , st_lng ) ;
	fputc ( lso16[0] , DB_handle ) ;
	fputc ( lso16[1] , DB_handle ) ;
	get_lso16 ( lso16 , at_lng ) ;
	fputc ( lso16[0] , DB_handle ) ;
	fputc ( lso16[1] , DB_handle ) ;
	get_lso16 ( lso16 , al_lng ) ;
	fputc ( lso16[0] , DB_handle ) ;
	fputc ( lso16[1] , DB_handle ) ;
	get_lso16 ( lso16 , gn_lng ) ;
	fputc ( lso16[0] , DB_handle ) ;
	fputc ( lso16[1] , DB_handle ) ;
	last_tag_offset += 10;
	
	fseek ( DB_handle , tags_start + last_tag_offset , SEEK_SET );
	fputs ( new_filename , DB_handle );
	fputc ( 0 , DB_handle );
	fputs ( songtitle , DB_handle );
	fputc ( 0 , DB_handle );
	fputs ( artist , DB_handle );
	fputc ( 0 , DB_handle );
	fputs ( album , DB_handle );
	fputc ( 0 , DB_handle );
	fputs ( genre , DB_handle );
	fputc ( 0 , DB_handle );
	last_tag_offset += fn_lng + st_lng + at_lng + al_lng + gn_lng ;

	fseek ( DB_handle , FAT_start + 4 * number_of_tags , SEEK_SET ) ;
	get_lso32 ( lso32 , previous_tag_offset ) ;
	fputc ( lso32[0] , DB_handle ) ;
	fputc ( lso32[1] , DB_handle ) ;
	fputc ( lso32[2] , DB_handle ) ;
	fputc ( lso32[3] , DB_handle ) ;


	number_of_tags++;
	fseek ( DB_handle , head_start , SEEK_SET ) ;
	get_lso32 ( lso32 , number_of_tags ) ;
	fputc ( lso32[0] , DB_handle ) ;
	fputc ( lso32[1] , DB_handle ) ;
	fputc ( lso32[2] , DB_handle ) ;
	fputc ( lso32[3] , DB_handle ) ;
	
	return 0;
}


int close_DB ( void )
{
	return fclose ( DB_handle );
}


int process_MP3 ( char* filename )

{
	ID3Tag* this_tag;
	ID3Frame* a_frame;
	ID3Field* a_field;
	char f_album[255],f_artist[255],f_songtitle[255],f_genre[255];
	int genre;

	if (be_verbose) printf ( "Processing file: %s\n" , filename );
	
	this_tag = ID3Tag_New ();
	ID3Tag_Link ( this_tag , filename ) ;

	/* TITLE */
	a_frame = ID3Tag_FindFrameWithID ( this_tag , ID3FID_TITLE );
	if ( !a_frame )
	{
		if (be_verbose) printf ( "No title frame found\n" ) ;
		if (!add_empty) {
			if (be_verbose) printf ( "File not added to DB.\n" ) ;
			return 1;
		}
		strcpy ( f_songtitle , "No title" ) ;
	}
	else
	{
		a_field = ID3Frame_GetField ( a_frame , ID3FN_TEXT ) ;
		ID3Field_GetASCII ( a_field , f_songtitle , 255 ) ;
	}
	if (be_verbose) printf ( "Songtitle: %s\n" , f_songtitle ) ;
	
	/* ARTIST */
	a_frame = ID3Tag_FindFrameWithID ( this_tag , ID3FID_LEADARTIST );
	if ( !a_frame )
	{
		if (be_verbose) printf ( "No artist frame found\n" ) ;
		strcpy ( f_artist , "Unknown" ) ;
	}
	else
	{
		a_field = ID3Frame_GetField ( a_frame , ID3FN_TEXT ) ;
		ID3Field_GetASCII ( a_field , f_artist , 255 ) ;
	}
	if (be_verbose) printf ( "Artist: %s\n" , f_artist ) ;
	
	/* ALBUM */
	a_frame = ID3Tag_FindFrameWithID ( this_tag , ID3FID_ALBUM );
	if ( !a_frame )
	{
		if (be_verbose) printf ( "No album frame found\n" ) ;
		strcpy ( f_album , "Unknown" ) ;
	}
	else
	{
		a_field = ID3Frame_GetField ( a_frame , ID3FN_TEXT ) ;
		ID3Field_GetASCII ( a_field , f_album , 255 ) ;
	}
	if (be_verbose) printf ( "Album: %s\n" , f_album ) ;
	
	/* GENRE */
	a_frame = ID3Tag_FindFrameWithID ( this_tag , ID3FID_CONTENTTYPE );
	if ( !a_frame )
	{
		if (be_verbose) printf ( "No genre frame found\n" ) ;
		strcpy ( f_genre , "Unknown" ) ;
	}
	else
	{
		a_field = ID3Frame_GetField ( a_frame , ID3FN_TEXT ) ;
		genre = ID3Field_GetINT ( a_field ) ;
		strcpy ( f_genre , ID3_v1_genre_description[genre] );
	}
	if (be_verbose) printf ( "Genre: %s\n" , f_genre ) ;
	
	update_DB ( filename ,
				f_songtitle ,
				f_artist   ,
				f_album    ,
				f_genre ) ;
	return 0;
}


int process_Ogg ( char* filename )
/* I'm lazy, structure and some code from easy tag */
{
	char *f_album,*f_artist,*f_songtitle,*f_genre;
	FILE*	OGG_FILE ;
    vcedit_state   *state;
    vorbis_comment *vc;

	
	if (be_verbose) printf ( "Processing file: %s\n" , filename ) ;

	OGG_FILE = fopen(filename,"rb") ;
    state = vcedit_new_state();    // Allocate memory for 'state'
    if ( vcedit_open ( state , OGG_FILE ) < 0 )
    {
        printf ( "ERROR: Failed to open file: '%s' as vorbis.\n" , filename );
        fclose ( OGG_FILE ) ;
        return 1;
    }

    /* Get data from tag */
    vc = vcedit_comments ( state ) ;

	/* TITLE */
	f_songtitle = vorbis_comment_query( vc , "title" , 0 ) ;
	if ( ( f_songtitle == NULL ) || ( *f_songtitle == 0x00 ) )
	{
		if (!add_empty) {
			if (be_verbose) printf ( "File not added to DB.\n" ) ;
			return 1;
		}
		f_songtitle = "No title";
	}
	if (be_verbose) printf ( "Songtitle: %s\n" , f_songtitle ) ;

	/* ARTIST */
	f_artist    = vorbis_comment_query( vc , "artist" , 0 ) ;
	if ( ( f_artist == NULL ) || ( *f_artist == 0x00 ) )
	{
		f_artist = "Unknown";
	}
	if (be_verbose) printf ( "Artist: %s\n" , f_artist ) ;
	
	/* ALBUM */
	f_album     = vorbis_comment_query( vc , "album" , 0 ) ;
	if ( ( f_album == NULL ) || ( *f_album == 0x00 ) )
	{
		f_album = "Unknown";
	}
	if (be_verbose) printf ( "Album: %s\n" , f_album ) ;
	
	/* GENRE */
	f_genre     = vorbis_comment_query( vc , "genre" , 0 ) ;
	if ( ( f_genre == NULL ) || ( *f_genre == 0x00 ) )
	{
		f_genre = "Unknown";
	}
	if (be_verbose) printf ( "Genre: %s\n" , f_genre ) ;
	
	fclose ( OGG_FILE );
	
	update_DB ( filename ,
				f_songtitle ,
				f_artist   ,
				f_album    ,
				f_genre ) ;
	return 0;
}


int process_file( char* filename, const struct stat* filestat, int file_type)
{
	char ext[4];

	if( (file_type == FTW_F) && S_ISREG(filestat->st_mode) )
	{
		strncpy ( ext , filename + strlen(filename) - 4 , 4 );
		str_ntoup ( ext , 4 );
		if ( !strncmp ( ext , ".MP3" , 4 ) )
		{
			process_MP3 ( filename ) ;
		}
		else if ( !strncmp ( ext , ".OGG" , 4 ) )
		{
			process_Ogg ( filename ) ;
		}
		else
		{
			if (be_verbose)
				printf ( "Warning: File [%s]\nFile format not supported.\n", filename );
		}
	}
	return 0;
}



#ifdef _WIN32

typedef void (*execfn)(char*, struct stat*, int file_type);

void win32_recurse_dir(char* path, execfn before)
{
    /* In depth traversal of the subdir tree */
  WIN32_FIND_DATA find_file_data;
  HANDLE hnd;

  struct stat stat_buf;	/* We have to have one local because of after */

  int path_len = strlen(path);

  /* current node */
  if (stat(path, &stat_buf))
    perror(path);

  /* execute before for the current node */
  if (before)
  {
    (*before)(path, &stat_buf, FTW_F);
  }

  /* if it is a directory, recurse through all sons */
  if ((stat_buf.st_mode & S_IFMT) == S_IFDIR)
  {
    strcat(path, "/*");
    hnd = FindFirstFile(path, &find_file_data);

    while (hnd != INVALID_HANDLE_VALUE && FindNextFile(hnd, &find_file_data) != FALSE)
	{ 
      if(!strcmp(find_file_data.cFileName, ".") ||
		 !strcmp(find_file_data.cFileName, "..")) continue;

      path[path_len+1] = '\0';
      strcat(path, find_file_data.cFileName);
      win32_recurse_dir(path, before);
    }
    path[path_len] = '\0';
    FindClose(hnd);
  }
}

#endif


int recurse_dir( char* dirname )
{
#ifdef _WIN32
	char temp_buffer[sizeof(start_dir)];
	strcpy(temp_buffer, dirname);
	win32_recurse_dir(temp_buffer, &process_file);
#else
	ftw( dirname, (__ftw_func_t) &process_file, 30 );
#endif
	return 0;
}


void print_help ( void )
{
	printf ( "iRipDB v0.1 - Create DB file for iRiver iHP players\n" ) ;
	printf ( "Usage: iripdb [-vhe] path\n\n" ) ;
	printf ( "\t-v\tverbose operation\n" ) ;
	printf ( "\t-h\tprint this help information\n" ) ;
	printf ( "\t-e\tadd also files not tagged (with generic tags)\n" ) ;
	printf ( "\tpath\tpath to the iHP's mount point\n" ) ;
}


void print_usage ( void )
{
	printf ( "Usage: iripdb [-vh] path_to_ihp\n" );
}


int main ( int argc, char **argv )

{
	int argpnt,charpnt;
	
	*start_dir = 0x00 ;
	/* By default it will be verbose-less */
	be_verbose = 0 ;
	/* By default it won't add empty files */
	add_empty = 0 ;
	
	/* Process arguments */
	for ( argpnt = 1 ; argpnt < argc ; argpnt++ )
	{
		if ( *argv[argpnt] == '-' )
		{
			for ( charpnt = 1 ; charpnt < strlen ( argv[argpnt] ) ; charpnt++ )
			{
				switch ( argv[argpnt][charpnt] )
				{
					case  'v':
						be_verbose = 1 ;
						break;
					case  'e':
						add_empty = 1 ;
						break;
					case  'h':
						print_help () ;
						return 0 ;
						break;
					default:
						printf ( "error: invalid argument -%c.\n" ,
							argv[argpnt][charpnt] ) ;
						print_usage () ;
						return 1 ;
						break;
				}

			}
		}
		else
		{
			strcpy ( start_dir , argv[argpnt] );
		}
	}
	
	if ( *start_dir != 0x00 )
	{
		if ( start_dir[strlen(start_dir)-1] == '/' ) {
			start_dir[strlen(start_dir)-1] = 0x00 ;
		}
		if (be_verbose) printf( "Building up the iRivDB in current dir.\n" );
		if (be_verbose) printf( "Writing header information.\n");
		startup_DB ();
		if (be_verbose) printf( "Start directory recursion.\n");
		recurse_dir ( start_dir );
		if (be_verbose) printf( "Closing database file.\n");
		close_DB ();
		printf ( " %i file(s) added at %s/%s\n" , number_of_tags ,
					start_dir , DB_file );
		return (0);
	}
	else
	{
		printf ( "error: no path provided\n" );
		print_usage () ;
		return 1;
	}
}
