/*******************************************************\
* irmp3-ncurses - An ncurses frontend for irmp3 using   *
* the Network Control Module                            *
* (C) 2003 Ross Axe                                     *
*                                                       *
* keyboard.c - keyboard and mouse handling              *
\*******************************************************/

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#include <signal.h>

#include "irmp3-ncurses.h"

vcid("$Id: keyboard.c,v 1.42 2006/01/13 06:09:53 ross Exp $");

static void direct_command(int sock)
{
    char buf[1024];

    sbar_getstr(buf, sizeof buf, ": ");
    if(buf[0])
	netputs(buf, sock);
    else
	sbar_printf(_("Cancelled"));
}

/* wrapper around getch() that is immune to signals put allows processing of
   network data and scrolling of titles while it waits */
int get_keystroke(void)
{
    int ret;

    unblock_all();
    do {
	ret = getch();
#ifdef SIGQUIT
	if(ret == 28) {		/* ctrl-\ */
	    endwin();
	    raise(SIGQUIT);
	}
#endif
    } while(ret == ERR && errno == EINTR);
    block_all();

    return ret;
}

#if USE_MOUSE
/*
 * statewin_mouse
 * Handles clicks on the state window
 */
static bool statewin_mouse(MEVENT *me, int sock)
{
    int r, c;
    int h, w;
    int row, col;

    getbegyx(statewin, r, c);
    getmaxyx(statewin, h, w);
    row = me->y - r;
    col = me->x - c;
    switch(row) {
    case BUTTON_ROW:
	if(me->bstate & BUTTON1_CLICKED) {
	    if((me->x - BUTTON_LEFT) % 3 < 2) {
		switch((me->x - BUTTON_LEFT) / 3) {
		case 0:		/* prev */
		    netputs("playlist jump -1", sock);
		    break;
		case 1:		/* rew */
		    netputs("seek -5", sock);
		    break;
		case 2:		/* play */
		    netputs("playlist jump +0", sock);
		    break;
		case 3:		/* pause */
		    netputs("pause", sock);
		    break;
		case 4:		/* stop */
		    netputs("stop", sock);
		    break;
		case 5:		/* ff */
		    netputs("seek +5", sock);
		    break;
		case 6:		/* next */
		    netputs("playlist jump +1", sock);
		    break;
		}
	    }
	}
	break;

    case TIME_ROW:
	if(me->bstate & BUTTON1_CLICKED) {
	    if(col == timeleft)
		netputs("seek -5", sock);
	    else if(col == w - 1)
		netputs("seek +5", sock);
	    else if(col > timeleft && col < w - 1) {
		if(songinfo.length >= 0)
		    netprintf(sock, "seek %d", songinfo.length *
			      (col - timeleft - 1) / (w - timeleft - 2));
	    }
	}
	break;

    case SHUFFLE_REPEAT_ROW:
	if(me->bstate & BUTTON1_CLICKED) {
	    if(col < repeatleft)
		netputs("playlist shuffle", sock);
	    else
		netputs("playlist repeat", sock);
	}
	break;

    case VOLUME_ROW:
	if(me->bstate & BUTTON1_CLICKED) {
	    if(col == volumeleft)
		netputs("mixer volume -1", sock);
	    else if(col == w - 1)
		netputs("mixer volume +1", sock);
	    else if(col > volumeleft && col < w - 1)
		netprintf(sock, "mixer volume %d",
			  100 * (col - volumeleft - 1) / (w - volumeleft - 3));
	}
	break;

    case BALANCE_ROW:
	if(me->bstate & BUTTON1_CLICKED) {
	    if(col == balanceleft)
		netputs("mixer balance -1", sock);
	    else if(col == w - 1)
		netputs("mixer balance +1", sock);
	    else if(col > balanceleft && col < w - 1) {
		netputs("mixer balance 0", sock);	/* hack! */
		netprintf(sock, "mixer balance %d",
			  100 * (col - balanceleft - 1) /
			  (w - balanceleft - 3) - 50);
	    }
	}
	break;

    case BASS_ROW:
	if(me->bstate & BUTTON1_CLICKED) {
	    if(col == bassleft)
		netputs("mixer bass -1", sock);
	    else if(col == w - 1)
		netputs("mixer bass +1", sock);
	    else if(col > bassleft && col < w - 1)
		netprintf(sock, "mixer bass %d",
			  100 * (col - bassleft - 1) / (w - bassleft - 3));
	}
	break;

    case TREBLE_ROW:
	if(me->bstate & BUTTON1_CLICKED) {
	    if(col == trebleleft)
		netputs("mixer treble -1", sock);
	    else if(col == w - 1)
		netputs("mixer treble +1", sock);
	    else if(col > trebleleft && col < w - 1)
		netprintf(sock, "mixer treble %d",
			  100 * (col - trebleleft - 1) / (w - trebleleft - 3));
	}
	break;
    }
    return true;
}

/*
 * handle_mouse
 * Handles mouse input
 */
static bool handle_mouse(int sock)
{
    MEVENT me;
    int h, w;

    getmaxyx(stdscr, h, w);

    if(OK == getmouse(&me)) {
	dbg_printf(2, "Mouse click at (%d,%d)", me.x, me.y);
	if(show_playlist && wenclose(plouterwin, me.y, me.x))
	    return playlist_mouse(&me, sock);
	else if(wenclose(statewin, me.y, me.x))
	    return statewin_mouse(&me, sock);
	else if(wenclose(debugwin, me.y, me.x))
	    direct_command(sock);
    } else
	sbar_printf("Error getting mouse event");
    return true;
}
#endif

/*
 * key_callback
 * Handles keyboard input
 * Returns false to exit program
 */
bool key_callback(int sock, int chr)
{
#define NUMLOCK_WARNING() sbar_printf(_("Please turn numlock on"))

    if(brwnd && browser_key_callback(sock, chr))
	return true;
    if(show_playlist && playlist_key_callback(sock, chr))
	return true;

    switch(chr) {
    case '?':
	showhelp();
	break;
    case 'a':
	about();
	break;
    case 'q':
    case 3:			/* ctrl+C */
    case 4:			/* ctrl+D */
	netputs("exit", sock);
	return false;
    case KEY_END:
	NUMLOCK_WARNING();
    case '1':
	netputs("playlist jump -1", sock);
	break;
#ifdef KEY_IC
    case KEY_IC:
	NUMLOCK_WARNING();
#endif
    case '0':
	netputs("playlist jump +0", sock);
	break;
    case KEY_B2:
	NUMLOCK_WARNING();
    case '5':
	netputs("pause", sock);
	break;
#ifdef KEY_DC
    case KEY_DC:
	NUMLOCK_WARNING();
#endif
    case '.':
	netputs("stop", sock);
	break;
    case '3':
	netputs("playlist jump +1", sock);
	break;
    case '9':
	netputs("playlist repeat", sock);
	break;
    case KEY_HOME:
	NUMLOCK_WARNING();
    case '7':
	netputs("playlist shuffle", sock);
	break;
    case KEY_RIGHT:
    case '6':
	netputs("seek +5", sock);
	break;
    case KEY_LEFT:
    case '4':
	netputs("seek -5", sock);
	break;
    case '=':
    case '+':
	netputs("mixer volume +1", sock);
	break;
    case '-':
	netputs("mixer volume -1", sock);
	break;
    case 'm':
	netputs("mixer mute", sock);
	break;
    case ']':
	netputs("mixer bass +1", sock);
	break;
    case '[':
	netputs("mixer bass -1", sock);
	break;
    case '#':
	netputs("mixer treble +1", sock);
	break;
    case '\'':
	netputs("mixer treble -1", sock);
	break;
    case '<':
    case '\\':
	netputs("mixer balance -1", sock);
	break;
    case '>':
    case '/':
	netputs("mixer balance +1", sock);
	break;
    case 'c':
	netputs("playlist clearnostop", sock);
	break;
#if PLAYLIST && HAVE_RESIZE_WINDOW
    case 'p':
	show_playlist = !show_playlist;
	if(show_playlist) {
	    int h, w;

	    getmaxyx(stdscr, h, w);
	    wclear(statewin);
	    wrefresh(statewin);
	    statewin = resize_window(statewin, STATE_HEIGHT,
				     PLAYLIST_LEFT - 1);
	    plwin_create(sock);
	} else {
	    plwin_destroy();
	    statewin = resize_window(statewin, STATE_HEIGHT, getmaxx(stdscr));
	}
	redraw_all();
	break;
#endif		/* PLAYLIST && HAVE_RESIZE_WINDOW */
    case 't':
	netputs("playlist sort", sock);
	break;
    case 'l':
    case ':':
	direct_command(sock);
	break;
    case 'b':
	browser_init(sock);
	break;
#if TERM_RESIZE
    case KEY_RESIZE:		/* terminal's just been resized */
	resize_terminal();
	break;
#endif
    case 'd':
	netputs("cd load", sock);
	break;
    case 'e':
	netputs("cd eject", sock);
	break;
    case 'j':
	netputs("cd eject2", sock);
	break;
#if USE_MOUSE
    case KEY_MOUSE:
	handle_mouse(sock);
	break;
#endif
    case 12:			/* ctrl+L */
	redraw_all();
	break;
#ifdef SIGTSTP
    case 26:			/* ctrl+Z */
	raise(SIGTSTP);
	break;
#endif
    case ERR:
	break;
    default:
	if(chr > 31 && chr < 256)
	    sbar_printf(_("Unknown key %c (code %d)"), chr, chr);
	else
	    sbar_printf(_("Unknown key code %d"), chr);
	break;
    }

    return true;
}
