#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>


#include "prototypes.h"

/*
   Copyright (C) 2003 Red Hat, Inc. All rights reserved.
                                                                                                             
   Usage and distribution of this file are subject to the Open Software License version 1.1
   that can be found at http://www.opensource.org/licenses/osl-1.1.txt and the COPYING file as
   distributed together with this file is included herein by reference.
                                                                                                                      
   Author: Arjan van de Ven   <arjanv@redhat.com>

*/

/*
 * This file contains the IRQ balancing policy code, eg assign the actual cpu to the irq's
 */


/*
 * cost function: the cost of the already assigned interrupts to a certain cpu 
 *
 * TODO: add irq number and a way to add physical layout for NUMA type optimisations
 *
 */
 
long native_costs[MAX_CPU];

static int irqcost(unsigned int cpu, unsigned int prevcpu, int policy)
{
	int total = 0;

	total = native_costs[cpu];
	if (cpu != cpubrother[cpu])
		total += native_costs[cpubrother[cpu]]/2;
		
	/* 
	 * give 100 credits bonus for staying on the same cpu. 100 is pretty arbitrary, 
	 * it's a small value but large enough to provide a limit affinity.
	 * Hyperthreading siblings get half the bonus in order to promote migration within
	 * the package.
	 *
	 * if the policy for this class is ROTATE, give bonus to the NEXT cpu instead
	 */
	if (policy != POLICY_ROTATE) {
		if (cpu == prevcpu)
			total -= 100;
		else if (cpu == cpubrother[prevcpu])
			total -= 50;
	} else {
		int distance;
		distance = cpu - prevcpu;
		while (distance <= 0)
			distance += cpucount;
			
		if (cpu != prevcpu) 
			total -= rotate_premium / distance;
	}
	return total;
}


void balance_policy(void)
{
	int maxpertype, totaltype;
	int typeassigned[MAX_CPU];
	int i;
	int type;
	
	struct irqdescriptor* lists[IRQ_GIGE+1], *current;
	
	
	memset(lists, 0, sizeof(lists));
	memset(native_costs, 0, sizeof(native_costs));
	
	/* first, create the per type lists of interrupts that need to be assigned*/
	for (i = 0; i < MAX_INTERRUPTS-1; i++) {
		struct irqdescriptor **prev, *current;
		
		if (interrupts[i].type >= 0) {
			prev = &lists[interrupts[i].type];
			current = lists[interrupts[i].type];
			interrupts[i].next = NULL;
			
			/* insertion sort, descending on count */
			do {
				if (!current) {
					/* end of list, so we just append */
					*prev = &interrupts[i];
					interrupts[i].next = NULL;
					break;
				}
				
				if (current->delta < interrupts[i].delta) {
					/* insert here */
					interrupts[i].next = current;
					*prev = &interrupts[i];
					break;
				}
				
				prev = &current->next;
				current=current->next;
			} while (1);
		}
		
	}
	
	/* then iterate over the interrupt classes for assigning */
	for (type = IRQ_GIGE; type >=0; type--) {
		int policy;
		
	
		/* first: find out how many interrupts have this type */
		totaltype = 0;
		for (i=0; i < MAX_INTERRUPTS; i++) {
			if (interrupts[i].type == type)
				totaltype++;
		}
		maxpertype = totaltype / cpucount + (totaltype%cpucount? 1 : 0);
		/* don't allow more than maxpertype irq's assigned to one cpu for this type */
		
		memset(typeassigned, 0, sizeof(typeassigned));
		
		current = lists[type];
		policy = class_policy[type];
		while (current!=NULL) {
			int cheapest = MAX_CPU;
			int cheapestcost = INT_MAX;
			int cost;
			if (policy == POLICY_IGNORE)
				cheapest = get_current_irqcpu(current->number);
			else for (i = 0; i < cpucount; i++) {
				cost = irqcost(i, current->oldcpu, policy);
				if ((cost < cheapestcost) && (typeassigned[i] < maxpertype))
				{
					cheapestcost = cost;
					cheapest = i;
				}
			}
			
			/* assign the irq */
			current->cpu = cheapest;
			native_costs[cheapest] += current->delta;
			typeassigned[cheapest]++;
			
			current = current->next;
		}
	}		
		
}
