--------------------------------------------------------------------------------
-- Project:     MicroBlaze on TE-XC2Se
-- Name:        te_mem_opb.vhd
-- Description: memory controller peripheral, OPB interface
--------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

library unisim;
use unisim.vcomponents.all; -- use FDR, FDRE

library Common_v1_00_a;
use Common_v1_00_a.all;     -- use pselect

entity te_mem is
	generic (
		C_OPB_AWIDTH : integer := 32;
		C_OPB_DWIDTH : integer := 32;
		C_BASEADDR   : std_logic_vector(0 to 31) := x"0080_0000";
		C_HIGHADDR   : std_logic_vector(0 to 31) := x"00FF_FFFF");
	port (
		-- Global signals
		OPB_Clk      : in    std_logic;
		OPB_Rst      : in    std_logic;
		Interrupt    : out   std_logic;
		-- OPB signals
		OPB_ABus     : in    std_logic_vector(0 to C_OPB_AWIDTH-1);
		OPB_BE       : in    std_logic_vector(0 to C_OPB_DWIDTH/8-1);
		OPB_DBus     : in    std_logic_vector(0 to C_OPB_DWIDTH-1);
		OPB_RNW      : in    std_logic;
		OPB_select   : in    std_logic;
		OPB_seqAddr  : in    std_logic;
		MEM_DBus     : out   std_logic_vector(0 to C_OPB_DWIDTH-1);
		MEM_errAck   : out   std_logic;
		MEM_retry    : out   std_logic;
		MEM_toutSup  : out   std_logic;
		MEM_xferAck  : out   std_logic;
		-- external memory bus
		mem_a        : out   std_logic_vector(21 downto 1); -- mem: address bus
		mem_d_i      : in    std_logic_vector(15 downto 0); -- mem: data bus
		mem_d_o      : out   std_logic_vector(15 downto 0); --      ...
		mem_d_t      : out   std_logic;                     --      ...
		mem_wen      : out   std_logic;                     -- mem: we#
		flash_cen    : out   std_logic;                     -- flash: ce#
		flash_oen    : out   std_logic;                     -- flash: oe#
		flash_rdy    : in    std_logic;                     -- flash: ready
		flash_8n     : out   std_logic;                     -- flash: byte#
		ram_csn      : out   std_logic;                     -- ram: cs#
		ram_oen      : out   std_logic;                     -- ram: oe#
		ram_blen     : out   std_logic;                     -- ram: ble#
		ram_bhen     : out   std_logic;                     -- ram: bhe#
		cpld_csn     : out   std_logic;                     -- cpld: cs#
		cpld_rwn     : out   std_logic);                    -- cpld: r/w#
end te_mem;

--------------------------------------------------------------------------------
architecture bhv of te_mem is
	component te_mem_core is
		generic (
			C_OPB_AWIDTH : integer := 32;
			C_OPB_DWIDTH : integer := 32;
			C_BASEADDR   : std_logic_vector(0 to 31) := x"0080_0000";
			C_HIGHADDR   : std_logic_vector(0 to 31) := x"00FF_FFFF"
			);
		port (
			-- Global signals
			OPB_Clk      : in    std_logic;
			OPB_Rst      : in    std_logic;
			-- OPB signals (mostly registered)
			OPB_ABus     : in    std_logic_vector(0 to C_OPB_AWIDTH-1);
			OPB_BE       : in    std_logic_vector(0 to C_OPB_DWIDTH/8-1);
			OPB_DBus     : in    std_logic_vector(0 to C_OPB_DWIDTH-1);
			OPB_RNW      : in    std_logic;
			-- simplified interface to OPB
			my_Select    : in    std_logic;
			my_Ack       : out   std_logic;
			my_DBus      : out   std_logic_vector(0 to C_OPB_DWIDTH-1);
			-- external memory bus
			mem_a        : out   std_logic_vector(21 downto 1); -- mem: address bus
			mem_d_i      : in    std_logic_vector(15 downto 0); -- mem: data bus
			mem_d_o      : out   std_logic_vector(15 downto 0); --      ...
			mem_d_t      : out   std_logic;                     --      ...
			mem_wen      : out   std_logic;                     -- mem: we#
			flash_cen    : out   std_logic;                     -- flash: ce#
			flash_oen    : out   std_logic;                     -- flash: oe#
			flash_rdy    : in    std_logic;                     -- flash: ready
			flash_8n     : out   std_logic;                     -- flash: byte#
			ram_csn      : out   std_logic;                     -- ram: cs#
			ram_oen      : out   std_logic;                     -- ram: oe#
			ram_blen     : out   std_logic;                     -- ram: ble#
			ram_bhen     : out   std_logic;                     -- ram: bhe#
			cpld_csn     : out   std_logic;                     -- cpld: cs#
			cpld_rwn     : out   std_logic);                    -- cpld: r/w#
	end component;
	
	-- internal signals
	signal ii_xfer_Ack:   std_logic;
	signal ii_CS_1:       std_logic;
	-- signals to interface with user function
	signal OPB_ABus_reg:  std_logic_vector(OPB_ABus'range);
	signal OPB_DBus_reg:  std_logic_vector(OPB_DBus'range);
	signal OPB_RNW_reg:   std_logic;
	signal my_Select:     std_logic;
	signal my_Ack:        std_logic;
	signal my_DBus:       std_logic_vector(0 to 31);
begin
	-----------------------------------------------------------------
	-- address decoding and chip select
	-----------------------------------------------------------------
	
	address_decode: block
		component pselect -- address comparator from Common_v1_00_a
			generic (
				C_AB   : integer := 9;
				C_AW   : integer := 32;
				C_BAR  : std_logic_vector(0 to 31) := x"33000000");
			port (
				A      : in   std_logic_vector(0 to C_AW-1);
				AValid : in   std_logic;
				PS     : out  std_logic);
		end component;
		
		-- calculate number of bits need to decode range from C_BASEADDR to C_HIGHADDR
		-- example: C_BASEADDR = X"FFFF_8000", C_HIGHADDR = X"FFFF_80FF", result = 24
		function Addr_Bits (x, y : std_logic_vector(0 to C_OPB_AWIDTH-1)) return integer is
			variable addr_nor : std_logic_vector(0 to C_OPB_AWIDTH-1);
		begin
			addr_nor := x xor y;
			for i in 0 to C_OPB_AWIDTH-1 loop
				if addr_nor(i) = '1' then return i; end if;
			end loop;
			return(C_OPB_AWIDTH);
		end function;
		
		constant C_AB :     integer:= Addr_Bits(C_HIGHADDR, C_BASEADDR);
		signal   ii_CS_x:   std_logic;
		--		signal   ii_CS_1:   std_logic;
		signal   ii_CS_2:   std_logic;
		signal   ii_CS_1_2: std_logic;
		
		begin
		
		-- OPB address decoding using pselect.vhd
		pselect_I : pselect	generic map (
			C_AB  => C_AB,
			C_AW  => OPB_ABus'length,
			C_BAR => C_BASEADDR)
		port map (
			A      => OPB_ABus,
			AValid => OPB_select,
			ps     => ii_CS_x);
		
		-- register pselect output twice...
		-- (note the sync-reset with xfer_Ack)
		CS_1_FF : FDR port map (
			Q => ii_CS_1,
			C => OPB_Clk,
			D => ii_CS_X,
			R => ii_xfer_Ack);
		
		CS_2_FF: FDR port map (
			Q => ii_CS_2,
			C => OPB_Clk,
			D => ii_CS_1,
			R => ii_xfer_Ack);
		
		-- ... and create a single cycle pulse
		ii_CS_1_2<= ii_CS_1 and not ii_CS_2;
		
		-- use it either registered...
		--	CS_2_DFF: FDR port map (
		--		Q => my_CS,
		--		C => OPB_Clk,
		--		D => my_CS_2ii,
		--		R => xfer_Ack);
		
		-- ... or "as is" which is one cycle faster
		my_Select<= ii_CS_1_2;
	end block;
	
	-----------------------------------------------------------------
	-- register signals from OPB
	-----------------------------------------------------------------
	
	from_OPB: block
		begin
		
		WRDBUS_FF_GENERATE: for i in 0 to C_OPB_DWIDTH-1 generate
			WRDBUS_FF_I: FDR port map (
				Q => OPB_DBus_reg(i),
				C => OPB_Clk,
				D => OPB_DBus(i),
				R => OPB_Rst);
		end generate;
		
		ABUS_FF_GENERATE: for i in 0 to C_OPB_AWIDTH-1 generate
			ABUS_FF_I: FDR port map (
				Q => OPB_ABus_reg(i),
				C => OPB_Clk,
				D => OPB_ABus(i),
				R => OPB_Rst);
		end generate;
		
		RNW_FF_I: FDR port map (
			Q => OPB_RNW_reg,
			C => OPB_Clk,
			D => OPB_RNW,
			R => OPB_Rst);
		
	end block;
	
	-----------------------------------------------------------------
	-- register signals to OPB
	-----------------------------------------------------------------
	
	to_OPB: block
		signal read_reg_rst: std_logic;
		signal toutSup:      std_logic;
		signal toutSup_rst:  std_logic;
		begin
		
		read_reg_rst<= not(my_Ack)          -- reset, when not the final cycle
		or             not(ii_CS_1)         -- reset, when not our device
		or             not(OPB_RNW_reg);    -- reset, when not a read transaction
		
		RDBUS_FF_GENERATE : for i in 0 to C_OPB_DWIDTH-1 generate
			RDBUS_FF_I : FDRE port map (
				Q  => Mem_DBus(i),
				C  => OPB_Clk,
				D  => my_DBus(i),
				R  => read_reg_rst,
				CE => OPB_BE(i/8));
		end generate;
		
		XFER_ACK_I: FDR port map (
			Q => ii_xfer_Ack,
			C => OPB_Clk,
			D => my_ACK,
			R => OPB_Rst);
		
		MEM_xferAck <= 	ii_xfer_Ack;
		MEM_errAck  <= '0';
		MEM_retry   <= '0';
		
		--		MEM_toutSup <= '0';
		
		toutSup    <= '1';
		toutSup_rst<= not(ii_CS_1) -- reset, when not our device
		or            ii_xfer_Ack; -- reset, when acknowledge is output
		
		TOUTSUP_FF: FDR port map (
			Q  => MEM_toutSup,
			C  => OPB_Clk,
			D  => toutSup,
			R  => toutSup_rst);
		
		Interrupt   <= '0';
	end block;
	
	--------------------------------------------------------
	-- 16-bit memory controller
	--------------------------------------------------------
	
	Ucore: te_mem_core generic map (
		C_OPB_AWIDTH => C_OPB_AWIDTH,
		C_OPB_DWIDTH => C_OPB_DWIDTH,
		C_BASEADDR   => C_BASEADDR,
		C_HIGHADDR   => C_HIGHADDR)
	port map (
		-- Global signals
		OPB_Clk      => OPB_Clk,
		OPB_Rst      => OPB_Rst,
		-- OPB signals (mostly registered)
		OPB_ABus     => OPB_ABus_reg,
		OPB_BE       => OPB_BE,
		OPB_DBus     => OPB_DBus_reg,
		OPB_RNW      => OPB_RNW_reg,
		-- simplified interface to OPB
		my_Select    => my_Select,
		my_Ack       => my_Ack,
		my_DBus      => my_DBus,
		-- external memory bus
		mem_a        => mem_a,
		mem_d_i      => mem_d_i,
		mem_d_o      => mem_d_o,
		mem_d_t      => mem_d_t,
		mem_wen      => mem_wen,
		flash_cen    => flash_cen,
		flash_oen    => flash_oen,
		flash_rdy    => flash_rdy,
		flash_8n     => flash_8n,
		ram_csn      => ram_csn,
		ram_oen      => ram_oen,
		ram_blen     => ram_blen,
		ram_bhen     => ram_bhen,
		cpld_csn     => cpld_csn,
		cpld_rwn     => cpld_rwn);
end bhv;

--------------------------------------------------------------------------------
-- end of file

